#include <ft2build.h>
#include FT_FREETYPE_H
#define STB_IMAGE_WRITE_IMPLEMENTATION
#include "stb_image_write.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define FIRST_CHAR 0
#define LAST_CHAR 127
#define COLS 16
#define ROWS ((LAST_CHAR - FIRST_CHAR + COLS) / COLS)

void save_png(int width, int height, unsigned char *image, const char *filename) {
    stbi_write_png(filename, width, height, 4, image, width * 4);
    printf("Saved %s (%dx%d)\n", filename, width, height);
}

void save_tga(int width, int height, unsigned char *image, const char *filename) {
    FILE *file = fopen(filename, "wb");
    if (!file) {
        printf("Error: Failed to create TGA file.\n");
        return;
    }

    // TGA header
    unsigned char header[18] = {0};
    header[2] = 10; // RLE compressed true-color image
    header[12] = width & 0xFF;
    header[13] = (width >> 8) & 0xFF;
    header[14] = height & 0xFF;
    header[15] = (height >> 8) & 0xFF;
    header[16] = 32; // Bits per pixel
    header[17] = 0x00; // Bottom-left origin

    fwrite(header, sizeof(header), 1, file);

    // Write image data with RLE compression
    for (int y = height - 1; y >= 0; y--) {
        int row_start = y * width * 4;
        int x = 0;
        while (x < width) {
            int i = row_start + x * 4;

            // Check if we can use RLE (run of identical pixels)
            int run_length = 1;
            while (x + run_length < width && run_length < 128) {
                int j = row_start + (x + run_length) * 4;
                if (memcmp(&image[i], &image[j], 4) != 0)
                    break;
                run_length++;
            }

            if (run_length > 1) {
                unsigned char packet_header = 0x80 | (run_length - 1);
                fwrite(&packet_header, 1, 1, file);
                fwrite(&image[i], 4, 1, file);
                x += run_length;
            } else {
                int raw_start = x;
                run_length = 1;
                while (x + run_length < width && run_length < 128) {
                    int a = row_start + (x + run_length - 1) * 4;
                    int b = row_start + (x + run_length) * 4;
                    if (memcmp(&image[a], &image[b], 4) == 0)
                        break;
                    run_length++;
                }

                unsigned char packet_header = run_length - 1;
                fwrite(&packet_header, 1, 1, file);
                fwrite(&image[row_start + raw_start * 4], run_length * 4, 1, file);
                x += run_length;
            }
        }
    }

    fclose(file);
    printf("Saved %s (RLE compressed, bottom-left origin, %dx%d)\n", filename, width, height);
}

// Function for input handling (handles empty input cases)
void get_input(const char *prompt, char *buffer, size_t size, const char *default_value) {
    printf("%s", prompt);
    fgets(buffer, size, stdin);

    // Remove newline character if present
    buffer[strcspn(buffer, "\n")] = 0;

    // If input is empty, use default value
    if (strlen(buffer) == 0) {
        strcpy(buffer, default_value);
    }

    // **Ensure input buffer is cleared** before the next prompt
    fflush(stdin);
}

int main() {
    char fontFile1[256], fontFile2[256], outputFile[256], outputFormat[4];
	int scale_factor, letter_size, r_shadow, g_shadow, b_shadow;

    // Get font filename
    get_input("Enter first font file (default: FreeMono.ttf): ", fontFile1, sizeof(fontFile1), "FreeMono.ttf");
    get_input("Enter second font file (default: FreeMono.ttf): ", fontFile2, sizeof(fontFile2), "FreeMono.ttf");

    FT_Library ft;
    if (FT_Init_FreeType(&ft)) {
        printf("Error: Could not initialize FreeType.\n");
        return 1;
    }

    FT_Face face1, face2;
    if (FT_New_Face(ft, fontFile1, 0, &face1)) {
        printf("Error loading font: %s\n", fontFile1);
        return 1;
    }
    if (FT_New_Face(ft, fontFile2, 0, &face2)) {
        printf("Error loading font: %s\n", fontFile2);
        return 1;
    }

    // Get filename
    get_input("Enter output filename (default: conchars): ", outputFile, sizeof(outputFile), "conchars");

    // Get output format
    get_input("Select output format (default: png) [png/tga]: ", outputFormat, sizeof(outputFormat), "png");

    // Validate format input
    if (!(strcmp(outputFormat, "png") == 0 || strcmp(outputFormat, "tga") == 0)) {
        printf("Error: Invalid format. Please enter 'png' or 'tga'.\n");
        return 1;
    }

    // Explain scale factor options
    printf("\nScale Factor Guide:\n");
    printf("  1 = 128x128 resolution\n");
    printf("  2 = 256x256 resolution\n");
    printf("  3 = 384x384 resolution\n");
    printf("  4 = 512x512 resolution\n");
    printf("  5 = 640x640 resolution\n");
    printf("  6 = 768x768 resolution\n");
    printf("  7 = 896x896 resolution\n");
    printf("  8 = 1024x1024 resolution\n");

    char scaleInput[10];
    get_input("Enter scale factor (default: 1, max: 8): ", scaleInput, sizeof(scaleInput), "1");
    scale_factor = atoi(scaleInput);
    if (scale_factor < 1) scale_factor = 1;
    if (scale_factor > 8) scale_factor = 8; // Ensure max is 8

    // Get letter size while keeping the fixed box (8x8)
    char letterSizeInput[10];
    get_input("Enter letter size (default: 8, min: 4, max: 8): ", letterSizeInput, sizeof(letterSizeInput), "8");
    letter_size = atoi(letterSizeInput);
    if (letter_size < 4) letter_size = 4; // Minimum size
    if (letter_size > 8) letter_size = 8; // Maximum size

    // Get color for Font 1
    char colorFont1[20];
    get_input("Enter text color RGB for Font 1 (default: 0 0 0, Black): ", colorFont1, sizeof(colorFont1), "0 0 0");
    int r1, g1, b1;
    sscanf(colorFont1, "%d %d %d", &r1, &g1, &b1);

    // Get color for Font 2
    char colorFont2[20];
    get_input("Enter text color RGB for Font 2 (default: 255 255 255, White): ", colorFont2, sizeof(colorFont2), "255 255 255");
    int r2, g2, b2;
    sscanf(colorFont2, "%d %d %d", &r2, &g2, &b2);


    printf("\nShadow Options:\n");
    printf("  0 = No shadow\n");
    printf("  1 = Simple shadow (offset +1, +1)\n");
    printf("  2 = Full shadow (outline around glyph), Kingpin use this one for default (best to use high scale).\n");

    char shadowInput[10];
    get_input("Select shadow option (default: 0): ", shadowInput, sizeof(shadowInput), "0");
    int shadow_mode = atoi(shadowInput);
    if (shadow_mode < 0 || shadow_mode > 2) shadow_mode = 0; // Ensure valid input

    if (shadow_mode)
    {
        // Get Shadow text color
        char ShadowcolorInput[20];
        get_input("Enter text shadow color RGB (default: 0 0 0): ", ShadowcolorInput, sizeof(ShadowcolorInput), "0 0 0");
        sscanf(ShadowcolorInput, "%d %d %d", &r_shadow, &g_shadow, &b_shadow);
    }

    // Create the full filename with the chosen format
    char finalOutputFile[260]; // Ensure enough space for filename + extension
    snprintf(finalOutputFile, sizeof(finalOutputFile), "%s.%s", outputFile, outputFormat);

    // Image dimensions based on scale factor
    int TILE_W = 8 * scale_factor;
    int TILE_H = 8 * scale_factor;
    int IMAGE_WIDTH = 128 * scale_factor;
    int IMAGE_HEIGHT = 128 * scale_factor;

    unsigned char *image = malloc(IMAGE_WIDTH * IMAGE_HEIGHT * 4);
    memset(image, 0, IMAGE_WIDTH * IMAGE_HEIGHT * 4);

    FT_Set_Pixel_Sizes(face1, 0, letter_size * scale_factor); // Scale letter while keeping the box fixed
    FT_Set_Pixel_Sizes(face2, 0, letter_size * scale_factor); // Scale letter while keeping the box fixed

    for (int font_index = 0; font_index < 2; font_index++) {
        FT_Face current_face = (font_index == 0) ? face1 : face2;
        int font_offset = font_index * (IMAGE_HEIGHT / 2);

	    for (int ch = FIRST_CHAR; ch <= LAST_CHAR; ch++) {
	        int actual_char = ch;

            if (ch == 16) actual_char = 91; //FREDZ [ Left bracket
            if (ch == 17) actual_char = 93; //FREDZ ] Right bracket

	        if (font_index == 1)//FREDZ both fonts?
            {
                if (ch >= 18 && ch <= 29) actual_char = '0' + (ch - 18);//FREDZ 0 to ;
                if (ch == 30) actual_char = 32;//FREDZ space
                if (ch == 31) actual_char = 32;//FREDZ space
            }

	        if (ch == 126) actual_char = 34; //FREDZ Kingpin Replace '~' with '"'
	        if (ch == 127) actual_char = 60; //FREDZ Kingpin Replace '' with '<'

	        if (FT_Load_Char(current_face, actual_char, FT_LOAD_RENDER | FT_LOAD_TARGET_LIGHT))
				continue;

	        FT_Bitmap *bmp = &current_face->glyph->bitmap;

	        int x = (ch - FIRST_CHAR) % COLS;
	        int y = (ch - FIRST_CHAR) / COLS;
	        int tile_x = x * TILE_W;
	        int tile_y = y * TILE_H + font_offset;

	        int x_offset = current_face->glyph->bitmap_left + (scale_factor * 2);
	        int y_offset = TILE_H - current_face->glyph->bitmap_top - (scale_factor * 2);

	        for (int row = 0; row < bmp->rows; row++) {
	            for (int col = 0; col < bmp->width; col++) {
	                int px = tile_x + x_offset + col;
	                int py = tile_y + y_offset + row;
	                if (px >= 0 && px < IMAGE_WIDTH && py >= 0 && py < IMAGE_HEIGHT)
	                {
	                    unsigned char alpha = bmp->buffer[row * bmp->pitch + col];

	                    if (shadow_mode == 1) {
	                        // Simple shadow (offset slightly)
	                        int shadow_x = px + 1;
	                        int shadow_y = py + 1;
	                        if (shadow_x >= 0 && shadow_x < IMAGE_WIDTH && shadow_y >= 0 && shadow_y < IMAGE_HEIGHT) {
	                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 0] = b_shadow;
	                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 1] = g_shadow;
	                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 2] = r_shadow;
	                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 3] = alpha * 0.8; // Soft shadow
	                        }
	                    }
	                    else if (shadow_mode == 2) {
	                        for (int dy = -2; dy <= 2; dy++) {
	                            for (int dx = -2; dx <= 2; dx++) {
	                                if (dx == 0 && dy == 0) continue;
	                                int shadow_x = px + dx;
	                                int shadow_y = py + dy;
	                                if (shadow_x >= 0 && shadow_x < IMAGE_WIDTH && shadow_y >= 0 && shadow_y < IMAGE_HEIGHT) {
	                                    float distance = dx * dx + dy * dy;
	                                    float strength = 1.0f - (distance / 8.0f); // distance 1: strong, 8: weak
	                                    if (strength > 0) {
	                                        unsigned char existing_alpha = image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 3];
	                                        if (existing_alpha < alpha * strength) {
	                                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 0] = r_shadow;
	                                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 1] = g_shadow;
	                                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 2] = b_shadow;
	                                            image[(shadow_y * IMAGE_WIDTH + shadow_x) * 4 + 3] = alpha * strength;
	                                        }
	                                    }
	                                }
	                            }
	                        }
	                    }

	                    image[(py * IMAGE_WIDTH + px) * 4 + 0] = (current_face == face1) ? b1 : b2; // Blue
                        image[(py * IMAGE_WIDTH + px) * 4 + 1] = (current_face == face1) ? g1 : g2; // Green
                        image[(py * IMAGE_WIDTH + px) * 4 + 2] = (current_face == face1) ? r1 : r2; // Red
                        image[(py * IMAGE_WIDTH + px) * 4 + 3] = alpha; // Alpha (opacity)
	                }
	            }
	        }
		}
    }

    // Save based on output format
    if (strcmp(outputFormat, "png") == 0) {
        save_png(IMAGE_WIDTH, IMAGE_HEIGHT, image, finalOutputFile);
    } else {
        save_tga(IMAGE_WIDTH, IMAGE_HEIGHT, image, finalOutputFile);
    }

    free(image);
    FT_Done_Face(face1);
    FT_Done_Face(face2);
    FT_Done_FreeType(ft);
    return 0;
}
