// bsp5.c

#include "bsp5.h"

void *Qmalloc(int Size, char Caller[]);
void Qfree(void *Ptr, int Size);

//
// command line flags
//
options_t options;

brushset_t	*brushset;

char		portfilename[1024];

//===========================================================================

/*
=======
Message
=======
*/
void Message (int MsgType, ...)
{
	va_list argptr;
	char	*Fmt;

	va_start (argptr, MsgType);

	if (MsgType == MSGWARN)
		return;

	Fmt = va_arg(argptr, char *);

	if (MsgType == MSGERR)
		Error (Fmt, argptr);

	vprintf (Fmt, argptr);
	printf ("\n");

	va_end (argptr);
}

/*
===============
GetCoord

Returns coordinates as a string
===============
*/
char *GetCoord(vec3_t v)
{
	static char Str[50];

	sprintf (Str, "(%.0f %.0f %.0f)", v[0], v[1], v[2]);

	return Str;
}

/*
=================
BaseWindingForPlane
=================
*/
winding_t *BaseWindingForPlane (plane_t *p)
{
	int		i, x;
	vec_t	max, v;
	vec3_t	org, vright, vup;
	winding_t	*w;

// find the major axis

	max = -BOGUS_RANGE;
	x = -1;
	for (i=0 ; i<3; i++)
	{
		v = fabs(p->normal[i]);
		if (v > max)
		{
			x = i;
			max = v;
		}
	}
	if (x==-1)
		Message (MSGERR, "BaseWindingForPlane: No axis found");

	VectorCopy (vec3_origin, vup);
	switch (x)
	{
	case 0:
	case 1:
		vup[2] = 1;
		break;
	case 2:
		vup[0] = 1;
		break;
	}

	v = DotProduct (vup, p->normal);
	VectorMA (vup, -v, p->normal, vup);
	VectorNormalize (vup);

	VectorScale (p->normal, p->dist, org);

	CrossProduct (vup, p->normal, vright);

	VectorScale (vup, 8192, vup);
	VectorScale (vright, 8192, vright);

// project a really big	axis aligned box onto the plane
	w = NewWinding (4);

	VectorSubtract (org, vright, w->points[0]);
	VectorAdd (w->points[0], vup, w->points[0]);

	VectorAdd (org, vright, w->points[1]);
	VectorAdd (w->points[1], vup, w->points[1]);

	VectorAdd (org, vright, w->points[2]);
	VectorSubtract (w->points[2], vup, w->points[2]);

	VectorSubtract (org, vright, w->points[3]);
	VectorSubtract (w->points[3], vup, w->points[3]);

	return w;
}


/*
==================
ClipWinding

Clips the winding to the plane, returning the new winding on the positive side
Frees the input winding.
If keepon is true, an exactly on-plane winding will be saved, otherwise
it will be clipped away.
==================
*/
winding_t *ClipWinding (winding_t *in, plane_t *split, qboolean keepon)
{
	vec_t	  dists[MAX_POINTS_ON_WINDING];
	int	  sides[MAX_POINTS_ON_WINDING];
	int	  counts[3];
	vec_t	  dot;
	int	  i, j;
	vec_t	  *p1, *p2;
	vec3_t	  mid;
	winding_t *neww;
	int	  maxpts;

	counts[0] = counts[1] = counts[2] = 0;

// determine sides for each point
	for (i=0 ; i<in->numpoints ; i++)
	{
		dot = DotProduct (in->points[i], split->normal);
		dot -= split->dist;
		dists[i] = dot;
		if (dot > ON_EPSILON)
			sides[i] = SIDE_FRONT;
		else if (dot < -ON_EPSILON)
			sides[i] = SIDE_BACK;
		else
		{
			sides[i] = SIDE_ON;
		}
		counts[sides[i]]++;
	}
	sides[i] = sides[0];
	dists[i] = dists[0];

	if (keepon && !counts[0] && !counts[1])
		return in;

	if (!counts[0])
	{
		FreeWinding (in);
		return NULL;
	}
	if (!counts[1])
		return in;

	maxpts = in->numpoints+4;	// can't use counts[0]+2 because
								// of fp grouping errors
	neww = NewWinding (maxpts);
	neww->numpoints = 0;

	for (i=0 ; i<in->numpoints ; i++)
	{
		p1 = in->points[i];

		if (sides[i] == SIDE_ON)
		{
			VectorCopy (p1, neww->points[neww->numpoints]);
			neww->numpoints++;
			continue;
		}

		if (sides[i] == SIDE_FRONT)
		{
			VectorCopy (p1, neww->points[neww->numpoints]);
			neww->numpoints++;
		}

		if (sides[i+1] == SIDE_ON || sides[i+1] == sides[i])
			continue;

	// generate a split point
		p2 = in->points[(i+1)%in->numpoints];

		dot = dists[i] / (dists[i]-dists[i+1]);
		for (j=0 ; j<3 ; j++)
		{	// avoid round off error when possible
			if (split->normal[j] == 1)
				mid[j] = split->dist;
			else if (split->normal[j] == -1)
				mid[j] = -split->dist;
			else
				mid[j] = p1[j] + dot*(p2[j]-p1[j]);
		}

		VectorCopy (mid, neww->points[neww->numpoints]);
		neww->numpoints++;
	}

	if (neww->numpoints > maxpts)
		Message (MSGERR, "ClipWinding: Points exceeded estimate");

// free the original winding
	FreeWinding (in);

	// Shrink the winding back to just what it needs ...
	neww = WindingResize (neww, neww->numpoints);

	return neww;
}


/*
==================
DivideWinding

Divides a winding by a plane, producing one or two windings.  The
original winding is not damaged or freed.  If only on one side, the
returned winding will be the input winding.  If on both sides, two
new windings will be created.
==================
*/
void DivideWinding (winding_t *in, plane_t *split, winding_t **front, winding_t **back)
{
	vec_t	  dists[MAX_POINTS_ON_WINDING];
	int	  sides[MAX_POINTS_ON_WINDING];
	int	  counts[3];
	vec_t	  dot;
	int	  i, j;
	vec_t	  *p1, *p2;
	vec3_t	  mid;
	winding_t *f, *b;
	int	  maxpts;

	counts[0] = counts[1] = counts[2] = 0;

// determine sides for each point
	for (i=0 ; i<in->numpoints ; i++)
	{
		dot = DotProduct (in->points[i], split->normal);
		dot -= split->dist;
		dists[i] = dot;
		if (dot > ON_EPSILON)
			sides[i] = SIDE_FRONT;
		else if (dot < -ON_EPSILON)
			sides[i] = SIDE_BACK;
		else
		{
			sides[i] = SIDE_ON;
		}
		counts[sides[i]]++;
	}
	sides[i] = sides[0];
	dists[i] = dists[0];

	*front = *back = NULL;

	if (!counts[0])
	{
		*back = in;
		return;
	}
	if (!counts[1])
	{
		*front = in;
		return;
	}

	maxpts = in->numpoints+4;	// can't use counts[0]+2 because
								// of fp grouping errors

	*front = f = NewWinding (maxpts);
	*back = b = NewWinding (maxpts);
	f->numpoints = b->numpoints = 0;

	for (i=0 ; i<in->numpoints ; i++)
	{
		p1 = in->points[i];

		if (sides[i] == SIDE_ON)
		{
			VectorCopy (p1, f->points[f->numpoints]);
			f->numpoints++;
			VectorCopy (p1, b->points[b->numpoints]);
			b->numpoints++;
			continue;
		}

		if (sides[i] == SIDE_FRONT)
		{
			VectorCopy (p1, f->points[f->numpoints]);
			f->numpoints++;
		}
		if (sides[i] == SIDE_BACK)
		{
			VectorCopy (p1, b->points[b->numpoints]);
			b->numpoints++;
		}

		if (sides[i+1] == SIDE_ON || sides[i+1] == sides[i])
			continue;

	// generate a split point
		p2 = in->points[(i+1)%in->numpoints];

		dot = dists[i] / (dists[i]-dists[i+1]);
		for (j=0 ; j<3 ; j++)
		{	// avoid round off error when possible
			if (split->normal[j] == 1)
				mid[j] = split->dist;
			else if (split->normal[j] == -1)
				mid[j] = -split->dist;
			else
				mid[j] = p1[j] + dot*(p2[j]-p1[j]);
		}

		VectorCopy (mid, f->points[f->numpoints]);
		f->numpoints++;
		VectorCopy (mid, b->points[b->numpoints]);
		b->numpoints++;
	}

	if (f->numpoints > maxpts || b->numpoints > maxpts)
		Message (MSGERR, "DivideWinding: Points exceeded estimate");

	// Shrink the windings back to just what they need ...
	*front = WindingResize (f, f->numpoints);
	*back = WindingResize (b, b->numpoints);
}

//===========================================================================

int			c_activefaces, c_peakfaces;
int			c_activesurfaces, c_peaksurfaces;
int			c_activewindings, c_peakwindings;
int			c_activeportals, c_peakportals;
int			c_activenodes, c_peaknodes;
int			c_activebrushes, c_peakbrushes;
int			c_activeother, c_peakother;
int			c_activetotalbytes, c_peaktotalbytes;

void AddBytes(int Size)
{
	c_activetotalbytes += Size;

	if (c_activetotalbytes > c_peaktotalbytes)
		c_peaktotalbytes = c_activetotalbytes;
}

void SubBytes(int Size)
{
	c_activetotalbytes -= Size;
}

void *Qmalloc(int Size, char Caller[])
{
	void *Ptr;

	Ptr = malloc(Size);

	if (Ptr == NULL)
		Message (MSGERR, "Memory allocation of %d bytes failed in %s", Size, Caller);

	AddBytes(Size);

	return(Ptr);
}

void Qfree(void *Ptr, int Size)
{
	SubBytes(Size);

	free(Ptr);
}

/*
==================
NewWinding
==================
*/
winding_t *NewWinding (int points)
{
	winding_t *w;
	int	  size;

	if (points > MAX_POINTS_ON_WINDING)
		Message (MSGERR, "NewWinding: %i points", points);

	c_activewindings++;
	if (c_activewindings > c_peakwindings)
		c_peakwindings = c_activewindings;

	size = (int)((winding_t *)0)->points[points];
	size += sizeof(int);

	w = Qmalloc (size, "NewWinding");
	memset (w, 0, size);

	// Save size just before the user area
	*(int *)w = size;
	w = (winding_t *)((char *)w + sizeof(int));

	w->numpoints = points;

	return w;
}

void FreeWinding (winding_t *w)
{
	c_activewindings--;

	// Size is saved just before the user area
	w = (winding_t *)((char *)w - sizeof(int));

	Qfree (w, *(int *)w);
}

/*
===========
AllocPortal
===========
*/
portal_t *AllocPortal (void)
{
	portal_t	*p;

	c_activeportals++;
	if (c_activeportals > c_peakportals)
		c_peakportals = c_activeportals;

	p = Qmalloc (sizeof(portal_t), "AllocPortal");
	memset (p, 0, sizeof(portal_t));

	return p;
}

void FreePortal (portal_t *p)
{
	c_activeportals--;
	Qfree (p, sizeof(portal_t));
}

/*
==================
BuildName
==================
*/
void BuildName(char Name[], char Orig[], char Ext[])
{
	strcpy (Name, Orig);
	StripExtension (Name);
	strcat (Name, Ext);
}

