/*
map.c file of the MergeT3D Source Code

Copyright 1997, 1998 Alexander Malmberg
Copyright Gyro Gearloose
Copyright 1996, Trey Harrison and Chris Carollo

This program is distributed under the GNU General Public License.
See legal.txt for more information.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "types.h"

#include "map.h"

#include "clip.h"
#include "entity.h"
#include "error.h"
#include "geom.h"
#include "globals.h"
#include "memory.h"
#include "misc.h"
#include "token.h"


/* Map loading stuff. */

#define ERROR_NO    0   // no error occured, assumed to be 0
#define ERROR_NOMEM 1   // out of memory
#define ERROR_PARSE 2   // error parsing file

static char *errors[]=
{
"No error!",
"Out of memory!",
"Error parsing file!"
};

#define GETTOKEN(x,y) if (!GetToken(x,y)) return ERROR_PARSE;

#define EXPECT(x,y,z) \
   { GETTOKEN(x,y) if (strcmp(token,z)) return ERROR_PARSE; }


static char buf[16384];


static int LoadTex(texdef_t *tex)
{
   int i;

   strcpy(tex->name,token);

   for (i=0;i<2;i++)
   {
      GETTOKEN(0,T_NUMBER);
      tex->shift[i]=atof(token);
   }

   GETTOKEN(0,T_NUMBER);
   tex->rotate=atof(token);

   for (i=0;i<2;i++)
   {
      GETTOKEN(0,T_NUMBER);
      tex->scale[i]=atof(token);
   }

   if (TokenAvailable(0))
   {
      GETTOKEN(0,T_NUMBER);
      tex->contents=atoi(token);

      GETTOKEN(0,T_NUMBER);
      tex->flags=atoi(token);

      GETTOKEN(0,T_NUMBER);
      tex->value=atoi(token);
   }
   else
   {
      tex->contents=0;
      tex->flags=0;
      tex->value=0;
   }

   return ERROR_NO;
}

static int LoadBrush(entity_t *ent,brush_t **res)
{
   int    i;
   vec3_t vec1,vec2,norm;

   brush_my_t B;
   face_my_t *f;

   brush_t *b;

   memset(&B,0,sizeof(brush_my_t));

   b=Q_malloc(sizeof(brush_t));
   if (!b)
      return ERROR_NOMEM;
   memset(b,0,sizeof(brush_t));

   *res=b;

   GETTOKEN(1,-1);
   if (!strcmp(token,"("))
   {
      while (strcmp(token,"}"))
      {
         B.faces = Q_realloc(B.faces,sizeof(face_my_t) * (B.numfaces+1));
         if (!B.faces)
            return ERROR_NOMEM;

         f=&B.faces[B.numfaces];

         for (i=0;i<3;i++)
         {
            if (strcmp(token,"("))
               return ERROR_PARSE;
   
            GETTOKEN(0,T_NUMBER);
            f->planepts[i].x = atof(token);

            GETTOKEN(0,T_NUMBER);
            f->planepts[i].y = atof(token);

            GETTOKEN(0,T_NUMBER);
            f->planepts[i].z = atof(token);

            EXPECT(0,T_MISC,")");

            if (i==2)
            {
               GETTOKEN(0,T_ALLNAME);
            }
            else
            {
               GETTOKEN(0,-1);
            }
         }

         i=LoadTex(&f->tex);
         if (i)
            return i;

         /* Calc normal */
   
         vec1.x = f->planepts[1].x - f->planepts[0].x;
         vec1.y = f->planepts[1].y - f->planepts[0].y;
         vec1.z = f->planepts[1].z - f->planepts[0].z;
   
         vec2.x = f->planepts[1].x - f->planepts[2].x;
         vec2.y = f->planepts[1].y - f->planepts[2].y;
         vec2.z = f->planepts[1].z - f->planepts[2].z;
   
         CrossProd(vec1, vec2, &norm);
         Normalize(&norm);
   
         /* Convert to normal / dist plane */
         f->normal.x = norm.x;
         f->normal.y = norm.y;
         f->normal.z = norm.z;
         
         f->dist = DotProd(f->normal,f->planepts[0]);
   
         B.numfaces++;

         GETTOKEN(1,-1);
      }
   
      if (!BuildBrush(&B,b))
      {
         return 0;
      }
      CalcBrushCenter(b);

      b->EntityRef=ent;

      return ERROR_NO;
   }
   else  // load a new style brush
   {
      int i,j,k;
      plane_t *p;

      if (strcmp(token,":"))
         return ERROR_PARSE;

      GETTOKEN(0,T_NUMBER);

      b->num_verts=atoi(token);

      b->verts=Q_malloc(sizeof(vec3_t)*b->num_verts);
      if (!b->verts)
         return ERROR_NOMEM;

      b->tverts=Q_malloc(sizeof(vec3_t)*b->num_verts);
      if (!b->tverts)
         return ERROR_NOMEM;

      for (i=0;i<b->num_verts;i++)
      {
         GETTOKEN(1,T_NUMBER);
         b->verts[i].x=atof(token);

         GETTOKEN(0,T_NUMBER);
         b->verts[i].y=atof(token);

         GETTOKEN(0,T_NUMBER);
         b->verts[i].z=atof(token);
      }

      b->num_planes=0;
      b->plane=NULL;

      GETTOKEN(1,-1);

      while (strcmp(token,"}"))
      {
         b->plane=Q_realloc(b->plane,sizeof(plane_t)*(b->num_planes+1));
         if (!b->plane)
            return ERROR_NOMEM;

         p=&b->plane[b->num_planes++];
         memset(p,0,sizeof(plane_t));

         p->num_verts=atoi(token);

         p->verts=Q_malloc(sizeof(int)*p->num_verts);
         if (!p->verts)
            return ERROR_NOMEM;

         EXPECT(0,T_MISC,"(");

         for (i=0;i<p->num_verts;i++)
         {
            GETTOKEN(0,T_NUMBER);
            p->verts[i]=atoi(token);
         }

         EXPECT(0,T_MISC,")");

         GETTOKEN(0,T_ALLNAME);
         LoadTex(&p->tex);

         GETTOKEN(1,-1);
      }

      b->num_edges=0;
      b->edges=NULL;

      for (i=0;i<b->num_planes;i++)
      {
         int v1,v2;

         p=&b->plane[i];

         for (j=0;j<p->num_verts;j++)
         {
            v1=p->verts[j];
            if (j==p->num_verts-1)
               v2=p->verts[0];
            else
               v2=p->verts[j+1];

            if (v2<v1)
            {
               k=v2;
               v2=v1;
               v1=k;
            }

            for (k=0;k<b->num_edges;k++)
            {
               if ((b->edges[k].startvertex==v1) &&
                   (b->edges[k].endvertex==v2))
                  break;
            }
            if (k==b->num_edges)
            {
               b->edges=Q_realloc(b->edges,sizeof(edge_t)*(b->num_edges+1));
               if (!b->edges)
                  return ERROR_NOMEM;
               b->edges[b->num_edges].startvertex=v1;
               b->edges[b->num_edges].endvertex=v2;
               b->num_edges++;
            }
         }
      }

      RecalcNormals(b);
      CalcBrushCenter(b);

      b->EntityRef=ent;
   
      return ERROR_NO;
   }
}

static int LoadEntity(entity_t **res,brush_t **bfirst,brush_t **blast,int *num)
{
   entity_t *e;
   int i;
   brush_t *first,*last;
   brush_t *b;

   if (strcmp(token,"{"))
      return ERROR_PARSE;

   GETTOKEN(1,-1);

   e=Q_malloc(sizeof(entity_t));
   if (!e)
      return ERROR_NOMEM;
   memset(e,0,sizeof(entity_t));
   InitEntity(e);

   *res=e;

   while (strcmp(token,"}") && strcmp(token,"{"))
   {
      strcpy(buf,&token[1]);
      buf[strlen(buf)-1]=0;

      GETTOKEN(0,T_STRING);

      strcpy(token,&token[1]);
      token[strlen(token)-1]=0;

      SetKeyValue(e,buf,token);

      GETTOKEN(1,-1);
   }

   first=last=NULL;
   *num=0;

   while (!strcmp(token,"{"))
   {
      i=LoadBrush(e,&b);
      if (i)
         return i;

      if (!last)
         last=b;

      if (first)
         first->Last=b;
      b->Next=first;
      b->Last=NULL;
      first=b;

      (*num)++;

      GETTOKEN(1,-1);
   }

   if (strcmp(token,"}"))
      return ERROR_PARSE;

   *bfirst=first;
   *blast=last;

   return ERROR_NO;
}

int LoadMap(char *filename)
{
   int i;
   entity_t *e;
   brush_t *b1,*b2;
   int n_brush;

   if (!LoadFile(filename,T_C|T_NUMBER|T_ALLNAME|T_STRING|T_MISC,NULL))
   {
      HandleError("LoadMap","Unable to load '%s'!",filename);
      exit(1);
   }

   num_entities = 0;
   num_brushes  = 0;
   BrushHead    = NULL;
   EntityHead   = NULL;
   WorldSpawn   = NULL;

   while (GetToken(1,-1))
   {
      i=LoadEntity(&e,&b1,&b2,&n_brush);
      if (i)
      {
         HandleError("LoadMap","'%s', line %i: %s",
            filename,linenum,errors[i]);
         return 0;
      }

      if (b1)
      {
         if (BrushHead) BrushHead->Last=b2;
         b2->Next=BrushHead;
         BrushHead=b1;
         num_brushes+=n_brush;
      }

      if (EntityHead) EntityHead->Last=e;
      e->Next = EntityHead;
      e->Last = NULL;
      EntityHead = e;
      num_entities++;

      if (!WorldSpawn)  // assume the first entity is the worldspawn entity
         WorldSpawn = e;
   }

   DoneFile();

   RecalcAllNormals();

   return 1;
}

