/*
t3d.c file of the MergeT3D Source Code

Copyright 1997, 1998 Alexander Malmberg
Copyright Gyro Gearloose
Copyright 1996, Trey Harrison and Chris Carollo

This program is distributed under the GNU General Public License.
See legal.txt for more information.
*/

// Quest is a very powerful and easy-to-use interface for constructing
// 3D geometry, and this is a hack to permit it to be used to create
// geometry data for UnrealEd.
//
// Quest does not, however, do texture alignment, and does not read .T3D
// files produced by UnrealEd.  To prevent texture alignment work done in
// UnrealEd from being lost, however, it is capable of scavenging alignment
// data from a .T3D file and applying it to identical unaligned brushes of
// its .MAP files.
//
// Designing Unreal levels using Quest is very different from that of Quake
// levels.  When a brush is added to the world, it is by default a subtractive
// brush.  That is, the player can be inside it.  Changing the brush to any
// kind of class causes it to become an additive brush.  So, creating a func_wall
// out of a brush, for example, will cause it to be an additive brush to UnrealEd.
//
// Player Starts and Lights are the only entities currently supported.
// I recommend creating func_wall scripts to specify an additive brush.
// I pass any spawnflags set on to the brush's PolyFlags field.
// Setting spawnflags 32 will set the func_wall to a semi-solid brush.
// Setting spawnflags 8 will set the func_wall to a non-solid brush.
// Setting spawnflags 256 will cause textures to be double-sided (use w/ 8 for water)
//
// Texture names may be specified, although all alignment and scaling will need to
// be performed in UnrealEd.
//
// The worlds of Quake and Unreal seem to reverse the sign of the y axis,
// so I must invert the sign of all Y coordinates when writing.  By leaving the
// sign alone and writing the polygon's vertices in the reverse order, a
// mirror image of the level can be had.
//
// Setting Flags=16 on an individual texture is "Environment Map" (sky)
// Setting Flags=8192 on an individual texture is "Small Wavy" (water)
// I am deriving some of the texture flags from the texture names, sky and
// water.  This is a kludge, and the right way to do it would be to add a
// bitfield to the faces and allow the edface dialog to modify it.

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>

#include "types.h"

#include "t3d.h"

#include "entity.h"
#include "error.h"
#include "globals.h"
#include "memory.h"


#define NONSOLID 8
#define SEMISOLID 32
#define PORTAL 4

//------------------------------
// Create the default red wire brush
//------------------------------
void SaveRootHullT3D(FILE * fp)
{
	fprintf(fp, "   Begin Brush Name=Brush\n");
	fprintf(fp, "      Settings  CSG=0 Flags=64 PolyFlags=0 Color=0\n");
	fprintf(fp, "      Location  +00000.000000,+00000.000000,+00000.000000\n");
	fprintf(fp, "      PrePivot  +00000.000000,+00000.000000,+00000.000000\n");
	fprintf(fp, "      PostPivot +00000.000000,+00000.000000,+00000.000000\n");
	fprintf(fp, "      Scale     X=+00001.000000 Y=+00001.000000 Z=+00001.000000 S=+00000.000000 AXIS=5\n");
	fprintf(fp, "      PostScale X=+00001.000000 Y=+00001.000000 Z=+00001.000000 S=+00000.000000 AXIS=5\n");
	fprintf(fp, "      Rotation  0,0,0\n");
	fprintf(fp, "      Begin PolyList Num=6 Max=20000\n");
	fprintf(fp, "         Begin Polygon Item=OUTSIDE Link=0\n");
	fprintf(fp, "            Origin   -00128.000000,-00128.000000,+00128.000000\n");
	fprintf(fp, "            Normal   +00000.000000,+00000.000000,+00001.000000\n");
	fprintf(fp, "            TextureU +00000.000000,+00001.000000,+00000.000000\n");
	fprintf(fp, "            TextureV -00001.000000,+00000.000000,+00000.000000\n");
	fprintf(fp, "            Vertex   -00128.000000,-00128.000000,+00128.000000\n");
	fprintf(fp, "            Vertex   +00128.000000,-00128.000000,+00128.000000\n");
	fprintf(fp, "            Vertex   +00128.000000,+00128.000000,+00128.000000\n");
	fprintf(fp, "            Vertex   -00128.000000,+00128.000000,+00128.000000\n");
	fprintf(fp, "         End Polygon\n");
	fprintf(fp, "         Begin Polygon Item=OUTSIDE Link=1\n");
	fprintf(fp, "            Origin   -00128.000000,+00128.000000,-00128.000000\n");
	fprintf(fp, "            Normal   +00000.000000,+00000.000000,-00001.000000\n");
	fprintf(fp, "            TextureU +00000.000000,-00001.000000,+00000.000000\n");
	fprintf(fp, "            TextureV -00001.000000,+00000.000000,+00000.000000\n");
	fprintf(fp, "            Vertex   -00128.000000,+00128.000000,-00128.000000\n");
	fprintf(fp, "            Vertex   +00128.000000,+00128.000000,-00128.000000\n");
	fprintf(fp, "            Vertex   +00128.000000,-00128.000000,-00128.000000\n");
	fprintf(fp, "            Vertex   -00128.000000,-00128.000000,-00128.000000\n");
	fprintf(fp, "         End Polygon\n");
	fprintf(fp, "         Begin Polygon Item=OUTSIDE Link=2\n");
	fprintf(fp, "            Origin   -00128.000000,+00128.000000,-00128.000000\n");
	fprintf(fp, "            Normal   +00000.000000,+00001.000000,+00000.000000\n");
	fprintf(fp, "            TextureU +00001.000000,+00000.000000,+00000.000000\n");
	fprintf(fp, "            TextureV +00000.000000,+00000.000000,-00001.000000\n");
	fprintf(fp, "            Vertex   -00128.000000,+00128.000000,-00128.000000\n");
	fprintf(fp, "            Vertex   -00128.000000,+00128.000000,+00128.000000\n");
	fprintf(fp, "            Vertex   +00128.000000,+00128.000000,+00128.000000\n");
	fprintf(fp, "            Vertex   +00128.000000,+00128.000000,-00128.000000\n");
	fprintf(fp, "         End Polygon\n");
	fprintf(fp, "         Begin Polygon Item=OUTSIDE Link=3\n");
	fprintf(fp, "            Origin   +00128.000000,-00128.000000,-00128.000000\n");
	fprintf(fp, "            Normal   +00000.000000,-00001.000000,+00000.000000\n");
	fprintf(fp, "            TextureU -00001.000000,+00000.000000,+00000.000000\n");
	fprintf(fp, "            TextureV +00000.000000,+00000.000000,-00001.000000\n");
	fprintf(fp, "            Vertex   +00128.000000,-00128.000000,-00128.000000\n");
	fprintf(fp, "            Vertex   +00128.000000,-00128.000000,+00128.000000\n");
	fprintf(fp, "            Vertex   -00128.000000,-00128.000000,+00128.000000\n");
	fprintf(fp, "            Vertex   -00128.000000,-00128.000000,-00128.000000\n");
	fprintf(fp, "         End Polygon\n");
	fprintf(fp, "         Begin Polygon Item=OUTSIDE Link=4\n");
	fprintf(fp, "            Origin   +00128.000000,+00128.000000,-00128.000000\n");
	fprintf(fp, "            Normal   +00001.000000,+00000.000000,+00000.000000\n");
	fprintf(fp, "            TextureU +00000.000000,-00001.000000,+00000.000000\n");
	fprintf(fp, "            TextureV +00000.000000,+00000.000000,-00001.000000\n");
	fprintf(fp, "            Vertex   +00128.000000,+00128.000000,-00128.000000\n");
	fprintf(fp, "            Vertex   +00128.000000,+00128.000000,+00128.000000\n");
	fprintf(fp, "            Vertex   +00128.000000,-00128.000000,+00128.000000\n");
	fprintf(fp, "            Vertex   +00128.000000,-00128.000000,-00128.000000\n");
	fprintf(fp, "         End Polygon\n");
	fprintf(fp, "         Begin Polygon Item=OUTSIDE Link=5\n");
	fprintf(fp, "            Origin   -00128.000000,-00128.000000,-00128.000000\n");
	fprintf(fp, "            Normal   -00001.000000,+00000.000000,+00000.000000\n");
	fprintf(fp, "            TextureU +00000.000000,+00001.000000,+00000.000000\n");
	fprintf(fp, "            TextureV +00000.000000,+00000.000000,-00001.000000\n");
	fprintf(fp, "            Vertex   -00128.000000,-00128.000000,-00128.000000\n");
	fprintf(fp, "            Vertex   -00128.000000,-00128.000000,+00128.000000\n");
	fprintf(fp, "            Vertex   -00128.000000,+00128.000000,+00128.000000\n");
	fprintf(fp, "            Vertex   -00128.000000,+00128.000000,-00128.000000\n");
	fprintf(fp, "         End Polygon\n");
	fprintf(fp, "      End PolyList\n");
	fprintf(fp, "   End Brush\n");
}

//------------------------------
// Write a single polygon.  This function may be called multiple times per
// brush.  One special case is that if the brush texture is named "portal",
// the texture name will not be written.
//------------------------------
void SavePolygonT3D(FILE *fp, brush_t *b, int k, int texflags, int portal, vec3_t offset, t3d_texture_t *t)
{
	int l;
	vec3_t vert;
	vec3_t origin;
	//
	// Write polygon header, texture, and number
	//
	fprintf(fp, "         Begin Polygon Item=OUTSIDE");
	
	if (!portal)
	{
		if (t)
		{
			fprintf(fp, " Texture=%s", t->texname);
		}
		else
		{
			fprintf(fp, " Texture=%s", b->plane[k].tex.name);
		}
	}
	if (texflags) {
		fprintf(fp, " Flags=%i", texflags);
	}
	if (!portal) {
		fprintf(fp, " Link=%i\n", k);
	} else {
		fprintf(fp, " Link=0\n");
	}
	//
	// Write polygon origin.  If we're using texture info from input file,
	// use the same origin that was in the input file.
	//
	if (t) {
		fprintf(fp, "            Origin   %+013.6f,%+013.6f,%+013.6f\n", t->origin.x, t->origin.y, t->origin.z);
	} else {
		// Make up an origin... duh?!?  Use last vertex I guess...
		origin.x = b->verts[b->plane[k].verts[b->plane[k].num_verts-1]].x - offset.x;
		origin.y = -1.0 * (b->verts[b->plane[k].verts[b->plane[k].num_verts-1]].y - offset.y);
		origin.z = b->verts[b->plane[k].verts[b->plane[k].num_verts-1]].z - offset.z;
		fprintf(fp, "            Origin   %+013.6f,%+013.6f,%+013.6f\n", origin.x, origin.y, origin.z);
	}
	//
	// Precomputed plane normal seems to work okay (except middle value needs to be negated?)
	//
	fprintf(fp, "            Normal   %+013.6f,%+013.6f,%+013.6f\n",
		b->plane[k].normal.x,
//		b->plane[k].normal.y,
-1.0 *		b->plane[k].normal.y,
		b->plane[k].normal.z);
	//
	// Write some texture alignment stuff that I don't understand
	//
	if (t) {
		if (t->panu || t->panv) {
			fprintf(fp, "            Pan      U=%i V=%i\n", t->panu, t->panv);
		}
		fprintf(fp, "            TextureU %+013.6f,%+013.6f,%+013.6f\n", t->textureu.x, t->textureu.y, t->textureu.z);
		fprintf(fp, "            TextureV %+013.6f,%+013.6f,%+013.6f\n", t->texturev.x, t->texturev.y, t->texturev.z);
	}
	//
	// Write vertices in same order, but change sign of Y coordinate.
	//
	for (l = 0; l < b->plane[k].num_verts; l++) {
		vert.x = b->verts[b->plane[k].verts[l]].x - offset.x;
		vert.y = b->verts[b->plane[k].verts[l]].y - offset.y;
		vert.z = b->verts[b->plane[k].verts[l]].z - offset.z;
		fprintf(fp, "            Vertex   %+013.6f,%+013.6f,%+013.6f\n",
			vert.x, -1.0 * vert.y, vert.z);
	}

	fprintf(fp, "         End Polygon\n");
}

// what must match: location, normal, identify
// if found, unlink it from the chain so we don't have to test it again
// returns NULL if not found
t3d_texture_t *matching_surface(vec3_t * offset, vec3_t * norm, vec3_t * identify)
{
	vec3_t location;
	vec3_t normal;
	t3d_texture_t * next;
	t3d_texture_t * prev;

	location.x = offset->x;
	location.y = -1.0 * offset->y;
	location.z = offset->z;
	normal.x = norm->x;
	normal.y = -1.0 * norm->y;
	normal.z = norm->z;

	next = first_t3d_texture;
	prev = NULL;

	while (next) {
		if (fabs(next->location.x - location.x) < 0.0001 &&	// Identify brush
			fabs(next->location.y - location.y) < 0.0001 &&
			fabs(next->location.z - location.z) < 0.0001 &&
			fabs(next->normal.x - normal.x) < 0.0001 &&		// Identify surface
			fabs(next->normal.y - normal.y) < 0.0001 &&
			fabs(next->normal.z - normal.z) < 0.0001 &&
			fabs(next->identify.x - identify->x) < 0.0001 &&// Nail it exactly
			fabs(next->identify.y - identify->y) < 0.0001 &&
			fabs(next->identify.z - identify->z) < 0.0001
		) {
			if (prev) {		// Unlink this entry
				prev->next = next->next;
			} else {
				first_t3d_texture = next->next;
			}
			num_tused++;
			return next;	// Indicate success
		} else {
			prev = next;
			next = next->next;
		}
	}
	return NULL;
}

//------------------------------
// Write the parameters for the given brush in T3D format
// CSG=2 is subtractive.  CSG=1 is additive.
// Things to think about:
//
// Location is currently always at origin.  We may want to compute the
// brush's center instead and adjust vertices accordingly.  For location,
// we'll write b->center.xyz,   
//
// Brush groups?  Compute centroid of entire group?
//
// ZonePortal sheets are done using a hack... we use a normal 3D brush, but
// one of its surfaces uses "portal" for a texture.  We write only that one
// surface to the .T3D file.
//------------------------------
void SaveBrushT3D(FILE * fp, brush_t * b, int csg)
{
	int k;
	int numplanes;
	int polyflags = 0;
	int texflags;
	int portal;
	char tempstr[128];
	vec3_t offset;
	t3d_texture_t * t;
	vec3_t identify;

	//CalcBrushCenter(b); // (Should have already been done)
	//
	// Compute the amount to add to vertex coordinates to make them relative
	// to the brush's center
	//
	offset.x = b->center.x;
	offset.y = b->center.y;
	offset.z = b->center.z;

	if (GetKeyValue(b->EntityRef, "spawnflags") != NULL)
	{
		sscanf(GetKeyValue(b->EntityRef, "spawnflags"), "%d", &polyflags);
	}
	portal = 0;				// Assume it's not a ZonePortal brush
	numplanes = b->num_planes;
	for (k = 0; k < numplanes; k++)
	{
      strcpy(tempstr,b->plane[k].tex.name);
		strupr(tempstr);
		if (strstr(tempstr, "PORTAL") != NULL)
		{
			portal = 1;		// It's a ZonePortal
			numplanes = 1;
			offset.x = b->plane[k].center.x;
			offset.y = b->plane[k].center.y;
			offset.z = b->plane[k].center.z;
		}
	}

	fprintf(fp, "   Begin Brush Name=QBrush\n");
	fprintf(fp, "      Settings  CSG=%i Flags=64 PolyFlags=%i Color=0\n", csg, polyflags);
	//fprintf(fp, "      Location  +00000.000000,+00000.000000,+00000.000000\n");
	fprintf(fp, "      Location  %+013.6f,%+013.6f,%+013.6f\n",
		offset.x,
		-1.0 * offset.y,
		offset.z);
	fprintf(fp, "      PrePivot  +00000.000000,+00000.000000,+00000.000000\n");
	fprintf(fp, "      PostPivot +00000.000000,+00000.000000,+00000.000000\n");
	fprintf(fp, "      Scale     X=+00001.000000 Y=+00001.000000 Z=+00001.000000 S=+00000.000000 AXIS=5\n");
	fprintf(fp, "      PostScale X=+00001.000000 Y=+00001.000000 Z=+00001.000000 S=+00000.000000 AXIS=5\n");
	fprintf(fp, "      Rotation  0,0,0\n");
	fprintf(fp, "      Begin Polylist Num=%i Max=%i\n", numplanes, numplanes);

	for (k = 0; k < b->num_planes; k++)
	{
		texflags = 0;
		//
		// If texture name contains "sky", set Flags=16 for "Environment Map"
		//
      strcpy(tempstr,b->plane[k].tex.name);
		strupr(tempstr);
		if (strstr(tempstr, "SKY") != NULL) {
			texflags = texflags | 16;
			texflags = texflags | 512;	// plus U-pan
			texflags = texflags | 1024;	// plus V-pan
		}
		//
		// If texture name contains "water", set Flags=8192 for "Small Wavy"
		//
		if (strstr(tempstr, "WATER") != NULL) {
			texflags = texflags | 8192;
			//
			// If any PolyFlags are set, set 256 to make water face 2-sided
			//
			if (polyflags) {
				texflags = texflags | 256;
			}
		}
		//
		// If any texture flags are set, add the PolyFlags as well
		//
		if (texflags) {
			texflags = texflags | polyflags;
		}
		if (!portal || (strstr(tempstr, "PORTAL") != NULL)) {
			//
			// Extract x,y,z from 1st, 2nd, and 3rd vertices of polygon
			//
			identify.x = b->verts[b->plane[k].verts[ 0 ]].x - offset.x;
			identify.y = -1.0 * (b->verts[b->plane[k].verts[ 1 ]].y - offset.y);
			identify.z = b->verts[b->plane[k].verts[ 2 ]].z - offset.z;
			//
			// Scan list of t3d_texture structures for a match
			//
			t = matching_surface(&offset, &b->plane[k].normal, &identify);
			//
			// Write a polygon
			//
			SavePolygonT3D(fp, b, k, texflags, portal, offset, t);
		}
	}
	fprintf(fp, "      End PolyList\n");
	fprintf(fp, "   End Brush\n");
}

void SaveActorListT3D(FILE * fp)
{
	int x, y, z;
	entity_t *e;
	int me = 0;
	time_t t;
	struct tm * tb;
	int light;		// Light level

	t = time(NULL);		// Get seconds since 1970
	tb = localtime(&t);	// Convert that absurd value into something good

	fprintf(fp, "   Begin ActorList Max=2500\n");
	fprintf(fp, "      Begin Actor Class=LevelDescriptor\n");
	fprintf(fp, "         Me=%i\n", me++);
	fprintf(fp, "         Time=+00000.505433\n");
	fprintf(fp, "         DayFraction=+00000.000168\n");
	fprintf(fp, "         Ticks=8249\n");
	fprintf(fp, "         Year=%i\n", tb->tm_year + 1900);
	fprintf(fp, "         Month=%i\n", tb->tm_mon + 1);
	fprintf(fp, "         Day=%i\n", tb->tm_mday);
	fprintf(fp, "         Hour=%i\n", tb->tm_hour);
	fprintf(fp, "         Minute=%i\n", tb->tm_min);
	fprintf(fp, "         Second=%i\n", tb->tm_sec);
	fprintf(fp, "         Millisecond=440\n");
	fprintf(fp, "      End Actor\n");
	//
	// Run throught the entity list and write any PlayerStarts or Lights we find
	//
	for (e=EntityHead; e; e=e->Next) {
		if (GetKeyValue(e, "classname") != NULL) {

			//
			// PlayerStart
			//
			if (strstr(GetKeyValue(e, "classname"), "info_player_start") != NULL)
			{
				sscanf(GetKeyValue(e, "origin"), "%d %d %d", &x, &y, &z);
				fprintf(fp, "      Begin Actor Class=PlayerStart\n");
				fprintf(fp, "         Location=(%+013.6f,%+013.6f,%+013.6f)\n",
					(float)x, -1.0 * (float)y, (float)z);
				fprintf(fp, "         Me=%i\n", me++);
				fprintf(fp, "      End Actor\n");
			} else
			//
			// Light
			//
			if (strstr(GetKeyValue(e, "classname"), "light") != NULL) {
				sscanf(GetKeyValue(e, "origin"), "%d %d %d", &x, &y, &z);
				fprintf(fp, "      Begin Actor Class=Light\n");
				fprintf(fp, "         Location=(%+013.6f,%+013.6f,%+013.6f)\n",
					(float)x, -1.0 * (float)y, (float)z);
				light = 0;
				sscanf(GetKeyValue(e, "light"), "%d", &light);
				if (light) {
					fprintf(fp, "         LightBrightness=%i\n", light);
				}
				light = 0;
				sscanf(GetKeyValue(e, "style"), "%d", &light);
				if (light) {
					fprintf(fp, "         LightRadius=%i\n", light);
				}
				light = 0;
				sscanf(GetKeyValue(e, "wait"), "%d", &light);
				if (light) {
					fprintf(fp, "         LightHue=%i\n", light);
				}
				light = 0;
				sscanf(GetKeyValue(e, "speed"), "%d", &light);
				if (light) {
					fprintf(fp, "         LightSaturation=%i\n", light);
				}
				//
				// These effects only seem to work in 3D mode
				//
				light = 0;
				sscanf(GetKeyValue(e, "angle"), "%d", &light);
				if (light) switch (light) {
					case 2: fprintf(fp, "         LightType=LT_Pulse\n"); break;
					case 3: fprintf(fp, "         LightType=LT_Blink\n"); break;
					case 4: fprintf(fp, "         LightType=LT_Flicker\n"); break;
					case 5: fprintf(fp, "         LightType=LT_Strobe\n"); break;
					case 6: fprintf(fp, "         LightType=LT_BackdropLight\n"); break;
					case 7: fprintf(fp, "         LightType=LT_SubtlePulse\n"); break;
					case 8: fprintf(fp, "         LightType=LT_TexturePaletteOnce\n"); break;
					case 9: fprintf(fp, "         LightType=LT_TexturePaletteLoop\n"); break;
				}
				//
				// These effects only seem to work in 2D mode
				//
				//light = 0;
				//sscanf(GetKeyValue(e, "angle"), "%d", &light);
				//if (light) switch (light) {
				//	case 1: fprintf(fp, "         LightEffect=LE_TorchWaver\n"); break;
				//	case 2: fprintf(fp, "         LightEffect=LE_FireWaver\n"); break;
				//	case 3: fprintf(fp, "         LightEffect=LE_WateryShimmer\n"); break;
				//	case 4: fprintf(fp, "         LightEffect=LE_SearchLight\n"); break;
				//	case 5: fprintf(fp, "         LightEffect=LE_SlowWave\n"); break;
				//	case 6: fprintf(fp, "         LightEffect=LE_FastWave\n"); break;
				//	case 7: fprintf(fp, "         LightEffect=LE_CloudCast\n"); break;
				//	case 8: fprintf(fp, "         LightEffect=LE_StaticSpot\n"); break;
				//	case 9: fprintf(fp, "         LightEffect=LE_Shock\n"); break;
				//	case 10:fprintf(fp, "         LightEffect=LE_Disco\n"); break;
				//	case 11:fprintf(fp, "         LightEffect=LE_Warp\n"); break;
				//	case 12:fprintf(fp, "         LightEffect=LE_SpotLight\n"); break;
				//	case 13:fprintf(fp, "         LightEffect=LE_StolenQuakeWater\n"); break;
				//	case 14:fprintf(fp, "         LightEffect=LE_ChurningWater\n"); break;
				//	case 15:fprintf(fp, "         LightEffect=LE_Satellite\n"); break;
				//	case 16:fprintf(fp, "         LightEffect=LE_Interference\n"); break;
				//	case 17:fprintf(fp, "         LightEffect=LE_Cylinder\n"); break;
				//	case 18:fprintf(fp, "         LightEffect=LE_Rotor\n"); break;
				//}
				fprintf(fp, "         Me=%i\n", me++);
				fprintf(fp, "      End Actor\n");
			}
			//
			// Fog
			//
			if (strstr(GetKeyValue(e, "classname"), "unreal_fogzone") != NULL) {
				sscanf(GetKeyValue(e, "origin"), "%d %d %d", &x, &y, &z);
				fprintf(fp, "      Begin Actor Class=ZoneInfo\n");
				fprintf(fp, "         Location=(%+013.6f,%+013.6f,%+013.6f)\n",
					(float)x, -1.0 * (float)y, (float)z);
				fprintf(fp, "         bFogZone=1\n");
				fprintf(fp, "         Me=%i\n", me++);
				fprintf(fp, "      End Actor\n");
			}
			//
			// Water
			//
			if (strstr(GetKeyValue(e, "classname"), "unreal_waterzone") != NULL) {
				sscanf(GetKeyValue(e, "origin"), "%d %d %d", &x, &y, &z);
				fprintf(fp, "      Begin Actor Class=ZoneInfo\n");
				fprintf(fp, "         Location=(%+013.6f,%+013.6f,%+013.6f)\n",
					(float)x, -1.0 * (float)y, (float)z);
				fprintf(fp, "         bWaterZone=1\n");
				fprintf(fp, "         Me=%i\n", me++);
				fprintf(fp, "      End Actor\n");
			}
		}
	}
	fprintf(fp, "   End ActorList\n");
}


int SaveT3D(char *filename)
{
	FILE *fp;
	entity_t *e;
	entity_t *LastEnt;
	brush_t *LastBrush;
	brush_t *b;

	//
	// Open file and write editor version header
	//
	if ((fp = fopen(filename, "wt")) == NULL)
	{
		HandleError("SaveT3D", "Unable to create file");
		return 0;
	}
	fprintf(fp, ";\n");
	fprintf(fp, "; UnQuest\n");
	fprintf(fp, ";\n");

	fprintf(fp, "Begin Map Name=QuestLev Brushes=%i\n", num_brushes+1);
	fprintf(fp, "   ;\n");

	SaveRootHullT3D(fp);

	// Locate last entity (will be worldspawn, since lists were built backwards)
	for (LastEnt=e=EntityHead; e; e=e->Next)
	{
		LastEnt = e;
	}

	for (LastBrush=b=BrushHead; b; b=b->Next) {
		LastBrush = b;
	}

	//
	// Write the CSG=2 (worldspawn subtractive) brushes
	//
	for (b=LastBrush; b; b=b->Last) {
		if (b->EntityRef == LastEnt) {
			SaveBrushT3D(fp, b, 2);
		}
	}

	//
	// Write the CSG=1 (entity additive) brushes
	//
	e=LastEnt->Last;
	for (; e; e=e->Last)
	{
		for (b=LastBrush; b; b=b->Last)
		{
			if (b->EntityRef == e)
			{
				SaveBrushT3D(fp, b, 1);
			}
		}
	}

	SaveActorListT3D(fp);

	fprintf(fp, "End Map\n");

	fclose(fp);

	return 1;
}

//----------------------------
// Loads texture data from an UnrealEd T3D file for use in merging by SaveT3D
//----------------------------
//----------------------------
// Given something like "Foo bar", advance to "bar"
//----------------------------

char * next_token(char * p) {
	while (*p != ' ') {
		p++;
	}
	while (*p == ' ' || *p == '\t') {
		p++;
	}
	return p;
}

//----------------------------
// Read a line into global variable t3dline and skip whitespace
//----------------------------
void read_t3dline(void) {
	fgets(t3dline, 255, t3dfp);
	t3d_linenumber++;
// printf("Line %i: %s", t3d_linenumber, t3dline);
	t3dpos = &t3dline[0];
	while (*t3dpos == ' ' || *t3dpos == '\t') {
		t3dpos++;
	}
	return;
}


int worth_preserving(vec3_t * textureu, vec3_t * texturev) {
	return 1;	// All surfaces worth preserving (will hurt performance)

	if (
		(textureu->x != 0.0 && textureu->x != 1.0 && textureu->x != -1.0) ||
		(textureu->y != 0.0 && textureu->y != 1.0 && textureu->y != -1.0) ||
		(textureu->z != 0.0 && textureu->z != 1.0 && textureu->z != -1.0) ||
		(texturev->x != 0.0 && texturev->x != 1.0 && texturev->x != -1.0) ||
		(texturev->y != 0.0 && texturev->y != 1.0 && texturev->y != -1.0) ||
		(texturev->z != 0.0 && texturev->z != 1.0 && texturev->z != -1.0)
	) {
		return 1;
	}
	return 0;
}


void load_polygon(void)
{
	char texname[16];
	vec3_t normal;
	vec3_t identify;
	vec3_t origin;
	int panu;
	int panv;
	vec3_t textureu;
	vec3_t texturev;
	char * p;
	t3d_texture_t * tp;
	float junk;
	int vertcount;

	vertcount = 0;
	panu = 0;
	panv = 0;
	normal.x = 0.0;
	normal.y = 0.0;
	normal.z = 0.0;
	origin.x = 0.0;
	origin.y = 0.0;
	origin.z = 0.0;
	identify.x = 0.0;	// x value of 1st vertex of poly
	identify.y = 0.0;	// y value of 2nd vertex of poly
	identify.z = 0.0;	// z value of 3rd vertex of poly
	textureu.x = 0.0;
	textureu.y = 0.0;
	textureu.z = 0.0;
	texturev.x = 0.0;
	texturev.y = 0.0;
	texturev.z = 0.0;
	load_texture(&texname[0]);	// copy texture name to local variable
	do {
		read_t3dline();
		if (memcmp("Normal", t3dpos, 6) == 0) {
			p = next_token(t3dpos);
			sscanf(p, "%f,%f,%f", &normal.x, &normal.y, &normal.z);
		}
		if (memcmp("Origin", t3dpos, 6) == 0) {
			p = next_token(t3dpos);
			sscanf(p, "%f,%f,%f", &origin.x, &origin.y, &origin.z);
		}
		if (memcmp("Pan", t3dpos, 3) == 0) {
			p = next_token(t3dpos);
			sscanf(p, "U=%i V=%i", &panu, &panv);
		}
		if (memcmp("TextureU", t3dpos, 8) == 0) {
			p = next_token(t3dpos);
			sscanf(p, "%f,%f,%f", &textureu.x, &textureu.y, &textureu.z);
		}
		if (memcmp("TextureV", t3dpos, 8) == 0) {
			p = next_token(t3dpos);
			sscanf(p, "%f,%f,%f", &texturev.x, &texturev.y, &texturev.z);
		}
		if (memcmp("Vertex", t3dpos, 6) == 0) {
			vertcount++;
			p = next_token(t3dpos);
			switch (vertcount) {
				case 1:	sscanf(p, "%f,%f,%f", &identify.x, &junk, &junk);
					break;
				case 2:	sscanf(p, "%f,%f,%f", &junk, &identify.y, &junk);
					break;
				case 3:	sscanf(p, "%f,%f,%f", &junk, &junk, &identify.z);
					break;
				default: break;
			}
		}
	} while (memcmp("End Polygon", t3dpos, 11) != 0);

	// if we have Pan, TextureU, or TextureV data, allocate a Texture Info
	// Structure, copy enough data into it to uniquely identify it, and
	// chain it to the list.
	if (panu || panv || worth_preserving(&textureu, &texturev))
	{
		tp = (t3d_texture_t *)Q_malloc(sizeof(t3d_texture_t));
		if (tp)
		{
			tp->next = NULL;
			tp->location.x = location.x;
			tp->location.y = location.y;
			tp->location.z = location.z;
			tp->normal.x = normal.x;
			tp->normal.y = normal.y;
			tp->normal.z = normal.z;
			tp->identify.x = identify.x;
			tp->identify.y = identify.y;
			tp->identify.z = identify.z;
			strcpy(tp->texname, texname);
			tp->origin.x = origin.x;
			tp->origin.y = origin.y;
			tp->origin.z = origin.z;
			tp->panu = panu;
			tp->panv = panv;
			tp->textureu.x = textureu.x;
			tp->textureu.y = textureu.y;
			tp->textureu.z = textureu.z;
			tp->texturev.x = texturev.x;
			tp->texturev.y = texturev.y;
			tp->texturev.z = texturev.z;

			if (last_t3d_texture)
			{
				last_t3d_texture->next = tp;
			} else
			{
				first_t3d_texture = tp;
			}
			last_t3d_texture = tp;
			num_tpolys++;
// printf("loaded %i polys by line %i\n", num_tpolys, t3d_linenumber);
		}
	}
	return;
}

void load_polylist(void)
{
	do {
		read_t3dline();
		if (memcmp("Begin Polygon", t3dpos, 13) == 0) {
			load_polygon();
		}
	} while (memcmp("End PolyList", t3dpos, 12) != 0);
	return;
}

void load_brush(void)
{
	do {
		read_t3dline();
		if (memcmp("Begin PolyList", t3dpos, 14) == 0) {
			load_polylist();
		}
	} while (memcmp("End Brush", t3dpos, 9) != 0);
	return;
}

//----------------------------
// Parse a .T3D Location=() statement
//----------------------------
void load_location(vec3_t * location) {
	char * p;

	location->x = 0.0;
	location->y = 0.0;
	location->z = 0.0;

	p = strstr(t3dpos, "X=");
	if (p) {
		sscanf(p, "X=%f", &location->x);
	}
	p = strstr(t3dpos, "Y=");
	if (p) {
		sscanf(p, "Y=%f", &location->y);
	}
	p = strstr(t3dpos, "Z=");
	if (p) {
		sscanf(p, "Z=%f", &location->z);
	}
	return;
}

//----------------------------
void load_texture(char * texture) {
	char * p;

	p = strstr(t3dpos, "Texture=");
	if (p) {
		p = p +	8;	// Point at actual texture name
		while (*p != ' ' && *p != '\t') {
			*texture++ = *p++;
		}
		*texture = '\0';
	} else {
		strcpy(texture, "Default");
	}
	return;
}


void load_actor(void)
{
	do {
		read_t3dline();
		if (memcmp("Location", t3dpos, 8) == 0) {
			load_location(&location);
		}
		if (memcmp("Begin Brush", t3dpos, 11) == 0) {
			load_brush();
		}
	} while (memcmp("End Actor", t3dpos, 9) != 0);
	return;
}

void load_actorlist(void)
{
	do {
		read_t3dline();
		if (memcmp("Begin Actor", t3dpos, 11) == 0) {
			load_actor();
		}
	} while (memcmp("End ActorList", t3dpos, 13) != 0);
	return;
}

void load_map(void)
{
	do
	{
		read_t3dline();
		if (memcmp("Begin ActorList", t3dpos, 15) == 0)
		{
			load_actorlist();
		}
	} while (memcmp("End Map", t3dpos, 7) != 0);
	return;
}

int LoadT3D(char *fname)
{
	if ((t3dfp = fopen(fname, "rt")) == NULL)
	{
		HandleError("LoadT3D", "Unable to open input T3D file.");
		return 0;
	}
	do
	{
		read_t3dline();
		if (memcmp("Begin Map", t3dpos, 9) == 0)
		{
			load_map();
		}
	} while (memcmp("End Map", t3dpos, 7) != 0);
	fclose(t3dfp);
	return 1;
}

