﻿using System;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Threading.Tasks;
using System.Windows.Forms;
using LogicAndTrick.Oy;
using Sledge.BspEditor.Commands;
using Sledge.BspEditor.Documents;
using Sledge.BspEditor.Editing.Properties;
using Sledge.BspEditor.Modification;
using Sledge.BspEditor.Modification.Operations;
using Sledge.Common.Shell.Commands;
using Sledge.Common.Shell.Menu;
using Sledge.Common.Translations;

namespace Sledge.BspEditor.Editing.Commands.Pointfile
{
    [AutoTranslate]
    [Export(typeof(ICommand))]
    [MenuItem("Map", "", "Pointfile", "B")]
    [CommandID("BspEditor:Map:LoadPointfile")]
    [MenuImage(typeof(Resources), nameof(Resources.Menu_LoadPointfile))]
    public class LoadPointfile : BaseCommand
    {
        public override string Name { get; set; } = "Load pointfile...";
        public override string Details { get; set; } = "Select a pointfile (.lin, .pts) generated by compile tools";

        public string InvalidPointfile { get; set; } = "{0} is not a valid pointfile!";

        protected override async Task Invoke(MapDocument document, CommandParameters parameters)
        {
            using (var ofd = new OpenFileDialog())
            {
                ofd.Filter = "Pointfiles (*.lin, *.pts)|*.lin;*.pts";
                ofd.InitialDirectory = Path.GetDirectoryName(document.FileName);
                ofd.Multiselect = false;
                if (ofd.ShowDialog() != DialogResult.OK) return;

                var file = ofd.FileName;
                var text = File.ReadAllLines(file);
                Pointfile point;
                try
                {
                    point = Pointfile.Parse(text);
                }
                catch
                {
                    MessageBox.Show(String.Format(InvalidPointfile, Path.GetFileName(file)));
                    return;
                }

                await MapDocumentOperation.Perform(document, new TrivialOperation(
                    d => d.Map.Data.Replace(point),
                    c => c.Add(c.Document.Map.Root)
                ));

                if (point.Lines.Any())
                {
                    var start = point.Lines[0].Start;
                    await Oy.Publish("MapDocument:Viewport:Focus2D", start);
                    await Oy.Publish("MapDocument:Viewport:Focus3D", start);
                }
            }
        }
    }
}