#include <Aztec3DPCH.h>

// Aztec2 includes
#include <controls/Aztec3DSceneCanvas.h>
#include <tools/MToolManager.h>
#include <utils/AztecGLUtils.h>
#include <views/AztecViewManager.h>

// AztecGUICommon includes
#include <config/UIConfig.h>
#include <config/ColourConstants.h>

// AztecLib includes
#include <MSystemManager.h>

// standard includes
#include <math.h>
#include <iostream>


namespace AztecGUI {

  using namespace Aztec;

  MRay clickRay;

  std::vector<MVector3> clickPoints;
  std::vector<MRay> clickRays;

  Aztec3DSceneCanvas::Aztec3DSceneCanvas(AztecView *parentView)
    : AztecGLCanvas(parentView)
  {
    perspectiveView = true;
    zoomFactor = 1.0;
    cameraDistance = 200;
    cameraRotation.set(-70,0,-30);

    textureMode = TextureLinear;
    shadingMode = ShadingSmooth;
    cullBackfaces = true;
    headlight = true;
  }

  Aztec3DSceneCanvas::~Aztec3DSceneCanvas() {
  }
  
  void Aztec3DSceneCanvas::draw3D() {
    std::vector<Aztec::MSelectionItem> sitem;
    draw3D(smNone, sitem, 0, 0);
  }

  bool Aztec3DSceneCanvas::onResize(int newWidth, int newHeight) {
    return AztecGLCanvas::onResize(newWidth, newHeight);
  }

  const Aztec::MVector3& Aztec3DSceneCanvas::getCameraRotation() const {
    return cameraRotation;
  }

  void Aztec3DSceneCanvas::setCameraRotation(const Aztec::MVector3 &rotation) {
    cameraRotation = rotation;
  }

  void Aztec3DSceneCanvas::setPerspective(bool persp) {
    perspectiveView = persp;
  } 

  bool Aztec3DSceneCanvas::isPerspective() const {
    return perspectiveView;
  }

  
  float Aztec3DSceneCanvas::getZoomFactor() const {
    return zoomFactor;
  }

  void Aztec3DSceneCanvas::setZoomFactor(float factor) {
    zoomFactor = factor;
  }

  
  float Aztec3DSceneCanvas::getDistance() const {
    return cameraDistance;
  }

  void Aztec3DSceneCanvas::setDistance(float distance) {
    cameraDistance = distance;
  }

  
  void Aztec3DSceneCanvas::getGate(float &left, float &top, float &right, float &bottom) const {
    left = gateLeft;
    top = gateTop;
    right = gateRight;
    bottom = gateBottom;
  }

  
  const Aztec::MVector3& Aztec3DSceneCanvas::getCameraFocus() const {
    return cameraFocus;
  }

  void Aztec3DSceneCanvas::setCameraFocus(const Aztec::MVector3 &focus) {
    cameraFocus = focus;
  }

  void Aztec3DSceneCanvas::getScreenVectors(Aztec::MVector3 &u, Aztec::MVector3 &v) {
    Aztec::MSize2D winsize = getSize();
    Aztec::MPlane plane(getCameraFocus(), getViewNormal());
    Aztec::MVector3 topLeft = getRay(0, 0).intersectWithPlane(plane);
    Aztec::MVector3 topRight = getRay(winsize.getWidth(), 0).intersectWithPlane(plane);
    Aztec::MVector3 bottomLeft = getRay(0, winsize.getHeight()).intersectWithPlane(plane);
    
    u = (topRight - topLeft) / winsize.getWidth(); 
    v = (bottomLeft - topLeft) / winsize.getHeight();
  }

  void Aztec3DSceneCanvas::setTextureMode(Aztec3DSceneCanvas::TextureModeEnum mode) {
    textureMode = mode;
  }

  Aztec3DSceneCanvas::TextureModeEnum Aztec3DSceneCanvas::getTextureMode() {
    return textureMode;
  }

  void Aztec3DSceneCanvas::setShadingMode(Aztec3DSceneCanvas::ShadingModeEnum mode) {
    shadingMode = mode;
  }

  Aztec3DSceneCanvas::ShadingModeEnum Aztec3DSceneCanvas::getShadingMode() {
    return shadingMode;
  }

  void Aztec3DSceneCanvas::setBackfaceCulling(bool cull) {
    cullBackfaces = cull;
  }

  bool Aztec3DSceneCanvas::getBackfaceCulling() {
    return cullBackfaces;
  }

  void Aztec3DSceneCanvas::setHeadlight(bool light) {
    headlight = light;
  }

  bool Aztec3DSceneCanvas::getHeadlight() {
    return headlight;
  }


  void Aztec3DSceneCanvas::draw3D(MSelectMethod method, 
                                    std::vector<Aztec::MSelectionItem> &items, 
                                    float left, float top, 
                                    float right, float bottom) 
  {
    makeCurrent();

    Aztec::MVector3 bgColour = UIConfig::getColour(Colours::VIEW_BACKGROUND);

    glClearColor(bgColour.x, bgColour.y, bgColour.z, 0.0f);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
    glEnable(GL_DEPTH_TEST);

    initProjectionMatrix(method, left, top, right, bottom);
 
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    if (getHeadlight()) {
      glLightModeli(GL_LIGHT_MODEL_TWO_SIDE , GL_TRUE);
      int LightTarget = (GLenum)(GL_LIGHT0 + 0);
      MVector3 Col(1,1,1);
      float Mult = 0.8f;
      MVector3 Spec(0,0,0);
    
      GLfloat  LightAmbient[] = { 0, 0, 0, 1 };
      GLfloat  LightDiffuse[] = { 0, 0, 0, 1 };
      GLfloat  LightSpecular[] = { 1, 1, 1, 1 };
      GLfloat  LightPosition[] = { 0, 0, 0, 1 };
    
      LightDiffuse[0] = Col.x * Mult;
      LightDiffuse[1] = Col.y * Mult;
      LightDiffuse[2] = Col.z * Mult;
    
      LightSpecular[0] = Spec.x;
      LightSpecular[1] = Spec.y;
      LightSpecular[2] = Spec.z;
    
      glLightfv(LightTarget, GL_AMBIENT, LightAmbient);
      glLightfv(LightTarget, GL_DIFFUSE, LightDiffuse);
      glLightfv(LightTarget, GL_SPECULAR, LightSpecular);
      glLightfv(LightTarget, GL_POSITION, LightPosition);
      glEnable(LightTarget);

      for (int i = 1; i < 32; ++i) {
        glDisable( (GLenum) (GL_LIGHT0 + i));
      }

      glEnable(GL_LIGHTING);
    }

    doCameraTransform();

    glColor3f(1,1,1);
    
    // now draw the scene.
    MSceneViewFlags flags;
    switch (getShadingMode()) {
      case ShadingWireframe : flags.m_ShadingMode = glWireframe; break;
      case ShadingFlat : flags.m_ShadingMode = glFlat; break;
      case ShadingSmooth : flags.m_ShadingMode = glSmooth; break;
    }
    flags.m_TexturingMode = glLinear;
    flags.m_LightingMode = getHeadlight() ? glHeadlight : glLit;
    flags.m_CullBackface = getBackfaceCulling();
    flags.m_SelectMethod = method;
    flags.m_SelectBuf = NULL;
    flags.m_WireColor.set(1,1,1,1);
    flags.m_FlatColor.set(1,1,1,1);

    if (method != smNone) {
      flags.m_SelectBuf = new UINT[16384];
      glSelectBuffer(16384, flags.m_SelectBuf);
      glRenderMode(GL_SELECT);
    }


    // draw the view
    Aztec::MScene::getGlobalScene()->drawScene(flags, &items);


    if (method == smNone) {
      draw3DExtras();
    }

    // if we had selection information, clear it out.
    if (method != smNone) {
      delete[] flags.m_SelectBuf;
      flags.m_SelectBuf = NULL;
    }
    

    if (method == smNone) {

      glFlush();
      glFinish();

      swapBuffers();
    }
  }

  void Aztec3DSceneCanvas::draw3DExtras() {
    // draw our grid.
	float gridRange = getGridRange();
	float gridSpacing = getGridSpacing();
	int gridMajorSpacing = getGridMajorSpacing();

    if (getGridVisible()) {
      drawGrid(gridBasisX, gridBasisY, gridRange, gridSpacing, gridMajorSpacing);
    }

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();

    // draw the current tool, but only if we are not in selection mode.
    drawManipulators(false);



    // set up all the drawing to be orthographic
    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
  
    glLoadIdentity();
  
    MSize2D size = this->getSize();
    glOrtho(0,size.getWidth(),0,size.getHeight(),-20,20);
  
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();



    // now draw the little cute axis icon in the bottom left
    glPushMatrix();
  
    glEnable(GL_DEPTH_TEST);
    glTranslatef(20,20,0);
    glRotateCamera();
    glDrawAxisIcon(getFont(), 20, 1, 1);
    glDisable(GL_DEPTH_TEST);
  
    glPopMatrix();

    // now draw our view title.
    glPushMatrix();
    glEnable(GL_BLEND);

    MVector3 titleColour(1,1,1);
    if (AztecViewManager::getCurrentView() == parentView) {
      titleColour = UIConfig::getColour(Colours::VIEW_TITLE_CURRENT);
    } else {
      titleColour = UIConfig::getColour(Colours::VIEW_TITLE);
    }
    getFont()->setForegroundColor(titleColour.x, titleColour.y, titleColour.z);

    // these postitions were just picked so they looked nice, they may appear 
    // dodgy in some circumstances!
    getFont()->draw(5, size.getHeight() - 13, parentView->getName().c_str());

    glPopMatrix();


    // Clean up this mess matrix wise
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();

    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    
  }

  int Aztec3DSceneCanvas::drawManipulators(bool selecting) {
    UINT *selectBuffer = NULL;
  
    // set up the gl rendering buffer
    if (selecting) {
      selectBuffer = new UINT[16384];
      glSelectBuffer(16384, selectBuffer);
      glRenderMode(GL_SELECT);
    
      glInitNames();
      glPushName(-1);
    }

    // draw the current tool
    MToolManager::getInstance()->GetTool(getViewGroup())->drawTool(selecting, AztecView::getViewForComponent(this));
    
    // if we have a temporary tool, also draw the non temporary tool for appearances sake.
    if (!selecting && MToolManager::getInstance()->isTemporary(getViewGroup())) {
      MToolTypePtr tool = MToolManager::getInstance()->GetTool(1, getViewGroup());
      if (tool != NULL) {
        tool->drawTool(false,  AztecView::getViewForComponent(this));
      }
    }

    // if we were in selection mode, extract the widget we clicked on
    if (selecting) {
      int widget = 0;
      GLuint *Ptr;
    
      int hitCount = glRenderMode(GL_RENDER);
    
      Ptr = selectBuffer;
    
      for (int i = 0; i < hitCount; i++)
      {
        float    Depth1, Depth2;
      
        int subHitCount = *Ptr; 
        Ptr++;
        Depth1 = (float) *Ptr/0x7fffffff; Ptr++;
        Depth2 = (float) *Ptr/0x7fffffff; Ptr++;
      
        for (int j = 0; j < subHitCount;) {
          if (*Ptr == -1) {     // we are doing a new hit
            Ptr++; j++;
            widget = *Ptr;
          }
          Ptr++; j++;
        
        }
      }
    
      delete[] selectBuffer;

      return widget;
    } else {
      return 0;
    }


  }


  void Aztec3DSceneCanvas::initProjectionMatrix(MSelectMethod method, 
                                        float left, float top,
                                        float right, float bottom)
  {
    int bigSize, smallSize;

    getBigAndSmallViewSize(bigSize, smallSize);
    
    float width, height;

    width = getSize().getWidth();
    height = getSize().getHeight();

    glMatrixMode(GL_PROJECTION);
    
    glLoadIdentity();

    // see if we have some selections to deal with
    if (method & smSelecting) {
      if (method & smBoxing) {
        glAdjustMatrixForSelection(left, top, right, bottom);
      } else {
        glAdjustMatrixForSelection(left, top);
      }
    }

    // set up the actual projection matrices
    if (perspectiveView) {

      float fov = 90;

      // TODO: add in a check to get the camera's fov here

      double factor = tan(DegToRad(fov / 2)) * aspectRatio / bigSize;
      factor /= 20;

      gateLeft = -factor * width;
      gateRight = factor * width;
      gateTop = factor * height;
      gateBottom = -factor * height;

      glFrustum( gateLeft, gateRight, gateBottom, gateTop, 0.1, 1024.0);
    } else {
      gateLeft = -32*width/smallSize*aspectRatio;
      gateRight = 32*width/smallSize*aspectRatio;
      gateTop = 32*height/smallSize*aspectRatio;
      gateBottom = -32*height/smallSize*aspectRatio;
      
      glOrtho(gateLeft, gateRight, gateBottom, gateTop, -512.0*zoomFactor, 512.0*zoomFactor);
      glScalef(zoomFactor, zoomFactor, zoomFactor);
    }
    
  }

  void Aztec3DSceneCanvas::doCameraTransform() {
    glTranslatef(0.0, 0.0, -cameraDistance);

    glRotateCamera();
    glTranslatef(-cameraFocus.x, -cameraFocus.y, -cameraFocus.z);

    double m_ModelMat[16], m_ProjMat[16];

    glGetDoublev(GL_PROJECTION_MATRIX, m_ProjMat);
    glGetDoublev(GL_MODELVIEW_MATRIX, m_ModelMat);

    modelViewMatrix = MMatrix4(m_ModelMat);
  }


  void Aztec3DSceneCanvas::drawGrid(const MVector3 &basisX, const MVector3 &basisY,float gridRange, float gridSpacing, int gridMajorSpacing) {

    // TODO: replace this so we areusing a display list.
    // Draw a grid using the given basis as the plane
    MVector3     Vec1,Vec2,Vec3,Vec4;
	    
    // Draw the most origin line
    glDisable(GL_LIGHTING);
    glDisable(GL_TEXTURE_2D);
    glDisable(GL_POLYGON_OFFSET_LINE);
    glDisable(GL_BLEND);
	
	//Draw the lines at the Origin.
	glColor3fv((float*)&UIConfig::getColour(Colours::GRID3D_AXIS));
    glBegin(GL_LINES);
    {
      Vec1 = (-gridRange)*basisX;
      Vec2 = (gridRange)*basisX;
      
      glVertex3fv((float*)&Vec1);
      glVertex3fv((float*)&Vec2);
      
      Vec1 = (-gridRange)*basisY;
      Vec2 = (gridRange)*basisY;
      
      glVertex3fv((float*)&Vec1);
      glVertex3fv((float*)&Vec2);
    }
    glEnd();

	//Draw the lines for major gris spacing
    
	glColor3fv((float*)&UIConfig::getColour(Colours::GRID3D_MAJOR));
    glBegin(GL_LINES);  
    {
      float n;
	  int gridLineCount = 1;
      for (n=gridSpacing; n <= gridRange; n+=gridSpacing) {    //grid spacing in Y and spread in X.
	
        if(gridLineCount++%gridMajorSpacing==0) {
			Vec1 = (-gridRange)*basisX;
			Vec1 += (n)*basisY;
			Vec2 = (gridRange)*basisX;
			Vec2 += (n)*basisY;
			Vec3 = -1*Vec1;
			Vec4 = -1*Vec2;

			glVertex3fv((float*)&Vec1);
			glVertex3fv((float*)&Vec2);
			glVertex3fv((float*)&Vec3);
			glVertex3fv((float*)&Vec4);

		}
		else continue;
      }
      gridLineCount = 1;
      for (n=gridSpacing; n <= gridRange; n+=gridSpacing)  {  //10 is the grid spacing
		if(gridLineCount++%gridMajorSpacing==0)  {
			Vec1 = (-gridRange)*basisY;
			Vec1 += (n)*basisX;
			Vec2 = (gridRange)*basisY;
			Vec2 += (n)*basisX;
			Vec3 = -1*Vec1;
			Vec4 = -1*Vec2;
        
			glVertex3fv((float*)&Vec1);
			glVertex3fv((float*)&Vec2);
			glVertex3fv((float*)&Vec3);
			glVertex3fv((float*)&Vec4);
		}
		else continue;
      }
    }
    glEnd();
    //Draw the rest of the lines.
    glColor3fv((float*)&UIConfig::getColour(Colours::GRID3D_MINOR));
    glBegin(GL_LINES);
    {
      float n;
	  int gridLineCount = 1;
      for (n=gridSpacing; n <= gridRange; n+=gridSpacing)   {    //grid spacing in Y and spread in X.
        if(gridLineCount++%gridMajorSpacing==0) {
          continue;
        }
        Vec1 = (-gridRange)*basisX;
        Vec1 += (n)*basisY;
        Vec2 = (gridRange)*basisX;
        Vec2 += (n)*basisY;
		Vec3 = -1*Vec1;
		Vec4 = -1*Vec2;

        
        glVertex3fv((float*)&Vec1);
        glVertex3fv((float*)&Vec2);
		glVertex3fv((float*)&Vec3);
		glVertex3fv((float*)&Vec4);

      }
      gridLineCount = 1;
      for (n=gridSpacing; n <= gridRange; n+=gridSpacing)  {  //10 is the grid spacing
        if(gridLineCount++%gridMajorSpacing==0)  {
          continue;
        }
        Vec1 = (-gridRange)*basisY;
        Vec1 += (n)*basisX;
        Vec2 = (gridRange)*basisY;
        Vec2 += (n)*basisX;
		Vec3 = -1*Vec1;
		Vec4 = -1*Vec2;

        
        glVertex3fv((float*)&Vec1);
        glVertex3fv((float*)&Vec2);
		glVertex3fv((float*)&Vec3);
		glVertex3fv((float*)&Vec4);

      }
    }
    glEnd();
    
  }

  void Aztec3DSceneCanvas::glRotateCamera() const {
    glRotatef(cameraRotation.y,0,1,0);
    glRotatef(cameraRotation.x,1,0,0);
    glRotatef(cameraRotation.z,0,0,1);
  }

  bool Aztec3DSceneCanvas::onMousePressed(const Aztec::MMouseEvent &event) {
    if (event.getShiftState().isKeyboardUp()) {
      clickRay = getRay(event.getX(), event.getY());
      
      MPlane plane(getCameraFocus(), getViewNormal());
      clickPoints.push_back(getRay(event.getX(), event.getY()).intersectWithPlane(plane));
      clickRays.push_back(getRay(event.getX(), event.getY()));
    }

    return AztecGLCanvas::onMousePressed(event);
  }


  const Matrix3& Aztec3DSceneCanvas::getInverseTransform() const {

//    m_ViewXForm.makeRotationMatrix((float)(-cameraRotation.z*M_PI/180), (-cameraRotation.x-180)*M_PI/180, 0);
    inverseTransform.makeRotationMatrix((float)(-cameraRotation.z*M_PI/180), (-cameraRotation.x-180)*M_PI/180, 0);
    inverseTransform.inverse();
    return inverseTransform;    
  }

  const MVector3& Aztec3DSceneCanvas::getViewNormal() const {

    getInverseTransform().transform(MVector3(0,0,1),viewNormal);
    
    viewNormal.x = -viewNormal.x;
    viewNormal.y = -viewNormal.y;
    viewNormal.z = viewNormal.z;
    
    return viewNormal;
  }

  const MMatrix4& Aztec3DSceneCanvas::getModelView() const {
    return modelViewMatrix;

  }

  float Aztec3DSceneCanvas::getScalingFactor(const MVector3 &Vec)
  {
    float    Dist;
    MVector3 VecScreenSpace;
    MMatrix4 ModelViewMat;
    float    Mat[16];
    
    if (isPerspective()) {
      double Factor;
      
      glGetFloatv(GL_MODELVIEW_MATRIX, Mat);
      ModelViewMat = MMatrix4(Mat);
      ModelViewMat.transpose();
      VecScreenSpace = ModelViewMat * Vec;
      
      Dist = -VecScreenSpace.z;
      
      Factor = Dist / 100.0;
      
      /*      if (m_WindowRect.right > m_WindowRect.bottom)
      Factor = 4*m_Dist/m_Aspect*(m_GateRight-m_GateLeft)/1024.0;
      else
      Factor = 4*m_Dist/m_Aspect*(m_GateTop-m_GateBottom)/1024.0;*/
      
      return Factor;
    } else {
      return 1.0/zoomFactor;
    }
  }  

  void Aztec3DSceneCanvas::getSelection(float left, float top, float right, float bottom, std::vector<Aztec::MSelectionItem> &items) {
    items.clear();
    draw3D(smSelectBox, items, left, top, right, bottom);

  }

  void Aztec3DSceneCanvas::getSelection(float x, float y, std::vector<Aztec::MSelectionItem> &items) {
    items.clear();
    draw3D(smSelect, items, x, y);
  }


}

