#include <Aztec3DPCH.h>

#include <functions/edit/SelectionFunctions.h>

// Aztec2 includes
#include <views/AztecViewManager.h>
#include <views/SelectingView.h>
#include <utils/SceneFunctions.h>

// AztecLib includes
#include <MScene.h>
#include <MSystemManager.h>
#include <MUIManager.h>
#include <misc/MSceneHelper.h>
#include <scripting/MScriptInterp.h>


#include <assert.h>

#include <algorithm>
#include <iterator>

namespace AztecGUI {

  /**
   * If an object is selected and it has a parent, then deselect it and then 
   * select its parent, otherwise do nothing.
   */
  static void selectParentFunc(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
    // if we have a node that is selected, select its parent
    if (node->getParent() != NULL) {
      scene->selectObject(node->getObject(), false);
      scene->selectObject(node->getParent()->getObject());
    }
  }

  /**
   * If an object is selected, then get its previous sibling. If we are at the
   * start of the sibling list, just wrap around to the end.
   */
  static void selectSiblingPrevFunc(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
    // If we have a previous sibling, use it.
    Aztec::MTreeObjectNodePtr newNode = node;
    if (node->getPrevSibling() != NULL) {
      newNode = node->getPrevSibling();
    } else {
      // otherwise, loop to the end.
      while (newNode->getNextSibling() != NULL) {
        newNode = newNode->getNextSibling();
      }
    }

    // if we ended up with a different node, do the selection.
    if (newNode != node) {
      scene->selectObject(node->getObject(), false);
      scene->selectObject(newNode->getObject());
    }
  }

  /**
   * If an object is selected and it has a parent, then deselect it and then 
   * select its parent, otherwise do nothing.
   */
  static void selectSiblingNextFunc(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
    // If we have a previous sibling, use it.
    Aztec::MTreeObjectNodePtr newNode = node;
    if (node->getNextSibling() != NULL) {
      newNode = node->getNextSibling();
    } else {
      // otherwise, loop to the start.
      while (newNode->getPrevSibling() != NULL) {
        newNode = newNode->getPrevSibling();
      }
    }

    // if we ended up with a different node, do the selection.
    if (newNode != node) {
      scene->selectObject(node->getObject(), false);
      scene->selectObject(newNode->getObject());
    }
  }

  /**
   * If an object is selected and it has a parent, then deselect it and then 
   * select its parent, otherwise do nothing.
   */
  static void selectChildFunc(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
    // if we have a node that is selected, select its parent
    if (node->getFirstChild() != NULL) {
      scene->selectObject(node->getObject(), false);
      scene->selectObject(node->getFirstChild()->getObject());
    }
  }

  /// Selects everything
  static void selectAllFunc(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
    switch (Aztec::MUIManager::getComponentMode()) {
    case Aztec::MComponentisedObject::OBJECT_TYPE:
      scene->selectObject(node->getObject());
      break;
    case Aztec::MComponentisedObject::POINT_TYPE:
    case Aztec::MComponentisedObject::FACET_TYPE:
    case Aztec::MComponentisedObject::EDGE_TYPE:
      {
        Aztec::MComponentisedObjectPtr compObj = Aztec::getComponentObject(node->getObject());
        
        if (compObj != NULL && compObj->isInComponentMode(Aztec::MUIManager::getComponentMode())) {
          int count = compObj->getComponentCount(Aztec::MUIManager::getComponentMode());
          for (int i = 0; i < count; ++i) {
            compObj->flagComponent(Aztec::MUIManager::getComponentMode(), i, Aztec::MComponentisedObject::COMPONENT_SELECTED, Aztec::MComponentisedObject::atSet);
          }
        }
      }
      break;
    }
  }

  /// Selects nothing
  static void selectNoneFunc(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
    switch (Aztec::MUIManager::getComponentMode()) {
    case Aztec::MComponentisedObject::OBJECT_TYPE:
      scene->selectObject(node->getObject());
      break;
    case Aztec::MComponentisedObject::POINT_TYPE:
    case Aztec::MComponentisedObject::FACET_TYPE:
    case Aztec::MComponentisedObject::EDGE_TYPE:
      {
        Aztec::MComponentisedObjectPtr compObj = Aztec::getComponentObject(node->getObject());
        
        if (compObj != NULL && compObj->isInComponentMode(Aztec::MUIManager::getComponentMode())) {
          int count = compObj->getComponentCount(Aztec::MUIManager::getComponentMode());
          for (int i = 0; i < count; ++i) {
            compObj->flagComponent(Aztec::MUIManager::getComponentMode(), i, Aztec::MComponentisedObject::COMPONENT_SELECTED, Aztec::MComponentisedObject::atUnset);
          }
        }
      }
      break;
    }
  }

  /// Just inverses the selection of each object.
  static void selectInverseFunc(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
    scene->selectObject(node->getObject(), !node->getObject()->isFlagged(OBJECTFLAG_SELECTED));
  }


  static void selectInversePointFunc(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
    Aztec::MComponentisedObjectPtr compObj = Aztec::getComponentObject(node->getObject());
      
    if (compObj != NULL && compObj->isInComponentMode(Aztec::MComponentisedObject::POINT_TYPE)) {
      for (int i = 0; i < compObj->getComponentCount(Aztec::MComponentisedObject::POINT_TYPE); ++i) {
        if (compObj->isComponentFlagged(Aztec::MComponentisedObject::POINT_TYPE, i, VERTEX_VISIBLE)) {
        compObj->flagComponent(Aztec::MComponentisedObject::POINT_TYPE, 
                               i, VERTEX_SELECTED, 
                               Aztec::MComponentisedObject::atToggle);
        }
      }
    }

  }

  static void selectInverseFaceFunc(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
    Aztec::MComponentisedObjectPtr compObj = Aztec::getComponentObject(node->getObject());

    // check to see if this is a mesh object, if it is, use the face functions directly, because it has funny handling of the component inversion.
    if (compObj != NULL && compObj->isInComponentMode(Aztec::MComponentisedObject::FACET_TYPE)) {
      Aztec::MMeshPtr mesh = AZTEC_CAST(Aztec::MMesh, compObj);

      if (mesh != NULL) {
        for (int i = 0; i < mesh->getNumTris(); ++i) {
          mesh->toggleTriangleFlag(i, TRIANGLE_SELECTED);
        }
      } else {
        for (int i = 0; i < compObj->getComponentCount(Aztec::MComponentisedObject::FACET_TYPE); ++i) {
          if (compObj->isComponentFlagged(Aztec::MComponentisedObject::EDGE_TYPE, i, TRIANGLE_VISIBLE)) {
            compObj->flagComponent(Aztec::MComponentisedObject::FACET_TYPE, 
                                   i, TRIANGLE_SELECTED, 
                                   Aztec::MComponentisedObject::atToggle);
          }
        }
      }
    }

  }

  static void selectInverseEdgeFunc(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
    Aztec::MComponentisedObjectPtr compObj = Aztec::getComponentObject(node->getObject());
      
    if (compObj != NULL && compObj->isInComponentMode(Aztec::MComponentisedObject::EDGE_TYPE)) {
      for (int i = 0; i < compObj->getComponentCount(Aztec::MComponentisedObject::EDGE_TYPE); ++i) {
        if (compObj->isComponentFlagged(Aztec::MComponentisedObject::EDGE_TYPE, i, EDGE_VISIBLE)) {
          compObj->flagComponent(Aztec::MComponentisedObject::EDGE_TYPE, 
                                 i, EDGE_SELECTED, 
                                 Aztec::MComponentisedObject::atToggle);
        }
      }
    }

  }

  /// If an object is a child of an already selected object, then select it.
  static void selectAllChildrenFunc(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
    if (node->isChildOfFlagged(OBJECTFLAG_SELECTED)) {
      scene->selectObject(node->getObject(), true, true);
    }
  }

  /**
   * For each object that is selected, the parent of that object is selected, 
   * and the original object deselected. If an object has no parent, then it 
   * stays selected.
   */
  static int selectObjectParent(const StringVector &args, std::string &result) {
    return applyFunction(Aztec::MScene::getGlobalScene(), "Select Parent", ifSelectedCriteria, selectParentFunc);
  }

  /**
   * For each object that is selected, the first child of that object is selected, 
   * and the original object deselected. If an object has no child, then it 
   * stays selected.
   */
  static int selectObjectChild(const StringVector &args, std::string &result) {
    return applyFunction(Aztec::MScene::getGlobalScene(), "Select Child", ifSelectedCriteria, selectChildFunc);
  }

  /**
   * For each object that is selected, the parent of that object is selected, 
   * and the original object deselected. If an object has no parent, then it 
   * stays selected.
   */
  static int selectObjectSiblingNext(const StringVector &args, std::string &result) {
    return applyFunction(Aztec::MScene::getGlobalScene(), "Select Next Sibling", ifSelectedCriteria, selectSiblingNextFunc);
  }

  /**
   * For each object that is selected, the previous sibling of that object is selected, 
   * and the original object deselected. If an object has no parent, then it 
   * stays selected.
   */
  static int selectObjectSiblingPrev(const StringVector &args, std::string &result) {
    return applyFunction(Aztec::MScene::getGlobalScene(), "Select Previous Sibling", ifSelectedCriteria, selectSiblingPrevFunc);
  }


  /**
   * This deselcts all objects in the scene
   */
  static int selectObjectNone(const StringVector &args, std::string &result) {
    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE) {
      Aztec::MScene::getGlobalScene()->selectNone();
      return FunctionManager::SUCCEED;
    } else {
      return applyFunction(Aztec::MScene::getGlobalScene(), "Select None", allObjectsCriteria, selectNoneFunc);
    }
  }

  /**
   * This selectes everything in the scene.
   */
  static int selectObjectAll(const StringVector &args, std::string &result) {
    return applyFunction(Aztec::MScene::getGlobalScene(), "Select All", allObjectsCriteria, selectAllFunc);
  }

  /**
   * Inverts the selection, i.e. anything that was selected before, is not 
   * selected node, and anything that wasn't selected before is selected now.
   */
  static int selectObjectInverse(const StringVector &args, std::string &result) {
    switch (Aztec::MUIManager::getComponentMode()) {
    case Aztec::MComponentisedObject::OBJECT_TYPE:
      return applyFunction(Aztec::MScene::getGlobalScene(), "Select Invserse", allObjectsCriteria, selectInverseFunc);
    case Aztec::MComponentisedObject::POINT_TYPE:
      return applyFunction(Aztec::MScene::getGlobalScene(), "Select Invserse", allObjectsCriteria, selectInversePointFunc);
    case Aztec::MComponentisedObject::EDGE_TYPE:
      return applyFunction(Aztec::MScene::getGlobalScene(), "Select Invserse", allObjectsCriteria, selectInverseEdgeFunc);
    case Aztec::MComponentisedObject::FACET_TYPE:
      return applyFunction(Aztec::MScene::getGlobalScene(), "Select Invserse", allObjectsCriteria, selectInverseFaceFunc);
    default:
      return 0;
    }
  }

  /**
   * For each object that is selected, the previous sibling of that object is selected, 
   * and the original object deselected. If an object has no parent, then it 
   * stays selected.
   */
  static int selectObjectAllChildren(const StringVector &args, std::string &result) {
    return applyFunction(Aztec::MScene::getGlobalScene(), "Select All Children", ifSelectedCriteria, selectAllChildrenFunc);
  }



  /// This calls a specific member function if we get a successful cast to a SelectingView. If that fails, we then call the default function instead.
  template <typename SpecialFunc, typename DefaultFunc>
  static int doSpecialSelection(SpecialFunc spec, DefaultFunc def, const StringVector &args, std::string &result)
  {
    AztecViewPtr curView = AztecViewManager::getCurrentView();
    SelectingView *selectionView = AZTEC_CAST(SelectingView, curView);

    bool selectResult = false;

    if (selectionView != NULL) {
      selectResult = (selectionView->*spec)();
    }

    // if we haven't had a successful call to the special function, call the standard one.
    if (!selectResult) {
      return def(args, result);
    } else {
      return FunctionManager::SUCCEED;
    }
  }


  /// If we have a view which has special selection implemented then we use that, otherwise we use the normal scene object selection.
  static int selectParent(const StringVector &args, std::string &result) {
    return doSpecialSelection(&SelectingView::selectParent, selectObjectParent, args, result);
  }

  /// If we have a view which has special selection implemented then we use that, otherwise we use the normal scene object selection.
  static int selectChild(const StringVector &args, std::string &result) {
    return doSpecialSelection(&SelectingView::selectChild, selectObjectChild, args, result);
  }
  
  /// If we have a view which has special selection implemented then we use that, otherwise we use the normal scene object selection.
  static int selectSiblingNext(const StringVector &args, std::string &result) {
    return doSpecialSelection(&SelectingView::selectSiblingNext, selectObjectSiblingNext, args, result);
  }
  
  /// If we have a view which has special selection implemented then we use that, otherwise we use the normal scene object selection.
  static int selectSiblingPrev(const StringVector &args, std::string &result) {
    return doSpecialSelection(&SelectingView::selectSiblingPrev, selectObjectSiblingPrev, args, result);
  }
  
  /// If we have a view which has special selection implemented then we use that, otherwise we use the normal scene object selection.
  static int selectNone(const StringVector &args, std::string &result) {
    return doSpecialSelection(&SelectingView::selectNone, selectObjectNone, args, result);
  }
  
  /// If we have a view which has special selection implemented then we use that, otherwise we use the normal scene object selection.
  static int selectAll(const StringVector &args, std::string &result) {
    return doSpecialSelection(&SelectingView::selectAll, selectObjectAll, args, result);
  }
  
  /// If we have a view which has special selection implemented then we use that, otherwise we use the normal scene object selection.
  static int selectInverse(const StringVector &args, std::string &result) {
    return doSpecialSelection(&SelectingView::selectInverse, selectObjectInverse, args, result);
  }

  /// If we have a view which has special selection implemented then we use that, otherwise we use the normal scene object selection.
  static int selectAllChildren(const StringVector &args, std::string &result) {
    return doSpecialSelection(&SelectingView::selectAllChildren, selectObjectAllChildren, args, result);
  }


  static int selectComponentsConnected(const StringVector &args, std::string &result) {
    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE) {
      Aztec::MSystemManager::getInstance()->logOutput("Error: selectComponentsConnected() - Should be in a component mode (points, faces etc) to use this function");
    }

    // iterate over all the selected objects that are in component mode.
    Aztec::MBaseObjectPtr obj;
    Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();
    scene->getObjectList()->beginIteration();
    while (( obj = scene->getObjectList()->getNext() ) != NULL ) {
      
      Aztec::MComponentisedObjectPtr compObj = Aztec::getComponentObject(obj);
      
      if (compObj != NULL) {
        compObj->flagConnected(Aztec::MUIManager::getComponentMode());
      }
    }
    scene->getObjectList()->endIteration();
    
    AztecViewManager::redrawAllViews();
  
    return FunctionManager::SUCCEED;
  }

  bool edgeEqual(int a1, int b1, int a2, int b2) {
    return ( (a1 == a2 && b1 == b2) || (a1 == b2 && b1 == a2) );
  }

  static void doMarkLoop(const Aztec::MMeshPtr &mesh, int triIndex, int edgeIndex, int startFrom, int startTo, Aztec::AztecFlags flag) {

    // first check to see if we are doign open edges.
    std::vector<int> trisOnEdge;
    mesh->getTrianglesWithEdge(startFrom, startTo, trisOnEdge);

    bool openEdgesOnly = (trisOnEdge.size() == 1);


    int vertFrom = startFrom;
    int vertTo = startTo;

    bool keepGoing = true;

    while (keepGoing) {
      mesh->setTriangleEdgeFlag(triIndex, edgeIndex, flag);

      Aztec::MVector3 vertNormal = mesh->getVertexNormal(vertTo);
      Aztec::MVector3 ab = mesh->getVertexPosition(vertTo) - mesh->getVertexPosition(vertFrom);
      float dot = (ab * vertNormal);
      Aztec::MVector3 abPlane = ab - vertNormal * dot;
      abPlane = ab;
      abPlane.normalize();

      // now find the vertices connecting with the to vertex, and find the 
      // one closest to our direction when all the vectors are mapped onto 
      // the plane defined by the vertex normal.
      std::vector<int> connectingVerts;
      mesh->getVerticesConnecting(vertTo, connectingVerts);

      int bestIndex = -1;
      // if we do not have 4 vertices, then stop
      if (openEdgesOnly) {
        // find another open edge that isn't the current one.
        for (int fromIndex = 0; fromIndex < connectingVerts.size(); ++fromIndex) {
          if (connectingVerts[fromIndex] != vertFrom) {
            mesh->getTrianglesWithEdge(vertTo, connectingVerts[fromIndex], trisOnEdge);
            if (trisOnEdge.size() == 1) {
              bestIndex = connectingVerts[fromIndex];
              break;
            }
          }
        }

      } else {
        if (connectingVerts.size() == 4) {
          // find where our from vertex is.
          int fromIndex = 0;
          for (fromIndex = 0; fromIndex < connectingVerts.size(); ++fromIndex) {
            if (connectingVerts[fromIndex] == vertFrom) {
              break;
            }
          }

          // make sure our from vertex is always found.
          assert(fromIndex != connectingVerts.size());

          bestIndex = connectingVerts[ (fromIndex + 2) % 4 ];
        }
      }

      // if we have a best index that is reasonable then use it, otherwise stop the looping
      if (bestIndex != -1) {
        // change our new vertFrom and vertTo  
        vertFrom = vertTo;
        vertTo = bestIndex;

        // now update our tri and edge index.
        std::vector<int> tris;
        mesh->getTrianglesWithEdge(vertFrom, vertTo, tris);

        // if we have no tris, something is seriously wrong.
        assert(tris.size() != 0);

        edgeIndex = mesh->getTriangleEdge(tris[0], vertFrom, vertTo);

        assert(edgeIndex != -1);

        triIndex = tris[0];

      } else {
        // we have no more edges to visit, break out.
        break;
      }

      if ((mesh->isTriangleEdgeFlagged(triIndex, edgeIndex, flag) ||
          edgeEqual(startFrom, startTo, vertFrom, vertTo))) 
      {
        break;
      }

    }
    
  }

  static void markLoop(const Aztec::MMeshPtr &mesh, int triIndex, int edgeIndex, Aztec::AztecFlags flag) {
    int startFrom, startTo;
    mesh->getVertsFromEdge(triIndex, edgeIndex, &startFrom, &startTo);

    doMarkLoop(mesh, triIndex, edgeIndex, startFrom, startTo, flag);
    doMarkLoop(mesh, triIndex, edgeIndex, startTo, startFrom, flag);
  }

  
  static void doMarkRing(const Aztec::MMeshPtr &mesh, int startFrom, int startTo, Aztec::AztecFlags flag) {

    Aztec::MSystemManager::getInstance()->logOutput("In doMarkRing: ");
    int vertFrom = startFrom;
    int vertTo = startTo;

    bool keepGoing = true;
    std::vector<int> p;

    while (keepGoing) {

      keepGoing = false;

      if (mesh->isEdgeFlagged(vertFrom, vertTo, EDGE_VISIBLE)) {
        mesh->setEdgeFlag(vertFrom, vertTo, flag);
        if (mesh->getPolygonVerticesFromEdge(vertFrom, vertTo, true, p)) {

          Aztec::MSystemManager::getInstance()->logOutput("polygon from edge (%i)", p.size());

          for (int i = 0; i < p.size(); ++i) {
            Aztec::MSystemManager::getInstance()->logOutput("  %i", p[i]);
          }

          // make sure we have 4 triangles. 
          if (p.size() == 4) {
            // find the index of our two points.
            int i1 = std::distance(p.begin(), std::find(p.begin(), p.end(), vertFrom));
            int i2 = std::distance(p.begin(), std::find(p.begin(), p.end(), vertTo));
            
            assert(i1 != i2);
            int dir = 0;
            
            dir = (i1 < i2) ? 1 : (p.size() - 1);
            
            vertTo = p[ (i1 + 2) % 4 ];
            vertFrom = p[ (i1 + 2 + dir) % 4];
            
            keepGoing = !mesh->isEdgeFlagged(vertFrom, vertTo, flag);
          }
        }
      }
    }
    
    Aztec::MSystemManager::getInstance()->logOutput("End doMarkRing");
  }

  static void markRing(const Aztec::MMeshPtr &mesh, int triIndex, int edgeIndex, Aztec::AztecFlags flag) {
    int startFrom, startTo;
    mesh->getVertsFromEdge(triIndex, edgeIndex, &startFrom, &startTo);

    doMarkRing(mesh, startFrom, startTo, flag);
    doMarkRing(mesh, startTo, startFrom, flag);
  }

  template <typename FUNC>
  class MeshFunction {
    public:
      MeshFunction(FUNC f) : func(f) { }
      MeshFunction(const MeshFunction &src) : func(src.func) { }

      void operator()(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
        Aztec::MMeshPtr mesh = Aztec::getMeshObject(node->getObject());

        if (mesh != NULL) {
          func(mesh);
        }
      }

      FUNC func;
  };

  template <typename FUNC, typename CONDITION>
  static void forEachMesh(FUNC f, CONDITION condition) {
    MeshFunction<FUNC> meshFunc(f);
    applyToObjects(Aztec::MScene::getGlobalScene(), Aztec::MScene::getGlobalScene()->getObjectList(), condition, meshFunc);
  }

  class ComponentsSelectedCriteria {
  public:
    ComponentsSelectedCriteria(Aztec::AztecFlags t) : componentType(t) { }
    ComponentsSelectedCriteria(const ComponentsSelectedCriteria &src) : componentType(src.componentType) { }

    inline bool operator()(const Aztec::MComponentisedObjectPtr &compObj) {
      if (compObj->getComponentMode() != componentType ||
          !compObj->hasComponentsFlagged(componentType) ||
          done.find(compObj) != done.end()) 
      {
        return false;
      } else {
        done.insert(compObj);
        return true;
      }
    }

    inline bool operator()(const Aztec::MTreeObjectNodePtr &node) {
      Aztec::MComponentisedObjectPtr compObj = Aztec::getComponentObject(node->getObject());
      return (compObj != NULL && (*this)(compObj));
    }

    std::set<Aztec::MComponentisedObjectPtr> done;
    const Aztec::AztecFlags componentType;
  };

  template <typename FUNC>
  class EdgeSelector {
  public:
    FUNC f;

    EdgeSelector(FUNC func, int *counter) : f(func), objectCount(counter) { }
    EdgeSelector(const EdgeSelector &src) : f(src.f), objectCount(src.objectCount) { }

    void operator()(const Aztec::MMeshPtr &mesh) {
      ++(*objectCount);

      // clear out any temp flags we might have.
      for (int triindex = 0; triindex < mesh->getNumTris(); ++triindex) {
        mesh->unsetTriangleEdgeFlag(triindex, 0, EDGE_FLAGFORCHANGE);
        mesh->unsetTriangleEdgeFlag(triindex, 1, EDGE_FLAGFORCHANGE);
        mesh->unsetTriangleEdgeFlag(triindex, 2, EDGE_FLAGFORCHANGE);
      }

      // okay, now go through the edges on the mesh, and find out where the 
      // connected ones are. Because we only deal with polygons, we ignore 
      // hidden edges.
      for (int triindex = 0; triindex < mesh->getNumTris(); ++triindex) {
        for (int edge = 0; edge < 3; ++edge) {
          if (mesh->isTriangleEdgeFlagged(triindex, edge, EDGE_VISIBLE) &&
              mesh->isTriangleEdgeFlagged(triindex, edge, EDGE_SELECTED)) {
            // we have a selected edge, try and construct a loop around it

            f(mesh, triindex, edge, EDGE_FLAGFORCHANGE);
          }
        }
      }


      // now loop over all the edges, and convert the flags into a selection
      for (int triindex = 0; triindex < mesh->getNumTris(); ++triindex) {
        for (int edge = 0; edge < 3; ++edge) {
          if (mesh->isTriangleEdgeFlagged(triindex, edge, EDGE_FLAGFORCHANGE)) {
            mesh->setTriangleEdgeFlag(triindex, edge, EDGE_SELECTED);
          }
        }
      }

    }

    int *objectCount;
  };

  template <typename FUNC>
  static int doSelectEdgeFunc(FUNC f) {
    // iterate over all the selected objects that are in component mode.
    Aztec::MBaseObjectPtr obj;
    Aztec::MScenePtr scene = Aztec::MScene::getGlobalScene();
    int objectCount = 0;
    EdgeSelector<FUNC> edgeSelector(f, &objectCount);

    forEachMesh(edgeSelector, ComponentsSelectedCriteria(Aztec::MComponentisedObject::EDGE_TYPE));

    if (objectCount == 0) {
      Aztec::MSystemManager::getInstance()->logOutput("Warning: selectEdgeLoop() - No edges were selected, so we couldn't do anything!");
    } else {
      AztecViewManager::redrawAllViews();
    }
  
    return FunctionManager::SUCCEED;
  }

  static int selectEdgeLoop(const StringVector &args, std::string &result) {
    return doSelectEdgeFunc(markLoop);

  }

  static int selectEdgeRing(const StringVector &args, std::string &result) {
    return doSelectEdgeFunc(markRing);

  }

  static void selectPoints(const Aztec::MMeshPtr &mesh, int triIndex, int edgeIndex, Aztec::AztecFlags flag) {
    int a, b;
    mesh->getVertsFromEdge(triIndex, edgeIndex, &a, &b);
    mesh->setVertexFlag(a, VERTEX_SELECTED);
    mesh->setVertexFlag(b, VERTEX_SELECTED);
  }

  void doEdgeToPoints(const Aztec::MMeshPtr &mesh) {
    for (int i = 0; i < mesh->getNumTris(); ++i) {
      for (int e = 0; e < 3; ++e) {
        if (mesh->isTriangleEdgeFlagged(i, e, EDGE_SELECTED)) {
          mesh->setVertexFlag(mesh->getTriangleVertex(i, e), VERTEX_SELECTED);
          mesh->setVertexFlag(mesh->getTriangleVertex(i, (e+1)%3), VERTEX_SELECTED);
        }
      }
    }
  }

  void doFaceToPoints(const Aztec::MMeshPtr &mesh) {
    for (int i = 0; i < mesh->getNumTris(); ++i) {
      if (mesh->isTriangleFlagged(i, TRIANGLE_SELECTED)) {
        mesh->setVertexFlag(mesh->getTriangleVertex(i, 0), VERTEX_SELECTED);
        mesh->setVertexFlag(mesh->getTriangleVertex(i, 1), VERTEX_SELECTED);
        mesh->setVertexFlag(mesh->getTriangleVertex(i, 2), VERTEX_SELECTED);
      }
    }
  }

  void doObjectToPoints(const Aztec::MMeshPtr &mesh) {
    mesh->setVertexFlags(VERTEX_SELECTED);
  }

  void doObjectToEdges(const Aztec::MMeshPtr &mesh) {
    for (int tri = 0; tri < mesh->getNumTris(); ++tri) {
      for (int edge = 0; edge < 3; ++edge) {
        if (mesh->isTriangleEdgeFlagged(tri, edge, EDGE_VISIBLE)) {
          mesh->setTriangleEdgeFlag(tri, edge, EDGE_SELECTED);
        }
      }
    }
  }

  void doObjectToFaces(const Aztec::MMeshPtr &mesh) {
    mesh->setTriangleFlags(TRIANGLE_SELECTED);
  }

  void doPointsToEdges(const Aztec::MMeshPtr &mesh) {
    for (int i = 0; i < mesh->getNumTris(); ++i) {
      for (int e = 0; e < 3; ++e) {
        if (mesh->isTriangleEdgeFlagged(i, e, EDGE_VISIBLE) &&
            mesh->isVertexFlagged(mesh->getTriangleVertex(i, e), VERTEX_SELECTED) &&
            mesh->isVertexFlagged(mesh->getTriangleVertex(i, (e+1)%3), VERTEX_SELECTED)) 
        {
          mesh->setTriangleEdgeFlag(i, e, EDGE_SELECTED);
        }
      }
    }
  }

  void doFacesToEdges(const Aztec::MMeshPtr &mesh) {
    for (int i = 0; i < mesh->getNumTris(); ++i) {
      if (mesh->isTriangleFlagged(i, TRIANGLE_SELECTED)) {
        for (int e = 0; e < 3; ++e) {
          if (mesh->isTriangleEdgeFlagged(i, e, EDGE_VISIBLE)) {
            mesh->setTriangleEdgeFlag(i, e, EDGE_SELECTED);
          }
        }
      }
    }
  }

  void doPointsToFaces(const Aztec::MMeshPtr &mesh) {
    for (int i = 0; i < mesh->getNumTris(); ++i) {
      if (mesh->isVertexFlagged( mesh->getTriangleVertex(i, 0), VERTEX_SELECTED) &&
          mesh->isVertexFlagged( mesh->getTriangleVertex(i, 1), VERTEX_SELECTED) &&
          mesh->isVertexFlagged( mesh->getTriangleVertex(i, 2), VERTEX_SELECTED) )
      {
        mesh->flagPolygon(i, TRIANGLE_SELECTED);
      }
    }
  }

  void doEdgesToFaces(const Aztec::MMeshPtr &mesh) {
    for (int i = 0; i < mesh->getNumTris(); ++i) {
      for (int e = 0; e < 3; ++e) {
        if (mesh->isTriangleEdgeFlagged(i, e, EDGE_SELECTED)) {
          mesh->flagPolygon(i, TRIANGLE_SELECTED);
        }
      }
    }
  }


  /**
   * This converts whatever selection we have at the moment, and changes it 
   * so all points to do with the current selection are selected. It also 
   * changes the current selection mode to point mode.
   */
  static int selectConvertToPoints(const StringVector &args, std::string &result) {
    int res = FunctionManager::FAIL;

    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::EDGE_TYPE) {
      forEachMesh(doEdgeToPoints, ComponentsSelectedCriteria(Aztec::MComponentisedObject::EDGE_TYPE));
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::FACET_TYPE) {
      forEachMesh(doFaceToPoints, ComponentsSelectedCriteria(Aztec::MComponentisedObject::FACET_TYPE));
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE) {
      forEachMesh(doObjectToPoints, ifSelectedCriteria);
    }

    Aztec::MScriptInterpreter::getInstance()->ExecuteScript("Scene.componentModeSet('point')");

    return res;
  }

  /**
   * This converts whatever selection we have at the moment, and changes it 
   * so all points to do with the current selection are selected. It also 
   * changes the current selection mode to point mode.
   */
  static int selectConvertToEdges(const StringVector &args, std::string &result) {
    int res = FunctionManager::FAIL;

    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::POINT_TYPE) {
      forEachMesh(doPointsToEdges, ComponentsSelectedCriteria(Aztec::MComponentisedObject::POINT_TYPE));
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::FACET_TYPE) {
      forEachMesh(doFacesToEdges, ComponentsSelectedCriteria(Aztec::MComponentisedObject::FACET_TYPE));
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE) {
      forEachMesh(doObjectToEdges, ifSelectedCriteria);
    }

    Aztec::MScriptInterpreter::getInstance()->ExecuteScript("Scene.componentModeSet('edge')");

    return res;
  }

  /**
   * This converts whatever selection we have at the moment, and changes it 
   * so all points to do with the current selection are selected. It also 
   * changes the current selection mode to point mode.
   */
  static int selectConvertToFaces(const StringVector &args, std::string &result) {
    int res = FunctionManager::FAIL;

    if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::POINT_TYPE) {
      forEachMesh(doPointsToFaces, ComponentsSelectedCriteria(Aztec::MComponentisedObject::POINT_TYPE));
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::EDGE_TYPE) {
      forEachMesh(doEdgesToFaces, ComponentsSelectedCriteria(Aztec::MComponentisedObject::EDGE_TYPE));
    } else if (Aztec::MUIManager::getComponentMode() == Aztec::MComponentisedObject::OBJECT_TYPE) {
      forEachMesh(doObjectToFaces, ifSelectedCriteria);
    }

    Aztec::MScriptInterpreter::getInstance()->ExecuteScript("Scene.componentModeSet('facet')");

    return res;
  }

  void doShrink(const Aztec::MComponentisedObjectPtr &compObj) {
    compObj->shrinkFlagged(Aztec::MUIManager::getComponentMode()); 
  }


  template <typename FUNC>
  class ComponentObjectFunction {
    public:
      ComponentObjectFunction(FUNC f) : func(f) { }
      ComponentObjectFunction(const ComponentObjectFunction &src) : func(src.func) { }

      void operator()(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
        Aztec::MComponentisedObjectPtr compObj = Aztec::getComponentObject(node->getObject());

        if (compObj != NULL) {
          func(compObj);
        }
      }

      FUNC func;
  };

  template <typename FUNC, typename CONDITION>
  static void forEachComponentObject(FUNC f, CONDITION condition) {
    ComponentObjectFunction<FUNC> compFunc(f);
    applyToObjects(Aztec::MScene::getGlobalScene(), Aztec::MScene::getGlobalScene()->getObjectList(), condition, compFunc);
  }

  static int selectShrink(const StringVector &args, std::string &result) {
    forEachComponentObject(doShrink, ComponentsSelectedCriteria(Aztec::MUIManager::getComponentMode()));
    return FunctionManager::SUCCEED;
  }

  static const char *SELECT_CATEGORY = "Selection";

  void registerSelectionFunctions(FunctionManager &man) {

    // These functions behave differently depending on what view is selected.
    // If the view doesn't define what to do withit, they just default back 
    // to the 3D object operation.
    man.registerFunction("selectParent", selectParent, SELECT_CATEGORY, "Select Parent");
    man.registerFunction("selectChild", selectChild, SELECT_CATEGORY, "Select Child");
    man.registerFunction("selectSiblingNext", selectSiblingNext, SELECT_CATEGORY, "Select the next Sibling");
    man.registerFunction("selectSiblingPrev", selectSiblingPrev, SELECT_CATEGORY, "Select the previous Sibling");
    man.registerFunction("selectNone", selectNone, SELECT_CATEGORY, "Deselect all objects or components");
    man.registerFunction("selectAll", selectAll, SELECT_CATEGORY, "Select all objects or components");
    man.registerFunction("selectInvserse", selectInverse, SELECT_CATEGORY, "Invert the current selection");
    man.registerFunction("selectAllChildren", selectAllChildren, SELECT_CATEGORY, "Select all the children");

    // these functions always operate on 3d objects.
    man.registerFunction("selectObjectParent", selectObjectParent);
    man.registerFunction("selectObjectChild", selectObjectChild);
    man.registerFunction("selectObjectSiblingNext", selectObjectSiblingNext);
    man.registerFunction("selectObjectSiblingPrev", selectObjectSiblingPrev);
    man.registerFunction("selectObjectNone", selectObjectNone);
    man.registerFunction("selectObjectAll", selectObjectAll);
    man.registerFunction("selectObjectInvserse", selectObjectInverse);
    man.registerFunction("selectObjectAllChildren", selectObjectAllChildren);

    // component selection functions
    man.registerFunction("selectComponentsConnected", selectComponentsConnected, SELECT_CATEGORY, "Select connected components");

    man.registerFunction("selectEdgeLoop", selectEdgeLoop, SELECT_CATEGORY, "Edge loop");
    man.registerFunction("selectEdgeRing", selectEdgeRing, SELECT_CATEGORY, "Edge ring");

    man.registerFunction("selectConvertToPoints", selectConvertToPoints, SELECT_CATEGORY, "Covnert the selection to Points");
    man.registerFunction("selectConvertToEdges", selectConvertToEdges, SELECT_CATEGORY, "Convert the selection to Edges");
    man.registerFunction("selectConvertToFaces", selectConvertToFaces, SELECT_CATEGORY, "Convert the select to Faces");

    man.registerFunction("selectShrink", selectShrink, SELECT_CATEGORY, "Shrink the current selection");
  }


}

