#ifndef MButton_Header
#define MButton_Header

#include "MRefCountedPtr.h"

namespace Aztec {

  class MButton;
  typedef MRefCountedPtr<MButton> MButtonPtr;

}

#include <MListener.h>

#include <gui/MComponent.h>
#include <gui/MLayoutManager.h>

#include <string>

#include <MImages.h>

namespace Aztec {

  class AZTECGUI_EXPORT MButtonListener : public MRefCountedObject {
  public:
    virtual void onClick(const MButtonPtr &button) = 0;
  };

  typedef MRefCountedPtr<MButtonListener> MButtonListenerPtr;



  /**
   * This is a simple button control, which allows the user to click on it. 
   * Each buttons has a caption and a command associated with it. The caption 
   * is the text that is displayed on the screen. The command is the internal 
   * text used to identify what the button can do. Buttons also have the 
   * option of being in an up or down state, allowing an application to have 
   * on/off like states for buttons.
   * 
   * You can use the button in two ways:
   * <UL>
   * <LI> new a normal MButton, and add it to a container. Then in the top level MWindow class, override the onCommand() event handler to respond to the button clicking.
   * <LI> Create a class that derives from MButton, and override the onClick() method. This is called before any of MWindow onCommand() methods are called. If the message is handled (i.e. the onClick() method returns true, then the onCommand() event handler is not called at all.   
   * </UL>
   */
  class AZTECGUI_EXPORT IMButton : public virtual IMComponent {
  public:
    /// Sets the caption to the given text. This will cause an update on the screen, and possible a layout of the parent container.
    virtual void setCaption(const std::string &caption) = 0;
    /// Gets the caption of this button.
    virtual std::string getCaption() = 0;

    /// Sets the command associated with this button.
    virtual void setCommand(const std::string &command) = 0;
    /// Gets the command associated with this butotn.
    virtual std::string getCommand() = 0;

    /**
     * Called when the user clicks the button. Returns true if the event was 
     * handled, false if it wasn't.
     */
    virtual bool onClick() = 0;

    virtual void setButtonDown(bool buttonDown) = 0;

    virtual bool getButtonDown() = 0;

    virtual void setImage(const Aztec::MImagePtr &image) = 0;

    virtual void addListener(const MButtonListenerPtr &listener) = 0;
  };

}


/**
 * The following header is included via the AZTEC_GUI
 * variable that should be defined for each platform
 * and used to specify an include directory
 * (ie. ../AztecGUICommon/include/gui/$(AZTEC_GUI)
 * when compiling this lib
 */
#include <MButtonImpl.h>


#endif

