#ifndef MMouseEvent_Header
#define MMouseEvent_Header

#include <gui/MGuiBase.h>

namespace Aztec {

  class MMouseEvent;
  typedef MRefCountedPtr<MMouseEvent> MMouseEventPtr;

  class MComponent;
  typedef MRefCountedPtr<MComponent> MComponentPtr;

}

#include <gui/MPoint2D.h>
#include <gui/MShiftState.h>

namespace Aztec {

  /**
   * The mouse event class represents the occurrance of some event happening 
   * with the mouse. It could be a mouse button press or release, or the mouse 
   * moving inside a control. The coordinates of the mouse event are the screen 
   * coordinates in pixels relative to the top left of the component the event 
   * occurrs in.
   */
  class AZTECGUI_EXPORT MMouseEvent {
  public:
    typedef enum { LBUTTON_DOWN, LBUTTON_UP, LBUTTON_DBLCLK, 
                   MBUTTON_DOWN, MBUTTON_UP, MBUTTON_DBLCLK,
                   RBUTTON_DOWN, RBUTTON_UP, RBUTTON_DBLCLK,
                   MOVE, WHEEL_UP, WHEEL_DOWN } MouseMessageType;


    typedef MMouseEventPtr Ptr;

    MMouseEvent(MouseMessageType type, unsigned long time, MPoint2D pos, const MShiftState &state, const MComponentPtr &component);
    MMouseEvent(const MMouseEvent &src) :
        type(src.type), 
        time(src.time) , 
        m_Pos(src.m_Pos), 
        m_State(src.m_State), 
        m_Component(src.m_Component) { }

    /**
     * Gets the type of event that occurred. For example, a mouse button press
     * or release, or a mouse movement.
     */
    MouseMessageType getType() const;

    /**
     * Gets the x coordinate of the mouse event relative to the control 
     * it occurred in.
     */
    int getX() const;

    /**
     * Gets the y coordinate of the mouse event relative to the control 
     * it occurred in.
     */
    int getY() const;

    /**
     * Gets the position the event occurred at relative to the control it 
     * occurred in.
     */
    MPoint2D getPos() const;

    /**
     * Returns true if the left mouse button is down.
     */
    bool isLeftButtonDown() const;
    /**
     * Returns true if the middle mouse button is down.
     */
    bool isMiddleButtonDown() const;
    /**
     * Returns true if the right mouse button is down.
     */
    bool isRightButtonDown() const;

    /**
     * Gets the state of various keys on the keyboard, such as contrl, shift, 
     * or alt. This is often use to perform different actinos when keys and 
     * mouse buttons are pressed in combination
     */
    MShiftState getShiftState() const;

    bool isShiftDown() const;
    bool isControlDown() const;
    bool isAltDown() const;

    /**
     * This gets the component that the event occurred in.
     */
    MComponentPtr getComponent() const;

  protected:
    /// The type of the mouse message.
    MouseMessageType type;

    /// The time the mouse event occurred.
    unsigned long time;

    /// Where the mouse event happened, relative to the control it happened in.
    MPoint2D m_Pos;


    /// What other keyboard keys are pressed at the same time.
    MShiftState m_State;

    /// The component that the event occurred in.
    MComponentPtr m_Component;
  };


  class AZTECGUI_EXPORT MMouseEventListener : public virtual IMGuiBase {
  public:
    virtual void onMouseEvent(const MComponentPtr &component, const MMouseEvent &event) = 0;
  };

  typedef MRefCountedPtr<MMouseEventListener> MMouseEventListenerPtr;
}

#include <gui/MComponent.h>


#endif

