#include <AztecGUICommonPCH.h>

#include <gui/win32/MAppImpl.h>

#include <gui/MButton.h>
#include <gui/MTextField.h>
#include <gui/MTreeControl.h>

#include <gui/win32/MTextFieldImpl.h>
#include <gui/win32/MSliderImpl.h>
#include <gui/win32/MCheckboxImpl.h>

#include <set>

#include <scripting/MScriptInterp.h>
#include <gui/scripting/AztecGUIScripting.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

using std::string;
using std::map;
using std::iterator;

namespace Aztec {

  LRESULT CALLBACK MAppWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);

  // this is the global set of application instances.
  std::set<IMApp::Ptr> g_allApps;

  MApp *g_CurrentApp;
  

  MApp::MApp() {
    lastMouseMove = NULL;

    // add this instance into the set of global instances.
    g_allApps.insert(this);
    g_CurrentApp = this;

    // add in the standard window's class atoms.
    addClass("BUTTON", ::FindAtom("BUTTON"));
    addClass("COMBOBOX", ::FindAtom("COMBOBOX"));
    addClass("EDIT", ::FindAtom("EDIT"));
    addClass("LISTBOX", ::FindAtom("LISTBOX"));
    addClass("MDICLIENT", ::FindAtom("MDICLIENT"));
    addClass("RichEdit", ::FindAtom("RichEdit"));
    addClass("RICHEDIT_CLASS", ::FindAtom("RICHEDIT_CLASS"));
    addClass("SCROLLBAR", ::FindAtom("SCROLLBAR"));
    addClass("STATIC", ::FindAtom("STATIC"));

    initGUIClasses(MScriptInterpreter::getInstance()->GetScriptContext());
  }

  MApp::~MApp() {
    // We clear out the window map before the menu one so that when components that have menus get cleaned up, the menus have a place to unregister themselves from it.
    hWndMap.clear();
    allWindows.clear();

    menuItemMap.clear();
  }

  MApp* MApp::getInstance() {
    return g_CurrentApp;
  }

  void MApp::win32Init(const string &appName, HINSTANCE hInst) {
    m_Name = appName;
    m_hInst = hInst;
  }

  HINSTANCE MApp::getHInstance() {
    return m_hInst;
  }

  ATOM MApp::getWindowClass(const string &name) {
    ATOM result;

    std::map<string,ATOM>::iterator it;

    it = classMap.find(name);

    // if we have found a class by that name,
    // just return it.
    if (it != classMap.end()) {
      return (*it).second;
    }

    // if we haven't found it, we need to create it.
    WNDCLASSEX wcex;
    
    wcex.cbSize = sizeof(WNDCLASSEX); 
    
    wcex.style			= CS_HREDRAW | CS_VREDRAW;
    wcex.lpfnWndProc	= (WNDPROC)MAppWndProc;
    wcex.cbClsExtra		= 0;
    wcex.cbWndExtra		= 0;
    wcex.hInstance		= m_hInst;
    wcex.hIcon			= NULL;
    wcex.hCursor		= LoadCursor(NULL, IDC_ARROW);
    wcex.hbrBackground	= (HBRUSH)(COLOR_WINDOW+1);
    wcex.lpszMenuName	= NULL;
    wcex.lpszClassName	= name.c_str();
    wcex.hIconSm		= NULL;

    result = RegisterClassEx(&wcex);

    addClass(name, result);

    return result;
  }

  MComponentPtr MApp::getComponent(HWND hWnd) {
    HWNDComponentMap::iterator it;
    it = hWndMap.find(hWnd);

    if (it != hWndMap.end()) {
      return (*it).second;
    }

    return NULL;
  }


  static int windowCount = 1;

  void MApp::registerComponent(HWND hWnd, const MComponentPtr &component) {
    MWindowPtr window = AZTEC_CAST(MWindow, component);
    if (window != NULL) {
      allWindows.insert(std::make_pair(window, windowCount++));
    }
    hWndMap.insert(HWNDComponentMap::value_type(hWnd, component));
  }

  void MApp::unregisterComponent(HWND hWnd) {
    allWindows.erase(AZTEC_CAST(MWindow, hWndMap[hWnd]));
    hWndMap.erase(hWnd);
  }

  void MApp::registerMenu(MMenuItem *item) {
    menuItemMap[item->getID()] = item;
    menuItemMap[item->getOptionID()] = item;
  }

  void MApp::unregisterMenu(MMenuItem *item) {
    menuItemMap.erase(item->getID());
    menuItemMap.erase(item->getOptionID());
  }

  MMenuItem* MApp::getItemFromID(WORD id) {
    IDMenuItemMap::iterator it = menuItemMap.find(id);

    return (it != menuItemMap.end()) ? it->second : NULL;
  }

  bool MApp::preTranslateMessage(MSG *msg) {
    // we iterate over the applications and try to find 
    // an application which is using the given window.

    std::set<IMApp::Ptr>::iterator it;

    for (it = g_allApps.begin(); it != g_allApps.end(); ++it) {

      MComponentPtr component;
      MApp* app;

      app = AZTEC_CAST(MApp, *it);

      if (app == NULL) {
        continue;
      }

      component = app->getComponent(msg->hwnd);

      if (component == NULL) {
        continue;
      }

      // see if we have a mouse move message. If we do, keep track of the last component we were over.
      if (msg->message == WM_MOUSEMOVE) {
        lastMouseMove = &*component;
      }

      return component->preTranslateMessage(msg);

    }

    return false;
  };



  // MApp methods
  bool MApp::initApp() {
    return true;
  }

  int MApp::run() {
  	MSG msg;

    while (::GetMessage(&msg, NULL, 0, 0)) 
    {
//      if (!::TranslateAccelerator(msg.hwnd, hAccelTable, &msg)) 
      {
        if (!preTranslateMessage(&msg)) {
          ::TranslateMessage(&msg);
          ::DispatchMessage(&msg);
        }
      }
    }

  	return msg.wParam;
  }

  void MApp::cleanup() {
    MScriptInterpreter::cleanInstance();


    hWndMap.clear();
    allWindows.clear();

    menuItemMap.clear();
    
    // remove us from the global list.
    g_allApps.erase(this);

  
  }

  std::string MApp::getName() {
    return m_Name;
  }

  std::string MApp::getApplicationPath() {
    return appPath;
  }

  MWindowPtr MApp::getTopLevelWindow() {
    MWindowPtr top;
    int best = 1000000;

    for (std::map<MWindowPtr, int>::iterator it = allWindows.begin(); it != allWindows.end(); ++it) {
      if (it->second < best) {
        top = it->first;
        best = it->second;
      }
    }

    return top;
  }

  bool MApp::onCommand(const std::string &command, const MComponentPtr &component) {
    // do nothing.
    return false;
  }

  bool MApp::onMouseMove(const MMouseEvent &event) {
    return false;
  }

  bool MApp::onMousePressed(const MMouseEvent &event) {
    return false;
  }

  bool MApp::onMouseReleased(const MMouseEvent &event) {
    return false;
  }

  bool MApp::onKeyPressed(const MKeyEvent &event) {
    return false;
  }

  bool MApp::onKeyReleased(const MKeyEvent &event) {
    return false;
  }


  // protected methods

  bool MApp::addClass(const std::string &name, ATOM atom) {
    // if it is not a valid atom, we don't add it.
    if (atom == 0) {
      return false;
    }

    classMap.insert(std::map<std::string, ATOM>::value_type(name, atom));

    return true;
  }



  LRESULT CALLBACK MAppWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
  {
    // we iterate over the applications and try to find 
    // an application which is using the given window.

    std::set<IMApp::Ptr>::iterator it;

    for (it = g_allApps.begin(); it != g_allApps.end(); ++it) {

      MComponentPtr component;
      MApp* app;

      app = AZTEC_CAST(MApp, *it);

      if (app == NULL) {
        continue;
      }

      component = app->getComponent(hWnd);

      if (component == NULL) {
        continue;
      }

      LRESULT lresult = 0;

      bool handled = component->wndProc(message, wParam, lParam, &lresult);

      if (message == WM_COMMAND) {
        WORD wNotifyCode = HIWORD(wParam); // notification code 
        WORD wID = LOWORD(wParam);         // item, control, or accelerator identifier 
        HWND hwndCtl = (HWND) lParam;      // handle of control 
        
        MComponentPtr comp;
        MButtonPtr but;
        
        comp = AZTEC_CAST(MComponent, app->getComponent(hwndCtl));
        
        if (wNotifyCode == BN_CLICKED) {
          but = AZTEC_CAST(MButton, comp);
          if (but != NULL) {
            MContainerPtr parent = but->getParent();
            while (parent != NULL) {
              handled = parent->onCommand(but->getCommand(), but);
              if (handled) break;
              parent = parent->getParent();
            }
            if (!handled) {
              handled = app->onCommand(but->getCommand(), but);
            }
            
            but->fireClickListeners();
            
            if (!handled) {
              handled = but->onClick();
            }
            
          } else {
            MCheckboxPtr checkbox = AZTEC_CAST(MCheckbox, comp);
            if (checkbox != NULL) {
              handled |= checkbox->wndProc(message, wParam, lParam, &lresult);
            }
          }
        }
        
        if (wNotifyCode == EN_CHANGE) {
          HWND editHWnd = (HWND)lParam;
          MTextFieldPtr editBox = AZTEC_CAST(MTextField, app->getComponent(editHWnd));
          
          if (editBox != NULL) {
            char buf[1024];
            ::GetWindowText(editHWnd, buf, 1024);
            editBox->setInternalValue(buf);
            
            editBox->onChanged();
          }
          
          // If this came from a menu or button
        } else if (wNotifyCode == 0) {
          WORD menuID = wID;
          
          MMenuItem *item = app->getItemFromID(menuID);
          
          if (item != NULL) {
            if (item->getID() == menuID) {
              handled = app->onCommand(item->getCommand(), component);

              if (!handled) {
                item->doClick();
                handled = true;
              }
            } else if (item->getOptionID() == menuID) {
              handled = app->onCommand(item->getOptionCommand(), component);
            }


          }
        } else if (comp != NULL) {
          handled = comp->handleWMCommandNotify(wNotifyCode, wID);
        }
        
        
      }  else if (message == WM_NOTIFY) {
        NMHDR *notify = (NMHDR*)lParam;
        MComponentPtr ctrl = app->getComponent(notify->hwndFrom);
        if (ctrl != NULL) {
          handled = ctrl->handleWMNotify(wParam, lParam);
        }
        
      } else if (message == WM_CTLCOLOREDIT) {
        // this message is sent to the parent of an edit control to request colours.
        HDC editDC = (HDC)wParam;
        HWND editHWND = (HWND)lParam;

        MComponentPtr comp = app->getComponent(editHWND);
        MTextFieldPtr editBox = AZTEC_CAST(MTextField, comp);

        if (editBox != NULL) {
          if (editBox->getBackgroundBrush() != NULL) {
            ::SetBkMode(editDC, TRANSPARENT);
            return (LRESULT)editBox->getBackgroundBrush();
          }
        }
      } else if (message == WM_CTLCOLORSTATIC) {
        // this message is sent to the parent of an edit control to request colours.
        HDC editDC = (HDC)wParam;
        HWND hWnd = (HWND)lParam;

        MComponentPtr comp = app->getComponent(hWnd);
        
        if (comp != NULL) {
          if (comp->getBackgroundBrush() != NULL) {
            ::SetBkMode(editDC, TRANSPARENT);
            return (LRESULT)comp->getBackgroundBrush();
          }
        }
      } else if (message == WM_HSCROLL) {
        // check to see if we have hit a slider control.
        HWND sliderWnd = (HWND)lParam;

        MSliderPtr slider = AZTEC_CAST(MSlider, app->getComponent(sliderWnd));
        if (slider != NULL) {
          slider->wndProc(message, wParam, lParam, &lresult);
        }
      } else if (message == WM_MOUSEWHEEL) {
        if (app->lastMouseMove != NULL) {
          if (app->lastMouseMove->wndProc(message, wParam, lParam, &lresult)) {
            return TRUE;
          }
        }
      }

      if (!handled) {
        int wmId, wmEvent;
        PAINTSTRUCT ps;
        HDC hdc;
    
        switch (message) 
        {
        case WM_COMMAND:
          wmId    = LOWORD(wParam); 
          wmEvent = HIWORD(wParam); 
          // Parse the menu selections:
          switch (wmId)
          {
          default:
            return DefWindowProc(hWnd, message, wParam, lParam);
          }
          break;
          case WM_PAINT:
            hdc = BeginPaint(hWnd, &ps);
            // TODO: Add any drawing code here...
            RECT rt;
            GetClientRect(hWnd, &rt);
            EndPaint(hWnd, &ps);
            break;
          case WM_DESTROY:
            // if the destroy is from the top level window, then quit
            if (component == app->getTopLevelWindow()) {
              PostQuitMessage(0);
              break;
            }
          default:
            return DefWindowProc(hWnd, message, wParam, lParam);
        }
      } else {
        return lresult;
      }

    }

    return DefWindowProc(hWnd, message, wParam, lParam);
  }

}
