#include <AztecGUICommonPCH.h>

#include <tools/MToolManager.h>

#include <assert.h>

namespace AztecGUI {
  MToolManager::MToolManager() {
    m_TempTool = false;
  }

  MToolManager::~MToolManager() {
    m_Listeners.clear();
    toolMap.clear();
  }

  static MToolManager *globalToolManager = NULL;

  MToolManager* MToolManager::getInstance() {
    if (globalToolManager == NULL) {
      globalToolManager = new MToolManager;
    }
    return globalToolManager;
  }

  void MToolManager::cleanup() {
    if (globalToolManager != NULL) {
      delete globalToolManager;
      globalToolManager = NULL;
    }
  }

  void MToolManager::registerTool(const MToolTypePtr &tool) {
    registerTool(tool, tool->getName());
  }

  void MToolManager::registerTool(const MToolTypePtr &tool, const std::string &name) {
    registeredTools[name] = tool;
  }

  MToolTypePtr MToolManager::getToolByName(const std::string &name) {
    return registeredTools[name];
  }

  void MToolManager::setTool(const MToolTypePtr &newTool, const std::string &group) {
    if (newTool == NULL) {
      setTool(getDefault(group), group);
      return;
    }

    MToolTypePtr oldTool;

    ToolList &toolList = getToolList(group);

    if (toolList.size() > 0) {
      oldTool = toolList.front();
      toolList.pop_front();
    }
    toolList.push_front(newTool);

    if (toolList.size() > TOOLMAN_MAXNODES) {
      toolList.pop_back();
    }
  
    notifyListeners(oldTool, group);

    newTool->initialise();
  }

  void MToolManager::PushTool(const MToolTypePtr &newTool, const std::string &group) {
    if (newTool == NULL) {
      PushTool(getDefault(group), group);
      return;
    }

    bool nothingChanged = false;
    ToolList &toolList = getToolList(group);

    MToolTypePtr oldTool;
    if (toolList.size() > 0) {
      // if our old tool had the same name, get rid of the old one, and
      // replace it with the new one. This will prevent dupliates from 
      // filling up the tool list.
      if (toolList.front()->getName() == newTool->getName()) {
        toolList.pop_front();
        nothingChanged = true;
      }
    }

    if (toolList.size() > 0) {
      oldTool = toolList.front();
    }

    toolList.push_front(newTool);
    if (toolList.size() > TOOLMAN_MAXNODES) {
      toolList.pop_back();
    }
  
    MToolTypePtr curTool = GetTool(group);

    if (!nothingChanged) {
      // Set the "last tool used" button (ignore select, move, rotate and scale tools)
      if ((!curTool->Is("KToolSelect")) && (!curTool->Is("KToolMove")) &&
          (!curTool->Is("KToolRotate")) && (!curTool->Is("KToolScale")))
      {
        // Assign the tool to the button and set it's image
/*        if (g_MainDlg) {
          g_MainDlg->m_TopToolDlg.m_LastToolBut.SetActionName(curTool->getName().c_str());
          MStr imageName = "Buttons/";
          imageName += curTool->getName().c_str();
          imageName += ".bmp";
          imageName.Replace('/','\\');
          g_MainDlg->m_TopToolDlg.m_LastToolBut.SetImage(g_ProgSet.m_PrefsPath + imageName);
          g_MainDlg->m_TopToolDlg.m_LastToolBut.InvalidateRect(NULL);
        }*/
      }
  
      notifyListeners(oldTool, group);
    }

    curTool->initialise();
  }

  MToolTypePtr MToolManager::PopTool(const std::string &group) {
    ToolList &toolList = getToolList(group);

    MToolTypePtr oldTool = toolList.front();
    toolList.pop_front();

    MToolTypePtr curTool = GetTool(group);
    // Set the "last tool used" button (ignore select, move, rotate and scale tools)
    if ((!curTool->Is("KToolSelect")) && (!curTool->Is("KToolMove")) &&
        (!curTool->Is("KToolRotate")) && (!curTool->Is("KToolScale")))
    {
      // Assign the tool to the button and set it's image
/*      if (g_MainDlg) {
        g_MainDlg->m_TopToolDlg.m_LastToolBut.SetActionName(curTool->getName().c_str());

        MStr imageName = "Buttons/";
        imageName += curTool->getName().c_str();
        imageName += ".bmp";
        imageName.Replace('/','\\');
        g_MainDlg->m_TopToolDlg.m_LastToolBut.SetImage(g_ProgSet.m_PrefsPath + imageName);
        g_MainDlg->m_TopToolDlg.m_LastToolBut.InvalidateRect(NULL);
      }*/
    }
  
    notifyListeners(oldTool, group);
  
    curTool->initialise();

    return oldTool;
  }

  MToolTypePtr MToolManager::GetTool(const std::string &group) {
    ToolList &toolList = getToolList(group);
    MToolTypePtr Tool;

    if (toolList.size() > 0) {
      Tool = toolList.front();
    }

    if (Tool == NULL) {
      setTool(NULL, group);
      Tool = toolList.front();
    }
    return Tool;
  }

  MToolTypePtr MToolManager::GetTool(int index, const std::string &group) {
    ToolList &toolList = getToolList(group);

    if (index < 0 || index >= toolList.size()) {
      return NULL;
    }

    ToolList::iterator it = toolList.begin();
    std::advance(it, index);
    return (*it);
  }

  void MToolManager::setDefault(const MToolTypePtr &tool, const std::string &group) {
    defaultTools[group] = tool;
  }

  MToolTypePtr MToolManager::getDefault(const std::string &group) {
    MToolTypePtr def;
    def = defaultTools[group];

    if (def == NULL) {
      def = defaultTools[""];
    }

    assert(def != NULL);

    return def;
  }


  void MToolManager::setTemporary(const MToolTypePtr &Tool, const std::string &group)
  {
    if (m_TempTool == false) {
      m_TempTool = true;
      PushTool(Tool, group);
      return;
    }
  
    setTool(Tool, group);
  }

  void MToolManager::unsetTemporary(const std::string &group) {
    if (m_TempTool == NULL)
      return;
  
    PopTool(group);
    m_TempTool = false;
  }

  bool MToolManager::isTemporary(const std::string &group) {
    return m_TempTool;
  };

  void MToolManager::addListener(MToolChangeListenerPtr listener, const std::string &group) {
    m_Listeners.insert(listener);
  }

  void MToolManager::removeListener(MToolChangeListenerPtr listener, const std::string &group) {
    m_Listeners.erase(listener);
  }

  void MToolManager::removeAllListeners(const std::string &group) {
    m_Listeners.clear();
  }

  void MToolManager::notifyListeners(MToolTypePtr oldTool, const std::string &group) {
    std::set<MToolChangeListenerPtr>::iterator it;

    for (it = m_Listeners.begin(); it != m_Listeners.end(); ++it) {
      (*it)->onToolChange(group, oldTool);
    }
  }

  MToolManager::ToolList& MToolManager::getToolList(const std::string &group) {
    return toolMap[group];
  }
}