// LogFile.cpp: implementation of the MLogFile class.
//
//////////////////////////////////////////////////////////////////////
#include "StdAfx.h"

#include "LogFile.h"
#include "HelperFuncs.h"

#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>

#ifdef WIN32
#include <direct.h>
#else
#include <sys/stat.h>
#endif

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

//----------------------------------------------------------------------------------------
//  MLogFile
//----------------------------------------------------------------------------------------
namespace Aztec {

  int makeDirectory(const char *directory) {
#ifdef WIN32
    return mkdir(directory);
#else
    // Create a directory and give ourselves read/write/exec access
    // The exec part allows the directory to be searched in (by the owner of the dir only)
    return mkdir(directory, S_IRWXU);
#endif

  }

  void getDirectory(const char *path, char *directory) {
    
#ifdef WIN32
    char	Drive[_MAX_DRIVE], Dir[_MAX_DIR];
    _splitpath((LPCTSTR)path, Drive, Dir, NULL, NULL);
    strcpy(directory, Drive);
    strcat(directory, Dir);
#else
    // find the last / or 
    char *start, *end;
    strcpy(directory, path);
    start = end = directory;
    while (*end != 0) {
      ++end;
    }

    while (end > start && *end != '/') {
      --end;
    }
    *end = '\0';
#endif
  }

  MLogFile::MLogFile() {
    m_Flags = LOG_PREFIXTIME | LOG_24HRTIME | LOG_PREFIXDATE;
  }
  
  MLogFile::MLogFile(MLogFile &Src) {
    listeners = Src.listeners;
    m_Flags = Src.m_Flags;
  }
  
  MLogFile::~MLogFile() {
    
  }
  
  void MLogFile::writeString(const char *text) {	
    for (ListenerList::iterator it = listeners.begin(); it != listeners.end(); ++it) {
      (*it)->writeString(this, (text == NULL) ? "" : text);
    }
  }
  
  void MLogFile::writeString(const MStr &text) {
    writeString(text.c_str());
  }

  void MLogFile::writeString(const std::string &text) {
    writeString(text.c_str());
  }

  void MLogFile::writeLine(const char *text) {
    for (ListenerList::iterator it = listeners.begin(); it != listeners.end(); ++it) {
      (*it)->writeLine(this, (text == NULL) ? "" : text);
    }
  }

  void MLogFile::writeLine(const MStr &text) {
    writeLine(text.c_str());
  }

  void MLogFile::writeLine(const std::string &text) {
    writeLine(text.c_str());
  }

  class LogFileWriter : public MLogFileListener {
  private:
    std::string filename;
  public:
    LogFileWriter(const std::string &filename) 
      : filename(filename)
    {
    }

    inline std::string getFilename() const {
      return filename;
    }

    void writeString(MLogFile *logfile, const std::string &text) {
      // for a physical file with time stamps, writing out a 
      // single string is the same as writing out a whole line
      writeLine(logfile, text );
    }

    void writeLine(MLogFile *logfile, const std::string &text) {
      FILE *FileHandle = NULL;
      MTimeObj	   CurTime;
      MStr		TimeStr, DateStr, EntireLine;
      
      CurTime.GetCurrentTime();
      
      if (logfile->m_Flags & LOG_PREFIXDATE) {
        DateStr = CurTime.Format("%d/%m/%Y");
        DateStr += " ";
        EntireLine += DateStr;
      }
      
      if (logfile->m_Flags & LOG_PREFIXTIME) {
        if (logfile->m_Flags & LOG_24HRTIME)
          TimeStr = CurTime.Format("%H:%M:%S");
        else
          TimeStr = CurTime.Format("%I:%M:%S%p");
        
        TimeStr += " :  ";
        EntireLine += TimeStr;
      }
      
      EntireLine += text.c_str();
      
      EntireLine += "\n";
      
      char *temp;
      temp = EntireLine.GetBuffer(1);
      
      FileHandle = fopen(filename.c_str(), "a+t");

      if (FileHandle == NULL) {
        // Path not found, so create it. Assume that
        // it will require only one make dir
        
        int		Result;
        char buffer[512];
        
        getDirectory(filename.c_str(), buffer);
        Result = makeDirectory(buffer);
        if (Result == 0) {
          FileHandle = fopen(filename.c_str(), "a+t");
        }
      }

      if (FileHandle != NULL) {
        fwrite(temp, EntireLine.GetLength(), 1, FileHandle);
        fclose(FileHandle);
      }
    }
  };

  int MLogFile::addLogFile(const char *Text, bool clear) {
    if (Text == NULL) {
      return 0;
    }
    
    // only bother adding a new log filename if we have a unique one.
    for (unsigned int n = 0; n < listeners.size(); ++n) {
      LogFileWriter *writer = AZTEC_CAST(LogFileWriter, listeners[n]);

      if (writer != NULL) {
        if (MStr(writer->getFilename().c_str()).compareNoCase(Text) == 0) {
          return 0;
        }
      }
    }
    
    LogFileWriter* writer = new LogFileWriter(Text);
    FILE* fhandle;

    addListener(writer);

    // clear content
    if (clear) {
      fhandle = fopen(writer->getFilename().c_str(), "w+t");
      if (fhandle != 0l) fclose(fhandle);
    }
    
    return 1;
  }

  void MLogFile::addListener(const MLogFileListenerPtr &listener) {
    listeners.push_back(listener);
  }

  void MLogFile::removeAllListeners() {
    listeners.clear();
  }

}
