#include "StdAfx.h"



#include "MBaseObject.h"
#include "MListsTrees.h"
#include "MScene.h"
#include "MMesh.h"
#include "MAnimMesh.h"
#include "MMeshShape.h"
#include "MLight.h"
#include "MSystemManager.h"
#include "MComponentisedObject.h"
#include "MUIManager.h"
#include <animation/MIKController.h>
#include "MLight.h"
#include <MBone.h>
#include <mesh/MMeshCreator.h>

#include <stdlib.h>
#include <stdio.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include <algorithm>
#include <set>

#include "ModelGLConst.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


namespace Aztec {

  using namespace std;

  MScenePtr globalScene;

  MScenePtr MScene::getGlobalScene() {
    return globalScene;
  }

  void MScene::setGlobalScene(const MScenePtr &scene) {
    globalScene = scene;
  }


  //---------
  //  MScene
  //---------
  int GLDrawAxisIcon(float Size, float Alpha, float LineWidth, bool ArrowHeads, bool Select)
  {
    GLenum   OldLighting;
    int      OldFill[2];
    float    Num;
    float    OldLineWidth;
    
    glMatrixMode(GL_MODELVIEW);
    
    glGetFloatv(GL_LINE_WIDTH, &OldLineWidth);
    glGetIntegerv(GL_LIGHTING, (int *)&OldLighting);
    glGetIntegerv(GL_POLYGON_MODE, OldFill);
    
    glEnable(GL_DEPTH_TEST);
    glDisable(GL_CULL_FACE);
    glDisable(GL_LIGHTING);
    
    glLineWidth(LineWidth);
    glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    
    for (Num = 1; Num <= 4; Num++)
    {
      glPushMatrix();
      
      if (Num == 1)
      {
        glColor4f(1,0,0,Alpha);
      }
      if (Num == 2)
      {
        glColor4f(0,1,0,Alpha);
        glRotatef(90, 0,0,1);
      }
      if (Num == 3)
      {
        glColor4f(0,0,1,Alpha);
        glRotatef(-90, 0,1,0);
      }
      if (Num == 4)
      {
        glLineWidth(LineWidth);
        
        glDisable(GL_LINE_STIPPLE);
        
        glPopMatrix();
        
        break;
      }
      
      glBegin(GL_LINES);
      glVertex3f(0,0,0);
      glVertex3f((float)(0.7*Size),0,0  );
      glEnd();
      
      if (ArrowHeads)      // Draw funky arrow heads
      {
        //         glPushMatrix();
        //         glTranslatef(Size*0.7,0,0);
        //         DrawArrowHead(Size*0.3, Size*0.1);
        //         glPopMatrix();
      }
      else
      {
        // Draw X Y Z labels at the end of the sticks
        char  Str[5];
        
        Str[0] = 'X'+(char)(Num-1);
        glRasterPos3f(Size, 0,0);
        glListBase(1000);
        glCallLists(1, GL_UNSIGNED_BYTE, Str);
        
      }
      glPopMatrix();
    }
    
    glLineWidth(OldLineWidth);
    glPolygonMode(GL_FRONT, (GLenum)OldFill[0]);
    glPolygonMode(GL_BACK, (GLenum)OldFill[1]);
    
    if (OldLighting)
      glEnable(OldLighting);
    else 
      glDisable(OldLighting);
    
    return 0;
  }
  
  MScene::MScene() {
    m_FPS = 30; // 30 Frames Per Second.
    m_StartTime = 0;
    m_EndTime = 3600 * 2; // have the end time default to 2 secons

    m_DoObjectUpdates = 0;
    m_DoingUpdate = false;

    m_Objects = new MBaseObjectTree;
    
    m_SelectedObj = new MBaseObjectList;
    
  }
  
  MScene::~MScene() {
    // clearScene();
    m_Lights.clear();
    m_TimeSegments.clear();

    m_Objects = NULL;
    m_SelectedObj = NULL;

    m_Objects = NULL;
    m_SelectedObj = NULL;
  }
  
  MBaseObjectPtr MScene::createNew()
  {
    MScenePtr Scene;
    
    Scene = new MScene;
    return Scene;
  }
  
  
  void MScene::setObjectList(MBaseObjectTreePtr Tree) {
    m_Objects = Tree;
    
    rebuildLightList();
  }
  
  
  void MScene::setTime(int Time) {
    MIntParameterPtr timeParam = MSystemManager::getTimeParameter();

    if (timeParam != NULL) {
      timeParam->setValueInteger(Time);
    }
  }
  
  
  void MScene::clearScene() {
    m_Lights.clear();
    m_TimeSegments.clear();

    m_Objects = NULL;
    m_SelectedObj = NULL;
    
    m_Objects = new MBaseObjectTree;
    m_SelectedObj = new MBaseObjectList;
    
    setTime(0);
  }
  
  int MScene::updateObject(MBaseObjectPtr Obj, bool UpdateDependants) {
    if (Obj == NULL) {
      m_Objects->beginIteration();
      while ((Obj = m_Objects->getNext()) != NULL) {
        MIKController *ik = AZTEC_CAST(MIKController, Obj);

        if (ik != NULL) {
          ik->updateObject();
        }
      }
      m_Objects->endIteration();

      m_Objects->beginIteration();
      while ((Obj = m_Objects->getNext()) != NULL) {
        MIKController *ik = AZTEC_CAST(MIKController, Obj);

        if (ik != NULL) {
          ik->unsetFlag(OBJECTFLAG_NEEDS_UPDATE);
        }
      }
      m_Objects->endIteration();

      return 1;
    }

    Obj->updateObject();

    return 1;
  }
  
  int MScene::updateSelectedObjectList() {
    int         Count = 0;
    MBaseObjectPtr Obj;
    
    m_SelectedObj->removeAllNodes();
    
    m_Objects->beginIteration();
    while ((Obj = m_Objects->getNext()) != NULL) {
      if (Obj->isFlagged(OBJECTFLAG_SELECTED)) {
        Count++;
        m_SelectedObj->addHead(Obj);
      }
    }
    m_Objects->endIteration();
    
    return Count;
  }
  
  void MScene::setObjectChanged(MBaseObject *obj) {
    m_PendingUpdates = true;
  }

  MTreeObjectNodePtr MScene::addObject(MNamedObjectPtr Obj, MTreeObjectNodePtr Parent) {
    if (Obj == NULL) {
      return NULL;
    }
    
    MTimeSegmentPtr segment = AZTEC_CAST(MTimeSegment, Obj);

    if (segment != NULL) {
      return addObjectSpecial(segment, Parent);
    }

    MSceneObjectPtr sceneObj = AZTEC_CAST(MSceneObject, Obj);

    if (sceneObj != NULL) {
      MLightPtr light = AZTEC_CAST(MLight, sceneObj->getShapeObject());

      if (light != NULL) {
        m_Lights.push_back(sceneObj);
      }

      MShapeObjectPtr shapeObj = sceneObj->getShapeObject();

      if (shapeObj != NULL) {
        if (getObjectList()->findObject(shapeObj) == NULL) {
          addObject(shapeObj);
        }
      }
    }

    Obj->flagOutputs(OBJECTFLAG_NEEDS_UPDATE);

    MTreeObjectNodePtr result = m_Objects->addNode(Obj);

    if (sceneObj != NULL) {
      if (Parent != NULL) {
        sceneObj->setParent(AZTEC_CAST(MSceneObject, Parent->getObject()));
      }
    }

    return result;
  }
  
  MTreeObjectNodePtr MScene::addObjectSpecial(MTimeSegmentPtr segment, MTreeObjectNodePtr parent) {
    MTreeObjectNodePtr newNode = m_Objects->addNode(segment, parent);

    // add the object to the segment list as well.
    m_TimeSegments.push_back(segment);

    return newNode;
  }
  
  int MScene::deleteObject(MNamedObjectPtr Obj, bool DeleteChildren) {
    if (Obj == NULL)
      return 0;
    
    if (AZTEC_CAST(MMaterial, Obj) != NULL)
      return deleteObjectSpecial(MMaterialPtr(AZTEC_CAST(MMaterial, Obj)), DeleteChildren);
    if (AZTEC_CAST(MTimeSegment, Obj) != NULL)
      return deleteObjectSpecial(MTimeSegmentPtr(AZTEC_CAST(MTimeSegment, Obj)), DeleteChildren);

    // check to see if the object was a light
    {
      vector<MSceneObjectPtr>::iterator it;

      for (it = m_Lights.begin(); it != m_Lights.end(); ++it) {
        if ((*it) == Obj) {
          m_Lights.erase(it);
          break;
        }
      }
    }

    MTreeObjectNodePtr Node;
    
    Node = m_Objects->findObject(Obj);
    
    return deleteObjectNode(Node, DeleteChildren);
    
  }
  
  int MScene::deleteObjectNode(MTreeObjectNodePtr Node, bool DeleteChildren) {
    if (Node == NULL)
      return 0;
    
    // Check to see if it is in the selection list, and delete it if necessary   
    
    m_SelectedObj->removeNode(m_SelectedObj->findNode(Node->getObject()));
    
    if (DeleteChildren) {
      m_Objects->deleteNodeAndChildren(Node);
      
      // Reconstruct the light
      rebuildLightList();
      rebuildMaterialList();
    } else {
      m_Objects->deleteNodeKeepChildren(Node);
    }
    
    return 1;
  }
  
  int MScene::deleteObjectSpecial(MMaterialPtr Material, bool DeleteChildren) {
    if (Material == NULL) {
      return 0;
    }
    
    // check to see if it occurs in any materials for any object
    // If it does occur, replace it with a NULL texture.
    {
      MBaseObjectPtr Obj;
      
      m_Objects->beginIteration();
      while (((Obj = m_Objects->getNext()) != NULL) != NULL) {
        MSceneObjectPtr SceneObj;
        SceneObj = AZTEC_CAST(MSceneObject, Obj);
        if (SceneObj != NULL) {
          if (SceneObj->getTextureMaterial() == Material)
            SceneObj->setTextureMaterial(NULL);
        }
      }
      m_Objects->endIteration();
    }
   
    return deleteObjectNode(m_Objects->findObject(Material), DeleteChildren);
  }
 
  
  int MScene::deleteObjectSpecial(MTimeSegmentPtr segment, bool DeleteChildren) {
    if (segment == NULL) {
      return 0;
    }
    
    // Delete it from the light list.
    {   
      vector<MTimeSegmentPtr>::iterator it;

      it = find(m_TimeSegments.begin(), m_TimeSegments.end(), segment);

      if (it != m_TimeSegments.end()) {
        m_TimeSegments.erase(it);
      }
    }
    
    return deleteObjectNode(m_Objects->findObject(segment), DeleteChildren);
  }
  
  // List maintenenace functions
  void MScene::rebuildLightList() {
    m_Lights.clear();
    
    MBaseObjectPtr Obj;
    MSceneObjectPtr sceneObj;
    
    m_Objects->beginIteration();
    
    while ((Obj = m_Objects->getNext()) != NULL) {
      sceneObj = AZTEC_CAST(MSceneObject, Obj);
      if (sceneObj == NULL) {
        continue;
      }

      MLightPtr Light = AZTEC_CAST(MLight, sceneObj->getShapeObject());
      
      if (Light != NULL) {
        m_Lights.push_back(sceneObj);
      }
      
    }

    m_Objects->endIteration();
  }
  
  void MScene::rebuildMaterialList() {
  }
  
  void MScene::rebuildSelectList() {
    std::vector<MBaseObjectPtr> toRemove;
    MBaseObjectPtr Obj;
    
    m_SelectedObj->beginIteration();
    while ((Obj = m_SelectedObj->getNext()) != NULL) {
      if (!Obj->isFlagged(OBJECTFLAG_SELECTED))
        toRemove.push_back(Obj);
    }
    m_SelectedObj->endIteration();
    
    for (unsigned i = 0; i < toRemove.size(); ++i) {
      m_SelectedObj->removeNode(m_SelectedObj->findNode(toRemove[i]));
    }
  }
  
  
  int MScene::drawScene(MSceneViewFlags ViewFlags, std::vector<MSelectionItem> *OutputList) {
    updateObject(NULL);
    
    
    GLuint            *SelectBuf;       // The buffer for the selection stack. For selection if necesary
    GLint             NumHits;                // The number of hits for the selection opeartion
    
    int               ObjCount;

    MTreeObjectNodePtr ObjNode;
    MBaseObjectPtr BaseObj;
    MSceneObjectPtr Obj;
    vector<MSceneObjectPtr> ObjArray;
    
    // Initialise OpenGL viewing parameters
    
    // Are we in select mode?
    SelectBuf = ViewFlags.m_SelectBuf;
    if (ViewFlags.m_SelectMethod != smNone) {
      SelectBuf = ViewFlags.m_SelectBuf;
      
      glInitNames();
      glPushName(-1);

      ObjArray.resize(getObjectList()->getCount(), NULL);
    }
    
    
    glEnable(GL_DEPTH_TEST);
    
    glEnable(GL_BLEND);
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    //   glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_COLOR);    // Additive transparency
    
    glDisable(GL_LINE_SMOOTH);
    glDisable(GL_POINT_SMOOTH);
    //   glEnable(GL_LINE_SMOOTH);
    //   glEnable(GL_POINT_SMOOTH);
    glLineWidth(1);
    glPointSize(4);
    
    if (ViewFlags.m_LightingMode == glLit) {
      setupGLLights();
    } else if (ViewFlags.m_LightingMode == glUnlit) {
      glDisable(GL_LIGHTING);
    }
    
    
    if (ViewFlags.m_CullBackface) {
      glEnable(GL_CULL_FACE);
    } else {
      glDisable(GL_CULL_FACE);
    }
    
    if (ViewFlags.m_ShadingMode == glWireframe) {
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    } else {
      glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
    }
    
    getObjectList()->beginIteration();
    ObjCount = 0;
    
    while ((BaseObj = getObjectList()->getNext()) != NULL) {
      Obj = AZTEC_CAST(MSceneObject, BaseObj);

      
      ViewFlags.m_ComponentMode = MUIManager::getComponentMode();
      
      if (Obj == NULL) {
        MIKController *ik = AZTEC_CAST(MIKController, BaseObj);

        if (ik == NULL) {
          continue;
        }

        if (ViewFlags.m_SelectMethod != smNone) {
//          ObjArray[ObjCount] = BaseObj;
          glPushName(ObjCount);
        }

        MBaseObjectPtr root = ik->getStartBone();
        MBaseObjectPtr target = ik->getEndEffector();

        if (root != NULL && target != NULL) {
          MVector3 start = objectToWorldSpace(root, MVector3(0,0,0));
          MVector3 end = objectToWorldSpace(target, MVector3(0,0,0));

          glColor3f(0.1f,0.5f,0.8f);
          glBegin(GL_LINES);
          glVertex3fv((float*)&start);
          glVertex3fv((float*)&end);
          glEnd();
        }

        ObjCount ++;
        if (ViewFlags.m_SelectMethod != smNone) {
          glPopName();
        }

        continue;
      }

      if (!BaseObj->isFlagged(OBJECTFLAG_VISIBLE)) {
        continue;
      }

      MSceneViewFlags oldViewFlags = ViewFlags;
      if (Obj->getParamInt("drawWire")) {
        ViewFlags.m_DrawWire = true;
        ViewFlags.m_ShadingMode = glWireframe;
        ViewFlags.m_TexturingMode = glNone;
      }
      
      ObjNode = getObjectList()->getCurrentNode();

      MComponentisedObjectPtr compObj = Obj->getComponentObject();
      
      if (ViewFlags.m_SelectMethod != smNone) {
        ObjArray[ObjCount] = Obj;
      }
      
      glPushMatrix();
      GLTransform();
      
      ViewFlags.m_WireColor.set(1,1,1,1);
      ViewFlags.m_FlatColor.set(1,1,1,1);
      MVector3 flatCol = Obj->getParamVec("viewCol");
      ViewFlags.m_FlatColor.set(flatCol.x, flatCol.y, flatCol.z, 1.0f);
      
      ViewFlags.m_DrawWire = ViewFlags.m_ShadingMode == glWireframe;
      
      if (ViewFlags.m_ShadingMode == glWireframe) {
        if (ObjNode->isChildOfFlagged(OBJECTFLAG_SELECTED)) {
          if (ObjNode->isChildOfObject(m_SelectedObj->getTail())) {
            ViewFlags.m_WireColor.set(0.8f, 1.0f, 0.8f, 1.0f);
          } else {
            ViewFlags.m_WireColor.set(0.1f, 1.0f, 0.1f, 1.0f);
          }
        } else {
          MVector3 col = Obj->getParamVec("viewCol");
          ViewFlags.m_WireColor.set(col.x, col.y, col.z, 1.0f);
//          ViewFlags.m_WireColor.set(1.0f, 1.0f, 1.0f, 1.0f);
        }
        
        if (compObj != NULL && compObj->isInComponentMode()) {
          ViewFlags.m_WireColor.set(0.5f, 0.8f, 0.8f, 1.0f);
        }
        
      } else {
        ViewFlags.m_WireColor.set(0,0,0,1);
      }
      
      if (ObjNode->isChildOfFlagged(OBJECTFLAG_SELECTED) && ViewFlags.m_ShadingMode != glWireframe) {
        ViewFlags.m_DrawWire = true;
        
        if (ObjNode->isChildOfObject(m_SelectedObj->getTail())) {
          ViewFlags.m_WireColor.set(0.8f, 1.0f, 0.8f, 1.0f);
        } else {
          ViewFlags.m_WireColor.set(0.1f, 1.0f, 0.1f, 1.0f);
        }
        
        if (compObj != NULL && compObj->isInComponentMode()) {
          ViewFlags.m_WireColor.set(0.5f, 0.8f, 0.8f, 1.0f);
        }
        
      }
      
      if (ViewFlags.m_ShadingMode == glWireframe) {
        glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
      } else { 
        glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
      }

      ViewFlags.m_GLLighting = (ViewFlags.m_LightingMode != glUnlit);
      
      if (ViewFlags.m_SelectMethod != smNone) {
        glPushName(ObjCount);
      }
   
      // Draw the object using its own implmentation.
      MShapeObjectPtr shapeObj = Obj->getShapeObject();

      if (ViewFlags.m_ShadingMode != glWireframe) {
        if (Obj->getTextureMaterial() != NULL) {
          glDisable(GL_COLOR_MATERIAL);
          Obj->getTextureMaterial()->setOpenGLMaterialAndTextures();
        } else {
          glEnable(GL_COLOR_MATERIAL);
          glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
        }
        if (ViewFlags.m_TexturingMode != glNone) {
          glEnable(GL_TEXTURE_2D);
        } else {
          glDisable(GL_TEXTURE_2D);
        }
      } else {
        glDisable(GL_TEXTURE_2D);

        if (ViewFlags.m_ShadingMode != glWireframe) {
          glEnable(GL_COLOR_MATERIAL);
          glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
        }
//        glEnable(GL_COLOR_MATERIAL);
//        glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE);
      }

      if (shapeObj != NULL) {
        if (shapeObj->drawObject(Obj, ViewFlags) == 0) {
          // If it failes, draw it using the converted mesh standard way.
          
          MMeshPtr Mesh;
          
          Mesh = AZTEC_CAST(MMesh, shapeObj->convertToMesh());
          
          if (Mesh != NULL) {
            Mesh->drawObject(Obj, ViewFlags);
          }
        }
      }

      bool drawAxis = false;
      Obj->findParameter("drawAxis")->getValueBoolean(drawAxis);
      if (drawAxis) {
        GLDrawAxisIcon(20, 1.0, 1, true, false);
      }

      ObjCount ++;
      glPopMatrix();
      
      if (ViewFlags.m_SelectMethod != smNone) {
        glPopName();
      }

      ViewFlags = oldViewFlags;
   }
   getObjectList()->endIteration();

   if (ViewFlags.m_SelectMethod != smNone)     // If we are selecting something, get and process the selection
   {
     int      Result;
     
     NumHits = glRenderMode(GL_RENDER);
     
     ViewFlags.m_ComponentMode = MUIManager::getComponentMode();
     Result = processSelection(SelectBuf, NumHits, ObjArray, OutputList);
     
     // if we ended up with nothing selected, and we are doing a component mode, 
     // try again but in object mode.
     if (ViewFlags.m_ComponentMode != MComponentisedObject::OBJECT_TYPE && OutputList->size() == 0) {
       int oldMode = ViewFlags.m_ComponentMode;
       ViewFlags.m_ComponentMode = MComponentisedObject::OBJECT_TYPE;
       Result = processSelection(SelectBuf, NumHits, ObjArray, OutputList);
       ViewFlags.m_ComponentMode = oldMode;
     }

     ObjArray.clear();

     return Result;
   }
   return 1;
  }

  int MScene::processSelection(UINT *SelectBuf, 
                               int NumHits, 
                               const std::vector<MSceneObjectPtr> &ObjArray, 
                               std::vector<MSelectionItem> *OutputList) {

    if (OutputList == NULL) {
      return 0;
    }

    OutputList->clear();
  
    int      i, Count, Result;
    GLuint   *Ptr, *end;
  
    Ptr = SelectBuf;
    Result = 2;
  
    for (i=0;i<NumHits;i++)
    {
      float    Depth1, Depth2;
    
      Count = *Ptr; 
      Ptr++;
      Depth1 = (float) *Ptr/0x7fffffff; Ptr++;
      Depth2 = (float) *Ptr/0x7fffffff; Ptr++;
    
      end = Ptr + Count;
      while (Ptr < end) {
        // we are doing a new hit
        if (*Ptr == -1) {
          Ptr++;
        
          int ObjNum = *Ptr;
          Ptr++;

          // if we have a null objet, we can't do anything.
          if (ObjArray[ObjNum] != NULL) {
            MSelectionItem SelItem(ObjArray[ObjNum]);

            // now loop over the rest of this hit to see if there are any components involved.
            while (*Ptr != -1 && Ptr < end) {
              // if we have a non -1 here, it means it is a component type.
              int type = *Ptr;

              Ptr++;

              if (*Ptr != -1 && Ptr < end) {
                int compNum = *Ptr;
                Ptr++;

                SelItem.addComponent(type, compNum);
              }
            
            }

            OutputList->insert(OutputList->begin(), SelItem);
          }
        
        } else {
          Ptr++;
        }
      
      }
    }
    return 0;
  }

  int MScene::selectNone() {
    MBaseObjectPtr BaseObj;
    MSceneObjectPtr sceneObj;
  
    m_Objects->beginIteration();
  
    while ((BaseObj = m_Objects->getNext()) != NULL) {
      MComponentisedObjectPtr compObj;
      sceneObj = AZTEC_CAST(MSceneObject, BaseObj);
      if (sceneObj != NULL) {
        compObj = sceneObj->getComponentObject();
      }
      if ( compObj == NULL || !compObj->isInComponentMode() ) {
        BaseObj->unsetFlag(OBJECTFLAG_SELECTED);
      }
    }
    m_Objects->endIteration();  

    m_SelectedObj->beginIteration();
  
    while ((BaseObj = m_SelectedObj->getNext()) != NULL) {
      MComponentisedObjectPtr compObj;
      sceneObj = AZTEC_CAST(MSceneObject, BaseObj);
      if (sceneObj != NULL) {
        compObj = sceneObj->getComponentObject();
      }

      if (compObj == NULL || !compObj->isInComponentMode()) {
        BaseObj->unsetFlag(OBJECTFLAG_SELECTED);
        m_SelectedObj->removeNode(m_SelectedObj->findNode(BaseObj));
        m_SelectedObj->restartIteration();
      } else if (compObj != NULL && compObj->isInComponentMode()) {
        MSceneObjectPtr Obj;
      
        Obj = AZTEC_CAST(MSceneObject, BaseObj);
        if (Obj == NULL) {
          continue;
        }
      
        MComponentisedObjectPtr compObj = Obj->getComponentObject();
      
        if (compObj == NULL) {
          continue;
        }

        for (int typeIndex = 0; typeIndex < MComponentisedObject::getTypeCount(); ++typeIndex) {
          MComponentisedObject::ComponentType mode =
            MComponentisedObject::ComponentType(MComponentisedObject::getType(typeIndex) & MUIManager::getComponentMode());
          
          if (mode == 0) {
            continue;
          }

          int compCount = compObj->getComponentCount(mode);
          for (int i = 0; i < compCount; ++i) {
            compObj->flagComponent(mode, i, MComponentisedObject::COMPONENT_SELECTED, MComponentisedObject::atUnset); 
          }
        }
      }
    }

    m_SelectedObj->endIteration();
  
    return 1;
  }

  int MScene::selectObject(MBaseObjectPtr Obj, bool TargetSel, bool ignoreParentSelection) {
    if (Obj == NULL) {
      return 0;
    }
    if (m_SelectedObj == NULL) {
      return 0;
    }
  
    if (TargetSel) {
      MTreeObjectNodePtr ObjNode;
    
      ObjNode = m_Objects->findObject(Obj);
      // Is the object or its parent are unselected. If so, make it selected
      if (ignoreParentSelection || !ObjNode->isChildOfFlagged(OBJECTFLAG_SELECTED)) {
        m_SelectedObj->addTail(Obj);
        Obj->setFlag(OBJECTFLAG_SELECTED);
      
        // Flag the object if we are in a component mode of some sort.
        MSceneObjectPtr sceneObj = AZTEC_CAST(MSceneObject, Obj);
        if (sceneObj != NULL) {
          MComponentisedObjectPtr compObj = sceneObj->getComponentObject();

          if (compObj != NULL) {
            compObj->setComponentMode( MUIManager::getComponentMode() );
          }
        }
      
        // !!! TODO: Deselect all children.
      }
    } else {
      // Is the object selected. If so, make it unselected
      if (Obj->isFlagged(OBJECTFLAG_SELECTED)) {
        // take this object out of component mode if it is in one
        MSceneObjectPtr sceneObj = AZTEC_CAST(MSceneObject, Obj);
        MComponentisedObjectPtr compObj;

        if (sceneObj != NULL) {
         compObj = sceneObj->getComponentObject();
        }
    
        if (compObj != NULL && compObj->isInComponentMode()) {
          compObj->setComponentMode(MComponentisedObject::OBJECT_TYPE);
        }
        m_SelectedObj->removeNode(m_SelectedObj->findNode(Obj));
        Obj->unsetFlag(OBJECTFLAG_SELECTED);
      }
    }
    return 1;
  }

  void MScene::getWorldTransformMatrix(MTreeObjectNodePtr Node, MMatrix4 &Mat)
  {
    if (Node == NULL) {
      Mat.identity();
      return;
    }
  
    std::vector<MTransformObjectPtr> list;
    MBaseObjectPtr Obj;
  
    while (Node != NULL) {
      if (Node->getObject() != NULL) {
        MSceneObjectPtr SceneObj;
      
        SceneObj = AZTEC_CAST(MSceneObject, Node->getObject());
      
        if (SceneObj != NULL) {
          list.push_back(SceneObj->getTransformObject());
        }
      }
    
      Node = Node->getParent();
    }
  
    int      Num = 0;
    MMatrix4 XFormMat;
  
    XFormMat.identity();
  
    // Obj will always be a MTransformObject
    while (list.size() > 0) {
      MVector3          Vec;
      MTransformObjectPtr XForm;
    
      XForm = list.back();
      list.pop_back();
    
      if (XForm != NULL) {
        MMatrix4    Mat44;
      
        XForm->getTransformMatrix(Mat44);
      
        Mat44 *= XFormMat;
        XFormMat = Mat44;
      
      }
      Num++;
    }
  
    Mat = XFormMat;
  }


  void MScene::getWorldTransformMatrix(MTreeObjectNodePtr Node, float* Mat) {
    if (Node == NULL)
      return;
  
    std::vector<MTransformObjectPtr> list;
    MBaseObjectPtr Obj;
  
    while (Node != NULL) {
      if (Node->getObject() != NULL) {
        MSceneObjectPtr SceneObj;
      
        SceneObj = AZTEC_CAST(MSceneObject, Node->getObject());
      
        if (SceneObj != NULL) {
          list.push_back(SceneObj->getTransformObject());
        }
      }
    
      Node = Node->getParent();
    }
  
    int      Num = 0;
    MMatrix4 XFormMat;
  
    XFormMat.identity();
  
    // Obj will always be a MTransformObject
    while (list.size() > 0) {
      MVector3          Vec;
      MTransformObjectPtr XForm;

      XForm = list.back();
      list.pop_back();
    
      if (XForm != NULL) {
        MMatrix4    Mat44;
      
        XForm->getTransformMatrix(Mat44);
        XFormMat *= Mat44;
        //         Mat44 *= XFormMat;
        //         XFormMat = Mat44;
      
      }
      Num++;
    }
    memcpy(Mat, (float*)&XFormMat.m, sizeof(float)*16);
  
    /*   int         OldMode;
  
      glGetIntegerv(GL_MATRIX_MODE, &OldMode);
      glMatrixMode(GL_MODELVIEW);
      glPushMatrix();
    
        glLoadIdentity();
        GLTransform(Node);
      
          glGetFloatv(GL_MODELVIEW_MATRIX, Mat);
        
            glPopMatrix();
    glMatrixMode(OldMode);*/
  }

  MVector3 MScene::getWorldTransformTranslate(MTreeObjectNodePtr Node) {
    MMatrix4    Mat;
  
    getWorldTransformMatrix(Node, Mat);
  
    return MVector3(Mat.m[3][0], Mat.m[3][1], Mat.m[3][2]);
  }


  void MScene::GLTransform(MBaseObjectPtr Obj, DWORD XFormChan) {
    GLTransform(m_Objects->findObject(Obj, NULL), XFormChan);
  }

  void MScene::GLTransform(MTreeObjectNodePtr Node, DWORD XFormChan) {
    if (Node == NULL)
      return;
  
    std::vector<MTransformObjectPtr> list;
    while (Node != NULL) {
      if (Node->getObject() != NULL) {
        MSceneObjectPtr SceneObj = AZTEC_CAST(MSceneObject, Node->getObject());
      
        if (SceneObj != NULL) {
          list.push_back(SceneObj->getTransformObject());
        }
      }
    
      Node = Node->getParent();
    }
  
    while (list.size() > 0) {
      MVector3          Vec;
      MTransformObjectPtr XForm = list.back();
      list.pop_back();
    
      if (XForm != NULL) {
        MMatrix4    Mat;
      
        XForm->getTransformMatrix(Mat);
        //         Mat.Transpose();
        glMultMatrixf((float*)Mat.m);
      
        glGetFloatv(GL_MODELVIEW_MATRIX, (float*)Mat.m);
      }
    }
  }

  void MScene::GLTransform(DWORD XFormChan)
  {
    GLTransform(m_Objects->getCurrentNode(), XFormChan);
  }

  void MScene::setupGLLights()
  {
    // loop over the scene and get all the lights
    // TODO: replace with with an object list in the actual scene
    // which contains all the active lights
    {
      m_Lights.clear();
      m_Objects->beginIteration();
      MBaseObjectPtr baseObj;
      while ((baseObj = m_Objects->getNext()) != NULL) {
        MSceneObject *sceneObj = AZTEC_CAST(MSceneObject, baseObj);

        if (sceneObj != NULL) {
          MLightPtr light = AZTEC_CAST(MLight, sceneObj->getShapeObject());

          if (light != NULL) {
            m_Lights.push_back(sceneObj);
          }
        }
      }
      m_Objects->endIteration();
    }

    GLenum LightTarget;
    unsigned int    i;
  
    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE , GL_TRUE);
  
    /*
    if (m_Lights.size() == 0)
    {
    for (i=1; i<32; i++)
    {
    glDisable( GL_LIGHT0 + i);
    }
    glEnable(GL_LIGHT0);
    glEnable(GL_LIGHTING);
    }
    */
  
    for (i=0 ; i < m_Lights.size() ; i++) {
      MParameterObjectPtr param;
      MLightPtr lightShape;
      MVector3 Col(1,1,1), Spec(0,0,0);
      float Mult = 1.0;
    
      lightShape = AZTEC_CAST(MLight, m_Lights[i]->getShapeObject());

      if (lightShape == NULL) {
        continue;
      }

      param = lightShape->findParameter("Col");
      if (param != NULL) {
        param->getValueVector(Col);
      }
      param = lightShape->findParameter("SpCol");
      if (param != NULL) {
        param->getValueVector(Spec);
      }
      param = lightShape->findParameter("Mul");
      if (param != NULL) {
        param->getValueFloat(Mult);
      }
    
    
      glPushMatrix();
      GLTransform(MBaseObjectPtr(m_Lights[i]));
    
      LightTarget = (GLenum)(GL_LIGHT0+i);
    
      GLfloat  LightAmbient[] = { 0, 0, 0, 1 };
      GLfloat  LightDiffuse[] = { 0, 0, 0, 1 };
      GLfloat  LightSpecular[] = { 1, 1, 1, 1 };
      GLfloat  LightPosition[] = { 0, 0, 0, 1 };
    
      LightDiffuse[0] = Col.x * Mult;
      LightDiffuse[1] = Col.y * Mult;
      LightDiffuse[2] = Col.z * Mult;
    
      LightSpecular[0] = Spec.x;
      LightSpecular[1] = Spec.y;
      LightSpecular[2] = Spec.z;
    
      glLightfv(LightTarget, GL_AMBIENT, LightAmbient);
      glLightfv(LightTarget, GL_DIFFUSE, LightDiffuse);
      glLightfv(LightTarget, GL_SPECULAR, LightSpecular);
      glLightfv(LightTarget, GL_POSITION, LightPosition);
      glEnable(LightTarget);
    
      float Shiny  = 10;
    
      glPopMatrix();
    }
    for (i=m_Lights.size(); i<32; i++)
    {
      glDisable( (GLenum) (GL_LIGHT0 + i));
    }
  
    glEnable(GL_LIGHTING);
  }

  bool MScene::anythingSelected() {
    bool Result = false;
  
    // Check to see if we have any components selected
    MBaseObjectPtr BaseObj;
    MSceneObjectPtr Obj;
  
    getObjectList()->beginIteration();
    while ((BaseObj = getObjectList()->getNext()) != NULL)
    {
      Obj = AZTEC_CAST(MSceneObject, BaseObj);
      if (Obj == NULL) {
        continue;
      }
      MComponentisedObjectPtr compObj = Obj->getComponentObject();
    
      if (compObj != NULL && compObj->isInComponentMode()) {
        if (compObj->hasComponentsFlagged(MUIManager::getComponentMode(), MComponentisedObject::COMPONENT_SELECTED)) {
          Result = true;
          break;
        }
      
      } else if (Obj->isFlagged(OBJECTFLAG_SELECTED) && Obj->isFlagged(OBJECTFLAG_VISIBLE)) {
        Result = true;
        break;
      }
    }
    getObjectList()->endIteration();
  
    return Result;
  }


  MVector3 MScene::getSelectionCentre(int MaxCount) {
    MVector3    Centre;
    int         SelCount;
  
    SelCount = 0;
  
    MBaseObjectPtr BaseObj;
    MSceneObjectPtr Obj;
  
    m_Objects->beginIteration();
    while ((BaseObj = m_Objects->getNext()) != NULL) {
      if (MaxCount == 0) {
        break;
      }
    
      if (MaxCount > 0) {
        MaxCount--;
      }
    
      Obj = AZTEC_CAST(MSceneObject, BaseObj);
      if (Obj == NULL) {
        continue;
      }

      bool hasPivot = false;

      MComponentisedObjectPtr compObj = Obj->getComponentObject();
      MVector3 pivot;

      if ( compObj != NULL && compObj->isInComponentMode()) {
        if (compObj->hasComponentsFlagged(MUIManager::getComponentMode()) ) {
          pivot = compObj->getFlaggedCentre(MUIManager::getComponentMode());
          hasPivot = true;
        }
      } else if (Obj->isFlagged(OBJECTFLAG_SELECTED)) {
        pivot = Obj->getPivotPoint();
        hasPivot = true;
      }
      
      if (hasPivot) {
        Centre += objectToWorldSpace(MBaseObjectPtr(Obj), pivot);
        ++SelCount;
      }
    }
    m_Objects->endIteration();
  
    if (SelCount == 0) {
      return Centre;
    }
  
    Centre.x /= SelCount;
    Centre.y /= SelCount;
    Centre.z /= SelCount;
  
    return Centre;
  }

  // Gets the Most closely related Transform Node for the given object
  MTransformObjectPtr MScene::getTransform(MBaseObjectPtr Obj) {
    MSceneObjectPtr SceneObj;
  
    SceneObj = AZTEC_CAST(MSceneObject, Obj);
  
    if (SceneObj == NULL)
      return NULL;
  
    return SceneObj->getTransformObject();
  }

  // Gets the Most closely related Transform Node for the given tree node
  MTransformObjectPtr MScene::getTransform(MTreeObjectNodePtr Node)
  {
    if (Node == NULL)
      return NULL;
  
    return getTransform(Node->getObject());
  }

  // Gets the Most closely related Transform Node for the given object's parent
  MTransformObjectPtr MScene::getTransformParent(MBaseObjectPtr Obj)
  {
    return getTransformParent(m_Objects->findObject(Obj));
  }

  // Gets the Most closely related Transform Node for the parent of the tree node
  MTransformObjectPtr MScene::getTransformParent(MTreeObjectNodePtr Node)
  {
    if (Node == NULL)
      return NULL;
  
    if (Node->getParent() != NULL)
      return getTransform(Node->getParent()->getObject());
  
    return NULL;
  }

  MVector3 MScene::worldToObjectSpace(MBaseObjectPtr Obj, const MVector3 &Vec) {
    return worldToObjectSpace(m_Objects->findObject(Obj), Vec);
  }

  MVector3 MScene::worldToObjectSpace(MTreeObjectNodePtr ObjNode, const MVector3 &Vec) {
    if (ObjNode == NULL)
      return Vec;
  
    MMatrix4             Mat;
    MVector4             Res4, Vec4(Vec);
    MVector3             Result;
  
    getWorldTransformMatrix(ObjNode, Mat);
    Mat.inverse();
  
    Res4 = Mat * Vec4;
  
    Result = Res4;
  
    return Result;
  }

  MVector3 MScene::objectToWorldSpace(MBaseObjectPtr Obj, const MVector3 &Vec) {
    return objectToWorldSpace(m_Objects->findObject(Obj), Vec);
  }

  MVector3 MScene::objectToWorldSpace(MTreeObjectNodePtr ObjNode, const MVector3 &Vec)
  {
    MMatrix4    Mat;
    MVector3    Result;
    MVector4    Res4, Vec4(Vec);
  
    getWorldTransformMatrix(ObjNode, Mat);
    Res4 = Mat * Vec4;
  
    Result = Res4;
  
    return Result;
  }

  MSceneObjectPtr MScene::createSceneObject(const std::string &name) {
    return AZTEC_CAST(MSceneObject, createObject("group", name));
  }

  MSceneObjectPtr MScene::createLight(const std::string &name) {
    return AZTEC_CAST(MSceneObject, createObject("light", name));
  }

  MSceneObjectPtr MScene::createMesh(const std::string &name) {
    return AZTEC_CAST(MSceneObject, createObject("mesh", name));
  }

  MNamedObjectPtr MScene::createObject(const std::string &primType, std::string name) {
    MNamedObjectPtr newObject;
    MShapeObjectPtr shape;
    MMeshCreatorPtr meshCreator;
    bool createEmptySceneObject = false;

    if (primType == "MMesh" || 
        primType == "MEditableMesh" || 
        primType == "MAnimMesh" || 
        primType == "mesh") {
      if (name == "") {
        name = "mesh";
      }
      MAnimMeshPtr mesh = new MAnimMesh();
      shape = new MMeshShape(mesh);
      mesh->setName((name+"Mesh").c_str());

	  // The mesh itself must be added to the scene to ensure it will be properly
	  // saved to /loaded from a file.
      addObject(mesh);

    // create an omni directional light
    } else if (primType == "light") {
      if (name == "") {
        name = "light";
      }
      shape = new MLight();

    // create a bone
    } else if (primType == "bone") {
      if (name == "") {
        name = "bone";
      }
      shape = new MBoneObject();

    // create an empty object
    } else if (primType == "group" || primType == "MSceneObject") {
      if (name == "") {
        name = "group";
      }
      createEmptySceneObject = true;

    } else {
      // create the object from scratch.
      newObject = AZTEC_CAST(MNamedObject, MSystemManager::getInstance()->getPluginManager()->createObjectFromDefault(primType.c_str()));

      // see if the newly made object is a shape object, and if it is, make 
      // a scene object for it to exist in.
      shape = AZTEC_CAST(MShapeObject, newObject);

      // try to convert to a mesh creator as well.
      meshCreator = AZTEC_CAST(MMeshCreator, newObject);

      // if we couldn't make anything, chuck an error
      if (newObject == NULL) {
        MSystemManager::getInstance()->logOutput("Error: Could not create object of type '%s', type not found.", primType.c_str());
        return NULL;
      }

      if (name == "") {
        name = primType;
      }
    }

    // just a last catch all to make sure we have a valid name.
    if (name == "") {
      name = "object";
    }

    // if we ended up with a shape object, add that in,
    // otherwise add in the newly created object as is.
    if (shape != NULL || createEmptySceneObject) {
      MSceneObjectPtr result = new MSceneObject();
      result->setName(name.c_str());

      if (shape != NULL) {
        shape->setName((name + "Shape").c_str());
        result->setShapeObject(shape);
        addObject(shape);
      }

      addObject(result);
      return result;
    } else if (meshCreator != NULL) {
      MShapeObjectPtr shapeObj = new MMeshShape();
      MSceneObjectPtr sceneObj = new MSceneObject();
      shapeObj->setName(MStr(name.c_str()) + "Shape");
      sceneObj->setName(name.c_str());
      sceneObj->setShapeObject(shapeObj);
      meshCreator->setName(MStr(name.c_str()) + "Creator");
      
      shapeObj->findParameter("inMesh")->setInputParameter(meshCreator->getOutputParameter());
      
      addObject(shape);
      addObject(sceneObj);
      addObject(meshCreator);

      return sceneObj;
    } else {
      addObject(newObject);
      newObject->setName(name.c_str());

      return newObject;
    }

  }

  int MScene::getTimeSegmentCount() {
    return m_TimeSegments.size();
  }

  MTimeSegmentPtr MScene::getTimeSegment(int index) {
    return m_TimeSegments[index];
  }

  void MScene::setStartTime(int startTime) {
    m_StartTime = startTime;

    // shift the end time if we have to
    if (m_StartTime > m_EndTime) {
      m_EndTime = m_StartTime;
    }

  }

  int MScene::getStartTime() {
    return m_StartTime;
  }

  void MScene::setEndTime(int endTime) {
    m_EndTime = endTime;

    // shift the end time if we have to
    if (m_StartTime > m_EndTime) {
      m_StartTime = m_EndTime;
    }
  }

  int MScene::getEndTime() {
    return m_EndTime;
  }

  void MScene::setTimeRange(int startTime, int endTime) {
    // do nothing if the times are invalid.
    if (endTime < startTime) {
      return;
    }

    m_StartTime = startTime;
    m_EndTime = endTime;
  }

  int MScene::getFramesPerSecond() const {
    return m_FPS;
  }

  void MScene::setFramesPerSecond(int fps) {
    m_FPS = fps;
  }

}
