
#include <AztecMainPCH.h>
#include "SegmentEditor.h"

#include "MdlGlobs.h"
#include "MdlMsgs.h"
#include "TimeScroller.h"

#include "resource.h"

#include "KeyFuncAnim.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif



//----------------------------------------------------------------------------------------
//  CSegmentEditor
//----------------------------------------------------------------------------------------

const int CSegmentEditor::NONE = 0;
const int CSegmentEditor::LEFT = 1;
const int CSegmentEditor::RIGHT = 2;
const int CSegmentEditor::MIDDLE = 3;

CSegmentEditor::CSegmentEditor() {
  m_Tracking = false;
  m_PrefHeight = -1;

  m_SegmentCount = 0;
  m_Segments = NULL;

  m_DownX = 0;
  m_Resizing = false;
  m_AdjustingSeg = NULL;

  m_Side = NONE;
  m_AutoOrganise = true;
  m_DrawTimeline = false;
  m_DrawHeadings = false;

  m_HeadingWidth = 110;

  m_YScroll = 0;
  m_LastSegmentCount = 0;
}

CSegmentEditor::~CSegmentEditor() {
  if (m_Segments != NULL) {
    delete[] m_Segments;
  }
}


BEGIN_MESSAGE_MAP(CSegmentEditor, CWnd)
//{{AFX_MSG_MAP(CSegmentEditor)
ON_WM_PAINT()
//}}AFX_MSG_MAP
END_MESSAGE_MAP()


BOOL CSegmentEditor::SetScrollRange(int Min, int Max) {
  m_ScrollMin = Min;
  m_ScrollMax = Max;
  
  return TRUE;
}

BOOL CSegmentEditor::SetScrollPos(int Pos) {
  return TRUE;
}

int CSegmentEditor::GetScrollPos() {
  return g_Scene->tickToFrame(g_Scene->getTime());
}


void CSegmentEditor::OnPaint() {
  CPaintDC    Paintdc(this); // device context for painting
  
  HDC         dc;
  HBITMAP     bitmap, oldBitmap;

  RECT        ClientRect, Rect;
  HBRUSH      hBrush;
  HFONT       hOldFont;
  float       tickSize;
  
  GetClientRect(&ClientRect);
  
  dc = ::CreateCompatibleDC(Paintdc.m_hDC);
  bitmap = ::CreateCompatibleBitmap(Paintdc.m_hDC, ClientRect.right, ClientRect.bottom);
  oldBitmap = (HBITMAP)::SelectObject(dc, bitmap);
  
  hOldFont = (HFONT)::SelectObject(dc, GetStockObject(DEFAULT_GUI_FONT));
  
  {
    HPEN        hPen, hOldPen;
    
    hPen = CreatePen(PS_SOLID, 1, 0x000000);
    hOldPen = (HPEN)::SelectObject(dc, hPen);
    ::Rectangle(dc, 0,0,ClientRect.right, ClientRect.bottom);
    ::SelectObject(dc, hOldPen);
    DeleteObject(hPen);
    
    ClientRect.left += 1;
    ClientRect.top += 1;
    ClientRect.right  -= 1;
    ClientRect.bottom -= 1;
  }
  
  hBrush = ::CreateSolidBrush( GetSysColor(COLOR_SCROLLBAR) );
  ::FillRect(dc, &ClientRect, hBrush);
  ::DeleteObject(hBrush);

  // Make it so text is transparent
  ::SetBkMode(dc, TRANSPARENT);
  
  tickSize = (float)(ClientRect.right - ClientRect.left) / (g_Scene->frameToTick(m_ScrollMax) - g_Scene->frameToTick(m_ScrollMin));

  retrieveTimeSegments();

  if (m_YScroll >= m_SegmentCount) {
    m_YScroll = m_SegmentCount - 1;
  }
  if (m_YScroll < 0) {
    m_YScroll = 0;
  }

  if (m_DrawHeadings) {
    HPEN oldPen;
    SIZE size;
    ::GetTextExtentPoint32(dc, "Wgj", 3, &size);

    oldPen = (HPEN)::SelectObject(dc, ::GetStockObject(BLACK_PEN));

    ::MoveToEx(dc, ClientRect.left, ClientRect.top + size.cy + 6 - 1, NULL);
    ::LineTo(dc, m_HeadingWidth, ClientRect.top + size.cy + 6 - 1);
    ::MoveToEx(dc, ClientRect.left, ClientRect.top + size.cy + 6 - 2, NULL);
    ::LineTo(dc, m_HeadingWidth, ClientRect.top + size.cy + 6 - 2);

    ::MoveToEx(dc, m_HeadingWidth, ClientRect.top, NULL);
    ::LineTo(dc, m_HeadingWidth, ClientRect.bottom);
    ::MoveToEx(dc, m_HeadingWidth-1, ClientRect.top, NULL);
    ::LineTo(dc, m_HeadingWidth-1, ClientRect.bottom);

    ::SelectObject(dc, oldPen);


    ::GetTextExtentPoint32(dc, "Wgj", 3, &size);

    int top = ClientRect.top + size.cy + 10;
    for (int index = m_YScroll; index < m_SegmentCount; ++index) {
      MStr name = m_Segments[index].timeSegment->getName();
      ::GetTextExtentPoint32(dc, name.c_str(), name.GetLength(), &size);

      // Draw the text clipped to the header size
      RECT clipRect;
      clipRect.left = ClientRect.left + 3;
      clipRect.right = ClientRect.left + m_HeadingWidth - 3;
      clipRect.top = top;
      clipRect.bottom = clipRect.top + size.cy + 8;
      top = clipRect.bottom + 1;

      ::SetTextColor(dc, 0x00000000);
      ::DrawText(dc, name.c_str(), name.GetLength(), &clipRect, DT_LEFT | DT_VCENTER);
      ::SetTextColor(dc, m_Segments[index].textColour);

      // count the number of visible segments.
      m_LastSegmentCount = index - m_YScroll;
      if (top > ClientRect.bottom) {
        break;
      }
    }

    
    ClientRect.left += m_HeadingWidth;
  }


  // draw the current cursor;
  {
  
    Rect.left = ClientRect.left + (int)(tickSize*g_Scene->frameToTick(GetScrollPos() - m_ScrollMin));
    Rect.right = ClientRect.left + (int)(tickSize*(g_Scene->frameToTick(GetScrollPos() - m_ScrollMin + 1)));
    if (Rect.left == Rect.right) {
      Rect.right ++;
    }
  
    Rect.top = ClientRect.top;
    Rect.bottom = ClientRect.bottom;

    // If we are drawing the headings, only draw the
    // cursor if we are greater than the heading.
    if (!m_DrawHeadings || Rect.left > m_HeadingWidth) {
      hBrush = ::CreateSolidBrush(0xFFAA00);
      ::FillRect(dc, &Rect, hBrush);
      ::DeleteObject(hBrush);
    }

  }

  // draw a time line if we need to
  if (m_DrawTimeline) {
    HPEN oldPen;
    SIZE size;
    ::GetTextExtentPoint32(dc, "Wgj", 3, &size);

    {
      HBRUSH headerBack;
      headerBack = ::CreateSolidBrush(0x00d0ffff);
      RECT rect;
      rect.left = ClientRect.left;
      rect.right = ClientRect.right;
      rect.top = ClientRect.top;
      rect.bottom = rect.top + size.cy + 6;
      ::FillRect(dc, &rect, headerBack);

      ::DeleteObject(headerBack);
    }

    int majorTick = CTimeScroller::calcMajorTickSize(ClientRect.right - ClientRect.left, m_ScrollMin,m_ScrollMax, dc);

    oldPen = (HPEN)::SelectObject(dc, ::GetStockObject(BLACK_PEN));
    ::SetTextColor(dc, 0x00000000);

    // draw the lines
    for (int i = 0; i < m_ScrollMax - m_ScrollMin; i += majorTick) {
      int tick = g_Scene->frameToTick(i);
      char numstr[16];

      ::MoveToEx(dc, ClientRect.left + tick * tickSize, ClientRect.top, NULL);
      ::LineTo(dc, ClientRect.left + tick * tickSize, ClientRect.bottom);

      sprintf(numstr, "%i", i + m_ScrollMin);

      ::TextOut(dc, ClientRect.left + tick * tickSize + 3, 3, numstr, strlen(numstr));
    }


    ClientRect.top += size.cy + 6;

    ::MoveToEx(dc, ClientRect.left, ClientRect.top - 1, NULL);
    ::LineTo(dc, ClientRect.right, ClientRect.top - 1);
    ::MoveToEx(dc, ClientRect.left, ClientRect.top - 2, NULL);
    ::LineTo(dc, ClientRect.right, ClientRect.top - 2);

    ::SelectObject(dc, oldPen);
  }

  // if we don't have any time sgments to draw, then 
  // draw a little message saying so.
  if (m_SegmentCount == 0) {
    SIZE  size;

    char *text = "There are no Time Segments defined";
    ::GetTextExtentPoint32(dc, text, strlen(text), &size);
    ::TextOut(dc, 5, ClientRect.top + 5, text, strlen(text));
    setPreferredHeight(size.cy + 10);
  } else {

    ::IntersectClipRect(dc, ClientRect.left, ClientRect.top, ClientRect.right, ClientRect.bottom);

    // otherwise draw our time segments.
    int lastYPos = ClientRect.top;
    int largestHeight = lastYPos;

    // erase out any unseen time segments
    for (int index = 0; index < m_YScroll; ++index) {
      m_Segments[index].drawRect.left = 0;
      m_Segments[index].drawRect.right = 20;
      m_Segments[index].drawRect.top = -150;
      m_Segments[index].drawRect.bottom = -140;
      m_Segments[index].boundingBox = m_Segments[index].drawRect;
    }

    for (int index = m_YScroll; index < m_SegmentCount; ++index) {
      hBrush = ::CreateSolidBrush( m_Segments[index].colour );
      MTimeSegmentPtr timeSeg = m_Segments[index].timeSegment;

      if (timeSeg == NULL) {
        continue;
      }

      RECT box;
      SIZE  nameSize;
      MStr name = timeSeg->getName();
      // if our name can't fit, try to see if our start and end frame numbers can fit.
      char startStr[10], endStr[10];
      SIZE startSize, endSize;
      sprintf(startStr, "%i", (int)g_Scene->tickToFrame(timeSeg->getStart()));
      sprintf(endStr, "%i", (int)g_Scene->tickToFrame(timeSeg->getEnd()));
      ::GetTextExtentPoint32(dc, startStr, strlen(startStr), &startSize);
      ::GetTextExtentPoint32(dc, endStr, strlen(endStr), &endSize);

      // if we are drawing the headings, then our name size is irrelevant
      ::GetTextExtentPoint32(dc, name.c_str(), name.GetLength(), &nameSize);
      if (m_DrawHeadings) {
        nameSize.cx = 0;
      }
      
      box.left = tickSize * (timeSeg->getStart() - g_Scene->frameToTick(m_ScrollMin)) + ClientRect.left;
      box.right = tickSize * (timeSeg->getEnd() - g_Scene->frameToTick(m_ScrollMin)) + ClientRect.left;
      box.top = ClientRect.top;
      box.bottom = box.top + nameSize.cy + 8;

      m_Segments[index].boundingBox = box;

      if (nameSize.cx + startSize.cx + endSize.cy > box.right - box.left - 10) {
        m_Segments[index].boundingBox.right = box.right + 8 + nameSize.cx;

        // if the numbers cant fit inside we need to put them outside to.
        if (startSize.cx + endSize.cx + 10 >= box.right - box.left) {
          m_Segments[index].boundingBox.right = box.right + 3 + nameSize.cx + 6 + startSize.cx + 5 + endSize.cx + 3;
        }

      }

      if (m_AutoOrganise) {
        // loop over all the previously drawn segments and see if 
        // we intersect any of them
        for (int other = m_YScroll; other < index; ++other) {
          CSegmentGUI *otherSeg = &m_Segments[other];

          // if our bounding boxes collide, then we need to shift down one row
          if (CSegmentGUI::RECTsIntersect(otherSeg->boundingBox, m_Segments[index].boundingBox)) {
            box.top = box.bottom + 1;
            box.bottom = box.top + nameSize.cy + 8;
            m_Segments[index].boundingBox.top = box.top;
            m_Segments[index].boundingBox.bottom = box.bottom;
            other = -1;
          }
        }
      } else {
        // If we aren't organising, each group as its own line
        if (index == m_YScroll) {
          box.top = ClientRect.top;
          box.bottom = box.top + nameSize.cy + 8;
        } else {
          box.top = m_Segments[index-1].boundingBox.bottom + 1;
          box.bottom = box.top + nameSize.cy + 8;
        }
        m_Segments[index].boundingBox.top = box.top;
        m_Segments[index].boundingBox.bottom = box.bottom;
      }


      // Draw a horizontal line for the bar
      {
        HPEN oldPen;
        oldPen = (HPEN)::SelectObject(dc, ::GetStockObject(BLACK_PEN));

        if (m_DrawHeadings) {
        ::MoveToEx(dc, ClientRect.left - m_HeadingWidth, box.bottom, NULL);
        } else {
        ::MoveToEx(dc, ClientRect.left, box.bottom, NULL);
        }

        ::LineTo(dc, ClientRect.right, box.bottom);

        ::SelectObject(dc, oldPen);
      }

      ::FillRect(dc, &box, hBrush);

      // if our object is selected, we need to draw a selection border.
      if (timeSeg->isFlagged(OBJECTFLAG_SELECTED)) {
        HPEN oldPen, selPen;
        HBRUSH oldBrush;
        selPen = ::CreatePen(PS_SOLID, 0, m_Segments[index].selectedCol);

        oldPen = (HPEN)::SelectObject(dc, selPen);
        oldBrush = (HBRUSH)::SelectObject(dc, hBrush);

        ::Rectangle(dc, box.left, box.top, box.right, box.bottom);
        ::Rectangle(dc, box.left+1, box.top+1, box.right-1, box.bottom-1);

        ::SelectObject(dc, oldPen);
        ::SelectObject(dc, oldBrush);
        ::DeleteObject(selPen);
      } else {
        HPEN oldPen, hiPen, loPen;
        hiPen = ::CreatePen(PS_SOLID, 0, m_Segments[index].hilightCol);
        loPen = ::CreatePen(PS_SOLID, 0, m_Segments[index].shadowCol);

        oldPen = (HPEN)::SelectObject(dc, loPen);
        ::MoveToEx(dc, box.left, box.bottom, NULL);
        ::LineTo(dc, box.right, box.bottom);
        ::LineTo(dc, box.right, box.top);
        ::SelectObject(dc, hiPen);
        ::LineTo(dc, box.left, box.top);
        ::LineTo(dc, box.left, box.bottom);

        ::SelectObject(dc, oldPen);
        ::DeleteObject(loPen);
        ::DeleteObject(hiPen);
      }

      ::SetTextColor(dc, m_Segments[index].textColour);

      if (!m_DrawHeadings) {
        // if our text can't fit in the little box, then we need to put it beside it.
        if (nameSize.cx + startSize.cx + endSize.cy > box.right - box.left - 10) {

          ::SetTextColor(dc, 0x00000000);
          ::TextOut(dc, box.right + 3, box.top + 3, name.c_str(), name.GetLength() );
          ::SetTextColor(dc, m_Segments[index].textColour);

        } else {
          ::TextOut(dc, box.left + (box.right - box.left - nameSize.cx) / 2, box.top + 3, name.c_str(), name.GetLength() );
        }
      }

      // try to see if our start and end frame numbers can fit.
      if (startSize.cx + endSize.cx + 10 < box.right - box.left) {
        ::TextOut(dc, box.left + 3, box.top + 3, startStr, strlen(startStr));
        ::TextOut(dc, box.right - endSize.cx - 3, box.top + 3, endStr, strlen(endStr));
      } else {
        // otherwise we draw the frame numbers beside it.
        ::SetTextColor(dc, 0x00000000);
        ::TextOut(dc, box.right + 3 + nameSize.cx + 6, box.top + 3, startStr, strlen(startStr));
        ::TextOut(dc, box.right + 3 + nameSize.cx + 6 + startSize.cx + 5, box.top + 3, endStr, strlen(endStr));
        ::SetTextColor(dc, m_Segments[index].textColour);
      }

      lastYPos = box.bottom + 1;

      if (lastYPos > largestHeight) {
        largestHeight = lastYPos;
      }

      m_Segments[index].drawRect = box;

  
      ::DeleteObject(hBrush);
    }

    setPreferredHeight(largestHeight);

  }

  ::SelectObject(dc, hOldFont);
  
  GetClientRect(&ClientRect);

  ::BitBlt(Paintdc.m_hDC, 0,0,ClientRect.right, ClientRect.bottom, dc, 0,0,SRCCOPY);

  ::SelectObject(dc, oldBitmap);
  ::DeleteObject(bitmap);
  ::DeleteDC(dc);
}

LRESULT CSegmentEditor::WindowProc(UINT message, WPARAM wParam, LPARAM lParam) 
{
  if (message == WM_MBUTTONDOWN) {
    setViewAsCurrent();
  } else if (message == WM_LBUTTONDOWN || message == WM_LBUTTONDBLCLK) {
    setViewAsCurrent();

    int xPos, yPos;
    
    xPos = (short)LOWORD(lParam);  // horizontal position of cursor 
    yPos = HIWORD(lParam);  // vertical position of cursor

    doLeftMouseDown(xPos, yPos, wParam);

    m_Tracking = true;
    ::SetCapture(m_hWnd);

    if (message == WM_LBUTTONDBLCLK) {
      KAnimZoomToTimeSegment();
    }
    
    return 1;
  }
  
  if (message == WM_LBUTTONUP) {
    m_Resizing = false;
    m_AdjustingSeg = NULL;
    if (m_Tracking) {
      m_Tracking = false;
      ::ReleaseCapture();
      
      // Notify the parent window that the time scroll has changed
      ::PostMessage(::GetParent(m_hWnd), SE_SEGMENTEDITOR_AFTERCHANGE, 0,GetScrollPos());
    }
  } 
  if (message == WM_RBUTTONDOWN) {
    setViewAsCurrent();

    int xPos = (short)LOWORD(lParam);  // horizontal position of cursor 
    int yPos = HIWORD(lParam);  // vertical position of cursor

    doRightMouseDown(xPos, yPos, wParam);
  }

  if (message == WM_MOUSEMOVE) {
//    if (m_Tracking) {
      int      NumTicks, xPos, yPos;
      float    TickStep;
      RECT     ClientRect;
      
      ::GetClientRect(m_hWnd, &ClientRect);
      
      NumTicks = m_ScrollMax - m_ScrollMin;
      TickStep = (float)ClientRect.right / NumTicks;
      
      xPos = (short)LOWORD(lParam);  // horizontal position of cursor 
      yPos = HIWORD(lParam);  // vertical position of cursor

      doMouseMove(xPos, yPos, wParam);

      return 1;
//    }
  }
  if (message == WM_ERASEBKGND) {
    return 1;
  }
  
  return CWnd::WindowProc(message, wParam, lParam);
}

void CSegmentEditor::setAutoOrganise(bool organise) {
  m_AutoOrganise = organise;
}

void CSegmentEditor::setDrawTimeline(bool drawTimeline) {
  m_DrawTimeline = drawTimeline;
}

void CSegmentEditor::setHeadings(bool headingsOn) {
  m_DrawHeadings = headingsOn;
}

int CSegmentEditor::GetHeight()
{
  RECT  ClientRect;
  if (!::IsWindow(m_hWnd))
    return 0;
  
  GetClientRect(&ClientRect);
  return ClientRect.bottom;
}

int CSegmentEditor::getPreferredHeight() {
  if (m_PrefHeight == -1) {
    return GetHeight();
  }
  return m_PrefHeight;
}

void CSegmentEditor::setPreferredHeight(int height) {
  if (height != m_PrefHeight) {
    // if we have changed, then tell the main
    // window to move around.
    m_PrefHeight = height;

    RECT rect;
    WINDOWPOS pos;

    ::GetWindowRect(::GetParent(m_hWnd), &rect);

    pos.x = rect.left;
    pos.y = rect.right;
    pos.flags = SWP_NOSIZE | SWP_NOZORDER | SWP_NOMOVE;
    ::SendMessage(::GetParent(m_hWnd), WM_WINDOWPOSCHANGED, 0, (long)&pos);
  }
}

void CSegmentEditor::scroll(int xAmount, int yAmount) {
  int min, max;

  min = GetScrollMin();
  max = GetScrollMax();

  int change = 0.2 * xAmount * (max - min);

  // If we have a fairly zoomed out view, make sure
  // we only stop on multipels of 10.
  if (m_ScrollMax - m_ScrollMin > 40) {
    change -= (min + change) % 10;
  } else if (m_ScrollMax - m_ScrollMin > 20) {
    change -= (min + change) % 5;
  }

  min += change;
  max += change;

  SetScrollRange(min, max);

  m_YScroll += yAmount;

  if (m_YScroll < 0) {
    m_YScroll = 0;
  } else if (m_YScroll >= m_SegmentCount) {
    m_YScroll = m_SegmentCount - 1;
  }

  ::InvalidateRect(m_hWnd, NULL, FALSE);
  ::SendMessage(m_hWnd, WM_PAINT, 0,0);
}


void CSegmentEditor::setSegmentCount(int count) {
  if (m_SegmentCount != count) {
    if (m_Segments != NULL) {
      delete[] m_Segments;
    }

    m_SegmentCount = count;

    if (count == 0) {
      m_Segments = NULL;
    } else {
      m_Segments = new CSegmentGUI[count];
    }
  }
}

void CSegmentEditor::retrieveTimeSegments() {
  setSegmentCount(g_Scene->getTimeSegmentCount());

  for (int index = 0; index < m_SegmentCount; ++index) {
    CSegmentGUI *seg = &m_Segments[index];
    seg->timeSegment = g_Scene->getTimeSegment(index);
    
    MVector3 colVec, white(1,1,1), black(0,0,0);
    MStr colStr;
    colStr = seg->timeSegment->getParamByName("clr");
    colVec.convertFromString(colStr);

    seg->colour = CSegmentGUI::vectorToColour(colVec);
    seg->hilightCol = CSegmentGUI::vectorToColour(colVec + MVector3(0.5, 0.5, 0.5));
    seg->shadowCol = CSegmentGUI::vectorToColour(colVec - MVector3(0.5, 0.5, 0.5));
    seg->selectedCol = 0x0011EF12;

    if ( (white - colVec).length2() > (black - colVec).length2() ) {
      // draw the text in white
      seg->textColour = 0x00FFFFFF;
    } else {
      // draw the text in black
      seg->textColour = 0x00000000;
    }
  }
}

CSegmentGUI* CSegmentEditor::getSegmentAt(int x, int y) {
  // go through the segments and select any that are under the mouse
  for (int index = 0; index < m_SegmentCount; ++index) {
    CSegmentGUI *seg = &m_Segments[index];
    RECT rect = seg->boundingBox;

    if ( (m_DrawHeadings && x < m_HeadingWidth && 
          y >= rect.top && y <= rect.bottom) ||
        x >= rect.left && x <= rect.right &&
        y >= rect.top && y <= rect.bottom) {

      return seg;
    }
  }

  return NULL;
}

void CSegmentEditor::doLeftMouseDown(int xPos, int yPos, DWORD wParam) {
  // if the CTRL key is pressed, we are toggling selections
  bool ctrlDown = (wParam & MK_CONTROL) != 0;
  bool toggleSel = ctrlDown;
  CSegmentGUI *seg = getSegmentAt(xPos, yPos);

  // if we aren't toggling selections,
  // see if we are pointing at no segment,
  // or an unselected segment
  if (!toggleSel && (seg == NULL || !seg->timeSegment->isFlagged(OBJECTFLAG_SELECTED))) {
    // go though the time segments and deselect them all.
    for (int index = 0; index < m_SegmentCount; ++index) {
      CSegmentGUI *seg = &m_Segments[index];
      g_Scene->selectObject(seg->timeSegment, false);
    }
  }

  if (seg != NULL) {
    // if we are toggling the selected, toggle it
    // otherwise set it to true
    if (toggleSel) {
      g_Scene->selectObject(seg->timeSegment, !seg->timeSegment->isFlagged(OBJECTFLAG_SELECTED));
    } else {
      g_Scene->selectObject(seg->timeSegment);
    }
  }

  // see if we can resize some of these wonderful scrollers
  if (m_Side != NONE) {
    m_DownX = xPos;
    takeSegmentSnapshot();
    m_Resizing = true;
  }

  ::InvalidateRect(m_hWnd, NULL, FALSE);
  ::SendMessage(m_hWnd, WM_PAINT, 0,0);

  ::SendMessage(::GetParent(m_hWnd), MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
  ::SendMessage(::GetParent(m_hWnd), MM_UPDATECHANNELBAR, 0, 0);
}

void CSegmentEditor::doRightMouseDown(int xPos, int yPos, DWORD wParam) {
  // make the popup menu appear if the right mouse and nothing elese is pressed 
  if (!(wParam & MK_LBUTTON || wParam & MK_MBUTTON || 
      wParam & MK_CONTROL || wParam & MK_SHIFT) ) {
      
    // Make a new popup menu
    POINT pnt;

    pnt.x = xPos;
    pnt.y = yPos;

    ::ClientToScreen(m_hWnd, &pnt);
    
    ViewPopupMenu(pnt.x, pnt.y);
  }
}

void CSegmentEditor::doMouseMove(int xPos, int yPos, DWORD wParam) {
  // if we are resizing a segment, do its resizing
  if (m_Resizing && m_AdjustingSeg) {
    int start, end;
    RECT ClientRect;
    
    ::GetClientRect(m_hWnd, &ClientRect);

    float tickSize = (float)(g_Scene->frameToTick(m_ScrollMax) - g_Scene->frameToTick(m_ScrollMin)) / ClientRect.right;
    int diff = tickSize * (xPos - m_DownX);


    // iterate over all the selected segments and move them
    for (int i = 0; i < g_Scene->getTimeSegmentCount(); ++i) {
      MTimeSegmentPtr seg = g_Scene->getTimeSegment(i);

      if (!seg->isFlagged(OBJECTFLAG_SELECTED)) {
        continue;
      }

      start = seg->getStart();
      end = seg->getEnd();

      if (m_Side == LEFT) {
        start = m_InitialValues[i].first + diff;
      } else if (m_Side == RIGHT) {
        end = m_InitialValues[i].second + diff;
      } else if (m_Side == MIDDLE) {
        start = m_InitialValues[i].first + diff;
        end = m_InitialValues[i].second + diff;
      }

      // round off the values to the nearest frame
      double startFrame, endFrame;
      startFrame = g_Scene->tickToFrame(start);
      endFrame = g_Scene->tickToFrame(end);

      if (startFrame < 0) {
        startFrame -= 0.5;
      } else {
        startFrame += 0.5;
      }
      if (endFrame < 0) {
        endFrame -= 0.5;
      } else {
        endFrame += 0.5;
      }

      start = g_Scene->frameToTick((int)startFrame);
      end = g_Scene->frameToTick((int)endFrame) - 1;

      if (end >= start) {
        seg->setRange(start, end);
      }
    }

    ::InvalidateRect(m_hWnd, NULL, FALSE);
    ::SendMessage(m_hWnd, WM_PAINT, 0,0);

    ::SendMessage(::GetParent(m_hWnd), MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
    ::SendMessage(::GetParent(m_hWnd), MM_UPDATECHANNELBAR, 0, 0);

  } else {
    HCURSOR cursor = ::LoadCursor(NULL, IDC_ARROW);

    // find a segment under the cursor and if we have one,
    // figure out where we are on that segment.
    m_AdjustingSeg = getSegmentAt(xPos, yPos);

    if (m_AdjustingSeg != NULL) {
      RECT rect = m_AdjustingSeg->drawRect;

      if (xPos >= rect.right - 5 && xPos <= rect.right) {
        cursor = ::LoadCursor(NULL, IDC_SIZEWE);
        m_Side = RIGHT;
      } else if (xPos >= rect.left && xPos <= rect.left + 5) {
        cursor = ::LoadCursor(NULL, IDC_SIZEWE);
        m_Side = LEFT;
      } else if (xPos >= rect.left && xPos <= rect.right) {
        cursor = ::LoadCursor(NULL, IDC_SIZEALL);
        m_Side = MIDDLE;
      }
    } else {
      m_Side = NONE;
      m_AdjustingSeg = NULL;
    }

    ::SetCursor(cursor);
  }
}


DWORD CSegmentEditor::ViewPopupMenu(int x, int y)
{
  CMenu    Menu, AppMenu;
  CMenu    *Popup;
  DWORD    Choice;
  
  Menu.LoadMenu(IDR_SEGMENTEDITOR_POPUP);
  Popup = Menu.GetSubMenu(0);
  
  // Check the carious menu items depending on the current flags etc.
  
  Choice = Popup->TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON | TPM_NONOTIFY | TPM_RETURNCMD, x, y, this);
  
  Popup->DestroyMenu();
  
  bool     Update;
  int      Result;
  
  Result = HandlePopupCommand(Choice);
  
  // The command was handled, but the view was changed, so we must return.
  if (Result == -1)
    return 1;
  
  if (Result == 1)
    Update = true;
  else 
    Update = false;
  
  if (Update) {
    ::InvalidateRect(m_hWnd, NULL, FALSE);
    ::SendMessage(m_hWnd, WM_PAINT, 0,0);

    ::SendMessage(::GetParent(m_hWnd), MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
    ::SendMessage(::GetParent(m_hWnd), MM_UPDATECHANNELBAR, 0, 0);
  }
  
  return Choice;
}

int CSegmentEditor::HandlePopupCommand(DWORD Cmd) {
  switch (Cmd) {

  case ID_POPUP_CREATEATIMESEGMENT:
    KAnimCreateTimeSegment();
    return 1;

  }
  
  return 0;
}

void CSegmentEditor::setViewAsCurrent() {
  MBaseViewWndPtr view;
  CWnd *parent, *window;

  window = this;

  while ((parent = window->GetParent()) != NULL) {
    view = dynamic_cast<MBaseViewWnd*>(parent);

    if (view != NULL) {
      view->setAsCurrentView();
      return;
    }

    window = parent;
  }
}

void CSegmentEditor::takeSegmentSnapshot() {
  m_InitialValues.resize(g_Scene->getTimeSegmentCount());

  for (int i = 0; i < g_Scene->getTimeSegmentCount(); i++) {
    m_InitialValues[i].first = g_Scene->getTimeSegment(i)->getStart();
    m_InitialValues[i].second = g_Scene->getTimeSegment(i)->getEnd();
  }
}
