#ifndef PRIMSUBDIV_H
#define PRIMSUBDIV_H

#include <mesh/MMeshModifier.h>
#include "MShapeObject.h"
#include "MMesh.h"
#include "MMath.h"
#include "MEditableMesh.h"

using namespace Aztec;

#include <set>

const int MAX_SUBDIVISION_DEPTH = 6;

//----------------------------------------------------------------------------------------
//  MPrimitiveSubdiv
//----------------------------------------------------------------------------------------
class CLoopSurface {
protected:
// Base (level 0) mesh used to create subdivision surface
	long m_Depth;
	MMeshPtr m_BaseMesh;

// Local types/classes
	typedef int IntVerts[3];  // Typically used for triangle indices
	class IntSet : public std::set< int > { };
	class VertTriArray : public std::vector< IntSet > { };
	class EdgeTriSet;

// Cached surface (already tesselated to appropriate depth).
	int m_LastDepth;		// Last depth used

	// Generated surface at various levels of detail
	MEditableMeshPtr m_Mesh;
	int *m_NumVerts;	// # of vertices the last time around
	MVector3 **m_Verts;	// Array of vertices of old control mesh

	int m_NumBaseTris;	// # of triangles in base mesh (might have degenerates!)
	int *m_NumTris;		// # of triangles the last time around
	IntVerts **m_Tris;	// Array of triangles of old control mesh

	EdgeTriSet *m_EdgeList;	// Array holding all unique edges in the mesh

	IntSet **m_VertXRef;		// List of vertices connecting to each vertex
	IntVerts **m_TriEdgeXRef;	// Indices of edges on the boundary of each triangle

	// Flags for tracking what parts of the surface need to be regenerated
	bool **m_VertFlags;

	// Do we smooth the boundary edges or not?
	bool m_RetainBoundary;

// Tesselator(s) routines
	// Comprehensive tesselation down to m_Depth levels
	void UniformTesselate(bool rebuildAll=true);

	void SplitInterior(int depth, int cnt, int t0, int t1, int *v, bool *tvertflags,
			  IntSet *verttris);
	// Yank base mesh vertices and triangles
	void PullVertsAndTris();
	// Generate the set of edges for this mesh
	void CollectUniqueEdges(int depth);
	// Collect the indices of the edges that bound each face
	void CollectFaceEdges(int depth);
	// Generate the new triangles that resulted from the subdivision
	void GenerateNewFaces(int depth, bool rebuildAll);
	// Generate the new vertices that result from subdivision
	int SplitEdges(int depth, bool rebuildAll);

// Internal management of params/mesh
	bool checkBaseMesh();		// Do we have a valid control mesh for this subdiv?

	const long VERTICES_CHANGED;
	const long TRIANGLES_CHANGED;
	long getMeshChanges();	// Are the vertices/faces of the base mesh still the same?

	void DeleteVertices();
	void DeleteTris();
	void DeleteVertexFlags();
	void DeleteEdges();

public:
	// Construction/Destruction
	CLoopSurface();
	virtual ~CLoopSurface();
	
// Debugging output...
	void ShowMeshInfo(int depth);

	MEditableMeshPtr convertToMesh(MMeshPtr baseMesh, long depth, bool retainBoundary = false);
};

class MPrimitiveSubdiv : public MMeshModifier {
protected:
	long m_Depth;
  MObjectParameterPtr m_InMeshParam;
	MMeshPtr m_BaseMesh;
  MEditableMeshPtr m_Mesh;
	CLoopSurface m_BaseSurf, m_TextureSurf;
	
	bool checkBaseMesh();
public:	
	// Construction/Destruction
	MPrimitiveSubdiv();
	virtual ~MPrimitiveSubdiv();
	
	// Class related
	virtual MStr getClassName() {return MStr("MPrimitiveSubdiv");};
	virtual MStr getParentClassName() {return MStr("MBaseObject");};
	virtual MBaseObjectPtr createNew();
	
	// Animation Keying
	void updateKey(int Time, DWORD Channel, bool CreateKey);

  // MBaseObject methods
  bool doUpdateObject();
protected:
	MMeshPtr convertToMesh();
};


#endif


