#include "Stdafx.h"

#include "PrimSubdiv.h"

#include "MBaseObject.h"
#include "MShapeObject.h"
#include "MMesh.h"

#include "MScene.h"
#include <MSystemManager.h>
#include <params/MParameterFactory.h>

#include <stdio.h>
#include <assert.h>
#include <set>


#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#ifndef WIN32
#define OutputDebugString(x) printf(x)
#endif

MPrimitiveSubdiv::MPrimitiveSubdiv() {
	addParameter(MParameterFactory::createInteger("SDep", "SDep", "Subdiv Depth"));
	
	m_Depth = 1;
	setParamByName("SDep", "1");

	setFlag(OBJECTFLAG_NOCOMPONENTS);
}

MPrimitiveSubdiv::~MPrimitiveSubdiv() {
}

MBaseObjectPtr MPrimitiveSubdiv::createNew() {
	MPrimitiveSubdiv *NewObj;
  MNamedObjectPtr baseObj;
  MStr baseName;
	
	NewObj = new MPrimitiveSubdiv;

	NewObj->m_ParamList->setFromList(getParamList());

  return NewObj;
}

void MPrimitiveSubdiv::updateKey(int Time, DWORD Channel, bool CreateKey) {
}

bool MPrimitiveSubdiv::checkBaseMesh() {
  if (getInputShape() != NULL) {
    m_BaseMesh = getInputShape()->getMeshObject();
  }

  if (m_BaseMesh != NULL) {
		return true;
	}

	return false;
}

bool MPrimitiveSubdiv::doUpdateObject() {
	MMeshCreator::doUpdateObject();
  getOutputParameter()->setValue(convertToMesh());
	return true;
}

MMeshPtr MPrimitiveSubdiv::convertToMesh()
{
	m_Depth = getParamByName("SDep").ToInt();
	if (m_Depth >= MAX_SUBDIVISION_DEPTH) {
		m_Depth = MAX_SUBDIVISION_DEPTH - 1;
		setParamByName("SDep", MStr(m_Depth));
	} else if (m_Depth < 0) {
		m_Depth = 0;
		setParamByName("SDep", MStr(m_Depth));
	}

	if (!checkBaseMesh()) {
		return NULL;
	}

	m_Mesh = m_BaseSurf.convertToMesh(m_BaseMesh, m_Depth);
//	m_Mesh->setTextureMaterial(getTextureMaterial());
	MEditableMeshPtr textureMesh = AZTEC_CAST(MEditableMesh, m_BaseMesh->getTextureMesh());
	if (textureMesh != NULL) {
#if 0
		// Set z component to zero...
		int vcnt = textureMesh->getNumVerts();
		for (int i=0;i<vcnt;i++) {
			MVector3 tvert = textureMesh->getVertexPosition(i);
			tvert.z = 0.0f;
      textureMesh->setVertexPosition(i);
		}
#endif
		MEditableMeshPtr texMesh = m_TextureSurf.convertToMesh(textureMesh, m_Depth);
		m_Mesh->setTextureMesh(texMesh);
	}

	return m_Mesh;
}

class EdgeTri {
public:
	EdgeTri() { v0 = -1; v1 = -1; t0 = -1; t1 = -1; split = false;}
	EdgeTri(int tv0, int tv1) {
		if (tv0 < tv1) {
			v0 = tv0;
			v1 = tv1;
		} else {
			v0 = tv1;
			v1 = tv0;
		}
		t0 = -1;
		t1 = -1;
		split = false;
	}
	int t0, t1;
	int v0, v1;
	int mid;    // Index of the result of inserting a vertex into this edge
	bool split; // Has this edge already been split?

	friend bool operator==(const EdgeTri &e1, const EdgeTri &e2) {
		return ((e1.v0 == e2.v0) && (e1.v1 == e2.v1));
	}
	friend bool operator<(const EdgeTri &e0, const EdgeTri &e1) {
		if (e0.v0 < e1.v0)
			return true;
		else if (e1.v0 < e0.v0)
			return false;
		else if (e0.v1 < e1.v1)
			return true;
		else
			return false;
	}
};

class EdgeTriRefCompare {
public:
	bool operator()(const EdgeTri *e0, const EdgeTri *e1) {
		if (e0->v0 < e1->v0)
			return true;
		else if (e1->v0 < e0->v0)
			return false;
		else if (e0->v1 < e1->v1)
			return true;
		else
			return false;
	}
};

class CLoopSurface::EdgeTriSet : public std::set< EdgeTri *, EdgeTriRefCompare > {
public:
	EdgeTriSet() { }
	~EdgeTriSet() {
		iterator eitr;

		for (eitr=begin();eitr!=end();eitr++) {
			delete *eitr;
		}

		clear();
	}
};

//----------------------------------------------------------------------------------------
//  CLoopSurface
//----------------------------------------------------------------------------------------
CLoopSurface::CLoopSurface() :
	VERTICES_CHANGED(0x01),
	TRIANGLES_CHANGED(0x02)
 {
	m_BaseMesh = NULL;

	m_LastDepth = -1;
	m_Mesh = NULL;
	m_NumVerts = new int[MAX_SUBDIVISION_DEPTH];
	m_NumTris = new int[MAX_SUBDIVISION_DEPTH];

	m_Verts = new MVector3 *[MAX_SUBDIVISION_DEPTH];
	m_Tris  = new IntVerts *[MAX_SUBDIVISION_DEPTH];

	m_VertFlags = new bool *[MAX_SUBDIVISION_DEPTH];
	m_VertXRef = new IntSet *[MAX_SUBDIVISION_DEPTH];
	m_TriEdgeXRef = new IntVerts *[MAX_SUBDIVISION_DEPTH];

	m_EdgeList = new EdgeTriSet[MAX_SUBDIVISION_DEPTH];

	m_RetainBoundary = false;

	for (int i=0;i<MAX_SUBDIVISION_DEPTH;i++) {
		m_NumVerts[i] = NULL;
		m_NumTris[i] = NULL;
		m_Verts[i] = NULL;
		m_Tris[i] = NULL;
		m_VertFlags[i] = NULL;
		m_VertXRef[i] = NULL;
		m_TriEdgeXRef[i] = NULL;
	}
}

void CLoopSurface::DeleteVertexFlags()
{
	for (int i=0;i<MAX_SUBDIVISION_DEPTH;i++) {
		if (m_VertFlags[i] != NULL)
			delete[] m_VertFlags[i];
		if (m_VertXRef[i] != NULL)
			delete[] m_VertXRef[i];
		m_VertFlags[i] = NULL;
		m_VertXRef[i] = NULL;
	}
}

void CLoopSurface::DeleteVertices()
{
	for (int i=0;i<MAX_SUBDIVISION_DEPTH;i++) {
		if (m_Verts[i] != NULL)
			delete[] m_Verts[i];
		m_Verts[i] = NULL;
		m_NumVerts[i] = -1;
	}
}

void CLoopSurface::DeleteTris()
{
	for (int i=0;i<MAX_SUBDIVISION_DEPTH;i++) {
		if (m_Tris[i] != NULL)
			delete[] m_Tris[i];
		if (m_TriEdgeXRef[i] != NULL)
			delete[] m_TriEdgeXRef[i];
		m_Tris[i] = NULL;
		m_TriEdgeXRef[i] = NULL;
		m_NumTris[i] = -1;
	}
}

void CLoopSurface::DeleteEdges()
{
	EdgeTriSet::iterator edgeitr;
	for (int i=0;i<MAX_SUBDIVISION_DEPTH;i++) {
		
		// Clear out the association between edges and triangles.  It is
		// only valid for a single iteration and we don't want any of it's
		// contents being reused in the next iteration.
		for (edgeitr=m_EdgeList[i].begin();edgeitr!=m_EdgeList[i].end();edgeitr++) {
			EdgeTri *etri = (*edgeitr);
			delete etri;
		}
		m_EdgeList[i].clear();
	}
}

CLoopSurface::~CLoopSurface()
{
	m_BaseMesh = NULL;
	m_Mesh = NULL;

	DeleteVertices();
	delete[] m_Verts;
	delete[] m_NumVerts;
	DeleteTris();
	delete[] m_Tris;
	delete[] m_NumTris;
	delete[] m_TriEdgeXRef;
	DeleteVertexFlags();
	delete[] m_VertFlags;
	delete[] m_VertXRef;
	DeleteEdges();
	delete[] m_EdgeList;
}

long CLoopSurface::getMeshChanges()
{
	// Examine the vertices, texture coordinates, and triangles to see if any
	// of them have been modified.
	bool vertChangedFlag = false;
	bool triChangedFlag = false;

	// We have already built a mesh.  What we have to do now is compare
	// that mesh with 
	int numBaseVerts = m_BaseMesh->getNumVerts();
	int numBaseTris = m_BaseMesh->getNumTris();
	if ((numBaseVerts != m_NumVerts[0]) || (numBaseTris != m_NumBaseTris)) {
		// Have a different # of vertices (or triangles) than last time around.
		// In the long run, there ought to be a way to determine if all that
		// happened was an edge split, or some added faces, and regenerate
		// locally.  For now, if the face structure of a mesh changes, we
		// regenerate the entire mesh.
		return TRIANGLES_CHANGED;
	} else {
		int i, j;

		// Compare vertices for equality - if any have changed, then
		// we need to rebuild the subdivision surface.
		MVector3 vert;
		for (i=0;i<m_NumVerts[0];i++) {
			m_VertFlags[0][i] = false;
			vert = m_BaseMesh->getVertexPosition(i);
			if (vert.x != m_Verts[0][i].x ||
				vert.y != m_Verts[0][i].y ||
				vert.z != m_Verts[0][i].z) {
				// Mismatch of vertices.
				vertChangedFlag = true;
				m_VertFlags[0][i] = true;
				m_Verts[0][i] = vert;
			}
		}

		// Compare triangle indices...
		for (i=0,j=0;i<m_NumTris[0];i++) {
			if (m_BaseMesh->getTriangleVertex(i, 0) == m_BaseMesh->getTriangleVertex(i, 1) ||
				m_BaseMesh->getTriangleVertex(i, 0) == m_BaseMesh->getTriangleVertex(i, 2) ||
				m_BaseMesh->getTriangleVertex(i, 1) == m_BaseMesh->getTriangleVertex(i, 2)) {
				// Skip degenerate triangles
				continue;
			}
			if (m_BaseMesh->getTriangleVertex(i, 0) != m_Tris[0][j][0] ||
				m_BaseMesh->getTriangleVertex(i, 1) != m_Tris[0][j][1] ||
				m_BaseMesh->getTriangleVertex(i, 2) != m_Tris[0][j][2]) {
				// Mismatch of faces
				return TRIANGLES_CHANGED;
			} else {
				j++;
			}
		}
	}

	long result = 0;
	result |= (vertChangedFlag ? VERTICES_CHANGED : 0);

	return result;
}

MEditableMeshPtr CLoopSurface::convertToMesh(MMeshPtr baseMesh, long depth, bool retainBoundary)
{
	bool needRebuild = false;
	bool needRebuildAll = false;

	if (baseMesh == NULL) {
		// TBD: Warning that base mesh is no good.
		return NULL;
	}

	m_RetainBoundary = retainBoundary;

	m_Depth = depth;	// Assumption: depth is already within allowed limits.
	if (m_Depth != m_LastDepth) {
		needRebuild = true;
	}

	m_BaseMesh = baseMesh;
	
	long checkFlags = getMeshChanges();
	if ((checkFlags & TRIANGLES_CHANGED) != 0) {
		needRebuild = true;
		needRebuildAll = true;
	} else if (checkFlags != 0) {
		// The toplogy is unchanged, but we still need to go and
		// recompute some or all of the vertex/texture values.
		needRebuild = true;
	} else {
		// No need to rebuild the subdivision surface - the control mesh is
		// unchanged from the last time around.
	}

	if (needRebuild) {
		UniformTesselate(needRebuildAll);
	} else {
		// OutputDebugString("Use cached mesh for subdiv\n");
	}

	// Since the base object might not be a mesh, we free the reference
	// to it's mesh representation.  This is important for non-mesh
	// prims, since they regenerate their mesh and this pointer (m_BaseMesh)
	// won't be valid next time around (can't cache it).
	m_BaseMesh == NULL;

	return m_Mesh;
}

static char sbuf[512];

void CLoopSurface::
ShowMeshInfo(int depth)
{
#if defined(_DEBUG) && 1
	int i;

	int num_sverts = m_NumVerts[depth];
	MVector3 *svertices = m_Verts[depth];
	int num_sfaces = m_NumTris[depth];
	IntVerts *sfaces = m_Tris[depth];
	EdgeTriSet &edges = m_EdgeList[depth];
	IntSet *verttris = m_VertXRef[depth];

	// Show the vertices
	OutputDebugString("Vertices:\n");
	for (i=0;i<num_sverts;i++) {
		sprintf(sbuf, "Vert[%d]: %g %g %g\n", i, svertices[i][0], svertices[i][1], svertices[i][2]);
		OutputDebugString(sbuf);
	}
	// Show the faces
	OutputDebugString("Faces:\n");
	for (i=0;i<num_sfaces;i++) {
		sprintf(sbuf, "Face[%d]: %d %d %d\n", i, sfaces[i][0], sfaces[i][1], sfaces[i][2]);
		OutputDebugString(sbuf);
	}
	// Show the edge and vertex associations
	OutputDebugString("Triangles touching each edge:\n");
	EdgeTriSet::iterator edgeitr;
	for (i=0,edgeitr=edges.begin();edgeitr!=edges.end();i++,edgeitr++) {
		sprintf(sbuf, "Edge %3d: [%d,%d] - %d, %d\n",
			i, (*edgeitr)->v0, (*edgeitr)->v1, (*edgeitr)->t0, (*edgeitr)->t1);
		OutputDebugString(sbuf);
	}

	if (verttris != NULL) {
		// At the bottommost depth, there may not be computed information about
		// connected vertices.
		OutputDebugString("Vertices adjacent to each vertex:\n");
		for (i=0;i<num_sverts;i++) {
			sprintf(sbuf, "Vertex %3d: ", i);
			OutputDebugString(sbuf);
			for (IntSet::iterator is=verttris[i].begin();is!=verttris[i].end();is++) {
				sprintf(sbuf, "%d ", (*is));
				OutputDebugString(sbuf);
			}
			OutputDebugString("\n");
		}
	}
#endif
}

void CLoopSurface::
SplitInterior(int depth, int cnt, int t0, int t1, int *v, bool *tvertflags,
			  IntSet *verttris)
{
	float x, y, z;
	int j;

	int num_sverts = m_NumVerts[depth];
	MVector3 *svertices = m_Verts[depth];
	MVector3 *tverts    = m_Verts[depth+1];
	IntVerts *sfaces   = m_Tris[depth];

	// Interior edge.  First we update the even vertices
	for (j=0;j<2;j++) {
		int vindex = v[j];
		if (!tvertflags[vindex]) {
			// Need to update this vertex
			float beta;
			IntSet::iterator vaitr;
			MVector3 tvert; // (0.0, 0.0, 0.0);
			// How many adjacent vertices to this one?
			int n = verttris[vindex].size();

			// Compute the mask value for 
			if (n == 6) {
				// This is the most typical case, especially after one or more
				// rounds of subdivision
				beta = 3.0f / 48.0f;	// Let's hope the compiler collapses this division
			} else if (n < 3) {
				// Must be a boundary vertex
				beta = 1.0f / 8.0f;
			} else if (n == 3) {
				// Special case
				beta = 3.0f / 16.0f;
			} else {
				beta = 3.0f / (8.0f * (float)n);
			}

			if (n >= 3) {
				float tmp = 1.0f - (float)n * beta;
				x = tmp * svertices[v[j]].x;
				y = tmp * svertices[v[j]].y;
				z = tmp * svertices[v[j]].z;
				tverts[v[j]] = svertices[v[j]];
				tverts[v[j]] *= (1.0f - (float)n * beta);
				for (vaitr=verttris[v[j]].begin();vaitr!=verttris[v[j]].end();vaitr++) {
					int vindex = *vaitr;
					x += beta * svertices[vindex].x;
					y += beta * svertices[vindex].y;
					z += beta * svertices[vindex].z;
				}
				tverts[v[j]].x = x;
				tverts[v[j]].y = y;
				tverts[v[j]].z = z;

#if 0
				sprintf(sbuf, "Update v[%d], (%g,%g,%g) -> (%g,%g,%g)\n",
					v[j], svertices[v[j]].x, svertices[v[j]].y, svertices[v[j]].z, x, y, z);
				OutputDebugString(sbuf);
#endif
			} else {
				// Do nothing - this vertex is on an isolated triangle
				tverts[v[j]] = svertices[v[j]];
			}

			tvertflags[v[j]] = true;
		}
	}

	// Now do the odd vertices (the one we just created in the middle of the edge)
	if (sfaces[t0][0] != v[0] && sfaces[t0][0] != v[1])
		v[2] = sfaces[t0][0];
	else if (sfaces[t0][1] != v[0] && sfaces[t0][1] != v[1])
		v[2] = sfaces[t0][1];
	else {
		assert(sfaces[t0][2] != v[0] && sfaces[t0][2] != v[1]);
		v[2] = sfaces[t0][2];
	}
	if (sfaces[t1][0] != v[0] && sfaces[t1][0] != v[1])
		v[3] = sfaces[t1][0];
	else if (sfaces[t1][1] != v[0] && sfaces[t1][1] != v[1])
		v[3] = sfaces[t1][1];
	else {
		assert(sfaces[t1][2] != v[0] && sfaces[t1][2] != v[1]);
		v[3] = sfaces[t1][2];
	}
	assert(v[2] != v[3]);

	tverts[cnt]  = (3.0f / 8.0f) * svertices[v[0]];
	tverts[cnt] += (3.0f / 8.0f) * svertices[v[1]];
	tverts[cnt] += (1.0f / 8.0f) * svertices[v[2]];
	tverts[cnt] += (1.0f / 8.0f) * svertices[v[3]];

#if 0
	sprintf(sbuf, "Split v[%d] - v[%d], (%g,%g,%g)-(%g,%g,%g) -> (%g,%g,%g)\n",
		v[0], v[1],
		svertices[v[0]].x, svertices[v[0]].y, svertices[v[0]].z,
		svertices[v[1]].x, svertices[v[1]].y, svertices[v[1]].z,
		tverts[cnt].x, tverts[cnt].y, tverts[cnt].z);
	OutputDebugString(sbuf);
	sprintf(sbuf, "  other points: (%g,%g,%g), (%g,%g,%g)\n",
		svertices[v[2]].x, svertices[v[2]].y, svertices[v[2]].z,
		svertices[v[3]].x, svertices[v[3]].y, svertices[v[3]].z);
	OutputDebugString(sbuf);
#endif
}

// Pull the vertex and triangle information from the control mesh into
// the base level of the subdivision surface.  This routine allocates
// the memory needed for depth=0 of the surface.
void CLoopSurface::
PullVertsAndTris()
{
	int i, j;

	// Pull all vertices & triangles - since we will be repeatedly iterating
	// over them, we can't go back to the base mesh once we start subdividing.
	m_NumVerts[0] = m_BaseMesh->getNumVerts();
	m_Verts[0] = new MVector3[m_NumVerts[0]];
	m_VertFlags[0] = new bool[m_NumVerts[0]];

	MVector3 *tverts;
	
	// Pull vertices from the mesh
	tverts = m_Verts[0];
	for (i=0;i<m_NumVerts[0];i++) {
		MVector3 vert = m_BaseMesh->getVertexPosition(i);
		tverts[i][0] = vert.x;
		tverts[i][1] = vert.y;
		tverts[i][2] = vert.z;
	}

	// Pull the triangles from the original mesh
	m_NumBaseTris = m_BaseMesh->getNumTris();
	m_Tris[0]    = new IntVerts[m_NumBaseTris];
	for (i=0,j=0;i<m_NumBaseTris;i++) {
		IntVerts *ttris = m_Tris[0];
		ttris[j][0] = m_BaseMesh->getTriangleVertex(i, 0);
		ttris[j][1] = m_BaseMesh->getTriangleVertex(i, 1);
		ttris[j][2] = m_BaseMesh->getTriangleVertex(i, 2);
		if (ttris[j][0] != ttris[j][1] &&
			ttris[j][0] != ttris[j][2] &&
			ttris[j][1] != ttris[j][2]) {
			// Not degenerate - ok to increment triangle count
			j++;
		}
	}
	// Set # of tris in the lowest level of subdivision to the number of
	// non-degenerate triangles.
	m_NumTris[0] = j;

}

// Generate the set of all unique edges (e.g., an edge on one tri that goes
// from v0 -> v1 is the same as an edge on another tri that goes v0->v1 or v1->v0).
// As the edges are generated, we also keep a cross reference from the edge back
// to the face(s) we pulled it from.
//
// Note:  There is an assumption that an edge is never shared by more than two
// tris.  If this assumption is invalid, then the subdivision code will fail in
// unpredictable ways.  An edge that is only on one tri is ok - it is a boundary
// edge.
void CLoopSurface::
CollectUniqueEdges(int depth)
{
	int v[3];
	EdgeTriSet::iterator edgeitr;
	std::pair< EdgeTriSet::iterator, bool> edge_insert_result;

	IntVerts *sfaces = m_Tris[depth];
	int num_sfaces = m_NumTris[depth];
	IntSet *verttris = m_VertXRef[depth];

	EdgeTriSet &edges = m_EdgeList[depth];

	// Collect all unique edges and make an association between each vertex and
	// the triangles it sits in.
	for (int i=0;i<num_sfaces;i++) {

		v[0] = sfaces[i][0];
		v[1] = sfaces[i][1];
		v[2] = sfaces[i][2];

		// Associate this triangle with each of it's edges
		for (int j=0;j<3;j++) {
			EdgeTri *newEdge = new EdgeTri(v[j], v[(j+1)%3]);
			edge_insert_result = edges.insert(newEdge);
			edgeitr = edge_insert_result.first;
			if (edge_insert_result.second) {
				// New edge
				(*edgeitr)->t0 = i;
			} else {
				// Edge already exists
				delete newEdge;
				(*edgeitr)->t1 = i;
			}
		}

		// Associate each vertex of this triangle with it's adjacent vertices
		verttris[v[0]].insert(v[1]);
		verttris[v[0]].insert(v[2]);
		verttris[v[1]].insert(v[0]);
		verttris[v[1]].insert(v[2]);
		verttris[v[2]].insert(v[0]);
		verttris[v[2]].insert(v[1]);
	}
}

// For each triangle we make a list of it's edges.  Note that some care needs
// to be taken so that the edges are in the same order as the vertices.  If
// we don't do this, then when subdividing, we will end up with tris that
// face the wrong way (or possibly reference the wrong vertices).
void CLoopSurface::
CollectFaceEdges(int depth)
{
	int i, t0, t1, v[3];
	EdgeTriSet::iterator edgeitr;

	IntVerts *sfaces = m_Tris[depth];
	int num_sfaces = m_NumTris[depth];
	IntSet *verttris = m_VertXRef[depth];

	EdgeTriSet &edges = m_EdgeList[depth];

	// Set cross reference array empty (-1 index means unset - all edge indices
	// start at 0, so is a safe value).  "teref" = Triangle/Edge Reference
	IntVerts *teref = m_TriEdgeXRef[depth];
	for (i=0;i<num_sfaces;i++) {
		teref[i][0] = -1;
		teref[i][1] = -1;
		teref[i][2] = -1;
	}

	for (i=0,edgeitr=edges.begin();edgeitr!=edges.end();i++,edgeitr++) {
		EdgeTri *etri = (*edgeitr);
		int ev0 = etri->v0;
		int ev1 = etri->v1;
		t0 = etri->t0;	// Index of first triangle using this edge
		// Grab in-order vertex indices for this triangle
		v[0] = sfaces[t0][0];
		v[1] = sfaces[t0][1];
		v[2] = sfaces[t0][2];

		if (((ev0 == v[0]) && (ev1 == v[1])) || ((ev0 == v[1]) && (ev1 == v[0]))) {
			assert(teref[t0][0] == -1);
			teref[t0][0] = i;
		} else if (((ev0 == v[1]) && (ev1 == v[2])) || ((ev0 == v[2]) && (ev1 == v[1]))) {
			assert(teref[t0][1] == -1);
			teref[t0][1] = i;
		} else if (((ev0 == v[0]) && (ev1 == v[2])) || ((ev0 == v[2]) && (ev1 == v[0]))) {
			assert(teref[t0][2] == -1);
			teref[t0][2] = i;
		} else {
			// This edge has to go between two vertices of the triangle.
			assert(false);
		}

		t1 = (*edgeitr)->t1;
		if (t1 >= 0) {
			// Grab in-order vertex indices for this triangle
			v[0] = sfaces[t1][0];
			v[1] = sfaces[t1][1];
			v[2] = sfaces[t1][2];

			if (((ev0 == v[0]) && (ev1 == v[1])) || ((ev0 == v[1]) && (ev1 == v[0]))) {
				assert(teref[t1][0] == -1);
				teref[t1][0] = i;
			} else if (((ev0 == v[1]) && (ev1 == v[2])) || ((ev0 == v[2]) && (ev1 == v[1]))) {
				assert(teref[t1][1] == -1);
				teref[t1][1] = i;
			} else if (((ev0 == v[0]) && (ev1 == v[2])) || ((ev0 == v[2]) && (ev1 == v[0]))) {
				assert(teref[t1][2] == -1);
				teref[t1][2] = i;
			} else {
				// This edge has to go between two vertices of the triangle.
				assert(false);
			}
		}
	}
}

void CLoopSurface::
GenerateNewFaces(int depth, bool rebuildAll)
{
	int i, j, v[6];

	if (!rebuildAll && m_Tris[depth+1] != NULL) {
		// We built all the faces at some previous time.  Since the topology
		// of the control mesh hasn't changed (e.g., rebuildAll == false), we
		// don't need to rebuild the face information for the next level.
		return;
	}

	int num_sfaces = m_NumTris[depth];
	IntVerts *sfaces = m_Tris[depth];
	IntVerts *teref = m_TriEdgeXRef[depth];

	// Split the triangles, creating new vertices and edges as we go
	int num_sverts = m_NumVerts[depth];
	IntVerts *new_faces = new IntVerts[4 * num_sfaces];
	m_Tris[depth+1] = new_faces;
	m_NumTris[depth+1] = 4 * m_NumTris[depth];

	for (i=0,j=0;i<num_sfaces;i++) {
		v[0] = sfaces[i][0];
		v[1] = sfaces[i][1];
		v[2] = sfaces[i][2];

		// Since we built the new vertices in the order of the edges, we simply
		// need to know the index of the edge to determine the index of the
		// new vertex that sits on it.
		v[3] = num_sverts + teref[i][0];
		v[4] = num_sverts + teref[i][1];
		v[5] = num_sverts + teref[i][2];

		// Build the four new triangles
		new_faces[j][0] = v[0];
		new_faces[j][1] = v[3];
		new_faces[j][2] = v[5];
		j++;

		new_faces[j][0] = v[3];
		new_faces[j][1] = v[1];
		new_faces[j][2] = v[4];
		j++;

		new_faces[j][0] = v[5];
		new_faces[j][1] = v[3];
		new_faces[j][2] = v[4];
		j++;

		new_faces[j][0] = v[2];
		new_faces[j][1] = v[5];
		new_faces[j][2] = v[4];
		j++;
	}
}

int CLoopSurface::
SplitEdges(int depth, bool rebuildAll)
{
	EdgeTriSet::iterator edgeitr;
	int i, t0, t1, cnt, vcnt;
	int v[4];  // SplitEdges() fills v[0], v[1] - SplitInterior() needs v[2], v[3].

	EdgeTriSet &edges = m_EdgeList[depth];

	// The total # of vertices after subdivision is all the ones we have, plus
	// a new one for each edge.
	int num_sverts = m_NumVerts[depth];
	vcnt = num_sverts + edges.size();

	MVector3 *svertices = m_Verts[depth];
	IntVerts *sfaces = m_Tris[depth];
	IntSet *verttris = m_VertXRef[depth];

	// Depth we are working on
	int new_depth = depth+1;

	// Allocate array to hold new vertices
	if (rebuildAll || (m_Verts[new_depth] == NULL)) {
		if (m_Verts[new_depth] != NULL) {
			delete[] m_Verts[new_depth];
		}
		m_Verts[new_depth] = new MVector3[vcnt];

		if (m_VertFlags[new_depth] != NULL) {
			delete[] m_VertFlags[new_depth];
		}
		m_VertFlags[new_depth] = new bool[vcnt];
	}

	MVector3 *tverts = m_Verts[new_depth];

	// Booleans to see which old vertices have been processed
	bool *tvertflags = new bool[vcnt];
	for (i=0;i<num_sverts;i++) {
		tvertflags[i] = false;
	}

	// Compute and flag all boundary vertices before any other computations.
	// If we wait, the vertex might get moved as a result of being on the
	// end of an interior edge (which can touch a vertex on a boundary).
	for (edgeitr=edges.begin();edgeitr!=edges.end();edgeitr++) {
		v[0] = (*edgeitr)->v0;
		v[1] = (*edgeitr)->v1;
		if ((*edgeitr)->t1 == -1) {
			// This is a boundary edge, therefore the vertices at each end
			// are boundary vertices.
			tvertflags[v[0]] = true;
			tverts[v[0]] = svertices[v[0]];
			tvertflags[v[1]] = true;
			tverts[v[1]] = svertices[v[1]];
		}
	}

	// Split each edge.  Since existing vertices will be in the range from
	// 0..num_sverts, we start the indices for the new (odd) vertices above
	// that range.
	for (i=0,cnt=num_sverts,edgeitr=edges.begin();edgeitr!=edges.end();i++,edgeitr++) {
		assert(cnt < vcnt);
		assert((*edgeitr)->t0 != -1); // Must have a triangle connected to this edge
		// See if the endpoints have been updated
		v[0] = (*edgeitr)->v0;
		v[1] = (*edgeitr)->v1;
		t0   = (*edgeitr)->t0;
		t1   = (*edgeitr)->t1;
		if ((*edgeitr)->t1 == -1) {
			// Boundary edge (for now do nothing).  Note that the locations of the
			// vertices on the ends of this edge have already been computed.  Since
			// we want to keep a hard boundary from moving, the intermediate point
			// is halfway between the ends (midpoint).
			tverts[cnt] = tverts[v[0]];
			tverts[cnt] += tverts[v[1]];
			tverts[cnt] *= 0.5;

		} else {
			// Interior edge.  First we update the even vertices
			SplitInterior(depth, cnt, t0, t1, v, tvertflags, verttris);
		}

#if 0
sprintf(sbuf, "Split %d - %d = v%d (%g, %g, %g)\n",
		v[0], v[1], cnt, tverts[cnt].x, tverts[cnt].y, tverts[cnt].z);
OutputDebugString(sbuf);
sprintf(sbuf, "v0: (%g, %g, %g), v1: (%g, %g, %g)\n",
		tverts[v[0]].x, tverts[v[0]].y, tverts[v[0]].z,
		tverts[v[1]].x, tverts[v[1]].y, tverts[v[1]].z);
OutputDebugString(sbuf);
#endif
		(*edgeitr)->split = true;
		(*edgeitr)->mid = cnt++;
	}

	// Make sure we have the right number of new vertices
	assert(cnt == vcnt);

	// Make sure that all new vertices were computed
	for (i=0;i<cnt;i++) {
#if 1
		if (!tvertflags[i]) {
			sprintf(sbuf, "Vert[%d] not computed\n", i);
			OutputDebugString(sbuf);
		}
#endif
		assert(tvertflags[i]);
	}
	delete[] tvertflags;

	// Return # of vertices after subdivision
	return vcnt;
}

void CLoopSurface::
UniformTesselate(bool rebuildAll)
{
	int depth, i;
	EdgeTriSet edges;
	EdgeTriSet::iterator edgeitr;

	// If we need to rebuild the mesh, then there's some memory allocation and
	// vertex/triangle extraction to be done.
	if (rebuildAll) {
		DeleteVertices();
		DeleteTris();
		DeleteVertexFlags();
		DeleteEdges();

		PullVertsAndTris();
	}
	assert(m_Verts != NULL);
	assert(m_Verts[0] != NULL);
	assert(m_Tris != NULL);

	// Start subdividing.
	for (depth=0;depth<m_Depth;depth++) {
		if (rebuildAll || (m_VertXRef[depth] == NULL)) {
			// Create the array that holds associations between all triangles and vertices
			m_VertXRef[depth] = new IntSet[m_NumVerts[depth]];
			m_TriEdgeXRef[depth] = new IntVerts[m_NumTris[depth]];

			// Collect all unique edges and make an association between each vertex and
			// the triangles it sits in.
			CollectUniqueEdges(depth);

			// For each triangle we make a list of it's edges
			CollectFaceEdges(depth);
		}

		// Debugging (when appropriate)
		// ShowMeshInfo(depth);

		// Create a new vertex in the middle of each edge
		int new_vcnt = SplitEdges(depth, rebuildAll);

		// Create the new set of triangles for the surface
		GenerateNewFaces(depth, rebuildAll);

		m_NumVerts[depth+1] = new_vcnt;
	}

	if (rebuildAll || m_LastDepth != m_Depth) {
		m_Mesh = new MEditableMesh();

		// Turn the face array into the MTriangles that Aztec wants to see
		MTriangle *triArray = new MTriangle[m_NumTris[m_Depth]];
		IntVerts *sfaces = m_Tris[m_Depth];
		for (i=0;i<m_NumTris[m_Depth];i++) {
			triArray[i].setVertex(0, sfaces[i][0]);
			triArray[i].setVertex(1, sfaces[i][1]);
			triArray[i].setVertex(2, sfaces[i][2]);
		}

		// Add all the newly created vertices and faces to the mesh
		m_Mesh->addVertsAndTriangles(m_Verts[m_Depth], triArray,
			m_NumVerts[m_Depth], m_NumTris[m_Depth]);

		delete[] triArray;	// Aztec copies the data, so need to free temporary triArray.
	} else {
		MVector3 *verts = m_Verts[m_Depth];
		for (i=0; i < m_NumVerts[m_Depth]; i++) {
      m_Mesh->setVertexPosition(i, *verts++);
		}
	}

	m_Mesh->calculateNormals();

	m_LastDepth = m_Depth;

	// ShowMeshInfo(0);
}

