#ifndef Aztec_MultiSplitter_Header
#define Aztec_MultiSplitter_Header

#include <vector>

#include <gui/MContainer.h>
#include <gui/MMouseCursor.h>
#include <gui/MRect2D.h>

namespace AztecGUI {

  class MultiSplitter;

  /**
   * The splitte method class desribes the layout and dragging behaviour for
   * a particular arrangement of windows inside a MultiSplitter. The class
   * that derives and implemented the SplitterMethod interface may have a 4
   * way split, or a T shape, or perhaps something stranger. 
   */
  class SplitterMethod {
  public:
    typedef enum { NOTHING = 0, Horizontal = 1, Vertical = 2, Both = 3 } HitTestResultEnum;

    /**
     * Gets the number of panes that this splitter method provides information
     * dragging for.
     * 
     * @return The number of panes that this splitter method describes.
     */
    virtual int getPaneCount() = 0;

    /**
     * Performs a hti test on the spltite rbars. Returns NOTHING (0) if 
     * nothing was under the mouse cursor, otherwise it returns a non 
     * zero value that is meaningful to this class.
     * 
     * @param splitterWnd The MultiSplitter window that we are laying out.
     */
    virtual HitTestResultEnum hitTest(MultiSplitter *splitterWnd, int x, int y) = 0;

    /**
     * This gets the rectangle for the given pane. 
     *
     * @param splitterWnd The MultiSplitter window that we are laying out.
     * @param index The index of the pane whos rectange we want.
     * @return The rectangle for the given pane in this splitter window.
     */
    virtual Aztec::MRect2D getRectFor(MultiSplitter *splitterWnd, int index) = 0;

    /**
     * Gets the cursor for the given hit result, which must have been 
     * retrieved from the method hitTest().
     *
     * @param hitResult The result from the method hitTest().
     */
    virtual Aztec::MMouseCursor getCursorForHit(HitTestResultEnum hitResult) = 0;

    /**
     * This tells the splitter method that we have begun dragging one of the 
     * splitters. This method should use the hitResult argument to determine
     * what action to take when dragging items.
     *
     * @param splitterWnd The MultiSplitter window that we are laying out.
     * @param hitResult The result from the method hitTest().
     * @param x The x coordinate of the mouse relative to the splitterWnd.
     * @param y The y coordinate of the mouse relative to the splitterwnd.
     */
    virtual void beginDrag(MultiSplitter *splitterWnd, HitTestResultEnum hitResult, int x, int y) = 0;

    /**
     * This tells the splitter method that we are doing some dragging on one 
     * of the splitters. This method should use the hitResult argument to 
     * determine what action to take when dragging items.
     *
     * @param splitterWnd The MultiSplitter window that we are laying out.
     * @param hitResult The result from the method hitTest().
     * @param x The x coordinate of the mouse relative to the splitterWnd.
     * @param y The y coordinate of the mouse relative to the splitterwnd.
     */
    virtual void doDrag(MultiSplitter *splitterWnd, HitTestResultEnum hitResult, int x, int y) = 0;

    /**
     * This tells the splitter method that we have finised dragging on one 
     * of the splitters. This method should use the hitResult argument to 
     * determine what action to take when dragging items.
     *
     * @param splitterWnd The MultiSplitter window that we are laying out.
     * @param hitResult The result from the method hitTest().
     * @param x The x coordinate of the mouse relative to the splitterWnd.
     * @param y The y coordinate of the mouse relative to the splitterwnd.
     */
    virtual void endDrag(MultiSplitter *splitterWnd, HitTestResultEnum hitResult, int x, int y) = 0;

    /**
     * This is called to indicate that we want the size of a pane to be the largest possible
     * 
     * @param pane The pane we want to make really big. -1 if we want to restore the panes to normal size.
     */
    virtual void maximisePane(int pane) = 0;

    SplitterMethod() {
    }

    virtual ~SplitterMethod() {
    }
  };

  /**
   * The MultiSplitter is a splitter window that allows a variety of windows
   * to be placed inside an unusual splitter. Instead of just a two pane 
   * splitter, the multi splitter allows any sort of configuration, such as
   * a four way splitter, two way, three way, or anything else that 
   * could be imagined.
   *
   * The actual laying out of the split up windows is controlled by a 
   * class derived from SplitterMethod. It provides the information necessary
   * to organise the child windows, and to also perform resizing on the panes.
   */
  class MultiSplitter : public Aztec::MContainer {
  public:
    MultiSplitter();
    ~MultiSplitter();

    void setSplitterMethod(SplitterMethod *method);
    SplitterMethod* getSplitterMethod() const;
    void toggleMaximisedComponent(const Aztec::MComponentPtr &component);

    // MComponent methods
    void onCreate();
    bool onPaint();
    bool onMousePressed(const Aztec::MMouseEvent &event);
    bool onMouseReleased(const Aztec::MMouseEvent &event);
    bool onMouseMove(const Aztec::MMouseEvent &event);

  private:
    SplitterMethod *method;
    SplitterMethod::HitTestResultEnum draggingHit;
    bool needsUpdating;
    Aztec::MPoint2D lastMouse;
    bool componentMaximised;

  };

  typedef Aztec::MRefCountedPtr<MultiSplitter> MultiSplitterPtr;

}

#endif

