#ifndef AztecSceneFunctions_Header
#define AztecSceneFunctions_Header

#include <MListsTrees.h>
#include <MUndoManager.h>
#include <views/AztecViewManager.h>
#include <functions/FunctionManager.h>

//#include <MMesh.h>
//#include <MEditableMesh.h>

namespace AztecGUI {

  /**
   * This function applies a given function to each object in a tree.
   * It first builds up a list of objects to visit if for each of true result of 
   * the criteria(node) function. After building up that list, it then calls 
   * func(scene, node) for each node that passed the criteria.
   *
   * @param scene The scene that we will operate in.
   * @param list the object list we are iterating over
   * @param criteria User defined function or class that returns true of the 
   *                 node we are on should be included, or false if it should 
   *                 be ignored. Must allow compilation of criteria(node).
   * @param func The function to apply to each of the object nodes that passed
   *             the criteria test. It takes a scene and an object node argument.
   */
  template <typename FUNC, typename CRITERIA> 
  static void applyToObjects(const Aztec::MScenePtr &scene, 
                             const Aztec::MBaseObjectTreePtr &list, 
                             CRITERIA criteria, 
                             FUNC func) {
    Aztec::MBaseObjectPtr object;
    Aztec::MTreeObjectNodePtr treeNode;


    // first build up a list of objects to apply to.
    std::vector<Aztec::MTreeObjectNodePtr> nodes; 

    // iterate over the lists and apply the function
    list->beginIteration();
    while ((object = list->getNext()) != NULL) {
      treeNode = list->getCurrentNode();
      if (treeNode != NULL) {
        if (criteria(treeNode)) {
          nodes.push_back(treeNode);
        }
      }
    }
    list->endIteration();

    // now apply our function
    for (unsigned int i = 0; i < nodes.size(); ++i) {
      func(scene, nodes[i]);
    }
  }

  template <typename FUNC>
  class ApplyToMeshFunc {
  public:
    ApplyToMeshFunc(FUNC *f) {
      func = f;
    }

    void operator()(const Aztec::MScenePtr &scene, 
                    const Aztec::MTreeObjectNodePtr &node) 
    {
      Aztec::MNamedObjectPtr namedObj = AZTEC_CAST(Aztec::MNamedObject, node->getObject());

      Aztec::MSceneObjectPtr sceneObj = AZTEC_CAST(Aztec::MSceneObject, namedObj);

      if (sceneObj == NULL) {
        return;
      }

      Aztec::MEditableMeshPtr mesh = Aztec::getEditableMeshObject(namedObj);
      if (mesh == NULL) {
        return;
      }

      (*func)(scene, node, sceneObj, mesh);
    }

    FUNC *func;
  };

  template <typename CRITERIA, typename FUNC>
  inline void applyToEditableMeshes(const Aztec::MScenePtr &scene, 
                                    const Aztec::MBaseObjectTreePtr &list, 
                                    CRITERIA criteria, 
                                    FUNC func) 
  {
    applyToObjects(scene, list, criteria, ApplyToMeshFunc<FUNC>(&func));      
  }


  /**
   * This function applies a function to all the objects in the scene. It 
   * wraps the call in an UndoSection using the given undo name. Then it calls
   * applyToObjects using the object list, and then redraws the views.
   */
  template <typename FUNC, typename CRITERIA> 
  static int applyFunction(const Aztec::MScenePtr &scene, 
                           const std::string &undoName, 
                           CRITERIA criteria, 
                           FUNC func) 
  {
    Aztec::MUndoManager::UndoSection section(undoName);
    Aztec::MBaseObjectTreePtr listCopy;
    
    applyToObjects(scene, scene->getObjectList(), criteria, func);
    
    AztecViewManager::redrawAllViews();

    return FunctionManager::SUCCEED;
  }

  /**
   * Simple criteria function thats tests if the ovject is selected. For use 
   * with applyToObjects and applyFunction
   *
   * @see applyToObjects and applyFunction for more information.
   */
  inline bool ifSelectedCriteria(const Aztec::MTreeObjectNodePtr &node) {
    return node->getObject()->isFlagged(OBJECTFLAG_SELECTED);
  }

  /**
   * Simple criteria function thats tests if the object is not selected. For use 
   * with applyToObjects and applyFunction
   *
   * @see applyToObjects and applyFunction for more information.
   */
  inline bool ifNotSelectedCriteria(const Aztec::MTreeObjectNodePtr &node) {
    return !node->getObject()->isFlagged(OBJECTFLAG_SELECTED);
  }

  /**
   * This is a simple criteria function that passes all objects. For use with 
   * applyToObjects and applyFunction
   *
   * @see applyToObjects and applyFunction for more information.
   */
  inline bool allObjectsCriteria(const Aztec::MTreeObjectNodePtr &node) {
    return true;
  }


}


#endif

