#ifndef Aztec2_SelecingView_Header
#define Aztec2_SelecingView_Header

namespace AztecGUI {

  /**
   * This class provides an interface that exposes specialised selection 
   * functionality for a view. For example, if a view wants to select all the 
   * UV's points instead of all the objects in the scene, it would provide a 
   * working implemtnation for selectAll(). 
   *
   * A view implements all of this interface if it wishes to take specific 
   * actions when the user executes a selectAll(), selectNone() or functions 
   * like that. For any function that returns false, it is assumed that no 
   * implementation for that method was there, and the dfault behaviour takes 
   * place.
   */
  class SelectingView {
  public:
    /**
     * This is called when the usre tries to do a selectParent().
     *
     * @returns true If the view has implemented this, false if it hasn't.
     */
    virtual bool selectParent() = 0;

    /**
     * This is called when the usre tries to do a selectParent().
     *
     * @returns true If the view has implemented this, false if it hasn't.
     */
    virtual bool selectChild() = 0;
    
    /**
     * This is called when the usre tries to do a selectSiblingNext(). 
     * An example implementation might be to select the graph keys that are 
     * next in line after the currently selected ones.
     *
     * @returns true If the view has implemented this, false if it hasn't.
     */
    virtual bool selectSiblingNext() = 0;
    
    /**
     * This is called when the usre tries to do a selectSiblingPrev().
     * An example implementation might be to select the graph keys that are 
     * previous in line after the currently selected ones.
     *
     * @returns true If the view has implemented this, false if it hasn't.
     */
    virtual bool selectSiblingPrev() = 0;
    
    /**
     * This is called when the usre tries to do a selectNone().
     * An example implementation might be to deselect all the current UV 
     * editing points, or all graph key's.
     *
     * @returns true If the view has implemented this, false if it hasn't.
     */
    virtual bool selectNone() = 0;
    
    /**
     * This is called when the usre tries to do a selectAll().
     * An example implementation might be to select all the current UV 
     * editing points, or all graph key's.
     *
     * @returns true If the view has implemented this, false if it hasn't.
     */
    virtual bool selectAll() = 0;

    /**
     * This is called when the usre tries to do a selectInverse().
     * An example implementation might be to invert the selection of all the 
     * current UV editing points, or all graph key's.
     *
     * @returns true If the view has implemented this, false if it hasn't.
     */
    virtual bool selectInverse() = 0;
    
    /**
     * This is called when the usre tries to do a selectAllChildren().
     *
     * @returns true If the view has implemented this, false if it hasn't.
     */
    virtual bool selectAllChildren() = 0;

    virtual bool anythingSelected() = 0;

    virtual Aztec::MVector3 getSelectionCentre() = 0;

  };

}

#endif 
