#include <Aztec3DPCH.h>

// Aztec2 includes
#include <functions/FunctionManager.h>
#include <views/AztecViewManager.h>
#include <utils/SceneFunctions.h>
#include <functions/mesh/MeshFunctions.h>

// AztecLib includes
#include <MSystemManager.h>
#include <MUIManager.h>
#include <MEditableMesh.h>
#include <misc/MSceneHelper.h>

// Standard includes
#include <algorithm>
#include <assert.h>
#include <math.h>

namespace AztecGUI {

  using namespace Aztec;


  void edgeConnectMesh(const MEditableMeshPtr &mesh, int divisions) {
    EdgeConnectEdges inputEdges;
    std::vector<Edge> newEdges;

    for (int t = 0; t < mesh->getNumTris(); ++t) {
      for (int e = 0; e < 3; ++e) {
        if (mesh->isTriangleEdgeFlagged(t, e, EDGE_VISIBLE) &&
            mesh->isTriangleEdgeFlagged(t, e, EDGE_SELECTED)) 
        {
          int a, b;
          mesh->getVertsFromEdge(t, e, &a, &b);
          inputEdges[Edge(a, b)] = -1;
        }
      }
    }

    edgeConnectMesh(mesh, divisions, inputEdges, newEdges);

    // now go and reselect our edges.
    for (EdgeConnectEdges::iterator it = inputEdges.begin(); it != inputEdges.end(); ++it) {
      mesh->setEdgeFlag(it->first.a, it->second, EDGE_SELECTED);
      mesh->setEdgeFlag(it->second, it->first.b, EDGE_SELECTED);
    }

    for (int i = 0; i < newEdges.size(); ++i) {
      mesh->setEdgeFlag(newEdges[i].a, newEdges[i].b, EDGE_SELECTED);
    }
  }

  template <class A, class B, class C, class D>
  void connectIterators(const Aztec::MEditableMeshPtr &mesh, A a, B b, C c, D d, std::vector<Edge> &newEdges) {
    --b;
    --d;
    ++a;
    ++c;

    while (a != b) {
      pointConnectMesh(mesh, *a, *c);
      newEdges.push_back(Edge(*a, *c));

      ++a;
      ++c;
    }

  }

  void edgeConnectMesh(const Aztec::MEditableMeshPtr &mesh, int divisions, EdgeConnectEdges &inputEdges, std::vector<Edge> &newEdges) {
    for (EdgeConnectEdges::iterator it = inputEdges.begin(); it != inputEdges.end(); ++it) {
      it->second = -1;
    }

    typedef std::set<Edge> EdgeMap;
    typedef std::pair<Edge, Edge> EdgePair;
    typedef std::set< EdgePair > EdgeConnectSet;
    typedef std::vector<int> IntVector;

    EdgeConnectSet toJoin;
    EdgeMap edges;

    IntVector poly;
    std::set<IntVector> oldPolygons;

    // loop over the edges, and try to find any two edges that lie on the same polygon.
    std::vector<Edge> tempEdges;
    for (EdgeConnectEdges::iterator it = inputEdges.begin(); it != inputEdges.end(); ++it) {
      for (int side = 0; side < 2; ++side) {
        int a = it->first.a;
        int b = it->first.b;
        if (side != 0) {
          std::swap(a,b);
        }
        mesh->getPolygonVerticesFromEdge(a, b, true, poly);

        // now walk along the poly to find a second flagged edge.
        tempEdges.clear();
        for (int i = 0; i < poly.size(); ++i) {
          int p1 = poly[i];
          int p2 = poly[(i + 1) % poly.size()];

          // check to see if the new edge conflicts with any currently selected.
          if (inputEdges.find(Edge(p1,p2)) != inputEdges.end()) {
            bool ok = true;
            for (int k = 0; k < tempEdges.size(); ++k) {
              int a = -1;
              int b = -1;
              int c = -1;
              if (tempEdges[k].a == p1) {
                a = p1;
                b = tempEdges[k].b;
                c = p2;
              }
              if (tempEdges[k].b == p1) {
                a = p1;
                b = tempEdges[k].a;
                c = p2;
              }
              if (tempEdges[k].a == p2) {
                a = p2;
                b = tempEdges[k].b;
                c = p1;
              }
              if (tempEdges[k].b == p2) {
                a = p2;
                b = tempEdges[k].a;
                c = p1;
              }


              if (a != -1) {
                Aztec::MVector3 ab, ac;
                ab = mesh->getVertexPosition(b) - mesh->getVertexPosition(a);
                ac = mesh->getVertexPosition(c) - mesh->getVertexPosition(a);

                ab.normalize();
                ac.normalize();

                // if the two edges are parallel, then we 
                if (fabs(ab * ac) > 0.999) {
                  ok = false;
                  break;
                }
              }
            }

            if (ok) {
              tempEdges.push_back(Edge(p1,p2));
              edges.insert(tempEdges[0]);
            }
          }
        }

        // if we only have two selected edges, then create a connection between them.

        for (int aa = 0; aa < (int)tempEdges.size() - 1; ++aa) {
          for (int bb = aa + 1; bb < tempEdges.size(); ++bb) {
            if (tempEdges[aa] < tempEdges[bb]) {
              toJoin.insert( EdgePair(tempEdges[aa], tempEdges[bb]) );
            } else {
              toJoin.insert( EdgePair(tempEdges[bb], tempEdges[aa]) );
            }
          }
        }

        // store our old polygons so we can recreate it later.
        if (tempEdges.size() > 0) {
          // reorder the polygon so it starts off with the smallest vertx
          int smallest = poly[0];
          int smallestIndex = 0;
          for (int i = 1; i < poly.size(); ++i) {
            if (poly[i] < smallest) {
              smallest = poly[i];
              smallestIndex = i;
            }
          }
          std::rotate(poly.begin(), poly.begin() + smallestIndex, poly.end());

          IntVector revPoly = poly;
          std::reverse(revPoly.begin() + 1, revPoly.end());

          if (oldPolygons.find(revPoly) == oldPolygons.end()) {
            oldPolygons.insert(poly);
          }
        }

      }
    }

    // now create our vertices
    MVector3 a, b, c;

    int divideCount = divisions;

    std::map<Edge, std::vector<int> > newPointsMap;

    for (EdgeMap::iterator it = edges.begin(); it != edges.end(); ++it) {
      a = mesh->getVertexPosition(it->a);
      b = mesh->getVertexPosition(it->b);

      std::vector<int> points;
      int newPoint = it->a;
      points.push_back(it->a);

      for (int i = 0; i < divideCount; ++i) {
        c = a + ((float)(i+1) / (float)(divideCount+1)) * (b - a);
        newPoint = mesh->divideEdge(newPoint, it->b, c);

        points.push_back(newPoint);
      }
      points.push_back(it->b);

      newPointsMap[*it] = points;
    }

    newEdges.clear();
    for (EdgeConnectSet::iterator it = toJoin.begin(); it != toJoin.end(); ++it) {
      Edge edge1 = it->first;
      Edge edge2 = it->second;

      std::vector<int> pointsa = newPointsMap[edge1];
      std::vector<int> pointsb = newPointsMap[edge2];



      if (pointsa.front() == edge1.a) {
        if (pointsb.front() == edge2.a) {
          connectIterators(mesh, 
                           pointsa.begin(), 
                           pointsa.end(), 
                           pointsb.rbegin(), 
                           pointsb.rend(), newEdges);
        } else {
          connectIterators(mesh, 
                           pointsa.begin(), 
                           pointsa.end(), 
                           pointsb.begin(), 
                           pointsb.end(), newEdges);
        }
      } else {
        if (pointsb.front() == edge2.a) {
          connectIterators(mesh, 
                           pointsa.rbegin(), 
                           pointsa.rend(), 
                           pointsb.rbegin(), 
                           pointsb.rend(), newEdges);
        } else {
          connectIterators(mesh, 
                           pointsa.rbegin(), 
                           pointsa.rend(), 
                           pointsb.begin(), 
                           pointsb.end(), newEdges);
        }
      }
    }
  }

  class doEdgeConnect {
  public:
    int divisions;

    doEdgeConnect(int divs) {
      divisions = divs;
    }
    void operator()(const Aztec::MScenePtr &scene, const Aztec::MTreeObjectNodePtr &node) {
      MNamedObjectPtr namedObj = AZTEC_CAST(MNamedObject, node->getObject());
      MEditableMeshPtr mesh = getEditableMeshObject(namedObj);
      if (mesh == NULL) {
        MSystemManager::getInstance()->logOutput("Warning: edgeConnect() failed on object '%s', only operates on editable meshes", namedObj != NULL ? namedObj->getName().c_str() : "");
        return;
      }

      if (mesh->getComponentMode() == MComponentisedObject::EDGE_TYPE) {
        edgeConnectMesh(mesh, divisions);
      }
    }
  };

  int edgeConnect(const StringVector &args, std::string &result) {
    int divCount = 1;

    if (args.size() > 0) {
      divCount = strToInt(args[0]);
      if (divCount < 1) {
        divCount = 1;
      }
    }

    applyToObjects(MScene::getGlobalScene(), 
                   MScene::getGlobalScene()->getObjectList(), 
                   ifSelectedCriteria, 
                   doEdgeConnect(divCount));

    AztecViewManager::redrawAllViews();
	  
    return FunctionManager::SUCCEED;
  }


}

