#include <Aztec3DPCH.h>
#include <tools/MGraphInsertKeyTool.h>

// AztecMain includes
#include <views/AztecGraphView.h>

namespace AztecGUI {

  template <typename A, typename B> 
  A round(A a, B b) {
    return (A)(b*(int)(a/(A)b + 0.5));
  }

  MGraphInsertKeyTool::MGraphInsertKeyTool() {
    m_Dragging = false;
    m_RequiresSel = false;
  }

  static GraphComponent* getGraph(const Aztec::MComponentPtr &component) {
    GraphComponent *graph = AZTEC_CAST(GraphComponent, component);

    if (graph != NULL) {
      return graph;
    }

    // try and get our graph component
    AztecGraphView *graphView = AZTEC_CAST(AztecGraphView, component);
    if (graphView != NULL) {
      return graphView->getGraphComponent();
    }

    return NULL;
  }

  std::string MGraphInsertKeyTool::getName() {
    return "GraphInsertKeyTool";
  }

  int MGraphInsertKeyTool::drawTool(bool select, const Aztec::MComponentPtr &component) {
    GraphComponent *graph = getGraph(component);

    if (graph == NULL) {
      return 0;
    }

    if (m_Dragging) {
      float x1,y1,x2,y2;

      graph->clientToGraph(m_DownX, m_DownY, x1,y1);
      graph->clientToGraph(m_CurX, m_CurY, x2,y2);

      glColor3f(0.1f, 1.0f, 0.7f);
      glBegin(GL_LINE_LOOP);
      glVertex2f(x1,y1);
      glVertex2f(x2,y2);

      glEnd();
    }

    return 1;
  }

  int MGraphInsertKeyTool::onMouseDown(const Aztec::MMouseEvent &event) {
    MToolType::onMouseDown(event);

    // restore our cursor so we can see what we are doing.
    event.getComponent()->setMouseCursor(Aztec::Mouse_Normal);

    m_Dragging = true;
    m_CurX = m_DownX = event.getX();
    m_CurY = m_DownY = event.getY();

    GraphComponent *graph = getGraph(event.getComponent());

    if (graph == NULL) {
      return TOOLRESULT_DRAWNONE;
    }
  
    std::vector<GraphComponent::ListKey> keys;
    std::vector<GraphPair> graphValues;

    graph->getGraphValues(graphValues);

    currentKeyList = NULL;

    // loop over all keys lists and store the values so we can revert 
    // back to them as needed.
    for (int i = 0; i < graphValues.size(); ++i) {
      MFloatKeyListPtr keyList = AZTEC_CAST(MFloatKeyList, graphValues[i].value);

      if (keyList == NULL) {
        continue;
      }

      keyList->storeValue();

      // loop over the keys and deselect them all.
      for (int i = 0; i < keyList->getKeyCount(); ++i) {
        MKeyPtr key = keyList->getKeyAtIndex(i);

        key->setSelected(false);
      }
    }

    currentKeyList = graph->getKeyListAtMouse(event.getX(), event.getY());

    if (currentKeyList != NULL) {
      float x1,y1;
    
      graph->clientToGraph(m_DownX, m_DownY, x1, y1);
    
      int roundedTime = round(x1, currentKeyList->getGranularity());
    
      currentKeyList->setKey(
        currentKeyList->getValueAtTime(roundedTime),
        roundedTime);
    }

    return TOOLRESULT_DRAWALL;
  }

  int MGraphInsertKeyTool::onMouseUp(const Aztec::MMouseEvent &event)
  {
    MToolType::onMouseUp(event);

    GraphComponent *graph = getGraph(event.getComponent());

    if (graph == NULL) {
      return TOOLRESULT_DRAWNONE;
    }
  
    m_Dragging = false;

    m_UpX = event.getX();
    m_UpY = event.getY();

    currentKeyList = NULL;
  
    return TOOLRESULT_DRAWALL;
  }

  int MGraphInsertKeyTool::onMouseMove(const Aztec::MMouseEvent &event) {
    MToolType::onMouseMove(event);
  
    m_CurX = event.getX();
    m_CurY = event.getY();
  
    GraphComponent *graph = getGraph(event.getComponent());

    if (graph == NULL) {
      return TOOLRESULT_DRAWNONE;
    }

    if (m_Dragging && currentKeyList != NULL) {
      currentKeyList->restoreValue();

      float x1,y1;
    
      graph->clientToGraph(m_CurX, m_CurY, x1, y1);
    
      int roundedTime = round(x1, currentKeyList->getGranularity());
    
      currentKeyList->setKey(y1,roundedTime);

      return TOOLRESULT_DRAWALL;
    } else {

      // if we don't have a point selected, we just update the mouse cursor
      // to say we can insert a key.
      MFloatKeyListPtr list = graph->getKeyListAtMouse(event.getX(), event.getY());

      event.getComponent()->setMouseCursor( list != NULL ? Aztec::Mouse_SizeAll : Aztec::Mouse_Normal );
    } 
  
    return TOOLRESULT_DRAWNONE;
  }


}