#ifndef MFileDialog_Header
#define MFileDialog_Header

#include <MRefCountedPtr.h>

namespace Aztec {

  class IMFileDialog;
  class MFileDialog;

  typedef MRefCountedPtr<IMFileDialog> IMFileDialogPtr;
  typedef MRefCountedPtr<MFileDialog> MFileDialogPtr;

}

#include <gui/MGuiBase.h>
#include <string>

namespace Aztec {

  class AZTECGUI_EXPORT IMFileDialog : public virtual IMGuiBase {
  protected:
    IMFileDialog() { }
    ~IMFileDialog() { }

  public:
    /**
     * This is the list some of the special return values that can be returned
     * by getSelectedFilter() or be used with setInitialFilter().
     *
     * @see getSelectedFilter,setInitialFilter 
     */
    typedef enum { ALL_FILES = -2, ALL_KNOWN_FILES = -1 } FilterEnum;

    /**
     * Sets the filename that will appear in the filename field in the dialog 
     * when the dialog first appears.
     */
    virtual void setFilename(const std::string &filename) = 0;

    /**
     * Sets the path that the dialog will open aup at
     */
    virtual void setPath(const std::string &path) = 0;

    /**
     * Adds a filter to appear in the file dialog.
     */
    virtual void addFilter(const std::string &description, const std::string &filter) = 0;

    /**       
     * This indicates what filter to use first when the dialog first appears. 
     * If this method is not called, the ALL_KNOWN_FILES filter is used by 
     * default.
     * 
     * @param The 0 based index of the filter to use, or one of the special 
     *        values from FilterEnum
     * @see The FilterEnum type, and the getSelectedFilter method.
     * 
     */
    virtual void setFilter(int index) = 0;

    /**
     * Shows the dialog in a modal style, so the user cannot do anything but 
     * choose a file, or cancel the dialog.
     *
     * @return true if the OK button was clicked, false if the CANCEL button 
     *         was chosen.
     */
    virtual bool doModal() = 0;

    /**
     * Gets the full path and filename of the file that was chosen. In the 
     * case of multiply selected files, the first filename is returned.
     *
     * @return the complete filename of the file selected.
     */
    virtual std::string getFilename() = 0;

    /**
     * This gets the selected filter that was used to select the file. It is 
     * the zero based index into the filters that were added using addFilter. 
     * In addition to that, it returns ALL_FILES if the "All Files *.*" 
     * filter was used, and ALL_KNOWN_FILES if the combination of all the 
     * known filters was used.
     */
    virtual int getFilter() = 0;

    /**
     * Sets the title that will appear in the dialog when it is opened.
     */
    virtual void setTitle(const std::string &title) = 0;

  };

}


/**
 * The following header is included via the AZTEC_GUI
 * variable that should be defined for each platform
 * and used to specify an include directory
 * (ie. ../AztecGUICommon/include/gui/$(AZTEC_GUI)
 * when compiling this lib
 */
#include <MFileDialogImpl.h>


#endif

