#ifndef MToolManager_Header
#define MToolManager_Header

#include "MToolType.h"

#include <list>
#include <map>

namespace AztecGUI {

  //----------------
  //  MToolManager
  //----------------
  #define TOOLMAN_MAXNODES      10

  class MToolChangeListener;
  typedef Aztec::MRefCountedPtr<MToolChangeListener> MToolChangeListenerPtr;

  class AZTECGUI_EXPORT MToolChangeListener : public Aztec::MRefCountedObject {
  public:
    typedef MToolChangeListenerPtr Ptr;

    virtual void onToolChange(const std::string &group, MToolTypePtr oldTool) = 0;
  };

  class AZTECGUI_EXPORT MToolManager {
  public:
    MToolManager();
    virtual ~MToolManager();

    static MToolManager* getInstance();
    static void cleanup();
    
    /// Registers a tool with the name tool->getName().
    void registerTool(const MToolTypePtr &tool);

    /// Registers a tool to a specific name, instead of using the default tool name.
    void registerTool(const MToolTypePtr &tool, const std::string &name);
    MToolTypePtr getToolByName(const std::string &name);

    void setTool(const MToolTypePtr &newTool, const std::string &group);
    void PushTool(const MToolTypePtr &newTool, const std::string &group);
    MToolTypePtr PopTool(const std::string &group);
  
    MToolTypePtr GetTool(const std::string &group);
    MToolTypePtr GetTool(int index, const std::string &group);
  
    /**
     * This sets the default tool to be used for the given viewgroup. If we 
     * set the default for the empty group, we set the global default tool.
     *
     * When a tool is requested for a view group, but we don't have any. 
     * First it checks for the view specific default tool, and if we don't 
     * have one of those, use the global default tool.
     *
     */
    void setDefault(const MToolTypePtr &tool, const std::string &group = "");

    /**
     * Gets the default tool for the given group. If no group is given, it 
     * gets the global default tool. Also, if there is no default tool set for
     * the givn group name, it will return the global default tool.
     *
     * In debug mode, it will assert if there is no tool found.
     */
    MToolTypePtr getDefault(const std::string &group = "");

    /**
     * Sets the current temporary tool. Temporary tools are those tools which 
     * are only used for a short time. Things like camera panning, dolly-ing 
     * and things like that are examples of common temporary tools.
     */
    void setTemporary(const MToolTypePtr &Tool, const std::string &group);

    /**
     * This removes the current temporary tool, if any, and reverts back to 
     * the previous tool.
     */
    void unsetTemporary(const std::string &group);

    /**
     * Check to see if the current tool is a temporary one.
     */
    bool isTemporary(const std::string &group);

    /**
     * This sets up the default tool to use for a category when no other tool
     * is selected.
     */
    void setDefaultTool(const std::string &group, const MToolTypePtr &defaultTool);
  
    void addListener(MToolChangeListenerPtr listener, const std::string &group);
    void removeListener(MToolChangeListenerPtr listener, const std::string &group);
    void removeAllListeners(const std::string &group);

  protected:
    typedef std::list<MToolTypePtr> ToolList;
    typedef std::map<std::string, ToolList> ToolListMap;

    typedef std::map<std::string, MToolTypePtr> StringToolMap;

    bool m_TempTool;    // True if topmost tool is temporary

    ToolListMap toolMap;
    std::set<MToolChangeListenerPtr> m_Listeners;
    StringToolMap registeredTools;
    StringToolMap defaultTools;

    void notifyListeners(MToolTypePtr oldTool, const std::string &group);

    ToolList& getToolList(const std::string &group);
  };

}
#endif

