#ifndef MFloatKeyList_Header 
#define MFloatKeyList_Header

#include "ModelGeneric.h"

namespace Aztec {

  class MFloatKeyList;
  typedef MRefCountedPtr<MFloatKeyList> MFloatKeyListPtr;

}

#include "MKeyList.h"
#include "MFloatKeyableValue.h"

#include "MMath.h"

#include <MBaseUndoNode.h>

#include <vector>

namespace Aztec {

  /**
   * This is an interface class for a list of Keys. Each type
   * of key list implements this interface.
   */
  class MGENEXPORT MFloatKeyList : public MKeyList,
                                   public MFloatKeyableValue
  {
  public:
    MFloatKeyList();
    virtual ~MFloatKeyList();

    /**
     * This gets the value of the key at the given index.
     */
    float getValueAtIndex(int index);
    /**
     * This gets the time of the key at the given index.
     */
    long getTimeAtIndex(int index);

    // MFloatKeyableValue methods
    void setInitialValue(float value);
    void setKey(float value, long time);

    // MFloatValue methods
    float getValueAtTime(long time);

    // MKeyableValue methods
    bool hasKeySet(long time);
    void shiftKeys(long onOrAfterTime, long beforeTime, long amount);
    void shiftSelectedKeys(long amount);
    void getKeyTimes(std::vector<int> &keyTimes);
    void getSelectedKeyTimes(std::vector<int> &keyTimes);
    void deleteKey(long time);
    void deleteSelectedKeys();
    bool hasKeys();

    // MKeyList methods
    int getKeyCount();
    MKeyPtr getKeyAtIndex(int index);
    MKeyPtr getKeyAtTime(long time);
    void setKey(MKeyPtr key, long time);
    void copyFromKeyList(MKeyListPtr source);
    int getGranularity();
    void setGranularity(int granularity);

    // MValue methods
    bool copyFromValue(MValuePtr source);
    void storeValue();
    void restoreValue();

    // MBaseObject methods
    MStr getClassName() {return MStr("MFloatKeyList");}
    MStr getParentClassName() {return MStr("MKeyList");}
    MBaseObjectPtr createNew();

    // MUndoableObject methods
    void finishWithUndoNode();

  private:
    friend class FloatKey;
    friend class MFloatKeyImpl;

    class FloatKey {
    public:
      FloatKey() { 
        flags.setFlag(IN_INVALID);
        flags.setFlag(OUT_INVALID);
        selected = false;
      }
      FloatKey(const FloatKey &src) 
        : time(src.time), value(src.value), in(src.in), out(src.out), flags(src.flags),
          selected(src.selected)
      {

      }
      /**
       * Says that the tangents should be computed from surrounding points.
       */
      static const AztecFlags IN_SMOOTH;
      /**
       * Indicates that the tangents are user editable.
       */
      static const AztecFlags IN_FREE;

      /**
       * says that the tangent in question goes in a straight line 
       * towards the adjoining point.
       */
      static const AztecFlags IN_LINEAR;

      /**
       * Indicates that the tangent is invalid, and needs updating. This
       * is only applicable when we have a smooth or linear tangent.
       */
      static const AztecFlags IN_INVALID;
      /**
       * Says that the tangents should be computed from surrounding points.
       */
      static const AztecFlags OUT_SMOOTH;
      /**
       * Indicates that the tangents are user editable.
       */
      static const AztecFlags OUT_FREE;

      /**
       * says that the tangent in question goes in a straight line 
       * towards the adjoining point.
       */
      static const AztecFlags OUT_LINEAR;

      /**
       * Indicates that the tangent is invalid, and needs updating. This
       * is only applicable when we have a smooth or linear tangent.
       */
      static const AztecFlags OUT_INVALID;

      /**
       * This is the time this key is at
       */
      long time;
      /**
       * The value of the key
       */
      float value;

      /**
       * The tangent as we go towards this key
       */
      float in;
      /**
       * The tangent as we head out of this key.
       */
      float out;

      /**
       * The flags determining what the tangents two. The first 4 bits are for 
       * the incoming tangent, the second 4 for the outgoing.
       * 
       */
      MFlagObject flags;

      bool selected : 1;

      /**
       * < operation using time
       */
      bool operator<(const FloatKey &rhs) const {
        return time < rhs.time;
      }

      bool operator<(int rhs) const {
        return time < rhs;
      }

    private:

    };

    std::vector<FloatKey> keys;

    int granularity;

    float initialValue;

    std::vector<FloatKey> storedKeys;

    int storedGranularity;

    float storedInitialValue;

    class FloatKeyListUndoNode : public MBaseUndoNode {
    public:
      FloatKeyListUndoNode(const MFloatKeyListPtr &keyList);

      // MBaseUndoNode Methods
      MUndoableObject* getObject();
      void undo();
      void redo();

    private:
      MFloatKeyListPtr list;

      std::vector<FloatKey> keys;
      int granularity;
      float initialValue;
    };

    typedef MRefCountedPtr<FloatKeyListUndoNode> FloatKeyListUndoNodePtr;

    FloatKeyListUndoNodePtr undoNode;

    /**
     * This gets the index of the first key that is >= time, i.e. not less than time.
     */
    int getNearestKeyIndex(long time);

    /**
     * Gets the index of the key at the given time. If no key exists at that
     * time, -1 is returned.
     */
    int getKeyIndexAtTime(long time);

    float getInTangent(int keyIndex);

    float getOutTangent(int keyIndex);

    void setInTangent(int keyIndex, float tangent);

    void setOutTangent(int keyIndex, float tangent);

    /**
     * This invalidates the tangents, and any other information, of
     * the key at the given index. 
     *
     * @param index The index of the key to invalidate.
     */
    void invalidateKey(int index);

    void ensureUndo();

    friend class FloatKeyTimeComparator;
    friend class FloatKeyListUndoNode;
  };
  

}

#endif 
