#ifndef MLineMesh_Header
#define MLineMesh_Header

#include "ModelGeneric.h"

namespace Aztec {

  class MLineMesh;
  typedef MRefCountedPtr<MLineMesh> MLineMeshPtr;

}

#define TRIFLAGS     Aztec::AztecFlags
#define VERTFLAGS    Aztec::AztecFlags

#define VERTEX_SELECTED       ((Aztec::AztecFlags)Aztec::MComponentisedObject::COMPONENT_SELECTED)
#define VERTEX_VISIBLE        (Aztec::AztecFlags)0x02
#define VERTEX_FLAGFORCHANGE  (Aztec::AztecFlags)0x04
#define VERTEX_DRAW_FEEDBACK  (Aztec::AztecFlags)0x08

#define EDGE_SELECTED   ((Aztec::AztecFlags)Aztec::MComponentisedObject::COMPONENT_SELECTED)
#define EDGE_VISIBLE    (Aztec::AztecFlags)0x02

#include "MShapeObject.h"
#include "MBaseObject.h"
#include "MComponentisedObject.h"

#include "MMath.h"
#include "MMaterial.h"

#include <MVertex.h>

#include <vector>
#include <set>

namespace Aztec {

  /**
   * A line mesh is contains a series of SubLineMeshes.
   */
  class MGENEXPORT MLineMesh : public MNamedObject, 
                               public MEditableComponentisedObject {
  public:
	  MLineMesh();
    MLineMesh(MLineMeshPtr src);
	  virtual ~MLineMesh();
    
    /**
     * Gets the number of vertices in the polyline
     */
    int getVertexCount() const;

    /**
     * This adds a vertex to the mesh. If the index parameter is omitted
     * then the point is added at the end of the list of points.
     */
    int addVertex(const MVector3 &pos);

    /**
     * Removes the vertex at the given index.
     */
    void removeVertex(int index);

    /**
     * Gets the position of the vertex at the given index.
     */
    MVector3 getVertexPosition(int index) const;

    /**
     * Gets the position of the vertex at the given index.
     */
    void setVertexPosition(int index, const MVector3 &newPosition);

    /**
     * Gets the number of edges in this poly line.
     */
    int getEdgeCount() const;

    /**
     * This adds an edge to the list. It will add the edge in sorted by the
     * vertices it connected so the indicies of some edges will be changed.
     */
    void addEdge(int from, int to);

    /**
     * Removes an edge fro the polyline. It will remove the edge at the 
     * given index.
     */
    void removeEdge(int index);

    /**
     * Removes an edge from the polyline. It will remove the edge that
     * connects the two given vertices.
     */
    void removeEdge(int from, int to);

    /**
     * This gets the vertices of the edge at the given index.
     */
    void getEdge(int index, int &from, int &to);

    /**
     * This gets the edges that are associated with the given point.
     * 
     */
    void getEdgesForVertex(int vertexIndex, std::vector<int> &edgeIndices);

    /**
     * This gets the flags for the edge at the given index.
     */
    MFlagObject& getEdgeFlags(int index);

    /**
     * This gets the flags for the edge at the given index.
     */
    const MFlagObject& getEdgeFlags(int index) const;

    /**
     * Is this polyline closed?
     */
    bool isClosed();
    
    MMeshPtr convertToMesh();
    bool drawObject(const MBaseObjectPtr &baseObj, MSceneViewFlags ViewFlags);
    
    // MBaseObject methods
    MStr getClassName() {return MStr("MLineMesh");};
    MStr getParentClassName() {return MStr("MBaseObject");};
    MBaseObjectPtr createNew();
    void setFrom(MBaseObjectPtr SrcObj);
    
    // MComponentisedObject methods
    bool isInComponentMode(AztecFlags type = POINT_TYPE | EDGE_TYPE | FACET_TYPE);
    void setComponentMode(AztecFlags type);
    AztecFlags getComponentMode();
    bool hasComponentType(AztecFlags type);
    int getComponentCount(AztecFlags type);
    bool flagComponent(AztecFlags type, int index, AztecFlags flag = COMPONENT_SELECTED, MAction action = atSet);
    bool isComponentFlagged(AztecFlags type, int index, AztecFlags flag = COMPONENT_SELECTED);
    bool hasComponentsFlagged(AztecFlags type, AztecFlags flag = COMPONENT_SELECTED);
    bool setComponentFeedbackColour(AztecFlags type, int index, const MRGBAFloat &colour);
    bool unsetComponentFeedbackColour(AztecFlags type, int index);
    MRGBAFloat getComponentFeedbackColour(AztecFlags type, int index);
    bool flagConnected(ComponentType type, AztecFlags flag = COMPONENT_SELECTED);
    bool growFlagged(ComponentType type, AztecFlags flag = COMPONENT_SELECTED);
    bool shrinkFlagged(ComponentType type, AztecFlags flag = COMPONENT_SELECTED);
    MVector3 getComponentCentre(AztecFlags type, int index);
    MVector3 getFlaggedCentre(AztecFlags type, AztecFlags flag = COMPONENT_SELECTED);
    MVector3 getComponentNormal(AztecFlags type, int index);

    // MEditableComponentisedObject methods
    void storeComponentPositions(AztecFlags type, AztecFlags flag = 0);
    void restoreComponentPositions(AztecFlags type, AztecFlags flag = 0);
    void transformComponents(AztecFlags type, 
                             const MMatrix4 &transform, 
                             AztecFlags flag = 0, 
                             bool localTransform = false,
                             TransformModifier *modifier = NULL);
    void updateComponentAnimation(AztecFlags type, AztecFlags flag, long time, bool createKey);
    void updateSingleComponentAnimation(AztecFlags type, int index, long time, bool createKey);

  protected:

  private:
    class Vertex : public MFlagObject {
    public:
      static const DWORD NormalValid;

      MVector3 m_Pos;
      MRGBAFloat m_FeedbackColor;
      MVector3 normal;

      Vertex();
      Vertex(const Vertex &src);

      Vertex(const MVector3 &src, DWORD flags = 0);
      Vertex(float x, float y, float z);
    };

    class Edge {
    public:
      static const DWORD Selected;
      static const DWORD Reversed;

      MFlagObject flags;

      Edge(int vertexA, int vertexB, AztecFlags flags = 0);
      Edge(const Edge &rhs);
      void getVertices(int &vertexA, int &vertexB) const;

      /**
       * Adjusts this edge according to the given vertex being removed.
       * If the edge has a vertex that is being removed, the function
       * returns true, which inicates this edge should be deleted.
       * If the edge does not contain a vertex equal to vertexIndex,
       * then false is returned, indicating we do not have to remove
       * the edge as well.
       *
       * @param vertexIndex The index of the vertex that has/will be removed.
       * @return True if the edge should be deleted, false if it shouldn't.
       */
      bool vertexRemoved(int vertexIndex);

      bool operator<(const Edge &rhs) const;
      bool operator<(int firstVertex) const;
      bool operator==(const Edge &rhs) const;

    private:
      int from;
      int to;

    };

    typedef std::vector<Vertex> VertexList;
    typedef std::vector<Edge> EdgeList;

    VertexList vertices;
    EdgeList edges;

    VertexList storedVertices;

    AztecFlags componentMode;

    void drawEdges(const MBaseObjectPtr &baseObj, const MSceneViewFlags &ViewFlags);
    void drawPoints(const MBaseObjectPtr &baseObj, const MSceneViewFlags &ViewFlags);
  };

}
#endif
