#ifndef MRefrenceCounted_Pointer_H
#define MRefrenceCounted_Pointer_H

#include "ModelGeneric.h"
#include <stdlib.h> // included for the NULL identifier

#ifdef WIN32
#pragma warning (disable : 4786)
#endif

namespace Aztec {

  /**
   * This is an automatic Ref counted object. It keeps a
   * Ref count, and has methods to increment and decrement
   * the refernce counter. It is a friend to the tmeplate class MRefCountedPtr
   */
  class MGENEXPORT MRefCountedObject {
  public:
    /**
     * Defualt constructor. Initialises the reference coutner to zero.
     */
    MRefCountedObject() : m_RefCount(0) {  }

    /**
     * Virtual destructor to ensure proper destruction
     */
    virtual ~MRefCountedObject() { };

    /**
     * Returns the reference count for this object.
     * @return the reference count.
     */
    int getRefCount() { return m_RefCount; }

    /**
     * This increments the reference count. 
     */
    void incrementRefCount() { m_RefCount++; }

    /**
     * This decrements the reference count. If the reference
     * count falls to zero or below, the object is detroyed
     * via a <code>delete this</code> call.
     */
    void decrementRefCount() {
      m_RefCount--;
      if (m_RefCount <= 0) {
        delete this;
      }
    }

  private:
    /**
     * This is the reference count for this object.
     */
    int m_RefCount;
  };

  /**
   * This is a template class for reference counted 'smart' pointers.
   * This is a wrapper class for all objects derived from 
   * <code>MRefCountedObject</code>. The template parameter T is
   * the class to wrap around. T must be derived from MRefCountedObject,
   * or basically must have the methods <code>incrementRefCount()</code> 
   * and <code>decrementRefCount()</code>.
   */
  template <class T> class MRefCountedPtr {
  public:
    typedef T PtrType;
    /**
     * default constructor. Initialise the pointer to NULL
     */
    MRefCountedPtr() : m_Ptr(NULL) { }

    /**
     * Constructor for taking an ordinary pointer type.
     */
    MRefCountedPtr(T *ptr) : m_Ptr(ptr) {
      if (m_Ptr != NULL) {
        m_Ptr->incrementRefCount();
      }
    }

    /**
     * Copy constructor using a template paramter so we can easily convert
     * ref counted ptr's from other types.
     */
    template <class A> 
      MRefCountedPtr( const MRefCountedPtr<A> &ptr ) { m_Ptr = &(*ptr); if (m_Ptr) m_Ptr->incrementRefCount(); }

    /**
     * Specialised copy constructor
     * @param ptr This is a reference counted pointer of the same
     * class T type.
     * type as this template 
     */
    MRefCountedPtr( const MRefCountedPtr<T> &ptr ) {
      m_Ptr = &(*ptr); 
      if (m_Ptr) {
        m_Ptr->incrementRefCount();
      }
    }

    /**
     * Destructor, so we decrement the reference count.
     */
    ~MRefCountedPtr() {
      if (m_Ptr != NULL) {
        m_Ptr->decrementRefCount();
      }
    }

    /**
     * Indirection operator, get the value that m_Ptr points to.
     */
    T& operator*(void) const { return *m_Ptr; }

    /**
     * Pointer to member operator
     */
    T* operator->(void) const { return m_Ptr; }

    /**
     * assignment operator for a reference counted pointer
     */
    template <class A> 
    MRefCountedPtr& operator=(const MRefCountedPtr<A> &p) {
      T *oldPtr = m_Ptr;
      m_Ptr = p.m_Ptr;
      if (m_Ptr != NULL) m_Ptr->incrementRefCount();
      if (oldPtr != NULL) oldPtr->decrementRefCount();

      return *this;
    }

    /**
     * Specialisation assignment operator. This is so
     * tempalte of the same class T type can be assigned.
     */
    MRefCountedPtr& operator=(const MRefCountedPtr<T> &p) {
      T *oldPtr = m_Ptr;
      m_Ptr = p.m_Ptr;
      if (m_Ptr != NULL) m_Ptr->incrementRefCount();
      if (oldPtr != NULL) oldPtr->decrementRefCount();

      return *this;
    }

    /**
     * assignment operator for a plain pointer type
     */
    MRefCountedPtr& operator=(T *p) {
      T *oldPtr = m_Ptr;
      m_Ptr = p;
      if (m_Ptr != NULL) m_Ptr->incrementRefCount();
      if (oldPtr != NULL) oldPtr->decrementRefCount();

      return *this;
    }

    /**
     * Comparison operator
     */
    bool operator==(const T *p) const {
      return (m_Ptr == p);
    }

    /**
     * Comparison operator for all refernce counted poitner types.
     */
    template <class A> 
    bool operator==(const MRefCountedPtr<A> &p) const {
      return (m_Ptr == p.m_Ptr);
    }

    /**
     * Specialised Comparison operator, for tempalte classes
     * of this same class T type.
     */
    bool operator==(const MRefCountedPtr<T> &p) const {
      return (m_Ptr == p.m_Ptr);
    }

    /**
     * inequality operator
     */
    bool operator!=(const T *p) const {
      return (m_Ptr != p);
    }

    /**
     * General inequality operator
     */
    template <class A> 
    bool operator!=(const MRefCountedPtr<A> &p) const {
      return (m_Ptr != &*p);
    }
    
    /**
     * This is the specialised inequality operator
     * for refernce counted poitner class of the class T type.
     */
    bool operator!=(const MRefCountedPtr<T> &p) const {
      return (m_Ptr != &*p);
    }

    bool operator<(const MRefCountedPtr<T> &p) const {
      return (m_Ptr < &*p);
    }
    
    bool operator>(const MRefCountedPtr<T> &p) const {
      return (m_Ptr > &*p);
    }


    T *m_Ptr;
  };

  typedef MRefCountedPtr<MRefCountedObject> MRefCountedObjectPtr;

}

/**
 * This is a helper macor to allow dynamic_cast's to be used
 * with smart pointers in a more compact and readable manner
 *
 * @param T The class type to cast TO.
 * @param Var the pointer to convert from. This may
 * be a smart or regular pointer.
 */
#define AZTEC_CAST(T, Var) dynamic_cast<T*>(&*Var)



#endif
