#include "StdAfx.h"

#include <MLineMesh.h>

#include <assert.h>
#include <algorithm>
#include <math.h>

namespace Aztec {


  MLineMesh::MLineMesh() {
  }

  MLineMesh::MLineMesh(MLineMeshPtr src) {
  }

	MLineMesh::~MLineMesh() {
  }
  
  int MLineMesh::getVertexCount() const {
    return vertices.size();
  }

  int MLineMesh::addVertex(const MVector3 &pos) {
    vertices.push_back(Vertex(pos));
    return vertices.size()-1;
  }

  void MLineMesh::removeVertex(int index) {
    // loop over all the edges and adjust the vertex indices
    if (edges.size() != 0) {
      for (int i = edges.size() - 1; i >= 0; --i) {
        if (edges[i].vertexRemoved(index)) {
          edges.erase(edges.begin() + i);
        }
      }
    }

    vertices.erase(vertices.begin() + index);
  }

  MVector3 MLineMesh::getVertexPosition(int index) const {
    return vertices[index].m_Pos;
  }

  void MLineMesh::setVertexPosition(int index, const MVector3 &newPosition) {
    vertices[index].m_Pos = newPosition;
  }

  int MLineMesh::getEdgeCount() const {
    return edges.size();
  }

  void MLineMesh::addEdge(int from, int to) {
    Edge edge(from,to);
    EdgeList::iterator it(std::lower_bound(edges.begin(), edges.end(), edge)); 

    // check to see if we already have that edge.
    if (it != edges.end() && (*it) == edge) {
      // if we do, do nothing.
    } else {
      edges.insert(it, edge);
    }
  }

  void MLineMesh::removeEdge(int index) {
    edges.erase(edges.begin() + index);
  }

  void MLineMesh::removeEdge(int from, int to) {
    Edge edge(from,to);
    EdgeList::iterator it(std::lower_bound(edges.begin(), edges.end(), edge)); 

    if (it != edges.end() && (*it) == edge) {
      edges.erase(it);
    }
  }

  void MLineMesh::getEdge(int index, int &from, int &to) {
    edges[index].getVertices(from,to);
  }

  void MLineMesh::getEdgesForVertex(int vertexIndex, std::vector<int> &edgeIndices) {
    // TODO: do a better search than this.

    edgeIndices.clear();
    int from, to;
    for (unsigned int i = 0; i < edges.size(); ++i) {
      getEdge(i, from, to);
      if (from == vertexIndex || to == vertexIndex) {
        edgeIndices.push_back(i);
      }
    }

  }

  MFlagObject& MLineMesh::getEdgeFlags(int index) {
    return vertices[index];
  }

  const MFlagObject& MLineMesh::getEdgeFlags(int index) const {
    return vertices[index];
  }

  bool MLineMesh::isClosed() {
    // TODO: cache this value
    
    // TODO: Calculate this better

    // go through and make sure each vertex as at least two edges. Three or 
    // more edges should be illegal, but we will worry about that another day.
    std::vector<int> vertexCount;

    vertexCount.resize(getVertexCount());

    // loop over all the edges, and count up how many edges are on each vertex
    EdgeList::iterator it;
    for (it = edges.begin(); it != edges.end(); ++it) {
      int a,b;

      (*it).getVertices(a,b);
      ++vertexCount[a];
      ++vertexCount[b];
    }

    // now loop over all the vertex counts, and make
    // sure each vertex has a count of two or more.
    for (unsigned int i = 0; i < vertexCount.size(); ++i) {
      if (vertexCount[i] < 2) {
        return false;
      }
    }

    return true;
  }

  
  MMeshPtr MLineMesh::convertToMesh() {
    return NULL;
  }

  bool MLineMesh::drawObject(const MBaseObjectPtr &baseObj, MSceneViewFlags ViewFlags) {

    drawEdges(baseObj, ViewFlags);

    bool doDrawPoints = isInComponentMode();

    if (doDrawPoints) {
      drawPoints(baseObj, ViewFlags);
    }

    return true;
  }
  
  // MBaseObject methods
  MBaseObjectPtr MLineMesh::createNew() {
    MLineMeshPtr newMesh = new MLineMesh();
    newMesh->setFrom(this);
    return newMesh;
  }

  void MLineMesh::setFrom(MBaseObjectPtr SrcObj) {
    MLineMesh *mesh = AZTEC_CAST(MLineMesh, SrcObj);

    if (mesh == NULL) {
      return;
    }

    vertices = mesh->vertices;
    edges = mesh->edges;
  }

  // MComponentisedObject methods
  bool MLineMesh::isInComponentMode(AztecFlags type) {
    return (componentMode & type) != 0;
  }

  void MLineMesh::setComponentMode(AztecFlags type) {
    componentMode = type;
  }

  AztecFlags MLineMesh::getComponentMode() {
    return componentMode;
  }

  bool MLineMesh::hasComponentType(AztecFlags type) {
    switch (type) {
    case MComponentisedObject::POINT_TYPE :
      return true;
    default:
      return 0;
    }
  }

  int MLineMesh::getComponentCount(AztecFlags type) {
    switch (type) {
    case MComponentisedObject::POINT_TYPE :
      return getVertexCount();
      break;

    default:
      return 0;
    }
  }
 
  bool MLineMesh::flagComponent(AztecFlags type, int index, AztecFlags flag, MAction action) {
    switch (type) {
    case MComponentisedObject::POINT_TYPE :
      switch (action) {
      case atSet:
        vertices[index].setFlag(flag);
        break;
      case atUnset:
        vertices[index].unsetFlag(flag);
        break;
      case atToggle:
        vertices[index].toggleFlag(flag);
        break;
      }
      return true;

    default:
      return false;
    }
  }

  bool MLineMesh::isComponentFlagged(AztecFlags type, int index, AztecFlags flag) {
    switch (type) {
    case MComponentisedObject::POINT_TYPE :
      return vertices[index].isFlagged(flag);

    default:
      return false;
    }
  }

  bool MLineMesh::hasComponentsFlagged(AztecFlags type, AztecFlags flag) {
    switch (type) {
    case MComponentisedObject::POINT_TYPE :
      for (unsigned int i = 0;i < vertices.size(); ++i) {
        if (vertices[i].isFlagged(flag)) {
          return true;
        }
      }
      return false;

    default:
      return false;
    }
  }

  bool MLineMesh::setComponentFeedbackColour(AztecFlags type, int index, const MRGBAFloat &colour) {
    // TODO:
    return false;
  }

  bool MLineMesh::unsetComponentFeedbackColour(AztecFlags type, int index) {
    // TODO:
    return false;
  }

  MRGBAFloat MLineMesh::getComponentFeedbackColour(AztecFlags type, int index) {
    // TODO:
    return MRGBAFloat(0,0,0,0);
  }

  bool MLineMesh::flagConnected(ComponentType type, AztecFlags flag) {
    // TODO:
    return false;
  }

  bool MLineMesh::growFlagged(ComponentType type, AztecFlags flag) {
    // TODO:
    return false;
  }

  bool MLineMesh::shrinkFlagged(ComponentType type, AztecFlags flag) {
    // TODO:
    return false;
  }

  MVector3 MLineMesh::getComponentCentre(AztecFlags type, int index) {
    switch (type) {
    case MComponentisedObject::POINT_TYPE :
      return vertices[index].m_Pos;

    default:
      return MVector3(0,0,0);
    }
  }

  MVector3 MLineMesh::getFlaggedCentre(AztecFlags type, AztecFlags flag) {
    MVector3 centre;
    int count = 0;

    switch (type) {
    case OBJECT_TYPE:
    case POINT_TYPE:
      for (VertexList::iterator it = vertices.begin(); it != vertices.end(); ++it) {
        if (type == OBJECT_TYPE || flag == 0 || (*it).isFlagged(flag)) {
          centre += (*it).m_Pos;
          ++count;
        }
      }
      break;
    case EDGE_TYPE:
      for (EdgeList::iterator it = edges.begin(); it != edges.end(); ++it) {
        if (flag == 0 || (*it).flags.isFlagged(flag)) {
          int from, to;
          (*it).getVertices(from,to);
          centre += 0.5*(vertices[from].m_Pos + vertices[to].m_Pos);
          ++count;
        }
      }
      break;
    default:
      return centre;
    }

    if (count != 0) {
      centre /= (float)count;
    }

    return centre;
  }

  MVector3 MLineMesh::getComponentNormal(AztecFlags type, int index) {
    switch (type) {
    case MComponentisedObject::POINT_TYPE :
      {
        if (!vertices[index].isFlagged(Vertex::NormalValid)) {

          std::vector<int> edgeIndices;
          getEdgesForVertex(index, edgeIndices);

          if (edgeIndices.size() == 0) {
            vertices[index].normal.set(0,0,1);
          } else if (edgeIndices.size() == 1)  {
            // if we only have one edge on this vertex,
            // out nomal must be pointing in the opposite direction
            // of that edge.
            int from, to;
            getEdge(edgeIndices[0], from, to);
            vertices[index].normal = vertices[from].m_Pos - vertices[to].m_Pos;
            vertices[index].normal.normalize();
            if (to == index) {
              vertices[index].normal *= -1.0;
            }
          } else if (edgeIndices.size() == 2) {
            // since we have two edges, the normal is the vector that bisects
            // the two edges.
            int from, to;
            MVector3 vecA, vecB;

            int point1, point2 = index, point3;

            getEdge(edgeIndices[0], from, to);

            // we want the first edge, so the first edge will have
            // the second vertex as our point of interest.
            if (to == index) {
              point1 = from;
            } else {
              point3 = to;
            }

            getEdge(edgeIndices[1], from, to);

            // we want the second edge, so the second edge will have
            // the first vertex as our point of interest.
            if (to == index) {
              point1 = from;
            } else {
              point3 = to;
            }

            vecA = vertices[point2].m_Pos - vertices[point1].m_Pos;
            vecB = vertices[point3].m_Pos - vertices[point2].m_Pos;

            vecA.normalize();
            vecB.normalize();
            MVector3 axis = vecA / vecB;
            bool normalFlipped = false;
            if (axis.z < 0) {
              axis *= -1.0;
              normalFlipped = true;
            }
//            vecA = axis / vecA;
//            vecB = axis / vecB;
            vecA.normalize();
            vecB.normalize();
            float dot = vecA * vecB;
            float ang = (float)acos(dot);

            if (normalFlipped) {
              ang = -ang;
            }
            rotateWithVector(vecA, axis, ang/2.0f + 3.14159f/2.0f, vertices[index].normal);
          } else {
            assert(0);
          }

        }

//        vertices[index].setFlag(Vertex::NormalValid);


        return vertices[index].normal;
      }

    default:
      return MVector3(0,0,1);
    }
  }

  void MLineMesh::storeComponentPositions(AztecFlags type, AztecFlags flag) {
    storedVertices = vertices;

  }

  void MLineMesh::restoreComponentPositions(AztecFlags type, AztecFlags flag) {
    if (storedVertices.size() == vertices.size()) {
      vertices = storedVertices;
    }
  }

  void MLineMesh::transformComponents(AztecFlags type, 
                                      const MMatrix4 &transform, 
                                      AztecFlags flag, 
                                      bool localTransform,
                                      TransformModifier *modifier) {
    switch (type) {
    case POINT_TYPE:
      {
        for (unsigned int i = 0; i < vertices.size(); ++i) {
          if (flag == 0 || vertices[i].isFlagged(flag)) {
/*            if (localTransform) {
              localMat.createRotationMatrix(MVector3(0,0,1), getVertexNormal(i));

              localMat = transform * localMat;
              vertices[i].m_Pos = vertices[i].m_Pos + localMat * MVector3(0,0,0);

            } else {*/
            MVector3 newPos = transform * (vertices[i].m_Pos);
            if (modifier != NULL) {
              modifier->adjustTransform(vertices[i].m_Pos, newPos);
            }
            vertices[i].m_Pos = newPos;
//            }
          }
        }
        break;
      }
    case FACET_TYPE:
      {
        break;
      }
    case EDGE_TYPE:
      {
        break;
      }
    }
  }

  void MLineMesh::updateComponentAnimation(AztecFlags type, AztecFlags flag, long time, bool createKey) {
  }

  void MLineMesh::updateSingleComponentAnimation(AztecFlags type, int index, long time, bool createKey) {
  }

  const DWORD MLineMesh::Vertex::NormalValid = 0x80;

  MLineMesh::Vertex::Vertex() { 
    setFlag(VERTEX_VISIBLE); 
  }

  MLineMesh::Vertex::Vertex(const MLineMesh::Vertex &src) 
    : m_Pos(src.m_Pos), m_FeedbackColor(src.m_FeedbackColor)
  {
  }

  MLineMesh::Vertex::Vertex(const MVector3 &src, AztecFlags flags) 
    : m_Pos(src), MFlagObject(flags)
  {
  }

  MLineMesh::Vertex::Vertex(float x, float y, float z) 
    : m_Pos(x,y,z)
  {
  }

  MLineMesh::Edge::Edge(int vertexA, int vertexB, AztecFlags srcFlags) 
    : flags(srcFlags)
  {
    if (vertexA < vertexB) {
      from = vertexA;
      to = vertexB;
      flags.unsetFlag(Reversed);
    } else {
      from = vertexB;
      to = vertexA;
      flags.setFlag(Reversed);
    }
  }

  MLineMesh::Edge::Edge(const MLineMesh::Edge &rhs) 
    : flags(rhs.flags), from(rhs.from), to(rhs.to)
  {

  }

  void MLineMesh::Edge::getVertices(int &vertexA, int &vertexB) const {
    if (flags.isFlagged(Reversed)) {
      vertexA = to;
      vertexB = from;
    } else {
      vertexA = from;
      vertexB = to;
    }
  }

  bool MLineMesh::Edge::vertexRemoved(int vertexIndex) {
    if (from == vertexIndex || to == vertexIndex) {
      return true;
    }

    if (from > vertexIndex) {
      --from;
    }
    if (to > vertexIndex) {
      --to;
    }
    return false;
  }


  const AztecFlags MLineMesh::Edge::Selected = (AztecFlags)MComponentisedObject::COMPONENT_SELECTED;
  const AztecFlags MLineMesh::Edge::Reversed = (AztecFlags)0x80;

  bool MLineMesh::Edge::operator==(const MLineMesh::Edge &rhs) const {
    return (from == rhs.from && to == rhs.to && 
      (flags.isFlagged(Reversed) == rhs.flags.isFlagged(Reversed)) );
  }

  bool MLineMesh::Edge::operator<(const MLineMesh::Edge &rhs) const {
    // when we compare edges, we always compare the edges if
    // vertexA is smaller than vertexB. Because an edge contains
    // potentially important direction information, we can't
    // store the two vertices in acending order, we just have to
    // cater for the possibility that it can happen.

    // first check to see if we have to flip things around for 
    // comparison purposes.
    if ( (from < to && rhs.from < rhs.to) ||
         (from > to && rhs.from > rhs.to) )

    {
      if (from < rhs.from) {
        return true;
      }
      if (from > rhs.from) {
        return false;
      }

      return to < rhs.to;
    } else if (from < to && rhs.from > rhs.to) {
      if (from < rhs.to) {
        return true;
      }
      if (from > rhs.to) {
        return false;
      }

      return to < rhs.from;
    } else if (from > to && rhs.from < rhs.to) {
      if (to < rhs.from) {
        return true;
      }
      if (to > rhs.from) {
        return false;
      }

      return from < rhs.to;
    } else {
      assert(0);
      return false;
    }

  }

  bool MLineMesh::Edge::operator<(int firstVertex) const {
    return from < firstVertex;
  }

  void MLineMesh::drawEdges(const MBaseObjectPtr &baseObj, const MSceneViewFlags &ViewFlags) {
    bool drawSelectedEdges = isInComponentMode(EDGE_TYPE);
    bool doSelectEdges = drawSelectedEdges && (ViewFlags.m_SelectMethod != smNone);

    glColor4fv((float*)&ViewFlags.m_WireColor);

    if (doSelectEdges) {
      glPushName(SELECTITEM_EDGE);
      glPushName(-1);
    } else {
      // TODO: Use a line strip
      glBegin(GL_LINES);
    }

    int edgeCount = 0;

    for (int i = 0; i < getEdgeCount(); ++i) {
      int from, to;

      getEdge(i, from, to);

      if (doSelectEdges) {
        glLoadName(edgeCount++);
        glBegin(GL_LINES);
      }

      if (drawSelectedEdges) {
        if (isComponentFlagged(EDGE_TYPE, i)) {
          glColor3f(0.9f, 0.9f, 0.1f);
        } else {
          glColor4fv((float*)&ViewFlags.m_WireColor);
        }
      }

      glVertex3fv((float*)&getVertexPosition(from));
      glVertex3fv((float*)&getVertexPosition(to));

      if (doSelectEdges) {
        glEnd();
      }

    }

    if (doSelectEdges) {
      glPopName();
      glPopName();
    } else {
      glEnd();
    }
  }

  void MLineMesh::drawPoints(const MBaseObjectPtr &baseObj, const MSceneViewFlags &ViewFlags) {
    bool drawSelectedPoints = isInComponentMode(POINT_TYPE);
    bool doSelectPoints = drawSelectedPoints && (ViewFlags.m_SelectMethod != smNone);
        
    glColor3f(0.1f, 0.7f, 0.1f);

    if (doSelectPoints) {
      glPushName(SELECTITEM_VERTEX);
      glPushName(-1);
    } else {
      glBegin(GL_POINTS);
    }

    int pointCount = 0;

    for (int i = 0; i < getVertexCount(); ++i) {
      if (doSelectPoints) {
        glLoadName(pointCount++);
        glBegin(GL_POINTS);
      }

      if (drawSelectedPoints) {
        if (isComponentFlagged(POINT_TYPE, i)) {
          glColor3f(0.9f, 0.9f, 0.1f);
        } else {
          glColor3f(0.1f, 0.7f, 0.1f);
        }
      }

      glVertex3fv((float*)&getVertexPosition(i));

      if (doSelectPoints) {
        glEnd();
      }

    }

    if (doSelectPoints) {
      glPopName();
      glPopName();
    } else {
      glEnd();
    }

    glBegin(GL_LINES);
    for (int i = 0; i < getVertexCount(); ++i) {
      MVector3 vertPos[2];

      vertPos[0] = getVertexPosition(i);
      vertPos[1] = vertPos[0] + 8.0 * getComponentNormal(POINT_TYPE, i);
      glVertex3fv((float*)&vertPos[0]);
      glVertex3fv((float*)&vertPos[1]);
    }
    glEnd();

  }


}
