// MSystemManager.cpp: implementation of the MSystemManager class.
//
//////////////////////////////////////////////////////////////////////
#include "StdAfx.h"

#include "MSystemManager.h"
#include "MMesh.h"
#include "MMeshShape.h"
#include "MAnimMesh.h"
#include "MEditableMesh.h"
#include "MClusterModifier.h"
#include "MBone.h"
#include "RegClass.h"
#include "MLight.h"
#include <MLineShape.h>
#include <MLineMesh.h>
#include <params/MParameterFactory.h>
#include <MLookAt.h>
#include <animation/MIKControllerCCD.h>
#include <MCamera.h>
#include <mesh/MEditMeshModifier.h>

#include <stdio.h>
#include <stdarg.h>
#include <assert.h>
#include <iostream>

#ifdef WIN32

#include <io.h>
#include <direct.h>

// Included for the user directory getting
#ifndef _WIN32_IE
#define _WIN32_IE 0x0400
#endif

#include <shlobj.h>
#include "Shlwapi.h"

#else

#include <sys/unistd.h>
#include <dlfcn.h>
#include <dirent.h>

#endif

#include <params/MUniqueNameParameter.h>
#include <params/MParentObjectParameter.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


namespace Aztec {

  static MSystemManagerPtr g_CurrentSysMan = NULL;
  static bool sysManagerCleaned = false;
  
  MSystemManagerPtr getSystemManager() {
    return MSystemManager::getInstance();
  }
  
  //------------------
  //  MSystemManager
  //------------------
  
  MSystemManager::MSystemManager() {
    m_LogFile = new MLogFile();
  }

  void MSystemManager::initSystemManager() {

    char     cwd[256];
    MStr     Str;
    
#ifdef WIN32
    ::GetModuleFileName(NULL, cwd, 256);
		char drive[_MAX_DRIVE];
		char dir[_MAX_DIR];
		_splitpath(cwd, drive, dir, NULL, NULL);
    strcpy(cwd, drive);
    strcat(cwd, dir);
#else
    // TODO: Do this correctly (see $AGUI/MAppImpl/Qt)
    getcwd(cwd, 256);
#endif
    
    Str = cwd;
    Str += "/Logs/SysMan.log";
    m_LogFile->addLogFile((LPCTSTR)Str);
    
    m_LogFile->writeLine("SysMan: Initialising");

    m_UndoManager = new MUndoManager;

    m_TimeParam = MParameterFactory::createInteger("t", "time", "Time");
    // we don't want the time parameter upating according to
    // its own changes, so fix it.
    m_TimeParam->unsetFlag(OBJECTFLAG_TIME_IS_INPUT);

    m_DllList = new MBaseObjectList;

    m_SystemSettings = new MGlobalSettings;
    
    m_LogFile->writeLine("SysMan: Registering Object Classes");

    pluginManager = new MPluginManager;

    pluginManager->registerObject( new MBaseObject() );
    
    pluginManager->registerObject( new MVector3KeyList() );
    pluginManager->registerObject( new MFloatKeyList() );
    
    pluginManager->registerObject( new MImage() );
    pluginManager->registerObject( new MMaterial() );
    pluginManager->registerObject( new MScene() );
    pluginManager->registerObject( new MTransformObject() );
    pluginManager->registerObject( new MSceneObject() );
    
    pluginManager->registerObject( new MLight() );
    
    pluginManager->registerObject( new MMesh() );
    pluginManager->registerObject( new MMeshShape() );
    pluginManager->registerObject( new MAnimMesh() );
    pluginManager->registerObject( new MEditableMesh() );
    pluginManager->registerObject( new MEditMeshModifier() );

    pluginManager->registerObject( new MLineShape() );
    pluginManager->registerObject( new MLineMesh() );
    
    pluginManager->registerObject( new MClusterModifier() );
    
    pluginManager->registerObject( new MBoneObject() );
    
    pluginManager->registerObject( new MTimeSegment() );

    pluginManager->registerObject( new MLookAt() );

    pluginManager->registerObject( new MIKControllerCCD() );

    pluginManager->registerObject( new MCamera() );

    m_LogFile->writeLine("SysMan: Initialisation Complete");
  }
  
  MSystemManager::~MSystemManager() {
    m_LogFile->writeLine("SysMan: Shutting Down");

    // assert if the plugin manager has a reference counte of more than one, because we required that it be cleaned up now
    assert(pluginManager->getRefCount() == 1);
    pluginManager = NULL;

    MScene::setGlobalScene(NULL);
    
    imageList.clear();
    
    m_SystemSettings = NULL;

    // this line will unload all the dll's, so everything must be cleaned
    // up before here.
    m_DllList = NULL;
    m_LogFile->writeLine("SysMan: Shutdown Complete");
  }

  MSystemManagerPtr MSystemManager::getInstance() {
    assert(!sysManagerCleaned);
    if (g_CurrentSysMan == NULL) {
      g_CurrentSysMan = new MSystemManager();
      g_CurrentSysMan->initSystemManager();
    }

    return g_CurrentSysMan;
  }

  void MSystemManager::cleanInstance() {
    g_CurrentSysMan = NULL;
    sysManagerCleaned = true;
  }

  int MSystemManager::findAndLoadDLLs(const MStr &directory, const MStr &FileMask, bool Init, bool RemoveOnFail) {
    typedef std::vector<std::string> FilenameList;
    
    FilenameList filenames;

    int Count = 0;

#ifdef WIN32
    _finddata_t    FindData;
    int            hFind, Result;

    // convert all the slashes to back slashes
    std::string realDirectory = directory.c_str();
    for (unsigned i = 0; i < realDirectory.length(); ++i) {
      if (realDirectory[i] == '/') realDirectory[i] = '\\';
    }

    MStr actualSearchMask = realDirectory + "\\" + FileMask;
    hFind = _findfirst((LPCTSTR)actualSearchMask, &FindData);
    
    if (hFind != -1) {
      Result = 0;
    } else {
      Result = 1;
    }
    
    while (Result == 0)
    {
      filenames.push_back(FindData.name);
      Result = _findnext(hFind, &FindData);
    }
    
    _findclose(hFind);
#else 
    // convert all the slashes to back slashes
    std::string realDirectory = directory.c_str();
    for (int i = 0; i < realDirectory.length(); ++i) {
      if (realDirectory[i] == '\\') realDirectory[i] = '/';
    }

    
    // TODO: use proper regular expression matching here.
    MStr expr = FileMask;
    expr.RemoveLeading('*');
    expr.RemoveTrailing('*');

    DIR *dirStream = opendir(realDirectory.c_str());
    struct dirent *dir;

    dir = readdir(dirStream);

    while (dir != NULL) {
      MStr filename(dir->d_name);
      if (filename.findSubstring(expr) != -1) {
        filenames.push_back(filename.c_str());
      }

      dir = readdir(dirStream);
    }

    closedir(dirStream);

#endif

    FilenameList::iterator it;
    for (it = filenames.begin(); it != filenames.end(); ++it) {
      if (loadDll((*it).c_str(), Init, RemoveOnFail)) {
        ++Count;
      }
    }
    return Count;
  }
  
  int MSystemManager::loadDll(MStr Name, bool Init, bool RemoveOnFail) {
    MDllFilePtr NewDll;
    MListObjectNodePtr DllNode;
    
    MStr  Str;
    
    Str.Format("SysMan: Loading plugin '%s' ...", (LPCTSTR)Name);
    m_LogFile->writeLine(Str);
    
    NewDll = new MDllFile();
    
    NewDll->setFilename(Name);
    // NewDll->setSysManager(this);
    
    DllNode = m_DllList->addHead(NewDll);
    if (DllNode == NULL) {
      return 0;
    }
    
    if (Init) {
      if ( NewDll->dllInit() != 1 ) {
        Str.Format("SysMan: Loading of plugin '%s' Failed", (LPCTSTR)Name);
        m_LogFile->writeLine(Str);
        
        if (RemoveOnFail) {
          m_DllList->removeNode(DllNode);
          DllNode = NULL;
        }
      }
    }
    
    return (DllNode != NULL);
  }


  MPluginManagerPtr MSystemManager::getPluginManager() {
    return pluginManager;
  }

  MImagePtr MSystemManager::loadImage(const MStr &Filename, bool force) {
    // Check to see if we have already imported this particular file.
    
    {
      MBaseObjectPtr Obj;
      MImagePtr Image;
      
      ImageMap::iterator it;
      for (it = imageList.begin(); it != imageList.end(); ++it) {
        // check for a matching filename.
        if (Filename.compareNoCase(it->second.getFilename().c_str()) == 0) {

          // if we have found it, see if the attributes have changed. If they 
          // have, remove the old one, and load in a new one.
          if (force || it->second.checkAttributes()) {
            imageList.erase(it);
            break;
          } else {
            return it->second.getImage();
          }
        }
      }
    }
    
    MImageTranslatorPtr Trans = pluginManager->getImageTranslatorThatImports(Filename);
    
    if (Trans != NULL) {
      MImagePtr NewImage;
      
      NewImage = new MImage;
      
      if (Trans->importFile(Filename, NewImage)) {
        NewImage->m_Filename = Filename;
        imageList.insert(ImageMap::value_type(Filename.c_str(), ImageListEntry(NewImage, Filename.c_str())));
      } else {
        logOutput("Translator could not load '%s'", (LPCTSTR)Filename);
      }
      
      return NewImage;
    } else {
      return NULL;
    }
    
    return NULL;
  }
  
  void MSystemManager::setScene(MScenePtr Scene) {
    MScene::setGlobalScene(Scene);
  }

  MScenePtr MSystemManager::getScene() {
    return MScene::getGlobalScene();
  }

  
  void MSystemManager::getTranslatorFromIndex(int Index, MSceneTranslatorPtr &Trans, bool addAllFileTypes) {
    if (addAllFileTypes) {
      // Check to see if we have it the "all files" filter
      if (Index == 1) {
        Trans = NULL;
        return;
      } else {
        Index--;
      }
    }
    
    Trans = pluginManager->getSceneTranslator(Index);
  }
  
  void MSystemManager::getTranslatorFromIndex(int Index, MImageTranslatorPtr &Trans) {
    // Check to see if we have it the "all files" filter
    if (Index == 1) {
      Trans = NULL;
      return;
    } else {
      Index--;
    }
    
    Trans = pluginManager->getImageTranslator(Index);
  }
  
  void MSystemManager::getTranslatorFromIndex(int Index, MMaterialTranslatorPtr &Trans) {
    // Check to see if we have it the "all files" filter
    if (Index == 1) {
      Trans = NULL;
      return;
    } else {
      Index--;
    }
    
    Trans = pluginManager->getMaterialTranslator(Index);
  }

  MStr MSystemManager::createSceneTranslatorFilterString(bool addAllFileTypes) {
    std::string result, filter, allFormats;

    for (int i = 0; i < pluginManager->getSceneTranslatorCount(); ++i) {
      MSceneTranslatorPtr Trans = pluginManager->getSceneTranslator(i);
      // try and extract the actual file mask
      // so we can generate an "All the formats" 
      // mask
      if (allFormats.length() > 0) {
        allFormats += ";";
      }
      allFormats += Trans->getFilter();

      filter += Trans->getFilterDescription();
      filter += " (";
      filter += Trans->getFilter();
      filter += ")";
      filter += '|';
      filter += Trans->getFilter();
      filter += '|';
    }
    
    filter += '|';

    if (addAllFileTypes) {
      // prepend the all the formats filter
      result = "All Known Formats (";
      result += allFormats;
      result += ")|";
      result += allFormats;
      result += "|";
    }

    result += filter;

    return result.c_str();
  }
  
  MStr MSystemManager::createImageTranslatorFilterString() {
    std::string result, filter, allFormats;

    for (int i = 0; i < pluginManager->getImageTranslatorCount(); ++i) {
      MImageTranslatorPtr Trans = pluginManager->getImageTranslator(i);
      // try and extract the actual file mask
      // so we can generate an "All the formats" 
      // mask
      if (allFormats.length() > 0) {
        allFormats += ";";
      }
      allFormats += Trans->getFilter();

      filter += Trans->getFilterDescription();
      filter += " (";
      filter += Trans->getFilter();
      filter += ")";
      filter += '|';
      filter += Trans->getFilter();
      filter += '|';
    }
    
    filter += '|';

    // prepend the all the formats filter
    result = "All Known Formats (";
    result += allFormats;
    result += ")|";
    result += allFormats;
    result += "|";
    result += filter;

    return result.c_str();
  }
  
  MStr MSystemManager::createMaterialTranslatorFilterString() {
    std::string result, filter, allFormats;

    for (int i = 0; i < pluginManager->getMaterialTranslatorCount(); ++i) {
      MMaterialTranslatorPtr Trans = pluginManager->getMaterialTranslator(i);
      // try and extract the actual file mask
      // so we can generate an "All the formats" 
      // mask
      if (allFormats.length() > 0) {
        allFormats += ";";
      }
      allFormats += Trans->getFilter();

      filter += Trans->getFilterDescription();
      filter += " (";
      filter += Trans->getFilter();
      filter += ")";
      filter += '|';
      filter += Trans->getFilter();
      filter += '|';
    }
    
    filter += '|';

    // prepend the all the formats filter
    result = "All Known Formats (";
    result += allFormats;
    result += ")|";
    result += allFormats;
    result += "|";
    result += filter;

    return result.c_str();
  }
  
  MLogFilePtr MSystemManager::getLogger() {
    return m_LogFile;
  }
  
  void MSystemManager::logOutput(const char *FormatStr, ...) {
    char        Str[1024];
    va_list     ParamList;
    
    va_start(ParamList, FormatStr);
    vsprintf(Str, FormatStr, ParamList);
    
#ifdef _WIN32
#ifdef _DEBUG 

    OutputDebugString("Log: ");
    OutputDebugString(Str);
    OutputDebugString("\n");

#endif
#endif

    m_LogFile->writeLine(Str);

  }
  
  MGlobalSettingsPtr MSystemManager::getSettings() {
    return m_SystemSettings;
  }

  /**
   * This gets the Undo Manager.
   * @see MUndoManager
   */
  MUndoManagerPtr MSystemManager::getUndoManager() { 
    if (g_CurrentSysMan == NULL) {
      return NULL;
    } else {
      return g_CurrentSysMan->m_UndoManager;
    }
  }

  MIntParameterPtr MSystemManager::getTimeParameter() {
    // getting the time parameter MUST NOT create a system
    // manager, as this might be called during clean up.
    if (g_CurrentSysMan != NULL) {
      return g_CurrentSysMan->m_TimeParam;
    }
    return NULL;
  }

  MBaseObjectListPtr MSystemManager::getDllList() { 
    return m_DllList; 
  }

  std::string MSystemManager::getGlobalSettingsPath() {
#ifdef _WIN32
    // For windows, we get the path that the executable is in.
    // TODO: Make this a bit cleaner, as its not very nice right now.
    char buf[512];
    ::GetModuleFileName(NULL, buf, 512);
    MStr directory = buf;
    MStr drive,dir;
    directory.SplitPath(&drive,&dir,NULL,NULL);
    directory = drive+dir;

    return directory.c_str();
#else
    // TODO: this needs to be changed to a common location where settings
    // can be kept. It is only using the users home directory for the moment.
    return "~";
#endif
  }

  std::string MSystemManager::getUserSettingsPath() {
#ifdef _WIN32
    char path[MAX_PATH] = "";

    HRESULT result = SHGetSpecialFolderPath(NULL, path, CSIDL_APPDATA, TRUE);
    PathAppend(path, "MartinTools");
    CreateDirectory(path, NULL);
    PathAppend(path, "Aztec-1.1");
    CreateDirectory(path, NULL);
    return path;
#else
    // TODO: Should do something bit better here, such as making a hidden
    // dot directory (eg: ~/.aztec) or something.
    return "~";
#endif
  }

  MStr MSystemManager::getClassName() {
    return MStr("MSystemManager");
  }

  MStr MSystemManager::getParentClassName() {
    return MStr("MBaseObject");
  }



  static void* dllGetFunc(HINSTANCE lib, const char *funcName) {
#ifdef WIN32
    return (void*)GetProcAddress(lib, funcName);
#else
    return (void*)dlsym(lib, funcName);
#endif
  }
  
  //----------------------------------------------------------------------------------------
  //  MDllFile
  //----------------------------------------------------------------------------------------
  
  MDllFile::MDllFile() {
    m_NumClasses = 0;
    m_DllInstance = NULL;
    m_SysMan = NULL;
  }
  
  MDllFile::~MDllFile() {
    dllFinish();
  }
  
  int MDllFile::setSysManager(MSystemManagerPtr SysMan) {
    if (SysMan == NULL) {
      return 0;
    }
    
    m_SysMan = SysMan;
    return 1;
  }
  
  int MDllFile::setFilename(MStr Name) {
    if (m_DllInstance) {
      return 0;
    }
    
    m_Filename = Name;
    return 1;
  }
  
  int MDllFile::dllInit() {
	  MSystemManagerPtr SysMan = getSystemManager();
	  if (SysMan == NULL)	{
      return 0;
    }
    
    dllFinish();
    
#ifdef _WIN32
    m_DllInstance = LoadLibrary(m_Filename);
#else
    // TODO: Remove this cruft asap, it's bad, annonying, hackerish and doesn't work for 'release' builds.
    /**
     * NOTE: We need an _absolute_ file path on Linux, this is due to the way how libdl is designed,
     *       that means; it does _not_ check for and load libraries from the current directory, like
     *       win32 does, so we're putting the absolute path in front of the actual library name when
     *       trying to load the lib.
     *
     *       See 'man dlopen' for more information about the whys and hows.
     */
    char cwd[256];
    getcwd(cwd, 256);
    std::string ourFileName = std::string(cwd) + std::string("/.libs/") + std::string(m_Filename.c_str());
    m_DllInstance = dlopen(ourFileName.c_str(), RTLD_LAZY);
#endif
    
    if (!m_DllInstance) {
    #ifndef _WIN32
      std::cerr << (char*)dlerror() << std::endl;
      std::cerr << "cannot open shared object (" << m_Filename.c_str() << ")" << std::endl;
    #endif
      return 0;
    }
    
    QueryVersionFunc  VersionFunc;
    QueryDescFunc     DescFunc;
    QueryCountFunc    CountFunc;
    RegisterFunc      RegFunc;

    VersionFunc = (QueryVersionFunc)dllGetFunc(m_DllInstance, "ModQueryVersion");
    if (!VersionFunc) {
      return VERSIONFUNCNOTFOUND;
    }

    DescFunc = (QueryDescFunc)dllGetFunc(m_DllInstance, "ModQueryDesc");
    
    CountFunc = (QueryCountFunc)dllGetFunc(m_DllInstance, "ModQueryCount");
    if (!CountFunc) {
      return COUNTFUNCNOTFOUND;
    }
    
    RegFunc = (RegisterFunc)dllGetFunc(m_DllInstance, "ModRegister");
    if (!RegFunc) {
      return REGISTERFUNCNOTFOUND;
    }
    
    long DllVersion;
    
    if (!VersionFunc(&DllVersion)) {
      return VERSIONDIFF;
    }
    
    if (DllVersion != *(long*)MODEL_API_VERSION) {
      return VERSIONDIFF;
    }
    
    m_NumClasses = CountFunc();
    // m_SysMan->logOutput("Registering plugin %s", (LPCTSTR)m_Filename);
    SysMan->logOutput("Registering plugin %s", (LPCTSTR)m_Filename);
    for (int n=0 ; n<m_NumClasses ; n++) {
      // m_SysMan->logOutput("Registering plugin %s - Class No. %i", (LPCTSTR)m_Filename, n);
      SysMan->logOutput("Registering plugin %s - Class No. %i", (LPCTSTR)m_Filename, n);
      if (!RegFunc(SysMan.m_Ptr, n)) {
        // m_SysMan->logOutput("Registering Class %i Failed", n);
        SysMan->logOutput("Registering Class %i Failed", n);
      }
    }
    
    return 1;
  }
  
  int MDllFile::dllFinish() {
    if (m_DllInstance == NULL) {
      return SUCCEED;
    }
    
#ifdef WIN32
    return FreeLibrary(m_DllInstance);
#else
    return dlclose(m_DllInstance);
#endif
  }
  
  HINSTANCE MDllFile::getInstance() { 
    return m_DllInstance; 
  }
  
  // MBaseObject methods
  MStr MDllFile::getClassName() {
    return MStr("MDllFile");
  }

  MStr MDllFile::getParentClassName() {
    return MStr("MBaseObject");
  }

  MSystemManager::ImageListEntry::ImageListEntry(const MImagePtr &newImage, const std::string &newFilename) {
    image = newImage;
    filename = newFilename;

    // call check attrbiutes to update any information
    checkAttributes();
  }

  MSystemManager::ImageListEntry::ImageListEntry(const ImageListEntry &src) 
    : image(src.image), filename(src.filename)
  {
    checkAttributes();
  }

  MImagePtr MSystemManager::ImageListEntry::getImage() {
    return image;
  }

  std::string MSystemManager::ImageListEntry::getFilename() {
    return filename;
  }

#ifdef WIN32
  bool operator!=(const FILETIME &lhs, const FILETIME &rhs) {
    return (lhs.dwHighDateTime != rhs.dwHighDateTime ||
      lhs.dwLowDateTime != rhs.dwLowDateTime);
  }
#endif

  /**
   * This checks the file attributes (date, size etc) and if they have
   * changed. Returns true if they have changed, false if they haven't.
   */
  bool MSystemManager::ImageListEntry::checkAttributes() {
#ifdef WIN32
    WIN32_FILE_ATTRIBUTE_DATA attributes;
    
    ::ZeroMemory(&attributes, sizeof(attributes));
    ::GetFileAttributesEx(filename.c_str(), GetFileExInfoStandard, &attributes);


    bool result = false;
    // check for any differences.
    if (attributes.ftLastWriteTime != lastAttributes.ftLastWriteTime ||
        attributes.nFileSizeHigh != lastAttributes.nFileSizeHigh ||
        attributes.nFileSizeLow != lastAttributes.nFileSizeLow) {
      result = true;
    }

    // copy the new attributes over the old ones.
    lastAttributes = attributes;

    return result;
#else
    // TODO: This should be changed so it does an actual check on the file,
    // because for the moment it reports it has always changed.
    return true;
#endif
  }
  
  //----------------------------------------------------------------------------------------
  //  MGlobalSettings
  //----------------------------------------------------------------------------------------
  
  MGlobalSettings::MGlobalSettings() {
    ResetToDefaults();
  }
  
  void MGlobalSettings::ResetToDefaults() {
    m_CFGFilename = "Aztec.cfg";
    m_KEYFilename = "Aztec.key";
    m_Animate = false;
  }
  
  
  int MGlobalSettings::ReadFilenamesFromReg() {
    MRegistry      Reg;
    
    if (!Reg.openKey("Software\\MartinTools\\Modeller\\"))
      return 0;
    
    Reg.readString("CFGFile", m_CFGFilename);
    Reg.readString("KEYFile", m_KEYFilename);
    
    Reg.closeKey();
    
    return 1;
  }
  
  int MGlobalSettings::SaveFilenamesToReg() {
    MRegistry      Reg;
    
    if (!Reg.openKey("Software\\MartinTools\\Modeller\\"))
      return 0;
    
    Reg.writeString("CFGFile", (LPCTSTR)m_CFGFilename);
    Reg.writeString("KEYFile", (LPCTSTR)m_KEYFilename);
    
    Reg.closeKey();
    
    return 1;
  }
  
  
  int MGlobalSettings::ReadConfig() {
    // Read the config file in.
    return 1;
  }
  
  int MGlobalSettings::WriteConfig() {
    return 1;
  }
 

}
