#include "StdAfx.h"

#include "scripting/ScriptArrayParam.h"
#include "scripting/MScriptInterp.h"
#include "scripting/ScriptUtils.h"
#include <scripting/ScriptVector3.h>

#include <params/MArrayParameter.h>

#include "jscntxt.h"

#include <assert.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {  
	
class ArrayEnumState {
public:
	int numParams;	// Total # of parameters to iterate over
	int index;		// Current property index
};

class AztecArrayParam {
public:
	AztecArrayParam() {
		parent = NULL;
		param = NULL;
	}
	MNamedObjectPtr parent;
	MArrayParameterPtr param;
};

static JSBool
aztecArrayLength(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	AztecArrayParam *param = (AztecArrayParam *)JS_GetPrivate(cx, obj);
	if (param == NULL || param->param == NULL) {
		return JS_FALSE;
	}

  // This method takes no parameters, so report an aerror if we are given any.
  if (argc != 0) {
		return JS_FALSE;
	}

  *rval = INT_TO_JSVAL(param->param->getElementCount());
  return JS_TRUE;
}


static JSPropertySpec aztecArray_props[] = {
//	{"x",	0,	JSPROP_ENUMERATE},
//	{"y",	1,	JSPROP_ENUMERATE},
//	{"z",	2,	JSPROP_ENUMERATE},
	{0}
};

static JSFunctionSpec aztecArray_methods[] = {
	{"size",	aztecArrayLength,	0},
	{0}
};
// See if a particular property exists in the Aztec object.  
static JSBool
aztec_lookupProperty(JSContext *cx, JSObject *obj, jsid id,
                     JSObject **objp, JSProperty **propp)
{
  AztecArrayParam *param = (AztecArrayParam*)JS_GetPrivate(cx, obj);
  MScriptInterpreter *interp = (MScriptInterpreter *)JS_GetContextPrivate(cx);
  if (param == NULL || interp == NULL) {
    return JS_TRUE;
  }
  
  // OutputDebugString("Aztec lookupProperty\n");
  
  jsval idval;
  if (!JS_IdToValue(cx, id, &idval)) {
    // Failed to resolve the property name
    *objp = NULL;
    *propp = NULL;
    return JS_TRUE;
  }
  
  if (JSVAL_IS_STRING(idval)) {
    // Look for a property by name
    char *prop_name = JS_GetStringBytes(JSVAL_TO_STRING(idval));
    // See if any of the properties on the list match prop_name
    int index = strToInt(prop_name);

    if (index >= 0 && index < param->param->getElementCount()) {
      *propp = (JSProperty *)1;
    } else {
      *propp = NULL;
    }
  } else if (JSVAL_IS_INT(idval)) {
    // Look for a property by index
    int index = JSVAL_TO_INT(idval);

    *objp = obj;

    if (index >= 0 && index < param->param->getElementCount()) {
      *propp = (JSProperty *)1;
    } else {
      *propp = NULL;
    }

  } else {
    // Failed to resolve the property name
    *objp = NULL;
    *propp = NULL;
  }
  
  return JS_TRUE;
}

static JSBool
aztec_defineProperty(JSContext *cx, JSObject *obj, jsid id, jsval value,
                         JSPropertyOp getter, JSPropertyOp setter,
                         uintN attrs, JSProperty **propp)
{
	// OutputDebugString("Attempt to define aztec property\n");

    return JS_TRUE;
}

static JSBool
aztec_getPropertyById(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	jsval idval;
	if (!JS_IdToValue(cx, id, &idval)) {
		return JS_FALSE;
	}

  AztecArrayParam *param = (AztecArrayParam *)JS_GetPrivate(cx, obj);
	if (param == NULL || param->param == NULL) {
		return JS_FALSE;
	}

	// If we are working from a parameter, then pull the current value
	// and use it.  Otherwise used the Array we are dragging around
	// inside this object.
  
  if (JSVAL_IS_STRING(idval)) {
    // Look for a property by name
    char *prop_name = JS_GetStringBytes(JSVAL_TO_STRING(idval));

#if 0
	// ARE: This code doesn't seem right - if we are given a string as a property,
	// then we aren't working with an index.

    // See if any of the properties on the list match prop_name
    int index = strToInt(prop_name);

	if (index >= 0) {
	    *vp = convertParam(cx, obj, param->parent, param->param, index);
		return JS_TRUE;
	}
#endif

	// May be a built-in method.
	for (int m=0;aztecArray_methods[m].name != NULL;m++) {
		if (!strcmp(prop_name, aztecArray_methods[m].name)) {
			// Yup - is a built-in function.
			JSFunction *jfunc =
				JS_NewFunction(cx, aztecArray_methods[m].call,
								aztecArray_methods[m].nargs,
								0, obj, prop_name);
			JSObject *jfobj = JS_GetFunctionObject(jfunc);
			*vp = OBJECT_TO_JSVAL(jfobj);
			return JS_TRUE;
		}
	}

  } else if (JSVAL_IS_INT(idval)) {
		int index = JSVAL_TO_INT(idval);

    *vp = convertParam(cx, obj, param->parent, param->param, index);

    return JS_TRUE;

	} else if (JSVAL_IS_OBJECT(idval)) {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, idval));
		OutputDebugString("Array prop: ");
		OutputDebugString(propStr);
		OutputDebugString("\n");
		return JS_FALSE;
	} else {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, idval));
		OutputDebugString("Array prop: ");
		OutputDebugString(propStr);
		OutputDebugString("\n");
		return JS_FALSE;
	}

  return JS_FALSE;
}

static JSBool
aztec_setPropertyById(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	jsval idval;
	if (!JS_IdToValue(cx, id, &idval)) {
		return JS_FALSE;
	}

  AztecArrayParam *param = (AztecArrayParam *)JS_GetPrivate(cx, obj);
	if (param == NULL || param->param == NULL) {
		return JS_FALSE;
	}

  assert(param->param->getElementType() == MParameterObject::TYPE_VECTOR);

  // make sure our index parameter is of an integer type
  int index = -1;

  if (JSVAL_IS_STRING(idval)) {
    // Look for a property by name
    char *prop_name = JS_GetStringBytes(JSVAL_TO_STRING(idval));
    // See if any of the properties on the list match prop_name
    index = strToInt(prop_name);

  } else if (JSVAL_IS_INT(idval)) {
		index = JSVAL_TO_INT(idval);
  }

  if (index != -1) {
    if (JSVAL_IS_STRING(*vp)) {
      // Look for a property by name
      char *value = JS_GetStringBytes(JSVAL_TO_STRING(*vp));

      param->param->setElement(index, MStr(value));

    } 
    
    if (JSVAL_IS_OBJECT(*vp)) {
      // This is an object param, so it is ok to hand it an object.
      JSObject *jsobj = JSVAL_TO_OBJECT(*vp);
      
      // attempt to convert it to a vector3 class.
      void *priv = JS_GetInstancePrivate(cx, jsobj, &aztecVector3_class, NULL);
      
      // if we have a successful convesion, then set the value appropriately
      if (priv != NULL) {
        MVector3 *nvec = (MVector3 *)priv;
        param->param->setElement(index, *nvec);
        return JS_TRUE;
      }
    }


	} else if (JSVAL_IS_OBJECT(idval)) {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, idval));
		OutputDebugString("Array prop: ");
		OutputDebugString(propStr);
		OutputDebugString("\n");
		return JS_FALSE;
	} else {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, idval));
		OutputDebugString("Array prop: ");
		OutputDebugString(propStr);
		OutputDebugString("\n");
		return JS_FALSE;
	}

  return JS_FALSE;
}

static JSBool
aztec_getAttributes(JSContext *cx, JSObject *obj, jsid id,
                    JSProperty *prop, uintN *attrsp)
{
	OutputDebugString("Aztec getAttributes\n");

    // We don't maintain JS property attributes for Aztec objects
    *attrsp = JSPROP_PERMANENT | JSPROP_ENUMERATE;
    return JS_FALSE;
}

static JSBool
aztec_setAttributes(JSContext *cx, JSObject *obj, jsid id,
                    JSProperty *prop, uintN *attrsp)
{
	OutputDebugString("Aztec setAttributes\n");

    // We don't maintain JS property attributes for Aztec objects
    if (*attrsp != (JSPROP_PERMANENT | JSPROP_ENUMERATE)) {
        return JS_FALSE;
    }

    // Silently ignore all setAttribute attempts
    return JS_TRUE;
}

static JSBool
aztec_deleteProperty(JSContext *cx, JSObject *obj, jsid id, jsval *vp)
{
	OutputDebugString("Aztec deleteProperty\n");

	return JS_FALSE;
}

static JSBool
aztec_defaultValue(JSContext *cx, JSObject *obj, JSType typ, jsval *vp)
{
	// OutputDebugString("Aztec defaultValue\n");
    switch (typ) {
    case JSTYPE_OBJECT:
      *vp = OBJECT_TO_JSVAL(obj);
      return JS_TRUE;
      
    case JSTYPE_FUNCTION:
      return JS_FALSE;

    case JSTYPE_VOID:
    case JSTYPE_STRING:
      {
        AztecArrayParam *param = (AztecArrayParam *)JS_GetPrivate(cx, obj);
        if (param == NULL) {
          JSString *str = JS_NewStringCopyZ(cx, "[Aztec Object - noparam]");
          *vp = STRING_TO_JSVAL(str);
        } else {
          MStr valStr;
          valStr = "[Aztec Array]";
          JSString *str = JS_NewStringCopyZ(cx, valStr);
          *vp = STRING_TO_JSVAL(str);
        }
        return JS_TRUE;
      }
      
    case JSTYPE_NUMBER:
      return JS_FALSE;
      
    case JSTYPE_BOOLEAN:
      return JS_FALSE;

    default:
        return JS_FALSE;
    }
}

static JSBool
aztec_newEnumerate(JSContext *cx, JSObject *obj, JSIterateOp enum_op,
                   jsval *statep, jsid *idp)
{
	AztecArrayParam *param = (AztecArrayParam *)JS_GetPrivate(cx, obj);

    // Check for native object
    if (param == NULL) {
		// No native object - say we don't have any properties.
        *statep = JSVAL_NULL;
        if (idp != NULL) {
            *idp = INT_TO_JSVAL(0);
		}
        return JS_TRUE;
    }

    switch(enum_op) {
    case JSENUMERATE_INIT:
		{
      int paramCount = param->param->getElementCount();
      
      ArrayEnumState *enumState = new ArrayEnumState;
      enumState->numParams = paramCount;
      enumState->index = 0;
      *statep = PRIVATE_TO_JSVAL(enumState);
      if (idp != NULL) {
        *idp = INT_TO_JSVAL(paramCount);
      }
      return JS_TRUE;
		}
        
    case JSENUMERATE_NEXT:
		{
      ArrayEnumState *enumState = (ArrayEnumState *)JSVAL_TO_PRIVATE(*statep);

      if (enumState->index < enumState->numParams) {
        JS_ValueToId(cx, INT_TO_JSVAL(enumState->index), idp);
        enumState->index++;
        return JS_TRUE;
      }
      // Drop through to destroy
    }

    case JSENUMERATE_DESTROY:
		{
      ArrayEnumState *enumState = (ArrayEnumState *)JSVAL_TO_PRIVATE(*statep);
      delete enumState;
      *statep = JSVAL_NULL;
      return JS_TRUE;
		}

    default:
		OutputDebugString("Bad enumeration state\n");
        return JS_FALSE;
    }
}

static JSBool
aztec_checkAccess(JSContext *cx, JSObject *obj, jsid id,
                      JSAccessMode mode, jsval *vp, uintN *attrsp)
{
	OutputDebugString("Aztec Array checkAccess\n");

    switch (mode) {
    case JSACC_WATCH:
        return JS_FALSE;

    case JSACC_IMPORT:
        return JS_FALSE;

    default:
        return JS_TRUE;
    }
}


static JSBool
aztecArray_convert(JSContext *cx, JSObject *obj, JSType typ, jsval *vp)
{
	AztecArrayParam *param = (AztecArrayParam *)JS_GetPrivate(cx, obj);
	if (param == NULL || param->param == NULL) {
		return JS_FALSE;
	}
	
  switch (typ) {
  case JSTYPE_OBJECT:
    *vp = OBJECT_TO_JSVAL(obj);
    return JS_TRUE;
    
  case JSTYPE_VOID:
  case JSTYPE_STRING:
		{
			char buf[128];
			sprintf(buf, "array");
			JSString *str = JS_NewStringCopyZ(cx, buf);
			*vp = STRING_TO_JSVAL(str);
			return JS_TRUE;
		}

  case JSTYPE_FUNCTION:
  case JSTYPE_NUMBER:
  case JSTYPE_BOOLEAN:
  default:
    return JS_FALSE;
  }
}

static void
aztecArray_finalize(JSContext *cx, JSObject *obj)
{
	AztecArrayParam *param = (AztecArrayParam *)JS_GetPrivate(cx, obj);
	if (param == NULL) {
		return;
	}

	delete param;
}

// Specialized object operations - these allow us to dynamically reflect
// Aztec properties into the script interpreter.
JSObjectOps aztecArray_ops = {
    // Mandatory non-null function pointer members.
    js_ObjectOps.newObjectMap,
    js_ObjectOps.destroyObjectMap,
    aztec_lookupProperty,
    aztec_defineProperty,
    aztec_getPropertyById,  // getProperty
    aztec_setPropertyById,  // setProperty
    aztec_getAttributes,
    aztec_setAttributes,
    aztec_deleteProperty,
    aztec_defaultValue,
    aztec_newEnumerate,
    aztec_checkAccess,

    // Optionally non-null members start here.
    NULL,                       /* thisObject */
    NULL,                       /* dropProperty */
    NULL,                       /* call */
    NULL,                       /* construct */
    NULL,                       /* xdrObject */
    NULL,                       /* hasInstance */
    NULL,                       /* setProto */
    NULL,                       /* setParent */
    NULL,                       /* mark */
    NULL,                       /* clear */
    NULL,                       /* getRequiredSlot */
    NULL                        /* setRequiredSlot */
};


static JSObjectOps *
aztecArray_getObjectOps(JSContext *cx, JSClass *clazz)
{
    return &aztecArray_ops;
}

JSClass aztecArray_class = { 
	"AztecArray", JSCLASS_HAS_PRIVATE, 
		JS_PropertyStub, JS_PropertyStub, // add/del property
		JS_PropertyStub, JS_PropertyStub, // get/set property
		JS_EnumerateStub, JS_ResolveStub,
		aztecArray_convert, aztecArray_finalize,
    aztecArray_getObjectOps
};

JSObject *newArrayParam(JSContext *cx, 
                         const MNamedObjectPtr &aztec_obj,
                         const MArrayParameterPtr &aztec_param)
{
	JSObject *jsvec = JS_NewObject(cx, &aztecArray_class, NULL, JS_GetGlobalObject(cx));
	AztecArrayParam *vparm = new AztecArrayParam;
	vparm->parent = aztec_obj;
	vparm->param = aztec_param;
	JS_SetPrivate(cx, jsvec, vparm);

	return jsvec;
}


void addArrayParamClass(JSContext *cx)
{
	// Add a vector class (reflection of MVector3) to the global namespace
    JS_InitClass(cx, JS_GetGlobalObject(cx), NULL,
		&aztecArray_class, 0, 0,
		aztecArray_props, aztecArray_methods, 
    0, 
    0);

}

}
