// ChannelBar.cpp : implementation file
//

#include <AztecMainPCH.h>
#include "resource.h"

#include "ChannelBar.h"

#include "MdlGlobs.h"
#include "HelperFuncs.h"
#include "ColorPickerDlg.h"

#include "DlgGlobs.h"

#include "UndoGeneral.h"

#include <params/MChoiceParameter.h>
#include <params/MVector3KeyParameter.h>
#include <MDAGraph.h>

#include "AztecGUIMPluginManager.h"

#include <assert.h>
#include <algorithm>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

static const COLORREF NULL_COLOUR = 0xFF000000;
static COLORREF getColourForParameter(const MParameterObjectPtr &param);

MChannelWindow::MChannelWindow() {
}

MChannelWindow::~MChannelWindow() {
}

LRESULT MChannelWindow::OnKeyDown(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled) {
	// Special handling for cr/tab/esc - we want to 
    if ((wParam == VK_RETURN) || (wParam == VK_TAB) || (wParam == VK_ESCAPE)) {
		SendMessage(m_hParent, uMsg, wParam, lParam);
		return 0;
	}
	
	return DefWindowProc(uMsg, wParam, lParam);
}

//----------------------------------------------------------------------------------------
//  MChannelParameter
//----------------------------------------------------------------------------------------
MChannelParameter::MChannelParameter() {
  m_NameLabel = NULL;
  m_ParamControl = NULL;
  m_CustomParamEditor = NULL;
  m_CustomParamControl = NULL;
  m_Object = NULL;
  m_Param = NULL;

  m_Height = 20;
  m_Width  = 64;

  editColour = NULL_COLOUR;
}

MChannelParameter::~MChannelParameter() {
  DeleteControls();
  
  m_Param = NULL;
  m_Object = NULL;
}

void MChannelParameter::setFromParameterObject(MParameterObjectPtr param) {
  m_ParamName = param->getShortName();
  m_DisplayName = param->getFriendlyName();
  param->getValueString(m_ParamValue);

    m_Param = param;
}

MChannelWindow * MChannelParameter::getControl() {
  return m_ParamControl;
}

MStr MChannelParameter::getParamName() {
  return m_ParamName;
}

MStr MChannelParameter::getDisplayName() {
  return m_DisplayName;
}

void MChannelParameter::setValue(const MStr &str) {
  m_ParamValue = str;
}

MStr MChannelParameter::getValue() {
  return m_ParamValue;
}

void MChannelParameter::setSelectedObject(MNamedObjectPtr obj) {
  m_Object = obj;
}

MNamedObjectPtr MChannelParameter::getSelectedObject() {
  return m_Object;
}

MParameterObjectPtr MChannelParameter::getParameterObject() {
  return m_Param;
}

void MChannelParameter::MoveControls(int Y, float MidMargin, int RightMargin) {
  int      MidPoint;
  RECT     ParentRect, EditRect;

  if (m_ParamControl == NULL || !m_ParamControl->IsWindow()) {
	  // Not a simple control - have to rely on custom processing
	  // to resize.
	  return;
  }
  
  ::GetClientRect(m_ParamControl->GetParent(), &ParentRect);
  
  ParentRect.left ++;
  ParentRect.right --;
  ParentRect.top ++;
  ParentRect.bottom --;
  
  MidPoint = (int)((ParentRect.right - 8)*MidMargin);
  
  EditRect.top = Y;
  EditRect.bottom = Y + 17;
  
  EditRect.left = ParentRect.left + 2;
  EditRect.right = MidPoint-1;
  
  m_NameLabel->MoveWindow(&EditRect, TRUE);
  
  if (MidMargin == 0.0)
    MidPoint += 10;
  
  EditRect.left = MidPoint+1;
  EditRect.right = ParentRect.right - 8 - RightMargin;
  
  m_ParamControl->MoveWindow(EditRect.left, EditRect.top,
	  EditRect.right - EditRect.left + 1, EditRect.bottom - EditRect.top + 1,
	  TRUE);
}

void MChannelParameter::CreateControls(CChannelParamWnd *Owner, int Y, float MidMargin, int RightMargin) {
	if (Owner == NULL) {
		return;
	}
	
	DeleteControls();
	
	m_NameLabel = new MChannelLabel(); // = Owner->CreateEditBox(2,2,2, 2, ES_RIGHT | ES_READONLY);//, FW_BOLD);
	m_NameLabel->m_hParent = Owner->m_hWnd;
	
	if (m_Param != NULL) {
		MChoiceParameterPtr choiceParam;
		choiceParam = AZTEC_CAST(MChoiceParameter, m_Param);
		
		// make a combo box if we have a choice parameter.
		if (choiceParam != NULL) {
			m_ParamControl = Owner->CreateComboBox(2,2,2,2);
			m_ParamControl->m_hParent = Owner->m_hWnd;
			int numChoices = choiceParam->getNumChoices();
			for (int i=0; i < numChoices; i++) {
				m_ParamControl->SendMessage(CB_ADDSTRING, 0, (long)choiceParam->getChoice(i).c_str());
			}
			// select out current value in the combo box.
			m_ParamControl->SendMessage(CB_SELECTSTRING, -1, (long)m_ParamValue.c_str());
		} else {
			// otherwise make an edit box
			m_ParamControl = Owner->CreateEditBox(2,2,2,2);
			m_ParamControl->m_hParent = Owner->m_hWnd;
		}
		
		// Make an edit box by default.
	} else {
		m_ParamControl = Owner->CreateEditBox(2,2,2,2);
		m_ParamControl->m_hParent = Owner->m_hWnd;
	}
	
	m_NameLabel->SetWindowText((LPCTSTR)m_DisplayName);
	m_ParamControl->SetWindowText((LPCTSTR)m_ParamValue);
	
	MoveControls(Y, MidMargin, RightMargin);
}

void MChannelParameter::UpdateControls() {
	m_Object->getParamByName(m_ParamName, m_ParamValue);

  bool controlNeedsPainting = false;
  {
    MParameterObjectPtr param = getParameterObject();
    COLORREF oldColour = editColour;
    editColour = getColourForParameter(param);
    if (editColour != oldColour) {
      controlNeedsPainting = true;
    }
  }

	if (m_NameLabel == NULL) {
		return;
	}
	
	char Str[256];
	
	m_NameLabel->GetWindowText(Str);
	if (Str != m_ParamName) {
		m_NameLabel->SetWindowText((LPCTSTR)m_DisplayName);
	}
	
	if (m_ParamControl->IsWindow()) {
		m_ParamControl->GetWindowText(Str, sizeof(Str));
		if (m_ParamValue != NULL && MStr(Str) != m_ParamValue) {
			m_ParamControl->SetWindowText((LPCTSTR)m_ParamValue);
		}
	}

  if (controlNeedsPainting) {
    m_ParamControl->Invalidate(FALSE);
  }
}

void MChannelParameter::DeleteControls() {
	m_NameLabel = NULL;
	
	if (m_CustomParamEditor != NULL) {
		// This editor was supplied by a plugin.  We need to have the plugin
		// remove it.
		m_CustomParamEditor->destroyChannelControls();
		m_CustomParamEditor = NULL;
		if (m_CustomParamControl != NULL) {
			if (m_CustomParamControl->IsWindow()) {
				m_CustomParamControl->DestroyWindow();
			}
			delete m_CustomParamControl;
			m_CustomParamControl = NULL;
		}
	} else if (m_ParamControl) {
		if (m_ParamControl->IsWindow()) {
			HWND hWnd = m_ParamControl->UnsubclassWindow();
			::DestroyWindow(hWnd);
		}
		// m_ParamControl->DestroyWindow();
		// m_ParamControl->m_hWnd = NULL;
		delete m_ParamControl;
		m_ParamControl = NULL;
	}
}

void MChannelParameter::focusControl() {
  // focus the control
	if (m_CustomParamEditor != NULL) {
		if (m_CustomParamControl != NULL) {
			m_CustomParamControl->SetFocus();
		}
	} else if (m_ParamControl != NULL) {
		m_ParamControl->SetFocus();
	}

#if 0
	// Determine what sort of control we have (text, combo, or custom)
  CEdit *edit;
  CComboBox *combo;

  edit = dynamic_cast<CEdit*>(m_ParamControl);

  if (edit != NULL) {
    edit->SetSel(0,500);
  } else {
    combo = dynamic_cast<CComboBox*>(m_ParamControl);
    if (combo != NULL) {
      combo->SetEditSel(0,500);
    }
  }
#endif
}

void MChannelParameter::setLabelVisible(bool visible) {
  m_NameLabel->m_Visible = visible;
}

void MChannelParameter::makeParameterChange() {
  // Create undo node
  char  str[64];
  sprintf(str, "Edit %s", getDisplayName().c_str());

  Aztec::getSystemManager()->getUndoManager()->beginUndo(str);

  getSelectedObject()->setParamByName(getParamName(), getValue(), g_SysMan->getSettings()->m_Animate);

  Aztec::getSystemManager()->getUndoManager()->endUndo();

  ::SendMessage(g_MainDlg->m_hWnd, MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
  ::SendMessage(g_MainDlg->m_hWnd, MM_UPDATECHANNELBAR, 0, 0);
}

COLORREF MChannelParameter::getEditColour() {
  return editColour;
}

int MChannelParameter::getHeight()
{
	return 20;
}

void MChannelParameter::setHeight(int h)
{
	m_Height = h;
}

int MChannelParameter::getWidth()
{
	return 128;
}

void MChannelParameter::setWidth(int w)
{
	m_Width = w;
}

AztecGUI::MParameterPluginPtr MChannelParameter::getCustomEditor()
{
	return m_CustomParamEditor;
}

void MChannelParameter::setCustomEditor(AztecGUI::MParameterPluginPtr cust)
{
	m_CustomParamEditor = cust;
}

MChannelWindow *MChannelParameter::getCustomControl()
{
	return m_CustomParamControl;
}

void MChannelParameter::setCustomControl(MChannelWindow *cust)
{
	m_CustomParamControl = cust;
}

//----------------------------------------------------------------------------------------
//  CChannelBar
//----------------------------------------------------------------------------------------

CChannelBar::CChannelBar() {
  m_LastSelected = NULL;
  
  m_ScrollX = 0;
  m_ScrollY = 0;
  m_Dragging = 0;
  m_VerticalScroll = true;
  m_HorizontalScroll = false;
}

CChannelBar::~CChannelBar() {
  DeleteAllParams();
}

void CChannelBar::OrganiseGroups() {
  int top = 0;
  RECT clientRect;

  ::GetClientRect(m_hWnd, &clientRect);

  for (int i = 0; i < channelGroups.size(); ++i) {
    channelGroups[i]->organiseGroup(top, clientRect.right - 8);
    top += channelGroups[i]->getHeight() + 2;
  }

  ::InvalidateRect(m_hWnd, NULL, FALSE);
}

int getVisibleParameterCount(MParameterObjectListPtr list) {
  int total = 0;
	for (int n=0; n < list->getNumParams(); n++) {
		// if our parameter is invisible, just ignore it.
		if (list->getParameter(n)->isVisible()) {
      total++;
    }
  }

  return total;
}

void CChannelBar::UpdateAllParams() {
  RECT     ClientRect;
  int      n;
  
  ::GetClientRect(m_hWnd, &ClientRect);
  ClientRect.right -= 8;
  
  for (n = 0; n < channelGroups.size(); n++) {
    channelGroups[n]->setMarkForDeletion(true);
  }
  
  MBaseObjectPtr SelObj;
  
#define ONLY_SINGLE_SELECTION
#ifdef ONLY_SINGLE_SELECTION

  // first check to see if our current object is still selected,
  // if so don't bother checking anything else.
  if (currentObject != NULL && 
      g_Scene->getSelectedObjectList()->findNode(currentObject) != NULL) 
  {
    addObject(currentObject);
  } else {
    // otherwise get the most recently selected object.
    SelObj = g_Scene->getSelectedObjectList()->getTail();
    currentObject = AZTEC_CAST(MNamedObject, SelObj);
    if (currentObject != NULL) {
      addObject(currentObject);
    }
  } 

#else
  g_Scene->getSelectedObjectList()->beginIteration();
  while ((SelObj = g_Scene->getSelectedObjectList()->getNext()) != NULL) {
    MNamedObjectPtr namedObj = AZTEC_CAST(MNamedObject, SelObj);
    if (SelObj != NULL && namedObj != NULL) {
      addObject(namedObj);
    }
  }
  g_Scene->getSelectedObjectList()->endIteration();
#endif

  DeleteMarkedParams();
  
  for (n = 0; n < channelGroups.size(); n++) {
    channelGroups[n]->updateWindow();
  }
  
  OrganiseGroups();
  InvalidateScrollRect();
}

void CChannelBar::addObject(const MNamedObjectPtr &object) {
  // loop over our current parameters, and see if we already have it
  // present. If we do, unmark it for deletion.
  for (int n = 0; n < channelGroups.size(); ++n) {
    if (channelGroups[n]->getObject() == object) {
      channelGroups[n]->setMarkForDeletion(false);
      return;
    }
  }

  // if we have gotten here, then we have not already found the object,
  // so just add it in.
  channelGroups.push_back(new ChannelGroup(this, object));
}

void CChannelBar::DeleteMarkedParams() {
  int NumToDelete, n;
  
  NumToDelete = 0;
  for (n = 0; n < channelGroups.size(); n++) {
    if (channelGroups[n]->isMarkedForDeletion()) {
      NumToDelete++;
    }
  }
  
  if (NumToDelete == 0) {
    return;
  }
  
  if (NumToDelete == channelGroups.size()) {
    DeleteAllParams();
    return;
  }
  
  if (channelGroups.size() > 0) {
    for (n = channelGroups.size() - 1 ; n >= 0; n--) {
      if (channelGroups[n]->isMarkedForDeletion()) {
        channelGroups.erase(channelGroups.begin() + n);
      }
    }
  }
  
}

void CChannelBar::DeleteAllParams() {
  channelGroups.clear();
  
  m_ScrollX = 0;
  m_ScrollY = 0;
  
  InvalidateScrollRect();
}

LRESULT CChannelBar::OnLButtonDown(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	SetCapture();
	POINT    P;
	
	XYFromDWORD(lParam, &P);
	ClientToScreen(&P);
	ScreenToClient(&P);
	m_OldX = P.x;
	m_OldY = P.y;
	m_Dragging = true;
	
	return 0;
}

LRESULT CChannelBar::OnLButtonUP(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	ReleaseCapture();
	m_Dragging = false;
	
	return 0;
}

LRESULT CChannelBar::OnMButtonDown(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	SetCapture();
	POINT    P;
	XYFromDWORD(lParam, &P);
	ClientToScreen(&P);
	ScreenToClient(&P);
	m_OldX = P.x;
	m_OldY = P.y;
	m_Dragging = true;

	return 0;
}

LRESULT CChannelBar::OnMButtonUp(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	m_Dragging = false;
	ReleaseCapture();

	return 0;
}

void adjustScrollingChange(int scrollY, RECT *ClientRect, RECT *childRect, int *DiffY) {
  // TODO: cap scrolling amounts here!!!!

  // if the childRect is smaller than the viewable rect, then we must
  // scroll to 0.
  int childHeight = childRect->bottom - childRect->top;
  int viewHeight = ClientRect->bottom - ClientRect->top;
  if (childHeight < viewHeight) {
    *DiffY = -childRect->top;
  } else {
    // the child rect should never move below the 0 mark
    // so test to see if the current shift will move it past that point.
    if (childRect->top > 0 || childRect->top + *DiffY > 0) {
      *DiffY = -childRect->top;
    }

    // the bottom of the child rect should never rise above the bottom of the client.

    if (childRect->bottom < ClientRect->bottom ||
        childRect->bottom + *DiffY < ClientRect->bottom) 
    {
      *DiffY = ClientRect->bottom - childRect->bottom;
    }
  }
}

LRESULT CChannelBar::OnMouseMove(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	::SetCursor(::LoadCursor(NULL, IDC_SIZEALL));
	if (!m_Dragging) {
		return 0;
	}

	int   DiffX, DiffY;
	RECT  childRect, ClientRect;
	int   X, Y;
	
	XYFromDWORD(lParam, X,Y);
	DiffX = X - m_OldX;
	DiffY = Y - m_OldY;
	
	GetChildExtents(&childRect);
	GetClientRect(&ClientRect);

  adjustScrollingChange(m_ScrollY, &ClientRect, &childRect, &DiffY);
	
  if (DiffX != 0 || DiffY != 0) {
    if (!m_VerticalScroll) {
      DiffY = 0;
    }
    if (!m_HorizontalScroll) {
      DiffX = 0;
    }
    ScrollWindow(DiffX, DiffY);
    m_ScrollX += DiffX;
    m_ScrollY += DiffY;
    
    InvalidateScrollRect();
  }
	
	m_OldX = X;
	m_OldY = Y;
	
	return 0;
}

LRESULT CChannelBar::ProcessSizingMsg(WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	int   DiffX, DiffY;
	RECT  childRect, ClientRect;
	
	DiffX = 0;
	DiffY = 0;
	
	GetChildExtents(&childRect);
	GetClientRect(&ClientRect);

  adjustScrollingChange(m_ScrollY, &ClientRect, &childRect, &DiffY);

	if (DiffX != 0 || DiffY != 0) {
		if (!m_VerticalScroll) {
            DiffY = 0;
		}
		if (!m_HorizontalScroll) {
            DiffX = 0;
		}
		ScrollWindow(DiffX, DiffY);
		m_ScrollX += DiffX;
		m_ScrollY += DiffY;
		
		InvalidateScrollRect();
	}

	return 0;
}

LRESULT CChannelBar::OnSize(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	return ProcessSizingMsg(wParam, lParam, bHandled);
}

LRESULT CChannelBar::OnWindowPosChanged(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	return ProcessSizingMsg(wParam, lParam, bHandled);
}

LRESULT CChannelBar::OnWindowPosChanging(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	return ProcessSizingMsg(wParam, lParam, bHandled);
}

LRESULT CChannelBar::OnKeyDown(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
    // Update the parmater in question.
    if ((wParam != VK_RETURN) && (wParam != VK_TAB) && (wParam != VK_ESCAPE)) {
		// bHandled = FALSE;
		return 0;
	}
	
	int CurrentEdit = -1;
  int CurrentGrp = -1;;
  int currentSubGroup = -1;
		// Get the item that currently has the focus
	HWND focusWnd = ::GetFocus();
	const MSG *msg = GetCurrentMessage();
	HWND tfocusWnd = msg->hwnd;

	char Str[256];
	CChannelParamWnd  *ParamWnd;
	for (int Grp = 0; Grp < channelGroups.size(); Grp++) {
    ChannelGroup::ChannelParamVector allWindows = channelGroups[Grp]->getAllWindows();

    for (int subGroup = 0; subGroup < allWindows.size(); ++subGroup) {
      ParamWnd = allWindows[subGroup];
        
      for (int n=0; n < ParamWnd->getNumParamObjects(); n++) {
        MChannelParameterPtr chanParam = ParamWnd->m_Params[n];
        MChannelWindow *chanWnd = chanParam->getControl();
        
        if (chanWnd != NULL) {
          if (chanWnd->m_hWnd == focusWnd) {
            chanWnd->GetWindowText(Str, sizeof(Str));
            chanParam->setValue(Str);
            
            // if the user didn't hit escape, we need to create
            // and undo node for whatever change they made
            if (wParam != VK_ESCAPE) {
              chanParam->makeParameterChange();
            }
            
            chanParam->UpdateControls();
            
            CurrentEdit = n;
            CurrentGrp = Grp;
            currentSubGroup = subGroup;
          }
        }
      }
    }
  }
	
	// Move to the next parameter
  if (wParam == VK_TAB && CurrentEdit != -1 && CurrentGrp != -1 && currentSubGroup != -1) {
    ChannelGroup::ChannelParamVector allWindows = channelGroups[CurrentGrp]->getAllWindows();
    
    ParamWnd = allWindows[currentSubGroup];
    
    MShiftState State;
    State.SetFromKeyState();
    
    // check to see if we are going backwads
    if (State.m_Shift) {
      CurrentEdit--;
    } else {
      CurrentEdit++;
    }
    
    // check to see if we have moved backwards beyond our first paramete
    if (CurrentEdit < 0) {
      // if so, adjust our current subgroup.
      currentSubGroup--;

      if (currentSubGroup < 0) {
        CurrentGrp--;
        if (CurrentGrp < 0) {
          CurrentGrp = channelGroups.size()-1;
        }
        allWindows = channelGroups[CurrentGrp]->getAllWindows();
        currentSubGroup = allWindows.size() - 1;
      }

      CurrentEdit = allWindows[currentSubGroup]->getNumParamObjects() - 1;
    } else if (CurrentEdit >= allWindows[currentSubGroup]->getNumParamObjects()) {
      currentSubGroup++;

      if (currentSubGroup >= allWindows.size()) {
        CurrentGrp++;
        currentSubGroup = 0;
      }

      CurrentEdit = 0;
      CurrentGrp %= channelGroups.size();

      allWindows = channelGroups[CurrentGrp]->getAllWindows();
    }

    allWindows[currentSubGroup]->m_Params[CurrentEdit]->focusControl();
  } else {
    // Set focus to first parent window that isn't itself a child window.
    // In MFC: GetParentOwner()->SetFocus();
    HWND pOwn = ::GetParent(focusWnd);
    while ((pOwn != NULL) && ((::GetWindowLong(pOwn, GWL_STYLE) & WS_CHILD) == 0)) {
      pOwn = ::GetParent(pOwn);
    }
    if (pOwn != NULL) {
      ::SetFocus(pOwn);
    }
  }

	return 0;
}

LRESULT CChannelBar::OnMouseWheel(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
    Sleep(1);
  
	return 0;
}

LRESULT CChannelBar::OnPaint(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
    PAINTSTRUCT    Paint;
    
    BeginPaint(&Paint);
    
    RECT     ClientRect;
    HBRUSH   hBrush;
    
    GetClientRect(&ClientRect);
    
    
    {
      RECT     Rect;
      
      hBrush = ::CreateSolidBrush( GetSysColor(COLOR_3DFACE) );
      
      Rect = ClientRect;
      Rect.right -= 4;
      ::FillRect(Paint.hdc, &Rect, hBrush);
      
      Rect = ClientRect;
      Rect.left = Rect.right - 2;
      ::FillRect(Paint.hdc, &Rect, hBrush);
      
      ::DeleteObject(hBrush);
    }

    {
      RECT ScrollRect;
      RECT AllChildrenRect;
      float ViewRatio;
      
      // Draw the tiny scrollbar
      GetChildExtents(&AllChildrenRect);
      
      ViewRatio = (float)ClientRect.bottom / (AllChildrenRect.bottom - AllChildrenRect.top);

      m_ScrollY = AllChildrenRect.top;
      
      // If the viewratio is smaller then 1, the children rect extends beyond what can be
      // Seen in one go.
      ScrollRect.left = ClientRect.right - 4;
      ScrollRect.right = ClientRect.right - 2;
      
      if (ViewRatio < 1) {
        ScrollRect.bottom = (long)(ClientRect.bottom * ViewRatio);
        ScrollRect.top = 0;
        
        ScrollRect.top -= (long)(AllChildrenRect.top * ViewRatio);
        ScrollRect.bottom -= (long)(AllChildrenRect.top * ViewRatio);
        
      } else {
        ScrollRect.top = 0;
        ScrollRect.bottom = ClientRect.bottom;
      }

      
      {
        RECT  OutsideRect;
        
        hBrush = (HBRUSH)::GetStockObject(BLACK_BRUSH);
        
        ::FillRect(Paint.hdc, &ScrollRect, hBrush);
        
        
        
        OutsideRect = ScrollRect;
        
        OutsideRect.top = 0;
        OutsideRect.bottom = ScrollRect.top;
        
        hBrush = ::CreateSolidBrush( GetSysColor(COLOR_SCROLLBAR) );
        ::FillRect(Paint.hdc, &OutsideRect, hBrush);
        
        OutsideRect.top = ScrollRect.bottom;
        OutsideRect.bottom = ClientRect.bottom;
        
        ::FillRect(Paint.hdc, &OutsideRect, hBrush);
        ::DeleteObject(hBrush);
      }
      
    }

    {
      if (channelGroups.size() > 0) {
        for (int i = 0; i < channelGroups.size(); ++i) {
          SetBkMode(Paint.hdc, TRANSPARENT);
          HFONT hFont = (HFONT)::GetStockObject(DEFAULT_GUI_FONT);
          HFONT hOldFont = (HFONT)SelectObject(Paint.hdc, hFont);

          TEXTMETRIC tm;
          char fontname[255];
          ::GetTextMetrics(Paint.hdc, &tm);
          ::GetTextFace(Paint.hdc, sizeof(fontname), fontname);
          hFont = ::CreateFont(tm.tmHeight, 0, 0, 0, FW_BOLD, FALSE, FALSE, FALSE, tm.tmCharSet, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH, fontname);
          ::SelectObject(Paint.hdc, hFont);

          ::TextOut(Paint.hdc, 3, channelGroups[i]->getInputLabelPos()+m_ScrollY, "Inputs", 6);
          ::TextOut(Paint.hdc, 3, channelGroups[i]->getOutputLabelPos()+m_ScrollY, "Outputs", 7);
          ::SelectObject(Paint.hdc, hOldFont);
          ::DeleteObject(hFont);
        }
      }
      // Draw the text labels
    
    }
   
    

    EndPaint(&Paint);

	return 0;
}

LRESULT CChannelBar::OnEraseBackground(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled) {
 

  return 0;
}


LRESULT CChannelBar::OnUpdateChannelBar(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
    UpdateAllParams();
  
	return 0;
}

void CChannelBar::GetChildExtents(RECT *lpRect) {
  lpRect->left = 10000;
  lpRect->right = 0;
  lpRect->bottom = 0;
  lpRect->top = 10000;
  
  GetChildExtents(m_hWnd, lpRect);
  lpRect->right += 10;
  lpRect->bottom += 25;
}


void CChannelBar::GetChildExtents(HWND hParent, RECT *lpRect) {
  RECT  Rect;
  HWND  LasthWnd;
  
  Rect.bottom = 0;
  Rect.left = 10000;
  Rect.top = 10000;
  Rect.right = 0;
  
  LasthWnd = NULL;
  
  while (LasthWnd = ::FindWindowEx(hParent, LasthWnd, NULL,NULL)) {
    // we now have a window handle, get the window position
    POINT       P;
    
    ::GetWindowRect(LasthWnd, &Rect);
    P.x = Rect.left;
    P.y = Rect.top;
    ::ScreenToClient(hParent, &P);
    Rect.left = P.x;
    Rect.top = P.y;
    P.x = Rect.right;
    P.y = Rect.bottom;
    ::ScreenToClient(hParent, &P);
    Rect.right = P.x;
    Rect.bottom = P.y;
    
    if (Rect.left < lpRect->left)
      lpRect->left = Rect.left;
    if (Rect.top < lpRect->top)
      lpRect->top = Rect.top;
    
    if (Rect.right > lpRect->right)
      lpRect->right = Rect.right;
    if (Rect.bottom > lpRect->bottom)
      lpRect->bottom = Rect.bottom;
    
  }
  
//  lpRect->right -= lpRect->left;
//  lpRect->bottom -= lpRect->top;
//  lpRect->left = 0;
//  lpRect->top = 0;
}

void CChannelBar::InvalidateScrollRect()
{
  if (m_hWnd == NULL) {
    return;
  }

  RECT  ScrollRect, ClientRect;
  
  ::GetClientRect(m_hWnd, &ClientRect);
  
  ScrollRect.right = ClientRect.right - 2;
  ScrollRect.left = ClientRect.right - 8;
  ScrollRect.top = 0;
  ScrollRect.bottom = ClientRect.bottom;
  
  ::InvalidateRect(m_hWnd, &ScrollRect, FALSE);
}

void CChannelBar::setCurrentObject(const MNamedObjectPtr &object) {
  currentObject = object;
  UpdateAllParams();
}

MNamedObjectPtr CChannelBar::getCurrentObject() {
  return currentObject;
}



/////////////////////////////////////////////////////////////////////////////
// CChannelParamWnd

CChannelParamWnd::CChannelParamWnd() {
  m_NameEdit = NULL;
  m_hFont = NULL;
  m_MarkForDelete = false;
}

CChannelParamWnd::~CChannelParamWnd() {
  DeleteAllParams();
}


UINT CChannelParamWnd::GetUnusedControlID() {
  HWND  hWnd;
  for (int n=1000; n<10000; n++) {
    hWnd = ::GetDlgItem(m_hWnd, n);
    if (hWnd == NULL) {
      return n;
    }
  }
  return 0;
}

MChannelWindow *CChannelParamWnd::CreateEditBox(int Left, int Top, int Width, int Height, UINT Styles, DWORD FontStyle) {
  
  HWND eWnd = ::CreateWindow("EDIT", NULL, WS_CHILD | WS_VISIBLE | WS_TABSTOP | Styles | ES_AUTOHSCROLL | ES_LEFT, 
    Left, Top, Width, Height, m_hWnd, (HMENU)GetUnusedControlID(),
    (HINSTANCE)GetWindowLong(GWL_HINSTANCE),
    NULL);                // pointer not needed 
  
  if (m_hFont == NULL) {
    HDC hdc = this->GetDC();
    
    TEXTMETRIC tm;
    char fontname[255];
    HFONT hfold = (HFONT)::SelectObject(hdc, GetStockObject(DEFAULT_GUI_FONT));
    
    ::GetTextMetrics(hdc, &tm);
    ::GetTextFace(hdc, sizeof(fontname), fontname);
    m_hFont = ::CreateFont(tm.tmHeight, 0, 0, 0, FontStyle, FALSE, FALSE, FALSE, tm.tmCharSet, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH, fontname);
    
    ::SelectObject(hdc, hfold);
    this->ReleaseDC(hdc);
  }
  
  ::SendMessage(eWnd, WM_SETFONT, (UINT)m_hFont, 0);
  ::ShowWindow(eWnd, SW_SHOW);
  
  MChannelWindow *chanWnd = new MChannelWindow();
  chanWnd->SubclassWindow(eWnd);
  
  return chanWnd;
}

MChannelWindow *CChannelParamWnd::CreateComboBox(int Left, int Top, int Width, int Height, UINT Styles, DWORD FontStyle) {
	// we adjust the height so the drop down is the correct size
	// This WILL change when MFC is no longer used.
	HWND cWnd = ::CreateWindow("COMBOBOX", NULL, WS_CHILD | WS_VISIBLE | WS_TABSTOP | Styles | CBS_AUTOHSCROLL | CBS_DROPDOWNLIST, 
		Left, Top, Width, Height+100, m_hWnd, (HMENU)GetUnusedControlID(),
		(HINSTANCE)GetWindowLong(GWL_HINSTANCE),
		NULL);                // pointer not needed 

  if (m_hFont == NULL) {
    HDC hdc = this->GetDC();
    
    TEXTMETRIC tm;
    char fontname[255];
    HFONT hfold = (HFONT)::SelectObject(hdc, GetStockObject(DEFAULT_GUI_FONT));
    ::GetTextMetrics(hdc, &tm);
    ::GetTextFace(hdc, sizeof(fontname), fontname);
    m_hFont = ::CreateFont(tm.tmHeight, 0, 0, 0, FontStyle, FALSE, FALSE, FALSE, tm.tmCharSet, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH, fontname);
    ::SelectObject(hdc, hfold);
    this->ReleaseDC(hdc);
  }

	::SendMessage(cWnd, WM_SETFONT, (UINT)m_hFont, 0);
	::ShowWindow(cWnd, SW_SHOW);
	
	MChannelWindow *chanWnd = new MChannelWindow();
	chanWnd->SubclassWindow(cWnd);

	return chanWnd;
}

MChannelWindow *CChannelParamWnd::CreateButton(int Left, int Top, int Width, int Height, UINT Styles, DWORD FontStyle) {
	HFONT    hFont;
	
	HWND bWnd = ::CreateWindow("BUTTON", NULL, WS_CHILD | WS_VISIBLE | WS_TABSTOP| Styles | BS_PUSHBUTTON, 
		Left, Top, Width, Height, m_hWnd, (HMENU)GetUnusedControlID(),
		(HINSTANCE)GetWindowLong(GWL_HINSTANCE),
		NULL);                // pointer not needed 
	
	hFont = (HFONT)GetStockObject(DEFAULT_GUI_FONT);
    HDC hdc = this->GetDC();

    TEXTMETRIC tm;
    char fontname[255];
    HFONT hfold = (HFONT)::SelectObject(hdc, hFont);
    ::GetTextMetrics(hdc, &tm);
    ::GetTextFace(hdc, sizeof(fontname), fontname);
    m_hFont = ::CreateFont(tm.tmHeight, 0, 0, 0, FontStyle, FALSE, FALSE, FALSE, tm.tmCharSet, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH, fontname);
    ::SelectObject(hdc, hfold);
    ::DeleteObject(hFont);
    this->ReleaseDC(hdc);

	::SendMessage(bWnd, WM_SETFONT, (UINT)m_hFont, 0);
	::ShowWindow(bWnd, SW_SHOW);
	
	MChannelWindow *chanWnd = new MChannelWindow();
	chanWnd->SubclassWindow(bWnd);

	return chanWnd;
}

int CChannelParamWnd::AddNameControl(MNamedObjectPtr SelectedObj, RECT &rect, int Y)
{
	MChannelParameterPtr Param;
	MStr      str;
	
	Param = new MChannelParameter;
	Param->setFromParameterObject(SelectedObj->findParameter("Name"));
	SelectedObj->getParamByName("Name", str);
	Param->setValue(str);
	
	Param->setSelectedObject(SelectedObj);
	
	Param->CreateControls(this, Y, 0.0f, 24);
	Param->setLabelVisible(false);
	
	Y += Param->getHeight();
	
	AddParameter(Param);

	return Y;
}

int CChannelParamWnd::AddStockControls(MNamedObjectPtr SelectedObj, RECT &rect, int Y)
{
	MParameterObjectListPtr ParamList;
	
	ParamList = SelectedObj->getParamList();
	for (int n=1; n<ParamList->getNumParams(); n++) {
		// if our parameter is invisible, just ignore it.
		if (ParamList->getParameter(n)->isVisible() == false) {
			continue;
		}
		
		MChannelParameterPtr Param;
		
		Param = new MChannelParameter;
		Param->setFromParameterObject(ParamList->getParameter(n));
		Param->setSelectedObject(SelectedObj);
		
		if (Param->getParamName().compareNoCase("Name") == 0) {
			Param->CreateControls(this, Y, 0.0f, 24);
		} else {
			Param->CreateControls(this, Y);
		}
		
		// Make the height of this parameter based on what the
		// parameter tells us.  This could be small for a simple
		// edit control, or large for a parameter that chooses
		// to build an interesting GUI.
		Y += Param->getHeight();
		
		AddParameter(Param);
	}
			
	return Y;
}

int CChannelParamWnd::AddCustomControls(MNamedObjectPtr SelectedObj, RECT &rect, int Y)
{
	AztecGUI::MPluginManager::ParameterPluginList plugins;
	AztecGUI::MPluginManager::getPluginsForObject(SelectedObj, plugins);
	AztecGUI::MPluginManager::ParameterPluginList::iterator pluginIt;
	for (pluginIt = plugins.begin(); pluginIt != plugins.end(); ++pluginIt) {
		AztecGUI::MParameterPluginPtr plugin = *pluginIt;
		if (plugin->hasChannelControls()) {
			// Build the wrapper for the custom window
			MChannelParameterPtr Param = new MChannelParameter;
			Param->setSelectedObject(SelectedObj);
			MChannelWindow *pluginWnd = new MChannelWindow();
			int pluginHeight = plugin->getChannelControlsHeight();
			int pluginWidth = 128;	// TBD: get width from plugin
			// Place/size the window into the channel
			RECT CtlRect = { rect.left+2, Y, rect.right - 32, Y+pluginHeight };

			pluginWnd->Create(m_hWnd, CtlRect, "customEdit");

			// Ask the plugin to controls onto the window
			plugin->createChannelControls(pluginWnd->m_hWnd, SelectedObj);

			// Remember how big this window wants to be.
			Param->setHeight(pluginHeight);
			Param->setWidth(pluginWidth);

			Param->setCustomEditor(plugin);
			Param->setCustomControl(pluginWnd);
			
			pluginWnd->ShowWindow(SW_SHOW);

			// Increment the position at which the next control will be placed.
			Y += pluginHeight;

			AddParameter(Param);
		}
	}

	return Y;
}

void CChannelParamWnd::UpdateAllParams() {
	bool RequireUpdate = false;
	MNamedObjectPtr SelectedObj;
	int n, Y;
	
	SelectedObj = m_Object;
	
	// Get the size of the window that will contain the editing controls.
	RECT ParamWndRect;
	GetClientRect(&ParamWndRect);

	// Take the size of the window that contains the editing controls and make sure
	// there is a little bit of a border.
	ParamWndRect.left++;
	ParamWndRect.top++;
	ParamWndRect.right--;
	ParamWndRect.bottom--;

	if (m_LastObject == SelectedObj) {
		// We are dealing with the same object, so just update the params.
		
		for (n = 0; n < getNumParamObjects(); n++) {
			m_Params[n]->UpdateControls();
		}
		
		return;
	}
	
	// Delete all the old parameters and such
	DeleteAllParams();
	
	if (SelectedObj == NULL) {
		m_Object = SelectedObj;
		return;
	}
	
	Y = 2;
	
	// Add in the object parameters
	{
		// Add the name parameter
		Y = AddNameControl(SelectedObj, ParamWndRect, Y);

		if (SelectedObj->isFlagged(OBJECTFLAG_CHANNELEXPAND)) {
			// Add controls for each stock parameter
			Y = AddStockControls(SelectedObj, ParamWndRect, Y);

			// Now add any plugins that want to help display things...
			Y = AddCustomControls(SelectedObj, ParamWndRect, Y);
		}
		
		m_Object = SelectedObj;
	}
	m_LastObject = m_Object;
}

void CChannelParamWnd::AddParameter(MChannelParameterPtr Param) {
  if (Param == NULL) {
    return;
  }
  
  m_Params.push_back(Param);
  
  //   InvalidateScrollRect();
}


void CChannelParamWnd::DeleteAllParams() {
  if (m_NameEdit != NULL) {
    ::DestroyWindow(m_NameEdit);
    m_NameEdit = NULL;
  }

  if (m_hFont != NULL) {
    ::DeleteObject(m_hFont);
    m_hFont = NULL;
  }

  std::vector<MChannelParameterPtr>::iterator citr;
  for (citr=m_Params.begin();citr!=m_Params.end();citr++) {
	  (*citr) = NULL;
  }
  m_Params.clear();
}


void CChannelParamWnd::GetChildExtents(RECT *lpRect) {
  lpRect->left = 10000;
  lpRect->right = 0;
  lpRect->bottom = 0;
  lpRect->top = 10000;
  
  GetChildExtents(m_hWnd, lpRect);
  lpRect->right += 10;

  if (this->m_Object->isFlagged(OBJECTFLAG_CHANNELEXPAND)) {
    lpRect->bottom += 10;
  } else {
    lpRect->bottom += 0;
  }
}


void CChannelParamWnd::GetChildExtents(HWND hParent, RECT *lpRect) {
  RECT  Rect;
  HWND  LasthWnd;
  
  Rect.bottom = 0;
  Rect.left = 10000;
  Rect.top = 10000;
  Rect.right = 0;
  
  LasthWnd = NULL;
  
  while (LasthWnd = ::FindWindowEx(hParent, LasthWnd, NULL,NULL)) {
    // we now have a window handle, get the window position
    POINT       P;
    
    ::GetWindowRect(LasthWnd, &Rect);
    P.x = Rect.left;
    P.y = Rect.top;
    ::ScreenToClient(hParent, &P);
    Rect.left = P.x;
    Rect.top = P.y;
    P.x = Rect.right;
    P.y = Rect.bottom;
    ::ScreenToClient(hParent, &P);
    Rect.right = P.x;
    Rect.bottom = P.y;
    
    if (Rect.left < lpRect->left) {
      lpRect->left = Rect.left;
    }
    
    if (Rect.top < lpRect->top) {
      lpRect->top = Rect.top;
    }
    
    if (Rect.right > lpRect->right) {
      lpRect->right = Rect.right;
    }
    if (Rect.bottom > lpRect->bottom) {
      lpRect->bottom = Rect.bottom;
    }
    
  }
  
  lpRect->right -= lpRect->left;
  lpRect->bottom -= lpRect->top;
  lpRect->left = 0;
  lpRect->top = 0;
}

void CChannelParamWnd::setMarkForDeletion(bool value) {
  m_MarkForDelete = value;
}
bool CChannelParamWnd::isMarkedForDeletion() {
  return m_MarkForDelete;
}

int CChannelParamWnd::getNumParamObjects() {
  return m_Params.size();
}

/////////////////////////////////////////////////////////////////////////////
// CChannelParamWnd message handlers
LRESULT CChannelParamWnd::OnAllMessages(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	// Suppress some messages
	switch (uMsg) {
	case WM_MOUSEFIRST:
	case WM_NCHITTEST:
	case WM_SETCURSOR:
	case WM_ERASEBKGND:
	case WM_NCPAINT:
	case WM_CTLCOLOREDIT:
		bHandled = FALSE;
		return 0;
	}

	char buf[255];
	sprintf(buf, "CPW - Msg: %x, wParam: %x, lParam: %x\n", uMsg, wParam, lParam);
	OutputDebugString(buf);

	if (uMsg == WM_GETDLGCODE) {
		OutputDebugString("WM_GETDLGCODE\n");
	}

	bHandled = FALSE;
	return 0;
}

LRESULT CChannelParamWnd::OnCommand(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
    // May be a change notification from an item in this parameter window
    int   code;
    HWND  hWnd;
    char  str[1024];
    
    code = HIWORD(wParam);
    hWnd = (HWND)lParam;

	// Start by assuming that we can't handle this command.
	bHandled = FALSE;
    
    if (code == CBN_SELCHANGE) {
      int curSel;
      
      curSel = ::SendMessage(hWnd, CB_GETCURSEL, 0, 0);
      if (curSel != CB_ERR) {
        ::SendMessage(hWnd, CB_GETLBTEXT, curSel, (DWORD)str);
        
        for (int n=0; n < getNumParamObjects(); n++) {
          if (m_Params[n]->getControl()->m_hWnd == hWnd) {
            m_Params[n]->setValue(str);
            m_Params[n]->makeParameterChange();
			bHandled = TRUE;
			bHandled = FALSE;
			break;
          }
        }
      }
    } else if (code == BN_CLICKED) {
        for (int n=0; n < getNumParamObjects(); n++) {
          if (m_Params[n]->getControl()->m_hWnd == hWnd) {
            m_Params[n]->setValue("");
            m_Params[n]->makeParameterChange();
			bHandled = TRUE;
			break;
          }
        }
	}

  return 0;
}

LRESULT CChannelParamWnd::OnDestroy(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{	
	return 0;
}

LRESULT CChannelParamWnd::OnComboBoxChange(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
{
    // May be a change notification from an item in this parameter window
    HWND  hWnd;
    char  str[1024];
    
	hWnd = hWndCtl;
	
	// Start by assuming that we can't handle this command.
	bHandled = FALSE;
    
	int curSel;
	
	curSel = ::SendMessage(hWnd, CB_GETCURSEL, 0, 0);
	if (curSel != CB_ERR) {
        ::SendMessage(hWnd, CB_GETLBTEXT, curSel, (DWORD)str);
        
        for (int n=0; n < getNumParamObjects(); n++) {
			if (m_Params[n]->getControl()->m_hWnd == hWnd) {
				m_Params[n]->setValue(str);
				m_Params[n]->makeParameterChange();
				bHandled = TRUE;
				break;
			}
        }
	}

  return 0;
}

LRESULT CChannelParamWnd::OnPaint(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
    PAINTSTRUCT    Paint;
    
	BeginPaint(&Paint);
    
    RECT     Rect;
    HBRUSH   hBrush, hOldBrush;
    HPEN     hPen, hOldPen;
    
    ::GetClientRect(m_hWnd, &Rect);

    if (this->m_Object->isFlagged(OBJECTFLAG_CHANNELEXPAND)) {
      Rect.bottom --;
    }
    
    hBrush = ::GetSysColorBrush(COLOR_3DFACE);
    FillRect(Paint.hdc, &Rect, hBrush);

    Rect.right --;

    if (this->m_Object->isFlagged(OBJECTFLAG_CHANNELEXPAND)) {
    
      hPen = ::CreatePen(PS_SOLID, 1, ::GetSysColor(COLOR_3DHILIGHT));
      hOldPen = (HPEN)::SelectObject(Paint.hdc, hPen);
      ::MoveToEx(Paint.hdc, Rect.right, Rect.bottom-2, NULL);
      ::LineTo(Paint.hdc, Rect.left, Rect.bottom-2);
      ::LineTo(Paint.hdc, Rect.left, Rect.bottom);
      ::SelectObject(Paint.hdc, hOldPen);
      ::DeleteObject(hPen);
    
      hPen = ::CreatePen(PS_SOLID, 1, ::GetSysColor(COLOR_3DDKSHADOW));
      hOldPen = (HPEN)::SelectObject(Paint.hdc, hPen);
      ::MoveToEx(Paint.hdc, Rect.right, Rect.bottom-2, NULL);
      ::LineTo(Paint.hdc, Rect.right, Rect.bottom);
      ::LineTo(Paint.hdc, Rect.left, Rect.bottom);
      ::SelectObject(Paint.hdc, hOldPen);
      ::DeleteObject(hPen);
    }    

    // Draw the arrow indicating if its dropped down or not.
    if (m_Object != NULL) {
		hPen = ::CreatePen(PS_SOLID, 1, ::GetSysColor(COLOR_3DSHADOW));
		hBrush = (HBRUSH)::GetStockObject(BLACK_BRUSH);
		
		hOldBrush = (HBRUSH)::SelectObject(Paint.hdc, hBrush);
		hOldPen = (HPEN)::SelectObject(Paint.hdc, hPen);
		
		if (m_Object->isFlagged(OBJECTFLAG_CHANNELEXPAND)) {
			// Draw the arrow down
			
			POINT    Triangle[3];
			
			Triangle[0].x = Rect.right - 15;
			Triangle[0].y = Rect.top + 5;
			Triangle[1].x = Rect.right - 5;
			Triangle[1].y = Rect.top + 5;
			Triangle[2].x = Rect.right - 10;
			Triangle[2].y = Rect.top + 15;
			
			Polygon(Paint.hdc, Triangle, 3);
		}
		else
		{
			// Draw the arrow sideways
			
			POINT    Triangle[3];
			
			Triangle[0].x = Rect.right - 5;
			Triangle[0].y = Rect.top + 5;
			Triangle[1].x = Rect.right - 5;
			Triangle[1].y = Rect.top + 15;
			Triangle[2].x = Rect.right - 15;
			Triangle[2].y = Rect.top + 10;
			
			Polygon(Paint.hdc, Triangle, 3);
		}
		::SelectObject(Paint.hdc, hOldBrush);
		::SelectObject(Paint.hdc, hOldPen);
		::DeleteObject(hPen);
    }
    
    // Go through the channel parameters, and draw the labels
    // Draw only from the second to ignore the name param
    for (int i=1; i<getNumParamObjects(); i++) {
		MChannelLabelPtr label = m_Params[i]->getLabel();
		if (label != NULL) {
			label->Draw(Paint.hdc);
		}
    }
    
    EndPaint(&Paint);
    
	return 0;
}

LRESULT CChannelParamWnd::OnEraseBackground(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled) {
  return 0;
}

LRESULT CChannelParamWnd::OnLButtonDown(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	m_ShiftState.ProcessMessage(uMsg, wParam, lParam);
	m_ShiftState.SetFromKeyState();

  m_SelFrom = 0;
  m_SelTo = 0;
  
  m_ShiftState.m_Left = true;
  int      XPos, YPos;
  RECT     ClientRect;
  
  ::GetClientRect(m_hWnd, &ClientRect);
  
  XPos = LOWORD(lParam);
  YPos = HIWORD(lParam);
  if (YPos < 17 && XPos > ClientRect.right - 17)
  {
    m_Object->toggleFlag(OBJECTFLAG_CHANNELEXPAND);
    
    m_LastObject = NULL;
    ::PostMessage(::GetParent(m_hWnd), MM_UPDATECHANNELBAR, 0 ,0);
    return 0;
  }

  // go through the params, and see if we select someting
  m_SelFrom = GetChannelParamNumFromPoint(XPos, YPos);
  m_SelTo = m_SelFrom;
  UpdateParamSelection(!m_ShiftState.m_Ctrl);
  
	return 0;
}

LRESULT CChannelParamWnd::OnLButtonUP(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	m_ShiftState.ProcessMessage(uMsg, wParam, lParam);
	m_ShiftState.SetFromKeyState();

	return 0;
}

LRESULT CChannelParamWnd::OnLButtonDblClk(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
    MChannelParameterPtr ChanParam;
    MParameterObjectPtr ParamObj;
    int                  n;
    
    n = GetChannelParamNumFromPoint(LOWORD(lParam), HIWORD(lParam));
    
    if (n != 0) {
      ChanParam = m_Params[n];
      ParamObj = ChanParam->getParameterObject();
      
      // If we have a filename based parameter
      if ( ParamObj->getDataMeaning() == MParameterObject::MEANING_FILENAME ||
           ParamObj->getDataMeaning() == MParameterObject::MEANING_IMAGE_FILENAME ) {

        ProcessPopup(ID_FILE_OPEN, ChanParam);

      // if we have a Colour based parameter
      } else if (ParamObj->getDataMeaning() == MParameterObject::MEANING_COLOUR) {
        ProcessPopup(ID_PICKCOLOUR, ChanParam);
      }
      ::SendMessage(g_MainDlg->m_hWnd, MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
      ::SendMessage(g_MainDlg->m_hWnd, MM_UPDATECHANNELBAR, 0, 0);
    }

	return 0;
}

LRESULT CChannelParamWnd::OnMButtonDown(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	m_ShiftState.ProcessMessage(uMsg, wParam, lParam);
	m_ShiftState.SetFromKeyState();

	return 0;
}

LRESULT CChannelParamWnd::OnMButtonUp(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	m_ShiftState.ProcessMessage(uMsg, wParam, lParam);
	m_ShiftState.SetFromKeyState();

	return 0;
}

LRESULT CChannelParamWnd::OnRButtonDown(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	m_ShiftState.ProcessMessage(uMsg, wParam, lParam);
	m_ShiftState.SetFromKeyState();

    DoPopup(LOWORD(lParam), HIWORD(lParam));

	return 0;
}

LRESULT CChannelParamWnd::OnRButtonUp(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	m_ShiftState.ProcessMessage(uMsg, wParam, lParam);
	m_ShiftState.SetFromKeyState();

	return 0;
}

LRESULT CChannelParamWnd::OnMouseMove(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	m_ShiftState.ProcessMessage(uMsg, wParam, lParam);
	m_ShiftState.SetFromKeyState();

	if (m_ShiftState.m_Left) {
		int      XPos, YPos;
		
		XPos = LOWORD(lParam);
		YPos = HIWORD(lParam);
		
		m_SelTo = GetChannelParamNumFromPoint(XPos, YPos);
		if (m_ShiftState.m_Ctrl) {
			if (m_SelTo != m_SelFrom) {
				m_SelFrom = m_SelTo;
				UpdateParamSelection(!m_ShiftState.m_Ctrl);
			}
		} else {
			UpdateParamSelection(!m_ShiftState.m_Ctrl);
		}
	}
	return 0;
}


LRESULT CChannelParamWnd::OnKeyDown(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	// We don't want to end processing here.  After a Tab/Cr/... we also want the
	// containing window (the channel bar itself) to do some processing.  To do that
	// we set bHandled to FALSE so that Windows knows that the message hasn't been
	// fully processed.
	// bHandled = FALSE;
	SendMessage(GetParent(), uMsg, wParam, lParam);
	return 0;

    // Update the parmater in question.
    if ((wParam == VK_RETURN) || (wParam == VK_TAB) || (wParam == VK_ESCAPE)) {
		return 0;
	}

	return 0;
}

static COLORREF getColourForParameter(const MParameterObjectPtr &param) {

  bool hasAnimation = false;
  bool hasKeySet = false;
  bool hasInput = false;

  if ((param != NULL) && param->isConnected()) {
    hasInput = true;
  } else {


    // First try to cast to a Vector3 parameter
    MVector3KeyParameter *vector3KeyParam = AZTEC_CAST(MVector3KeyParameter, param);

    if (vector3KeyParam != NULL) {
      hasKeySet = vector3KeyParam->getKeyableValue()->hasKeySet(g_Scene->getTime());

      // if we don't have a key set, see if we have an animation
      if (!hasKeySet) {
        std::vector<int> keyTimes;

        vector3KeyParam->getKeyableValue()->getKeyTimes(keyTimes);
        hasAnimation = keyTimes.size() > 0;
      }


    }
  }

  if (hasInput) {
    return RGB(230,230,255);
  } else if (hasKeySet) {
    return RGB(255,200,200);
  } else if (hasAnimation) {
    return RGB(255,230,230);
  }

  return NULL_COLOUR;
}

MChannelParameter* CChannelParamWnd::getChannelParameterForHWnd(HWND hWnd) {
  for (int i = 0; i < m_Params.size(); ++i) {
    if (m_Params[i]->getControl()->m_hWnd == hWnd) {
      return &*m_Params[i];
    }
  }
  return NULL;
}

HBRUSH CChannelParamWnd::getBrushForHWnd(HWND hWnd, HDC hdc) {
  // loop over the parameters, and see if we can find the parameter
  // for our window
  MChannelParameter *channelParam = getChannelParameterForHWnd(hWnd);

  if (channelParam != NULL) {
    COLORREF colour = channelParam->getEditColour();

    if (colour != NULL_COLOUR) {
      ::SetBkColor(hdc, colour);
      ::SetTextColor(hdc, RGB(0,0,0));
      return ::CreateSolidBrush(colour);
    }
  }

  return NULL;
}


LRESULT CChannelParamWnd::OnGetEditControlColour(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled) {

  HWND hWnd = (HWND)lParam;

  return (LRESULT)getBrushForHWnd(hWnd, (HDC)wParam);
}


int CChannelParamWnd::GetChannelParamNumFromPoint(int X, int Y) {
	for (int i=1; i<getNumParamObjects(); i++) {
		MChannelLabelPtr label = m_Params[i]->getLabel();
		if (label != NULL) {
			if (label->MouseIn(X, Y) || 
				label->MouseIn(X, Y-2) || 
				label->MouseIn(X, Y+2))
			{
				return i;
			}
		}
	}
	return 0;
}

void CChannelParamWnd::InitPopup(HMENU Popup, MChannelParameterPtr ChanParam) {
  MParameterObjectPtr ParamObj;
  
  ParamObj = m_Object->findParameter(ChanParam->getParamName());
  
  if (ParamObj->getDataMeaning() == MParameterObject::MEANING_FILENAME ||
      ParamObj->getDataMeaning() == MParameterObject::MEANING_IMAGE_FILENAME) {
	  ::AppendMenu(Popup, MF_SEPARATOR, 0, NULL);
	  ::AppendMenu(Popup, MF_STRING, ID_FILE_OPEN, "Choose File...");
  }
  if (ParamObj->getDataMeaning() == MParameterObject::MEANING_COLOUR) {
	  ::AppendMenu(Popup, MF_SEPARATOR, 0, NULL);
	  ::AppendMenu(Popup, MF_STRING, ID_PICKCOLOUR, "Pick Colour...");
  }
}

void CChannelParamWnd::DoPopup(int X, int Y) {
  HMENU    Menu;
  HMENU    Popup;
  DWORD    Choice;
  
  MChannelParameterPtr ChanParam;
  {
    int   n;
    bool  AnythingSel = false;
    
    n = GetChannelParamNumFromPoint(X, Y);
    ChanParam = m_Params[n];
    ChanParam->getLabel()->m_Selected = true;
  }
  
  Menu = ::LoadMenu(0, MAKEINTRESOURCE(IDR_CHANNELBAR_POPUP));
  Popup = ::GetSubMenu(Menu, 0);
  
  InitPopup(Popup, ChanParam);
 
  POINT   Pnt;
  Pnt.x = X;
  Pnt.y = Y;
  
  ClientToScreen(&Pnt);
  
  Choice = ::TrackPopupMenu(Popup, TPM_LEFTALIGN | TPM_RIGHTBUTTON | TPM_NONOTIFY | TPM_RETURNCMD, Pnt.x, Pnt.y, 0, m_hWnd, NULL);
  
  ProcessPopup(Choice, ChanParam);
}

void CChannelParamWnd::ProcessPopup(DWORD MenuID, MChannelParameterPtr ChanParam) {
  if (MenuID == ID_POPUP_COLLAPSE) {
    m_Object->unsetFlag(OBJECTFLAG_CHANNELEXPAND);
    m_LastObject = NULL;
    ::PostMessage(::GetParent(m_hWnd), MM_UPDATECHANNELBAR, 0 ,0);
    
  } else if (MenuID == ID_POPUP_EXPAND) {
    m_Object->setFlag(OBJECTFLAG_CHANNELEXPAND);
    m_LastObject = NULL;
    ::PostMessage(::GetParent(m_hWnd), MM_UPDATECHANNELBAR, 0 ,0);
    
  } else if (MenuID == ID_POPUP_COLLAPSEALLOBJECTS) {
/*    HWND           Parent;
    CChannelBar    ChanParent;
    
    Parent = GetParent();
    ChanParent.Attach(Parent);
    
    if (ChanParent != NULL) {
      for (int n = 0; n < ChanParent.size(); n++) {
        allObjects[n]->m_Object->setFlag(OBJECTFLAG_CHANNELEXPAND);
        allObjects[n]->m_LastObject = NULL;
      }
      ChanParent->collapseAll();
      ::PostMessage(::GetParent(m_hWnd), MM_UPDATECHANNELBAR, 0 ,0);
    }*/
  } else if (MenuID == ID_POPUP_EXPANDALLOBJECTS) {
/*    HWND		Parent;
    CChannelBar	ChanParent;
    
    Parent = GetParent();
    ChanParent.Attach(Parent);
    
    if (ChanParent != NULL) {
      int   n;
      ChanParent->expandAll();
      ::PostMessage(::GetParent(m_hWnd), MM_UPDATECHANNELBAR, 0 ,0);
    }*/
  } else if (MenuID == ID_POPUP_SETKEY) {
    // loop thorugh the selected parms, and set keys where possible.
    int   n;
    for (n=0; n<getNumParamObjects(); n++) {
      if (!m_Params[n]->getLabel()->m_Selected) {
        continue;
      }
      
      MStr     ParamName, Value;
      
      ParamName = m_Params[n]->getParamName();
      Value = m_Params[n]->getValue();
      
      m_Object->setParamByName(ParamName, Value, true);
    }
  } else if (MenuID == ID_FILE_OPEN) {
    CFileDialog    Dlg(TRUE, NULL, (LPCTSTR)ChanParam->getValue(), 0, "All Files (*.*)|*.*||");
    MParameterObjectPtr ParamObj = m_Object->findParameter(ChanParam->getParamName());
    MStr filter, title;

    // Check to see if we have an image based filename. If we do, change our 
    // file filter to allow all our image types.
    if (ParamObj != NULL && ParamObj->getDataMeaning() == MParameterObject::MEANING_IMAGE_FILENAME) {
      filter = g_SysMan->createImageTranslatorFilterString();
      filter.Replace('|', '\x0');
      title = "Choose an Image...";
  
      Dlg.m_ofn.lpstrTitle = title.c_str();
      Dlg.m_ofn.lpstrFilter = filter.c_str();
    }

    if (Dlg.DoModal() == IDOK) {
      m_Object->setParamByName(ChanParam->getParamName(), (LPCTSTR)Dlg.GetPathName());
      ::PostMessage(::GetParent(m_hWnd), MM_UPDATECHANNELBAR, 0 ,0);
      ::SendMessage(g_MainDlg->m_hWnd, MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
      ::SendMessage(g_MainDlg->m_hWnd, MM_UPDATECHANNELBAR, 0, 0);
    }
  } else if (MenuID == ID_PICKCOLOUR) {
    g_ClrPickDlg->SetChannelParameter(ChanParam);
    g_ClrPickDlg->ShowWindow(SW_SHOW);
  }
}

void CChannelParamWnd::UpdateParamSelection(bool ClearSel) {
	int      i, Min, Max;
	bool     Changed = false;
	
	if (m_SelFrom <= m_SelTo) {
		Min = m_SelFrom;
		Max = m_SelTo;
	} else {
		Max = m_SelFrom;
		Min = m_SelTo;
	}
	
	MChannelLabelPtr label;
	if (ClearSel) {
		for (i=1; i<Min; i++) {
			label = m_Params[i]->getLabel();
			if (label == NULL) {
				continue;
			}
			if (label->m_Selected != false) {
				label->m_Selected = false;
				label->Draw();
			}
		}
		for (i=Max+1; i<getNumParamObjects(); i++) {
			label = m_Params[i]->getLabel();
			if (label == NULL) {
				continue;
			}
			if (label->m_Selected != false) {
				label->m_Selected = false;
				label->Draw();
			}
		}
		for (i=Min; i<=Max; i++) {
			label = m_Params[i]->getLabel();
			if (label == NULL) {
				continue;
			}
			if (label->m_Selected != true) {
				label->m_Selected = true;
				label->Draw();
			}
		}
	} else {
		for (i=Min; i<=Max; i++)
		{
			label = m_Params[i]->getLabel();
			if (label == NULL) {
				continue;
			}
			label->m_Selected = !label->m_Selected;
			label->Draw();
		}
	}
}


//----------------------------------------------------------------------------------------
//  MChannelParameter
//----------------------------------------------------------------------------------------
MChannelLabel::MChannelLabel()
{
  m_hParent = NULL;
  m_Selected = false;
  m_Visible = true;
}

void MChannelLabel::SetWindowText(LPCTSTR lpszString) {
  m_String = lpszString;
  Draw();
}

void MChannelLabel::GetWindowText(CString Str) {
  Str = (LPCTSTR)m_String;
}


void MChannelLabel::SetParent(HWND hParent) {
  m_hParent = hParent;
}


void MChannelLabel::Draw(HDC hDC) {
  if (!m_Visible) {
    return;
  }
  
  bool     bReleaseDC = false;
  
  if (hDC == NULL) {
    hDC = ::GetDC(m_hParent);
    bReleaseDC = true;
  }
  
  HBRUSH      hBrush;
  if (m_Selected) {
    hBrush = ::CreateSolidBrush(0x11a011);
  } else {
    hBrush = ::CreateSolidBrush(::GetSysColor(COLOR_3DFACE));
  }
  
  FillRect(hDC, &m_Rect, hBrush);
  ::DeleteObject(hBrush);
  
  HFONT       hFont, hOldFont;

  hFont = (HFONT)::GetStockObject(DEFAULT_GUI_FONT);

  TEXTMETRIC tm;
  char fontname[255];
  hOldFont = (HFONT)::SelectObject(hDC, hFont);
  ::GetTextMetrics(hDC, &tm);
  ::GetTextFace(hDC, sizeof(fontname), fontname);
  ::SelectObject(hDC, hOldFont);
  hFont = ::CreateFont(tm.tmHeight, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, tm.tmCharSet, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY, DEFAULT_PITCH, fontname);

  hOldFont = (HFONT)SelectObject(hDC, hFont);
  
  if (m_Selected) {
    SetBkMode(hDC, TRANSPARENT);
  } else {
    SetBkMode(hDC, OPAQUE);
  }
  
  SetBkColor(hDC, GetSysColor(COLOR_3DFACE));
  ::TextOut(hDC, m_Rect.left + 2, m_Rect.top + 2, (LPCTSTR)m_String, m_String.GetLength());
  
  
  ::SelectObject(hDC, hOldFont);
  ::DeleteObject(hFont);
  
  if (bReleaseDC) {
    ::ReleaseDC(m_hParent, hDC);
  }
}

void MChannelLabel::MoveWindow(RECT *lpRect, BOOL Repaint) {
  m_Rect = *lpRect;
  if (Repaint) {
    Draw();
  }
}

bool MChannelLabel::MouseIn(int X, int Y) {
  if (!m_Visible) {
    return false;
  }
  
  if (X >= m_Rect.left && X <= m_Rect.right && Y >= m_Rect.top && Y <= m_Rect.bottom) {
    return true;
  }
  
  return false;
}


//---------------
//  ChannelGroup
//---------------
ChannelGroup::ChannelGroup(CChannelBar *parentChannelBar, 
                           const MNamedObjectPtr &newObject) {
  lastGroupHeight = 0;
  channelBar = parentChannelBar;
  object = newObject;
  inputLabelPos = -50;
  outputLabelPos = -50;
  shouldDelete = false;

  updateWindow();

}

ChannelGroup::~ChannelGroup() {
  ChannelParamVector::iterator it;

  for (it = allObjects.begin(); it != allObjects.end(); ++it) {
    if (*it != NULL) {
      if ((*it)->IsWindow()) {
        (*it)->DestroyWindow();
      }
      delete *it;
    }
  }
  
}


MNamedObjectPtr ChannelGroup::getObject() {
  return object;
}


void ChannelGroup::organiseGroup(int top, int width) {
  int topStart = top;
  organiseGroups(normalObjects, top, width);
  inputLabelPos = top;
  top += 15;
  organiseGroups(inputObjects, top, width);
  outputLabelPos = top;
  top += 15;
  organiseGroups(outputObjects, top, width);

  for (int n = 0; n < allObjects.size(); n++) {
    ::InvalidateRect(allObjects[n]->m_hWnd, NULL, TRUE);
    ::UpdateWindow(allObjects[n]->m_hWnd);
  }

  lastGroupHeight = top - topStart;
}

void ChannelGroup::organiseGroups(ChannelParamVector &group, int &ypos, int width) {
  for (int n = 0; n < group.size(); n++) {
    moveGroup(group[n], ypos, width);
  }
}

void ChannelGroup::moveGroup(CChannelParamWnd* window, int &yPos, int width) {
  RECT ChildRect;

  window->GetChildExtents(&ChildRect);
  ::MoveWindow(window->m_hWnd, 0, yPos, width, ChildRect.bottom, TRUE);
  
  yPos += ChildRect.bottom + 3;
}


int ChannelGroup::getHeight() {
  return lastGroupHeight;
}

void ChannelGroup::updateWindow() {
  // loop over all our objects, and mark them for deletion.
  for (int i = 0; i < allObjects.size(); ++i) {
    allObjects[i]->setMarkForDeletion(true);
  }

  // add in ourselves
  addObject(object, Category_Normal);

  MSceneObjectPtr sceneThis = AZTEC_CAST(MSceneObject, object);
  MSceneObjectPtr sceneParent;

  if (sceneThis != NULL) {
    sceneParent = sceneThis->getParent();
  }

  // iterate over the inputs, and add them into the channel bar.
  std::vector<MDAGNode*> ignoreList;
  ignoreList.push_back(&*sceneParent);
  for (MDAGraphUpstreamIterator it(&*object, ignoreList); !it.atEnd(); ++it) {
    MNamedObject *inputNamedObj = AZTEC_CAST(MNamedObject, *it);
    
    if (inputNamedObj != NULL && inputNamedObj != &*object) {
      addObject(inputNamedObj, Category_Input);
    }
  }

  // iterate over the outpust, and add those objects as well.
  for (MDAGraphDownstreamIterator it(&*object); !it.atEnd(); ++it) {
    MNamedObject *inputNamedObj = AZTEC_CAST(MNamedObject, *it);
    
    if (inputNamedObj != NULL && object != inputNamedObj) {
      addObject(inputNamedObj, Category_Output);
    }
  }

  // now loop over all the objects and delete any that shuld be deleted
  std::set<CChannelParamWnd*> windowsToRemove;

  for (int i = allObjects.size() - 1; i >= 0; --i) {
    if (allObjects[i]->isMarkedForDeletion()) {
      windowsToRemove.insert(allObjects[i]);
      allObjects.erase(allObjects.begin() + i);
    }
  }
  for (int i = normalObjects.size() - 1; i >= 0; --i) {
    if (normalObjects[i]->isMarkedForDeletion()) {
      windowsToRemove.insert(normalObjects[i]);
      normalObjects.erase(normalObjects.begin() + i);
    }
  }
  for (int i = inputObjects.size() - 1; i >= 0; --i) {
    if (inputObjects[i]->isMarkedForDeletion()) {
      windowsToRemove.insert(inputObjects[i]);
      inputObjects.erase(inputObjects.begin() + i);
    }
  }
  for (int i = outputObjects.size() - 1; i >= 0; --i) {
    if (outputObjects[i]->isMarkedForDeletion()) {
      windowsToRemove.insert(outputObjects[i]);
      outputObjects.erase(outputObjects.begin() + i);
    }
  }

  // now actually delete the windows
  std::set<CChannelParamWnd*>::iterator it;
  for (it = windowsToRemove.begin(); it != windowsToRemove.end(); ++it) {
    if (*it != NULL) {
      if ((*it)->IsWindow()) {
        (*it)->DestroyWindow();
      }
      delete *it;
    }
  }

  for (int n = 0; n < allObjects.size(); ++n) {
    allObjects[n]->UpdateAllParams();
    allObjects[n]->ShowWindow(SW_SHOW);
  }
}

int ChannelGroup::getInputLabelPos() {
  return inputLabelPos;
}

int ChannelGroup::getOutputLabelPos() {
  return outputLabelPos;
}

void ChannelGroup::setMarkForDeletion(bool value) {
  shouldDelete = value;
}

bool ChannelGroup::isMarkedForDeletion() {
  return shouldDelete;
}

const ChannelGroup::ChannelParamVector& ChannelGroup::getAllWindows() {
  return allObjects;
}

void ChannelGroup::collapseAll() {
  for (int n = 0; n < allObjects.size(); n++) {
    allObjects[n]->m_Object->unsetFlag(OBJECTFLAG_CHANNELEXPAND);
    allObjects[n]->m_LastObject = NULL;
  }
}

void ChannelGroup::expandAll() {
  for (int n = 0; n < allObjects.size(); n++) {
    allObjects[n]->m_Object->setFlag(OBJECTFLAG_CHANNELEXPAND);
    allObjects[n]->m_LastObject = NULL;
  }
}

void ChannelGroup::addObject(const MNamedObjectPtr &object, Category category) {

  CChannelParamWnd *controls = addObjectControls(object);
  if (controls != NULL) {
    if (category == Category_Output) {
      object->unsetFlag(OBJECTFLAG_CHANNELEXPAND);
    } else {
      object->setFlag(OBJECTFLAG_CHANNELEXPAND);
    }
    categoryMap.insert(CategoryMap::value_type(object, category));
    getParamVectorFor(object)->push_back(controls);
  }
}

CChannelParamWnd* ChannelGroup::addObjectControls(const MNamedObjectPtr &object) {
  RECT ClientRect;
  
  // If we only have one parameter visible (its name), then do not
  // bother adding it, since it is useless.
  if (getVisibleParameterCount(object->getParamList()) <= 1) {
    return NULL;
  }
  
  // loop over our currently known objects. If it already exists, then do not
  // bother adding it, just unmark it for deletion
  for (int i = 0; i < allObjects.size(); ++i) {
    if (allObjects[i]->m_Object == object) {
      allObjects[i]->setMarkForDeletion(false);
      return NULL;
    }
  }

  ::GetClientRect(channelBar->m_hWnd, &ClientRect);
  ClientRect.right -= 8;
  
  CChannelParamWnd *newWindow = new CChannelParamWnd();
  newWindow->Create(channelBar->m_hWnd, ClientRect, "ChannelBar", WS_CHILD | WS_VISIBLE | WS_CLIPCHILDREN | WM_VSCROLL);
  newWindow->m_Object = object;

  allObjects.push_back(newWindow);
  
  return newWindow;
}

ChannelGroup::ChannelParamVector* 
ChannelGroup::getParamVectorFor(const MNamedObjectPtr &object) {
  // we should always have a valid window that has been added properly.
  assert(categoryMap.find(object) != categoryMap.end());

  Category cat = categoryMap.find(object)->second;

  switch (cat) {
  case Category_Normal:
    return &normalObjects;
  case Category_Input:
    return &inputObjects;
  case Category_Output:
    return &outputObjects;
  }
  return NULL;
}

// CChannelBarContainer

LRESULT CChannelBarContainer::ProcessSizingMsg(WPARAM wParam, LPARAM lParam, BOOL& bHandled) {
  RECT clientRect;
  GetClientRect(&clientRect);

  if (::IsWindow(menuBar.m_hWnd)) {
    menuBar.SetWindowPos(NULL, clientRect.left, clientRect.top, clientRect.right, menuBar.getMenuHeight(), SWP_NOZORDER | SWP_NOMOVE);
    clientRect.top += menuBar.getMenuHeight();
  }

  if (channelBar.IsWindow()) {
    channelBar.SetWindowPos(NULL, &clientRect, SWP_NOZORDER);
  }

	return 0;
}

class ObjectsMenu : public MMenuBar::MenuCreator {
public:
  typedef CChannelBarContainer::ObjectMap ObjectMap;

  ObjectsMenu(CChannelBar *channelBar, ObjectMap &objects) 
    : channelBar(channelBar),
      objectMap(objects)
  {
  }

  HMENU createMenu() {
    HMENU popup = ::CreatePopupMenu();

    MBaseObjectListPtr objects = g_Scene->getSelectedObjectList();
    MBaseObjectPtr object;

    objectMap.clear();
    objects->beginIteration();

    int id = 1000;

    if (channelBar->getCurrentObject() != NULL) {
      objectMap.insert(ObjectMap::value_type(id, &*channelBar->getCurrentObject()));
      ::AppendMenu(popup, MF_STRING, id, channelBar->getCurrentObject()->getName().c_str());
      ++id;

      ::AppendMenu(popup, MF_SEPARATOR, id, NULL);
      ++id;
    }

    while ((object = objects->getNext()) != NULL) {
      MNamedObject *namedObj = AZTEC_CAST(MNamedObject, object);

      if (namedObj != NULL) {
        objectMap.insert(ObjectMap::value_type(id, namedObj));
        ::AppendMenu(popup, MF_STRING, id, namedObj->getName().c_str());
        ++id;
      }
    }

    objects->endIteration();

    // If we ended up with no objects in our map, it means our channel bar is 
    // empty, so add in a disabled menu item to say so.

    ::AppendMenu(popup, MF_STRING | MF_DISABLED | MF_GRAYED, 0, "No Objects Selected!");

    return popup;
  }

  void destroyMenu(HMENU menu) {
    ::DestroyMenu(menu);
  }

private:
  CChannelBar *channelBar;
  ObjectMap &objectMap;
};

LRESULT CChannelBarContainer::OnCreate(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled) {
  RECT clientRect, buttonRect;
  
  GetClientRect(&clientRect);

  CWnd *parent = CWnd::FromHandle(m_hWnd);

  menuBar.Create(NULL, 
                    "MenuBar", 
                    WS_VISIBLE | WS_CHILD | WS_CLIPSIBLINGS, 
                    CRect(clientRect.left,clientRect.top,clientRect.right,24), 
                    parent, 
                    0);
  menuBar.addMenu("Selected Objects", new ObjectsMenu(&channelBar, objectMap));
  menuBar.SetWindowPos(NULL, clientRect.left, clientRect.top, clientRect.right, menuBar.getMenuHeight(), SWP_NOZORDER | SWP_NOMOVE);

  clientRect.top += menuBar.getMenuHeight();

  buttonRect = clientRect;
  buttonRect.top += 3;
  buttonRect.bottom = buttonRect.top + 22;
  buttonRect.left += 5;
  buttonRect.right -= 5;
  clientRect.top += 25;

  channelBar.Create(m_hWnd, clientRect, "ChannelBar");
  channelBar.ShowWindow(SW_SHOW);
	return 0;
}

LRESULT CChannelBarContainer::OnWindowPosChanged(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled) {
	return ProcessSizingMsg(wParam, lParam, bHandled);
}

LRESULT CChannelBarContainer::OnWindowPosChanging(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled) {
	return ProcessSizingMsg(wParam, lParam, bHandled);
}

LRESULT CChannelBarContainer::OnSize(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled) {
	return ProcessSizingMsg(wParam, lParam, bHandled);
}

LRESULT CChannelBarContainer::OnPaint(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled) {
  PAINTSTRUCT paint;

  ::BeginPaint(m_hWnd, &paint);

  HDC hdc = (HDC)wParam;

  if (hdc == NULL) {
    hdc = paint.hdc;
  }

  RECT rect;

  ::GetClientRect(m_hWnd, &rect);

  ::FillRect(hdc, &rect, ::GetSysColorBrush(COLOR_3DFACE));

  ::EndPaint(m_hWnd, &paint);

  return 0;
}

LRESULT CChannelBarContainer::OnEraseBackground(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled) {
  

  return 0;
}

LRESULT CChannelBarContainer::OnCommand(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled) {

  HWND hwnd = (HWND)lParam;

  if (hwnd == NULL) {
    assert(objectMap.size() > 0);

    // convert the menu id into an object
    ObjectMap::iterator objIter = objectMap.find(wParam);

    channelBar.setCurrentObject(objIter->second);

    // now clear out the object map now that we are done with it.
    objectMap.clear();
  }

  return 0;
}