#include <AztecMainPCH.h>
#include <tools/MGraphMoveTool.h>

// AztecMain includes
#include <MDLGlobs.h>

#include <views/GraphViewWnd.h>

#include <MFloatKey.h>

MGraphMoveTool::MGraphMoveTool() {
  setName("GraphMoveTool");
  m_Dragging = false;
  m_RequiresSel = true;
}

static CGraphComponent* getGraph(MBaseViewWndPtr View) {
  CGraphComponent *graph = AZTEC_CAST(CGraphComponent, View);

  if (graph != NULL) {
    return graph;
  }

  // try and get our graph component
  CGraphViewWnd *graphView = AZTEC_CAST(CGraphViewWnd, View);
  if (graphView != NULL) {
    return graphView->getGraphComponent();
  }

  return NULL;
}

int MGraphMoveTool::DrawTool(bool Select, MShiftState ShiftState, MBaseViewWndPtr View) {
  CGraphComponent *graph = getGraph(View);

  if (graph == NULL) {
    return 0;
  }

  if (m_Dragging) {
    float x1,y1,x2,y2;

    graph->clientToGraph(m_DownX, m_DownY, x1,y1);
    graph->clientToGraph(m_CurX, m_CurY, x2,y2);

    glColor3f(0.1f, 1.0f, 0.7f);
    glBegin(GL_LINE_LOOP);
    glVertex2f(x1,y1);
    glVertex2f(x2,y2);

    glEnd();
  }

  return 1;
}

int MGraphMoveTool::onMouseDown(int mouseX, int mouseY, const MShiftState &Shift) {
  MToolType::onMouseDown(mouseX, mouseY, Shift);

  m_Dragging = true;
  m_CurX = m_DownX = mouseX;
  m_CurY = m_DownY = mouseY;

  CGraphComponent *graph = getGraph(g_CurView);

  if (graph == NULL) {
    return TOOLRESULT_DRAWNONE;
  }
  
  std::vector<CGraphComponent::ListKey> keys;
  std::vector<GraphPair> graphValues;

  graph->getGraphValues(graphValues);

  // loop over all keys lists and store the values so we can revert 
  // back to them as needed.
  for (int i = 0; i < graphValues.size(); ++i) {
    MFloatKeyListPtr keyList = AZTEC_CAST(MFloatKeyList, graphValues[i].value);

    if (keyList == NULL) {
      continue;
    }

    keyList->storeValue();
  }

  return TOOLRESULT_DRAWALL;
}

int MGraphMoveTool::onMouseUp(int mouseX, int mouseY, const MShiftState &Shift)
{
  MToolType::onMouseUp(mouseX, mouseY, Shift);

  CGraphComponent *graph = getGraph(g_CurView);

  if (graph == NULL) {
    return TOOLRESULT_DRAWNONE;
  }
  
  m_Dragging = false;

  m_UpX = mouseX;
  m_UpY = mouseY;
  
  return TOOLRESULT_DRAWALL;
}

int MGraphMoveTool::onMouseMove(int mouseX, int mouseY, const MShiftState &Shift)
{
  MToolType::onMouseMove(mouseX, mouseY, Shift);
  
  m_CurX = mouseX;
  m_CurY = mouseY;
  
  CGraphComponent *graph = getGraph(g_CurView);

  if (graph == NULL) {
    return TOOLRESULT_DRAWNONE;
  }

  if (m_Dragging) {
    std::vector<CGraphComponent::ListKey> keys;
    std::vector<GraphPair> graphValues;

    graph->getGraphValues(graphValues);

    float shiftX;
    float shiftY;
    {
      float x1,y1,x2,y2;
      graph->clientToGraph(m_DownX, m_DownY, x1, y1);
      graph->clientToGraph(m_CurX, m_CurY, x2, y2);

      shiftX = x2 - x1;
      shiftY = y2 - y1;

      // if we have shift depressed, we contraining the movement to 
      // one axis only.
      if (Shift.m_Shift) {
        int diffX = m_CurX - m_DownX;
        int diffY = m_CurY - m_DownY;

        // if we have moved X more, then make it so we only move in 
        // the X direction.
        if (abs(diffX) > abs(diffY)) {
          shiftY = 0;
        } else {
          shiftX = 0;
        }
      }

    }

    // if we are not holding down shift or control, we are just doing a normal drag
    if (!Shift.m_Ctrl) {
      for (int i = 0; i < graphValues.size(); ++i) {
        MFloatKeyListPtr keyList = AZTEC_CAST(MFloatKeyList, graphValues[i].value);
    
        if (keyList == NULL) {
          continue;
        }

        keyList->restoreValue();

        int roundedShift = shiftX / (float)keyList->getGranularity();
        keyList->shiftSelectedKeys(roundedShift * (float)keyList->getGranularity());

        for (int keyIndex = 0; keyIndex < keyList->getKeyCount(); ++keyIndex) {
          MKeyPtr key = keyList->getKeyAtIndex(keyIndex);
          MFloatKey *floatKey = AZTEC_CAST(MFloatKey, key);
          if (floatKey != NULL && floatKey->isSelected()) {
            floatKey->setValue(floatKey->getValue() + shiftY);
          }
        }

      }
    } else if (Shift.m_Ctrl) {
      // if we have control pressed, then we are copying keys. First we iterate over the 
      // keys, to find out which ones need copying, and then copy them.
      for (int i = 0; i < graphValues.size(); ++i) {
        MFloatKeyListPtr keyList = AZTEC_CAST(MFloatKeyList, graphValues[i].value);
    
        if (keyList == NULL) {
          continue;
        }

        keyList->restoreValue();

        int roundedShift = shiftX / (float)keyList->getGranularity();
        roundedShift *= keyList->getGranularity();

        std::vector<std::pair<long, float> > values;

        for (int keyIndex = 0; keyIndex < keyList->getKeyCount(); ++keyIndex) {
          MKeyPtr key = keyList->getKeyAtIndex(keyIndex);
          MFloatKey *floatKey = AZTEC_CAST(MFloatKey, key);
          if (floatKey != NULL && floatKey->isSelected()) {
            values.push_back(std::make_pair(key->getKeyTime(), floatKey->getValue()));
            floatKey->setSelected(false);
          }
        }


        // now go an set some keys.
        for (int i = 0; i < values.size(); ++i) {
          keyList->setKey(values[i].second + shiftY, values[i].first + roundedShift);
          keyList->getKeyAtTime(values[i].first + roundedShift)->setSelected(true);
        }
      }
    }

    return TOOLRESULT_DRAWALL;
  } else {
    std::vector<CGraphComponent::ListKey> keys;

    graph->getKeysAtPoint(mouseX, mouseY, keys);

    HCURSOR cursor = ::LoadCursor(NULL, IDC_ARROW);

    // loop over the keys underneath our cursor and see if any are
    // selected. If they are, show the moving cursor to say that we can move
    // them somewhere.
    if (keys.size() > 0) {
      for (int i = 0; i < keys.size(); ++i) {
        if (keys[i].first->getKeyAtIndex(keys[i].second)->isSelected()) {
          cursor = ::LoadCursor(NULL, IDC_SIZEALL);
          break;
        }
      }
    }
    ::SetCursor(cursor);
  }
  
  return TOOLRESULT_DRAWNONE;
}


