// UVEditViewWnd.cpp : implementation file
//

#include "AztecMainPCH.h"
#include "UVEditViewWnd.h"

#include "OpenGLWnd.h"

#include "MainFunc.h"
#include "KeyFuncView.h"
#include "MUIManager.h"

#include "MdlGlobs.h"
#include "MdlMsgs.h"

#include "DlgGlobs.h"

#include <gl/gl.h>
#include <gl/glu.h>

#include <MEditableMesh.h>

using std::vector;

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


//----------------------------------------------------------------------------------------
//  CUVEditViewWnd
//----------------------------------------------------------------------------------------
CUVEditViewWnd::CUVEditViewWnd() {
  m_Zoom = 1;
  m_Pan.x = 0.5;
  m_Pan.y = 0.5;
  
  m_PopupRect.left = 0;
  m_PopupRect.right = 50;
  m_PopupRect.top = 0;
  m_PopupRect.bottom = 16;
  
  m_ImageAspect = 1.0;
}

CUVEditViewWnd::~CUVEditViewWnd() {
}


BEGIN_MESSAGE_MAP(CUVEditViewWnd, CWnd)
//{{AFX_MSG_MAP(CUVEditViewWnd)
ON_WM_LBUTTONDOWN()
ON_WM_LBUTTONUP()
ON_WM_MBUTTONDOWN()
ON_WM_MBUTTONUP()
ON_WM_RBUTTONDOWN()
ON_WM_RBUTTONUP()
ON_WM_PAINT()
ON_WM_SIZE()
ON_WM_MOUSEMOVE()
ON_WM_DESTROY()
ON_WM_CLOSE()
//}}AFX_MSG_MAP
END_MESSAGE_MAP()


MBaseObjectPtr CUVEditViewWnd::createNew() {
  CUVEditViewWnd    *Wnd;
  
  Wnd = new CUVEditViewWnd;
  Wnd->setName(getName());
  
  Wnd->SetViewName(m_ViewName);
  Wnd->m_Pan = m_Pan;
  Wnd->m_Zoom = m_Zoom;
  
  
  Wnd->m_Flags = m_Flags;
  
  return (MBaseObject*)Wnd;
}

void CUVEditViewWnd::ViewCreate() {
  MBaseOpenGLWnd::ViewCreate();
  
  int Error;
  
  Error = InitOpenGL(m_hWnd);
  
  if (Error!=0)	{// Did OpenGL fail to intiialise
    if (Error == WGL_CREATECONTEXT_FAILED) {
      AfxMessageBox("wglCreateContext Failed");
    }
    if (Error == WGL_MAKECURRENT_FAILED) {
      AfxMessageBox("wglMakeCurrent Failed");
    }
  }
}

void CUVEditViewWnd::DrawView() {
  MBaseOpenGLWnd::DrawView();

  MSceneViewFlags   ViewFlags;
  MSelectMethod     SelectMethod;
  
  SelectMethod = smNone;
  if (m_Action == atSelect) {
    SelectMethod = smSelect;
  } else if (m_Action == atDeselect) {
    SelectMethod = smDeselect;
  } else if (m_Action == atSelectBox) {
    SelectMethod = smSelectBox;
  } else if (m_Action == atDeselectBox) {
    SelectMethod = smDeselectBox;
  }
  
  if (g_CurView != MBaseViewWndPtr(this)) {
    SelectMethod = smNone;
  }
  
  // check to see if tehe viewport is 0 size, then don't draw it
  
  if (m_WindowRect.bottom == 0 || m_WindowRect.right == 0) {
    return;
  }
  
  GLMakeCurrent();
  
  AdjustProjectionMatrix(SelectMethod);
  
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  DoViewportTransform();
  
  glColor4f(1,1,1,1);
  
  if (SelectMethod != smNone) {
    ViewFlags.m_SelectBuf = new UINT[16384];
    glSelectBuffer(16384, ViewFlags.m_SelectBuf);
    glRenderMode(GL_SELECT);
    glPushName(-1);
  }
  
  glDisable(GL_DEPTH_TEST);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  
  glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
  glClearColor(0.4f, 0.4f, 0.4f, 0.0f);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  
  glDisable(GL_LIGHTING);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glDisable(GL_CULL_FACE);
  
  glLineWidth(2);
  
  // Draw the texture of hte last selected object
  {
    MSceneObjectPtr Obj;
    MMeshPtr Mesh;
    
    Obj = AZTEC_CAST(MSceneObject, g_Scene->getSelectedObjectList()->getTail());
    
    if (Obj != NULL) {
      MSceneObjectPtr sceneObj;
      if ((sceneObj = AZTEC_CAST(MSceneObject, Obj)) != NULL) {
        if (sceneObj->getShapeObject() != NULL) {
          Mesh = AZTEC_CAST(MMesh, sceneObj->getShapeObject()->convertToMesh());
        }
      }
      if (Mesh != NULL) {
        MMaterialPtr Material;
        
        Material = Obj->getTextureMaterial();
        
        if (Material != NULL) {
          MImagePtr Image;
          Image = Material->getDiffuseMapImage();
          if (Image != NULL) {
            glEnable(GL_TEXTURE_2D);
            
            // Scale the image so it is in the correct aspect ratio.
            m_ImageAspect = (float)Image->getWidth()/Image->getHeight();
            
            //                  Image->SetGLImageAsCurrent(GL_TEXTURE_2D);
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
            Material->setOpenGLMaterialAndTextures();
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
            glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
            
            glBegin(GL_TRIANGLES);
            
            glTexCoord2f(0,0);
            glVertex2f(0,0);
            glTexCoord2f(1,0);
            glVertex2f(1,0);
            glTexCoord2f(1,1);
            glVertex2f(1,1);
            
            glTexCoord2f(1,1);
            glVertex2f(1,1);
            glTexCoord2f(0,1);
            glVertex2f(0,1);
            glTexCoord2f(0,0);
            glVertex2f(0,0);
            
            glEnd();
            
            glDisable(GL_TEXTURE_2D);
          }
        }
      }
    }
    
  }
  
  glColor4f(1,1,1,1);
  glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
  
  // Draw the skin mesh border
  glBegin(GL_LINE_STRIP);
  glVertex2f(0,0);
  glVertex2f(1,0);
  glVertex2f(1,1);
  glVertex2f(0,1);
  glVertex2f(0,0);
  glEnd();
  
  
  glColor4f(1,1,1,1);
  // Draw the texture mesh
  {
    MBaseObjectPtr Obj;
    MMeshPtr Mesh;  
    int meshNumber = 0;
    
    g_Scene->getSelectedObjectList()->beginIteration();
    
    while ((Obj = g_Scene->getSelectedObjectList()->getNext()) != NULL) {
      MSceneObjectPtr sceneObj;
      if ((sceneObj = AZTEC_CAST(MSceneObject, Obj)) == NULL) {
        continue;
      }
      if (sceneObj->getShapeObject() != NULL) {
        Mesh = AZTEC_CAST(MMesh, sceneObj->getShapeObject()->convertToMesh());
      }
      meshNumber++;

      if (Mesh != NULL) {
        // put our object number on the name stack if necessary.
        if (SelectMethod != smNone) {
          glPushName(meshNumber);
        }

        MMeshPtr TextureMesh;
      
        TextureMesh = Mesh->getTextureMesh();
      
        if (TextureMesh != NULL) {
          int   nTri, nVert;
        
          // Only draw triangles if not in a selection mode
          if (SelectMethod == smNone) {
            glDisable(GL_LINE_SMOOTH);
            glLineWidth(1);
          
            glColor4f(1,1,1,1);
            for (int n=0; n<3; n++) {
              glBegin(GL_TRIANGLES);
              for (nTri = TextureMesh->getNumTris()-1; nTri>=0; nTri--) {
                MTriangle const *Tri = TextureMesh->getTriangle(nTri);
                MVector3 V[3];
              
                if (!Tri) continue;
              
                V[0] = TextureMesh->getVertexPosition(Tri->getVertex(0));
                V[1] = TextureMesh->getVertexPosition(Tri->getVertex(1));
                V[2] = TextureMesh->getVertexPosition(Tri->getVertex(2));
              
                glVertex2fv((float*)&V[0]);
                glVertex2fv((float*)&V[1]);
                glVertex2fv((float*)&V[2]);
              }
              glEnd();
            }
          }
        
          // Object mode is basically groupd vertex selecting, so only deal with 
          // the selection buffer here
          if (MUIManager::getComponentMode() == MUIManager::OBJECT_MODE) {
            if (SelectMethod != smNone) {
              for (nVert = TextureMesh->getNumVerts()-1; nVert>=0; nVert--) {
                MVector3 pos = TextureMesh->getVertexPosition(nVert);
              
                glPushName(nVert);
                glBegin(GL_POINTS);
                glVertex2f(pos.x, pos.y);
                glEnd();
                glPopName();
              }
            }
          }
        
          // If we are in point mode, draw points
          if (MUIManager::getComponentMode() == MUIManager::POINT_MODE) {
            glColor3f(0.0f,0.8f,1.0f);
            glPointSize(3);
          
            if (SelectMethod == smNone) {
              glBegin(GL_POINTS);
            }
          
            for (nVert = TextureMesh->getNumVerts()-1; nVert>=0; nVert--) {
              glColor3f(0.0f,0.8f,1.0f);
            
              if (TextureMesh->isVertexFlagged(nVert, VERTEX_SELECTED)) {
                glColor3f(1,1,0);
              }
            
              if (SelectMethod != smNone) {
                glPushName(nVert);
                glBegin(GL_POINTS);
              }

              MVector3 pos = TextureMesh->getVertexPosition(nVert);
              glVertex2f(pos.x, pos.y);
            
              if (SelectMethod != smNone) {
                glEnd();
                glPopName();
              }
            }
          
            if (SelectMethod == smNone) {
              glEnd();
            }
          
          }
        
          // Face mode deals with whole triangles. This handles both selection buffers
          // and drawing the selected triangles.
          if (MUIManager::getComponentMode() == MUIManager::FACET_MODE) {
            glEnable(GL_BLEND);
            //                  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
            glBlendFunc(GL_SRC_ALPHA, GL_SRC_COLOR);
            glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
            glColor4f(1.0f,0.8f,0.3f, 0.2f);
          
            if (SelectMethod == smNone) {
              glBegin(GL_TRIANGLES);
            }
          
            for (nTri = TextureMesh->getNumTris()-1; nTri>=0; nTri--) {
              MTriangle const *T;
              MVector3 V[3];
            
              T = TextureMesh->getTriangle(nTri);
            
              if (!T) continue;
            
              V[0] = TextureMesh->getVertexPosition(T->getVertex(0));
              V[1] = TextureMesh->getVertexPosition(T->getVertex(1));
              V[2] = TextureMesh->getVertexPosition(T->getVertex(2));
            
              if (SelectMethod != smNone) {
                glPushName(nTri);
                glBegin(GL_TRIANGLES);
              }
            
              // Only draw the triangle  if we are drawing the select buffer, or if the triangle is selected
              if (SelectMethod != smNone || T->isFlagged(TRIANGLE_SELECTED)) {
                glVertex2fv((float*)&V[0]);
                glVertex2fv((float*)&V[1]);
                glVertex2fv((float*)&V[2]);
              }
            
              if (SelectMethod != smNone) {
                glEnd();
                glPopName();
              }
            }
            if (SelectMethod == smNone) {
              glEnd();
            }
          }
        
          // Wsfw mode deals with only single edges of triangles. This handles
          // both selection and drawing the selected triangles.
          if (MUIManager::getComponentMode() == MUIManager::EDGE_MODE)
          {
            int      NumEdges;
          
            NumEdges = 0;
          
            glColor4f(1.0f,0.0f,0.0f, 1.0f);
          
            if (SelectMethod == smNone) {
              glBegin(GL_LINES);
            }
          
            //                  for (nTri = TextureMesh->getNumTris()-1; nTri>=0; nTri--)
            for (nTri = 0; nTri<TextureMesh->getNumTris(); nTri++) {
              MTriangle const *T;
              MVector3 V[3];
            
              T = TextureMesh->getTriangle(nTri);
              if (!T) {
                continue;
              }
            
              V[0] = TextureMesh->getVertexPosition(T->getVertex(0));
              V[1] = TextureMesh->getVertexPosition(T->getVertex(1));
              V[2] = TextureMesh->getVertexPosition(T->getVertex(2));
            
              if (SelectMethod != smNone) {
                glPushName(NumEdges++);
                glBegin(GL_LINES);
                glVertex2fv((float*)&V[0]);
                glVertex2fv((float*)&V[1]);
                glEnd();
                glPopName();
              
                glPushName(NumEdges++);
                glBegin(GL_LINES);
                glVertex2fv((float*)&V[1]);
                glVertex2fv((float*)&V[2]);
                glEnd();
                glPopName();
              
                glPushName(NumEdges++);
                glBegin(GL_LINES);
                glVertex2fv((float*)&V[2]);
                glVertex2fv((float*)&V[0]);
                glEnd();
                glPopName();
              
              } else {
              
                if (T->isEdgeFlagged(0, EDGE_SELECTED)) {
                  glVertex2fv((float*)&V[0]);
                  glVertex2fv((float*)&V[1]);
                }
                if (T->isEdgeFlagged(1, EDGE_SELECTED)) {
                  glVertex2fv((float*)&V[1]);
                  glVertex2fv((float*)&V[2]);
                }
                if (T->isEdgeFlagged(2, EDGE_SELECTED)) {
                  glVertex2fv((float*)&V[2]);
                  glVertex2fv((float*)&V[0]);
                }
              }
            }
            if (SelectMethod == smNone) {
              glEnd();
            }
          }
        }
        // take our object number on the name stack if necessary.
        if (SelectMethod != smNone) {
          glPopName();
        }
      }
    }
    g_Scene->getSelectedObjectList()->endIteration();
  }
  
  DrawViewExtras();
  
  glFlush();
  glFinish();
  
  HDC  hDC;
  hDC = ::GetDC(m_hWnd);
  
  GLSwapBuffers();
  
  ::ReleaseDC(m_hWnd, hDC);
  
  if (SelectMethod != smNone) {
    // check to see if we have to clear the selection first.
    if (!getShiftState().m_Ctrl) {
      MBaseObjectPtr Obj;
      MMeshPtr Mesh, TextureMesh;
      
      TextureMesh = NULL;
      g_Scene->getSelectedObjectList()->beginIteration();
    
      while ((Obj = g_Scene->getSelectedObjectList()->getNext()) != NULL) {
      MSceneObjectPtr sceneObj;
        if ((sceneObj = AZTEC_CAST(MSceneObject, Obj)) == NULL) {
          continue;
        }
        if (sceneObj->getShapeObject() != NULL) {
          Mesh = AZTEC_CAST(MMesh, sceneObj->getShapeObject()->convertToMesh());
        }
        if (Mesh != NULL) {
          TextureMesh = Mesh->getTextureMesh();
        }
        
        if (TextureMesh != NULL) {
          if (MUIManager::getComponentMode() == MUIManager::POINT_MODE) {
            // Deselect the vertices
            TextureMesh->unsetVertexFlags(VERTEX_SELECTED);
          }
          if (MUIManager::getComponentMode() == MUIManager::FACET_MODE) {
            // Deselect the vertices
            for (int i=0; i<TextureMesh->getNumTris(); i++) {
              TextureMesh->unsetTriangleFlag(i, TRIANGLE_SELECTED);
            }
          }
          if (MUIManager::getComponentMode() == MUIManager::EDGE_MODE) {
            // Deselect the vertices
            for (int i=0; i<TextureMesh->getNumTris(); i++) {
              TextureMesh->unsetTriangleEdgeFlag(i, 0, EDGE_SELECTED);
              TextureMesh->unsetTriangleEdgeFlag(i, 1, EDGE_SELECTED);
              TextureMesh->unsetTriangleEdgeFlag(i, 2, EDGE_SELECTED);
            }
          }
          
        }
      }
      g_Scene->getSelectedObjectList()->endIteration();
    }
     
     
     int      NumHits;
     
     NumHits = glRenderMode(GL_RENDER);
     
     // Process selection here.
     
     int      i, Count, j, Result;
     GLuint   *Ptr;
     
     Ptr = ViewFlags.m_SelectBuf;
     Result = 2;
     
     for (i=0;i<NumHits;i++) {
       float    Depth1, Depth2;
       
       Count = *Ptr; 
       Ptr++;
       Depth1 = (float) *Ptr/0x7fffffff; Ptr++;
       Depth2 = (float) *Ptr/0x7fffffff; Ptr++;
       
       
       j = 0;
       while (j<Count) {
         if (*Ptr == -1) {
           Ptr++; j++;
           if (j >= Count) {
             break;
           }
           
           int VertNum, ObjNum;
           
           ObjNum = *Ptr;
           Ptr++; j++;
           if (j >= Count) {
             break;
           }

           VertNum = *Ptr;
           
           MBaseObjectPtr Obj;
           MMeshPtr Mesh;
           
           int objectCount = 0;

           g_Scene->getSelectedObjectList()->beginIteration();

           // try and get the object involved in the selection.
           while ((Obj = g_Scene->getSelectedObjectList()->getNext()) != NULL) {
             objectCount++;
             if (objectCount == ObjNum) break;
           }
           g_Scene->getSelectedObjectList()->endIteration();
           
           if (Obj != NULL) {
             MSceneObjectPtr sceneObj;
             if ((sceneObj = AZTEC_CAST(MSceneObject, Obj)) == NULL) {
               continue;
             }
             if (sceneObj->getShapeObject() != NULL) {
               Mesh = AZTEC_CAST(MMesh, sceneObj->getShapeObject()->convertToMesh());
             }
             if (Mesh != NULL) {
               MMeshPtr TextureMesh;
               bool SelectTarget;
               
               if (g_MainAction == atSelect || g_MainAction == atSelectBox) {
                 SelectTarget = true;
               } else {
                 SelectTarget = false;
               }
               
               TextureMesh = Mesh->getTextureMesh();
               
               if (TextureMesh != NULL) {
                 TextureMesh->selectComponent( MUIManager::getComponentMode(), VertNum, SelectTarget);
               }
             }
           }
           
           Ptr++; 
           j++;
         } else {
           Ptr++;
           j++;
         }
       }
     }
     
     
     delete[] ViewFlags.m_SelectBuf;
   }
}

void CUVEditViewWnd::DrawViewExtras() {
  glDisable(GL_LIGHTING);
  glDisable(GL_DEPTH_TEST);
  
  glClear(GL_DEPTH_BUFFER_BIT);
  // Draw the gizmos for the various tools
  
  glMatrixMode(GL_MODELVIEW);
  
  DrawManipulators(false);
  
  // Push the projection stack, so we can draw using screen coordinates
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  
  glViewport(0,0,m_WindowRect.right, m_WindowRect.bottom);
  
  glLoadIdentity();
  
  glOrtho(0,m_WindowRect.right,0,m_WindowRect.bottom,-20,20);
  // glOrtho(0,m_WindowRect.right,m_WindowRect.bottom,0,20,20);
  
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();
  
  MStr     ViewTitle;
  
  ViewTitle = m_ViewName;
  {
    MBaseObjectPtr Obj;
    MNamedObjectPtr namedObj;
    
    g_Scene->getSelectedObjectList()->beginIteration();
    
    while ((Obj = g_Scene->getSelectedObjectList()->getNext()) != NULL) {
      namedObj = AZTEC_CAST(MNamedObject, Obj);
      if (namedObj != NULL) {
        ViewTitle += " - ";
        ViewTitle += namedObj->getName();
      }
    }
    g_Scene->getSelectedObjectList()->endIteration();
  }
  
  glColor3f(1.0,1.0,1.0);
  glRasterPos2d(4,m_WindowRect.bottom-14);
  glListBase(GLVIEW_FONTSTART);
  glCallLists(ViewTitle.GetLength(), GL_UNSIGNED_BYTE, (LPCTSTR)ViewTitle);
  
  if (g_MainAction == atSelectBox || g_MainAction == atDeselectBox) {
    // Draw the current selection box
    if (g_CurView == MBaseViewWndPtr(this)) {
      glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
      glLineWidth(1);
      glColor3f(0.0f, 0.8f, 0.6f);
      
      glBegin(GL_LINE_LOOP);
      
      glVertex2f((float)m_DownX,(float)(m_WindowRect.bottom-m_DownY));
      glVertex2f((float)m_DownX,(float)(m_WindowRect.bottom-m_UpY));
      glVertex2f((float)m_UpX,(float)(m_WindowRect.bottom-m_UpY));
      glVertex2f((float)m_UpX,(float)(m_WindowRect.bottom-m_DownY));
      
      glEnd();
    }
  }
  
  // Put the viewports projection matrix on the thingy
  //   AdjustProjectionMatrix();
  //   DoViewportTransform();
  
  glPushMatrix();
  
  /*   glEnable(GL_DEPTH_TEST);
  glTranslatef(20,20,0);
  glRotatef(m_Rot.z,0,1,0);
  glRotatef(m_Rot.y+180,1,0,0);
  glRotatef(-m_Rot.x,0,0,1);
  DrawAxisIcon(15, 1, 1);
  glDisable(GL_DEPTH_TEST);*/
  
  glPopMatrix();
  
  
  // Draw the border if its the current view
  
  if (g_CurView == MBaseViewWndPtr(this)) {
    glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
    glLineWidth(3);
    glColor3f(0.0f, 0.3f, 0.0f);
    
    glBegin(GL_LINE_LOOP);
    
    glVertex2f(1.0f,1.0f);
    glVertex2f(1.0f,(float)(m_WindowRect.bottom-2));
    glVertex2f((float)(m_WindowRect.right-2),(float)(m_WindowRect.bottom-2));
    glVertex2f((float)(m_WindowRect.right-2),1.0f);
    
    glEnd();
  }
  
  // Clean up this mess matrix wise
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  
  glMatrixMode(GL_MODELVIEW);
  
  // I know, this shouldn't be here.. I'll change this at 18/07/01 :)
  updateSelectionFeedback();
}


void CUVEditViewWnd::onMouseUp(int X, int Y, MShiftState Shift) {
  if (!Shift.m_Left && !Shift.m_Mid && !Shift.m_Right) {
    m_UpX = X;
    m_UpY = Y;
    
    m_Action = g_MainAction;
    if (m_Action == atSelectBox) {
      if (m_UpX == m_DownX || m_UpY == m_DownY) {
        m_Action = atSelect;
      }
    } else if (m_Action == atDeselectBox) {
      if (m_UpX == m_DownX || m_UpY == m_DownY) {
        m_Action = atDeselect;
      }
    }
    
    DrawView();
    
    m_Action = atNone;
    g_MainAction = atNone;
    
    if (m_Action == atSelect || m_Action == atDeselect || m_Action == atSelectBox || m_Action == atDeselectBox) {
    }
    
    g_MainDlg->SendMessage(MM_UPDATEVIEWPORTS, MMC_UPDATE_ALL, 0);
    g_MainDlg->SendMessage(MM_UPDATECHANNELBAR, 0, 0);
  }
}

void CUVEditViewWnd::onMouseDown(int X, int Y, MShiftState Shift) {
  
  // Draw the manipulators so we know what manipulator has been pushed
  {
    GLMakeCurrent();
    
    m_UpX = X;
    m_UpY = Y;
    AdjustProjectionMatrix(smSelect);
    
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    DoViewportTransform();
    g_PickedManip = DrawManipulators(true);
  }
  
  if (Shift.m_Mid) {
    g_PickedManip = 4;
  }
  
  MBaseObjectPtr Obj;
  MMeshPtr Mesh;
  MEditableMeshPtr TextureMesh;
  
  Obj = NULL;
  Mesh = NULL;
  TextureMesh = NULL;
  
  Obj = g_Scene->getSelectedObjectList()->getTail();
  if (Obj != NULL) {
    MSceneObjectPtr sceneObj;
    if ((sceneObj = AZTEC_CAST(MSceneObject, Obj)) != NULL) {
      if (sceneObj->getShapeObject() != NULL) {
        Mesh = AZTEC_CAST(MMesh, sceneObj->getShapeObject()->convertToMesh());
      }
    }
    if (Mesh != NULL) {
      TextureMesh = AZTEC_CAST(MEditableMesh, Mesh->getTextureMesh());
    }
  }

  g_MainAction = atNone;
  
  if (g_MainAction == atNone) {
    
    if (g_ToolMan.GetTool()->Is("KToolSelect")) {
      if (Shift.m_Shift) {
        g_MainAction = atDeselect;
      } else {
        g_MainAction = atSelect;
      }
    }
    if (g_ToolMan.GetTool()->Is("KToolMove")) {
      g_MainAction = atMove;
    }
    if (g_ToolMan.GetTool()->Is("KToolRotate")) {
      g_MainAction = atRotate;
    }
    if (g_ToolMan.GetTool()->Is("KToolScale")) {
      g_MainAction = atScale;
    }
    
    // If our current too requires a selection, and we have none, put our tool to at select
    if (g_ToolMan.GetTool()->RequiresSelection()) {
      if (g_Scene->getNumSelectedObjects() == 0)
        g_MainAction = atSelect;
    }
    
  }
  
  if (g_MainAction == atMove) {
    if (g_PickedManip == 0) {
      g_MainAction = atSelect;
    }
  } else if (g_MainAction == atRotate) {
    if (g_PickedManip == 0) {
      g_MainAction = atSelect;
    }
  } else if (g_MainAction == atScale) {
    if (g_PickedManip == 0) {
      g_MainAction = atSelect;
    }
  } else if (Shift.m_Shift && g_MainAction == atSelect) {
    g_MainAction = atDeselect;
  }
  
  
  if (TextureMesh != NULL) {
    if (g_ToolMan.GetTool()->RequiresSelection() && g_MainAction != atNone) {
      bool AnythingSelected;
      
      AnythingSelected = false;
      
      TextureMesh->unsetVertexFlags(VERTEX_FLAGFORCHANGE);

      // If we ar ein point mode, we just go and flag
      // selected vertices
      if (MUIManager::getComponentMode() == MUIManager::POINT_MODE) {
        int      nVert;
        for (nVert = TextureMesh->getNumVerts()-1; nVert>=0; nVert--) {
          if (TextureMesh->isVertexFlagged(nVert, VERTEX_SELECTED)) {
            TextureMesh->setVertexFlag(nVert, VERTEX_FLAGFORCHANGE);
            AnythingSelected = true;
          } else {
            TextureMesh->unsetVertexFlag(nVert, VERTEX_FLAGFORCHANGE);
          }
        }
      }
      
      // If we are in face mode, we need to flag the vertices
      // that the faces are made up of
      if (MUIManager::getComponentMode() == MUIManager::FACET_MODE) {
        TextureMesh->unsetVertexFlags(VERTEX_FLAGFORCHANGE);
        
        for (int nTri = TextureMesh->getNumTris()-1; nTri>=0; nTri--) {
          MTriangle const *T;
          T = TextureMesh->getTriangle(nTri);
          
          if (!T) continue;
          
          if (!T->isFlagged(TRIANGLE_SELECTED)) continue;
          
          AnythingSelected = true;
          for (int n=0; n<3; n++) {
            TextureMesh->setVertexFlag(T->getVertex(n), VERTEX_FLAGFORCHANGE);
          }
        }
      }
      // If we are in edge mode, we need to flag the vertices
      // that the edges are composed of
      if (MUIManager::getComponentMode() == MUIManager::EDGE_MODE) {
        TextureMesh->unsetVertexFlags(VERTEX_FLAGFORCHANGE);

        for (int nTri = TextureMesh->getNumTris()-1; nTri>=0; nTri--) {
          MTriangle const *T = TextureMesh->getTriangle(nTri);
          
          if (!T) {
            continue;
          }
          
          for (int edge = 0; edge < 3; ++edge) {
            if (T->isEdgeFlagged(0, EDGE_SELECTED)) {
              TextureMesh->setVertexFlag(T->getVertex(edge), VERTEX_FLAGFORCHANGE);
              TextureMesh->setVertexFlag(T->getVertex((edge+1)%3), VERTEX_FLAGFORCHANGE);
              AnythingSelected = true;
            }
          }
        }
      }
      
      if (AnythingSelected == false) {
        g_MainAction = atSelect;
      }
    }
   }
   
   if (getShiftState() == g_ProgSet.m_PanOrbitKey) {
     g_MainAction = atPanOrbit;
   } else if (getShiftState() == g_ProgSet.m_PanTrackKey) {
     g_MainAction = atPanTrack;
   } else if (getShiftState() == g_ProgSet.m_PanZoomKey) {
     g_MainAction = atPanZoom;
   }
   
   if (Shift.m_Left && Shift.m_Shift) {
     g_MainAction = atDeselect;
   }
   if (Shift.m_Left && Shift.m_Ctrl) {
     g_MainAction = atSelect;
   }
}

void CUVEditViewWnd::onMouseMove(int x, int y, MShiftState Shift) {
  ::SetCursor(LoadCursor(NULL, IDC_ARROW));

  {
    MVector3 vec;
    screenToUV(x, y, vec.x, vec.y);
    MUIManager::setWorldCoordinates(vec);
  }

  static int     ox,oy;
  bool           Update = false;
  
  if (g_MainAction != atNone) {
    int nx,ny;
    nx = x;
    ny = y;
    
    WrapMouseToScreen(m_hWnd, x,y,ox, oy);
    
    x = nx;
    y = ny;
  }
  if (g_MainAction == atSelect) {
    g_MainAction = atSelectBox;
  }

  if (g_MainAction == atDeselect) {
    g_MainAction = atDeselectBox;
  }
  
  if (g_MainAction == atMove) {
    // Check to see if there is anything selected.
    bool          AnythingSelected;
    MBaseObjectPtr Obj;
    MMeshPtr Mesh;  
    vector<MMeshPtr> meshes;
    
    AnythingSelected = false;
    g_Scene->getSelectedObjectList()->beginIteration();
    while ((Obj = g_Scene->getSelectedObjectList()->getNext()) != NULL) {
      MSceneObjectPtr sceneObj;
      if ((sceneObj = AZTEC_CAST(MSceneObject, Obj)) == NULL) {
        continue;
      }
      if (sceneObj->getShapeObject() != NULL) {
        Mesh = AZTEC_CAST(MMesh, sceneObj->getShapeObject()->convertToMesh());
      }
      
      if (Mesh != NULL) {
        meshes.push_back(Mesh);
      }
    }
    g_Scene->getSelectedObjectList()->endIteration();

    for (int i = 0; i < meshes.size(); ++i) {
      Mesh = meshes[i];
      MEditableMeshPtr TextureMesh;
      
      TextureMesh = AZTEC_CAST(MEditableMesh, Mesh->getTextureMesh());
      
      if (TextureMesh != NULL) {
        
        for (int nVert = TextureMesh->getNumVerts()-1; nVert >= 0; nVert--) {
          if (TextureMesh->isVertexFlagged(nVert, VERTEX_FLAGFORCHANGE)) {
            AnythingSelected = true;
            break;
          }
        }
        
        if (AnythingSelected == false) {
          g_MainAction = atSelectBox;
        } else {
          MVector3    MoveVec;
          int         nVert;

          float Mult;
          
          if (m_WindowRect.right > m_WindowRect.bottom) {
            Mult = 1.0f/m_Zoom*(m_GateLeft-m_GateRight)/m_WindowRect.right;
          } else {
            Mult = 1.0f/m_Zoom*(m_GateTop-m_GateBottom)/m_WindowRect.bottom;
          }
          
          MoveVec.x = Mult*(ox - x) / m_ImageAspect;
          MoveVec.y = Mult*(oy - y);
          
          if (g_PickedManip == 1) {
            MoveVec.y = 0.0f;
          }
          if (g_PickedManip == 2) {
            MoveVec.x = 0.0f;
          }
          
          Update = true;
          
          for (nVert = TextureMesh->getNumVerts()-1; nVert >= 0; nVert--) {
            if (TextureMesh->isVertexFlagged(nVert, VERTEX_FLAGFORCHANGE)) {
              MVector3 pos = TextureMesh->getVertexPosition(nVert);
              pos.x += MoveVec.x;
              pos.y += MoveVec.y;
              TextureMesh->setVertexPosition(nVert, pos);
            }
          }
          
        }
      }
    }

    if (Update) {
      // Update only texture views
      g_ViewList.getList()->beginIteration();
      MBaseObjectPtr Obj;
      while ((Obj = g_ViewList.getList()->getNext()) != NULL ) {
        MBaseOpenGLWndPtr view;
      
        view = AZTEC_CAST(MBaseOpenGLWnd, Obj);
        if (view != NULL) {
          view->DrawView();
        }
      }
      g_ViewList.getList()->endIteration();
    }
  }
  
  
  if (g_MainAction == atSelectBox || g_MainAction == atDeselectBox) {
    m_UpX = x;
    m_UpY = y;
    Update = true;
  }
  
  if (g_MainAction == atPanTrack) {
    float Mult;
    
    if (m_WindowRect.right > m_WindowRect.bottom) {
      Mult = 1.0f/m_Zoom*(m_GateLeft-m_GateRight)/m_WindowRect.right;
    } else {
      Mult = 1.0f/m_Zoom*(m_GateTop-m_GateBottom)/m_WindowRect.bottom;
    }
    
    m_Pan.x -= Mult*(ox - x);
    m_Pan.y -= Mult*(oy - y);
    Update = true;
  }

  if (g_MainAction == atPanZoom) {
    m_Zoom /= (float)(1.0 + 0.02*(ox-x));
    
    AdjustProjectionMatrix();
    
    Update = true;
  }
  
  if (Update) {
    DrawView();
  }
  
  ox = x;
  oy = y;
}

DWORD CUVEditViewWnd::ViewPopupMenu(int x, int y) {
  CMenu    Menu;
  CMenu    *Popup;
  DWORD    Choice;
  
  Menu.LoadMenu(IDR_UVEDIT_POPUP);
  
  Popup = Menu.GetSubMenu(0);
  InitPopupMenu(Popup);
  
  // Check the carious menu items depending on the current flags etc.
  
  Choice = Popup->TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON | TPM_NONOTIFY | TPM_RETURNCMD, x, y, this);
  
  Popup->DestroyMenu();
  
  bool     Update;
  int      Result;
  
  Result = HandlePopupCommand(Choice);
  
  // The command was handled, but the view was changed, so we must return.
  if (Result == -1) {
    return 1;
  }
  
  if (Result == 1) {
    Update = true;
  } else {
    Update = false;
  }
  
  if (Update) {
    DrawView();
  }
  
  return Choice;
}

int CUVEditViewWnd::HandlePopupCommand(DWORD Cmd) {
  int Result;
  
  Result = MBaseOpenGLWnd::HandlePopupCommand(Cmd);
  
  if (Result) {
    return Result;
  }
  
  if (Cmd == ID_POPUP_GRID_VISIBLE) {
  }

  switch (Cmd) {
  case ID_POPUP_GRID_VISIBLE:
    
    return 1;
  default:
    break;
  }
  
  return 0;
}

// MZoomableWindow methods
void CUVEditViewWnd::zoomIn() {
  m_Zoom *= 1.4f;
}

void CUVEditViewWnd::zoomOut() {
  m_Zoom /= 1.4f;
}

void CUVEditViewWnd::zoomToFitAll() {
  // TODO: Do something here
}

void CUVEditViewWnd::zoomToFitSelected() {
  MVector3          Min, Max;
  int               NumSel;
  
  MBaseObjectPtr Obj;
  
  Obj = g_Scene->getSelectedObjectList()->getTail();
  if (Obj != NULL) {
    if (Obj->getClassName() == "MMesh" || Obj->getClassName() == "MAnimMesh") {
      MMeshPtr Mesh, TextureMesh;
      
      Mesh = AZTEC_CAST(MMesh, Obj);
      TextureMesh = Mesh->getTextureMesh();
      
      if (TextureMesh != NULL) {
        Min.set(10,10,10);
        Max.set(-10,-10,-10);
        
        NumSel = 0;
        for (int n = 0; n < TextureMesh->getNumVerts(); n++) {
          
          if (!TextureMesh->isVertexFlagged(n, VERTEX_SELECTED)) {
            continue;
          }
          
          MVector3 Vert(TextureMesh->getVertexPosition(n));
          
          NumSel++;
          
          if (Vert.x < Min.x) {
            Min.x = Vert.x;
          }

          if (Vert.y < Min.y) {
            Min.y = Vert.y;
          }

          if (Vert.z < Min.z) {
            Min.z = Vert.z;
          }
          
          if (Vert.x > Max.x) {
            Max.x = Vert.x;
          }

          if (Vert.y > Max.y) {
            Max.y = Vert.y;
          }

          if (Vert.z > Max.z) {
            Max.z = Vert.z;
          }
        }
      }
    }
  }
  
  m_Pan.x = (Max.x + Min.x) / 2;
  m_Pan.y = (Max.y + Min.y) / 2;
  m_Pan.z = (Max.z + Min.z) / 2;
  
  float Big;
  Big = Max.x - Min.x;
  if (Max.y - Min.y > Big)
    Big = Max.y - Min.y;
  
  Big += 0.1f;
  
  if (NumSel == 0)
  {
    Big = 1.5;
    m_Pan.x = 0.5;
    m_Pan.y = 0.5;
    m_Pan.z = 0.0;
    
  }
  
  m_Zoom = 2.0f/Big;
}


void CUVEditViewWnd::AdjustProjectionMatrix(MSelectMethod PickMode,
                                            int left, int top,
                                            int right, int bottom)
{
  GLMakeCurrent();
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  glViewport(0,0,m_WindowRect.right, m_WindowRect.bottom);
  
  float big, Small;
  
  if (m_WindowRect.right>m_WindowRect.bottom) {
    big = (float)m_WindowRect.right;
    Small = (float)m_WindowRect.bottom;
  } else {
    big = (float)m_WindowRect.bottom;
    Small = (float)m_WindowRect.right;
  }
  
  double fov = 45;
  
  m_Aspect = (float)big/Small;
  
  // set up the picking projection if necessary.
  if (PickMode != smNone) {
    GLint       ViewportInfo[4];
    
    int         PickWidth, PickHeight, PickX, PickY;
    
    PickWidth = 4;
    PickHeight = 4;
    PickX = m_UpX;
    PickY = m_UpY;
    
    if (g_MainAction == atSelectBox || g_MainAction == atDeselectBox) {
      PickX = (m_DownX + m_UpX) / 2;
      PickY = (m_DownY + m_UpY) / 2;
      PickWidth = (m_DownX - m_UpX);
      PickHeight = (m_DownY - m_UpY);
      
      if (PickWidth < 0)
        PickWidth = -PickWidth;
      if (PickHeight < 0)
        PickHeight = -PickHeight;
      if (PickWidth == 0)
        PickWidth = 1;
      if (PickHeight == 0)
        PickHeight = 1;
      
    }
    
    glGetIntegerv(GL_VIEWPORT, ViewportInfo);
    
    gluPickMatrix(PickX, ViewportInfo[3] - PickY, PickWidth,PickHeight, ViewportInfo);
  }
  
  m_GateLeft = -1*m_WindowRect.right/big*m_Aspect;
  m_GateRight = 1*m_WindowRect.right/big*m_Aspect;
  m_GateTop = -1*m_WindowRect.bottom/big*m_Aspect;
  m_GateBottom = 1*m_WindowRect.bottom/big*m_Aspect;
  
  
  glOrtho(m_GateLeft, m_GateRight, m_GateBottom, m_GateTop, -512*m_Zoom,512*m_Zoom);
  glScalef(m_Zoom, m_Zoom, m_Zoom);
  
  glMatrixMode(GL_MODELVIEW);
}

void CUVEditViewWnd::DoViewportTransform() {
  glTranslatef(-m_Pan.x, -m_Pan.y, m_Pan.z);
  
  {
    MSceneObjectPtr Obj;

    // Get the last object's image to use
    Obj = AZTEC_CAST(MSceneObject, g_Scene->getSelectedObjectList()->getTail());
    
    if (Obj != NULL) {
      MMaterialPtr Material = Obj->getTextureMaterial();
        
      if (Material != NULL) {
        MImagePtr Image = Material->getDiffuseMapImage();
        
        if (Image != NULL) {
          // Scale the image so it is in the correct aspect ratio.
          m_ImageAspect = (float)Image->getWidth()/Image->getHeight();
        }
      }
    }
  }
  
  
  glScalef(m_ImageAspect, 1.0, 1.0);
}

void CUVEditViewWnd::OnPaint() {
  CPaintDC dc(this); // device context for painting
  
  DrawView();	
}

void CUVEditViewWnd::OnSize(UINT nType, int cx, int cy) {
  MBaseOpenGLWnd::OnSize(nType, cx, cy);

  RECT clientRect;
  GetClientRect(&clientRect);
  if (glCanvas != NULL) {
    glCanvas->MoveWindow(&clientRect);
  }

  GLResizeToClient();
}

void CUVEditViewWnd::OnDestroy() {
  MBaseOpenGLWnd::OnDestroy();
}

BOOL CUVEditViewWnd::PreTranslateMessage(MSG* pMsg) {
  return MBaseOpenGLWnd::PreTranslateMessage(pMsg);
}

LRESULT CUVEditViewWnd::WindowProc(UINT message, WPARAM wParam, LPARAM lParam) {
  if (message == WM_MOUSEWHEEL) {
    short zDelta;
    
    zDelta = (short) HIWORD(wParam);
    if (zDelta < 0) {
      KViewportZoomOut();
    } else if (zDelta > 0) {
      KViewportZoomIn();
    }
    return 1;
  }
  
  return CWnd::WindowProc(message, wParam, lParam);
}

void CUVEditViewWnd::OnClose() {
  g_ViewList.DeleteView(this);
  
  //	CWnd::OnClose();
}


// Draws the manipulatrs. If Select is true, it draws using selection buffers, and returns 0 on none,
// x,y,z are 1,2,3 respectivley, 4 is parallel to the screen.
int CUVEditViewWnd::DrawManipulators(bool Select) {
  MVector3    PivotPoint;
  UINT        *SelectBuf;
  
  if (Select) {
    SelectBuf = new UINT[16384];
    glSelectBuffer(16384, SelectBuf);
    glRenderMode(GL_SELECT);
    
    glInitNames();
    glPushName(-1);
  }
  
  glPushMatrix();
  
  // Normalise the model view matrix so we don't draw skewed manips
  {
    float       Mat[16];
    MVector3    XVec, YVec, ZVec;
    
    glGetFloatv(GL_MODELVIEW_MATRIX, Mat);
    
    XVec.set(Mat[0], Mat[4], Mat[8]);
    YVec.set(Mat[1], Mat[5], Mat[9]);
    ZVec.set(Mat[2], Mat[6], Mat[10]);
    
    XVec.normalize();
    YVec.normalize();
    ZVec.normalize();
    
    Mat[0] = XVec.x; Mat[4] = XVec.y; Mat[8] = XVec.z;
    Mat[1] = YVec.x; Mat[5] = YVec.y; Mat[9] = YVec.z;
    Mat[2] = ZVec.x; Mat[6] = ZVec.y; Mat[10] = ZVec.z;
    
    glLoadMatrixf(Mat);
  }
  
  
  // Draw the move arrow if we need to
  if (g_ToolMan.GetTool()->Is("KToolMove")) {
    // Check to see if we have anything selected
    MBaseObjectPtr Obj;
    MMeshPtr Mesh, TextureMesh;
    int selectionCount = 0;
    
    TextureMesh = NULL;
    g_Scene->getSelectedObjectList()->beginIteration();
    
    while ((Obj = g_Scene->getSelectedObjectList()->getNext()) != NULL) {
    
      MSceneObjectPtr sceneObj;
      if ((sceneObj = AZTEC_CAST(MSceneObject, Obj)) == NULL) {
        continue;
      }
      if (sceneObj->getShapeObject() != NULL) {
        Mesh = AZTEC_CAST(MMesh, sceneObj->getShapeObject()->convertToMesh());
      }
      if (Mesh != NULL) {
        TextureMesh = Mesh->getTextureMesh();
      }
    
      if (TextureMesh != NULL) {
        if (TextureMesh->hasComponentsFlagged(MUIManager::getComponentMode())) {
          PivotPoint += TextureMesh->getFlaggedCentre(MUIManager::getComponentMode(), MComponentisedObject::COMPONENT_SELECTED);
          selectionCount++;
        }
      }
    }
    g_Scene->getSelectedObjectList()->endIteration();

    if (TextureMesh != NULL && selectionCount > 0) {
      PivotPoint *= (float)1.0 / selectionCount;
      glScalef(m_ImageAspect, 1, 1);
      glTranslatef(PivotPoint.x, PivotPoint.y, PivotPoint.z);
      glScalef(1.0f / m_ImageAspect, 1, 1);
      glScalef(0.015f, 0.015f, 0.015f);
      COpenGLWnd::DrawAxisIcon(g_IconSize / m_Zoom, 0.5, 2, Select, DRAWAXIS_X | DRAWAXIS_Y | DRAWAXIS_W | DRAWAXIS_ARROWS);
    }
  }
  
  glPopMatrix();
  
  if (Select) {    // Have we beein drawing in select mode
    int      NumHits;
    int      ObjNum;
    int      i, Count, j;
    GLuint   *Ptr;
    
    ObjNum = 0;
    
    NumHits = glRenderMode(GL_RENDER);
    
    Ptr = SelectBuf;
    
    //      for (i=0;(i<NumHits && ObjNum == 0);i++)
    for (i=0;(i<NumHits);i++) {
      float    Depth1, Depth2;
      
      Count = *Ptr; 
      Ptr++;
      Depth1 = (float) *Ptr/0x7fffffff; Ptr++;
      Depth2 = (float) *Ptr/0x7fffffff; Ptr++;
      
      
      j = 0;
      while (j<Count) {
        if (*Ptr == -1) {      // we are doing a new hit
          Ptr++; j++;
          ObjNum = *Ptr;
        }
        Ptr++; j++;
        
      }
    }
    
    delete[] SelectBuf;
    
    return ObjNum;
  }
  
  return 0;
}


// Updates the feedback colors for use in the 2D/3D views
void CUVEditViewWnd::updateSelectionFeedback() {
  MBaseObjectPtr Obj;
  MMeshPtr       Mesh;  
  
  g_Scene->getSelectedObjectList()->beginIteration();
  
  while ((Obj = g_Scene->getSelectedObjectList()->getNext()) != NULL) {
    MSceneObjectPtr sceneObj;
    if ((sceneObj = AZTEC_CAST(MSceneObject, Obj)) == NULL) {
      continue;
    }
    if (sceneObj->getShapeObject() != NULL) {
      Mesh = AZTEC_CAST(MMesh, sceneObj->getShapeObject()->convertToMesh());
    }

    // if we don't have a mesh, we can't do anything, so continue on
    if (Mesh == NULL) {
      continue;
    }
    MMeshPtr TextureMesh;
    
    TextureMesh = Mesh->getTextureMesh();
    
    // if we don't have a texture mesh, we can't do anything.
    if (TextureMesh == NULL) {
      continue;
    }

    // Give some feedback for the selected UV points
    if (MUIManager::getComponentMode() == MUIManager::POINT_MODE) {
      
      for (int vertexIndex = Mesh->getComponentCount(MComponentisedObject::POINT_TYPE) - 1; 
           vertexIndex >= 0; 
           --vertexIndex) 
      {
        Mesh->unsetComponentFeedbackColour(MComponentisedObject::POINT_TYPE, vertexIndex);
      }
      for (int faceIndex = TextureMesh->getNumTris() - 1; faceIndex >= 0; --faceIndex) {
        for (int corner = 0; corner < 3; ++corner) {
          if (TextureMesh->isVertexFlagged(TextureMesh->getTriVert(faceIndex, corner),
                                           VERTEX_SELECTED)) 
          {
            MRGBAFloat C;
            C.set(1,0,1,1);
            Mesh->setComponentFeedbackColour(MComponentisedObject::POINT_TYPE, Mesh->getTriVert(faceIndex, corner), C);
          }
        }
      }
    }
  }
  g_Scene->getSelectedObjectList()->endIteration();
}

void CUVEditViewWnd::screenToUV(int x, int y, float &u, float &v) {
  float Mult;
  
  if (m_WindowRect.right > m_WindowRect.bottom) {
    Mult = 1.0f/m_Zoom*(m_GateLeft-m_GateRight)/m_WindowRect.right;
  } else {
    Mult = 1.0f/m_Zoom*(m_GateTop-m_GateBottom)/m_WindowRect.bottom;
  }

  CRect clientRect;
  GetClientRect(&clientRect);

  x = -x + clientRect.Width() / 2;
  y = -y + clientRect.Height() / 2;

  u = x * Mult;
  v = y * Mult;

  u += 0.5;
  v += 0.5;
}
