#include "StdAfx.h"

#include <math.h>
#include "scripting/ScriptSceneObject.h"
#include "scripting/ScriptUtils.h"
#include "scripting/ScriptFuncs.h"
#include "ParticleScript.h"

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

namespace Aztec {

class AztecParticleInfo {
public:
	AztecParticleInfo() {
		particle = NULL;
	}
	~AztecParticleInfo() {
		particle= NULL;
	}

	MParticlePtr particle;
};

static JSBool
addAction(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	// we can accept 2 parameters - type type of action followed by an array of
	// parameters.
	if (argc != 2) {
		return JS_FALSE;
	}

	AztecParticleInfo *pinfo = (AztecParticleInfo *)JS_GetInstancePrivate(cx, obj, &aztecParticle_class, NULL);
	if (pinfo == NULL || pinfo->particle == NULL) {
		return JS_FALSE;
	}

	int actionTypeValue = JSVAL_TO_INT(argv[0]);
	if (actionTypeValue < MPrimitiveParticle::actionType::actionEnum::firstActionEnum ||
		actionTypeValue > MPrimitiveParticle::actionType::actionEnum::lastActionEnum) {
		// Action is not in allowed range.  TBD: Report error.
		return JS_FALSE;
	}

	// Make sure the second argument is an array.
	if (!JSVAL_IS_OBJECT(argv[1])) {
		// Not an object, therefore not an array
		return JS_FALSE;
	}

	JSObject *paramsObj = JSVAL_TO_OBJECT(argv[1]);
	if (!JS_IsArrayObject(cx, paramsObj)) {
		// Not an array object.
		return JS_FALSE;
	}

	jsuint paramCount;
	if (!JS_GetArrayLength(cx, paramsObj, &paramCount)) {
		// Failed to get length of array?  TBD: what sort of error should be reported?
		return JS_FALSE;
	}

	// Convert input array into array of floats
	float *paramVals = new float[paramCount];
	double td;
	jsval vp;
	for (unsigned int i=0;i<paramCount;i++) {
		if (!JS_GetElement(cx, paramsObj, i, &vp)) {
			// Failed to get an entry from the array.
			return JS_FALSE;
		}
		if (!getDoubleVal(cx, vp, td)) {
			// Failed to translate one of the params into a double.
			return JS_FALSE;
		}

		paramVals[i] = (float)td;
	}

	// Build the action
	MPrimitiveParticle::actionType action;
	action.action = (MPrimitiveParticle::actionType::actionEnum)actionTypeValue;
	action.m_params.clear();
	for (unsigned int j=0;j<paramCount;j++) {
		action.m_params.push_back(paramVals[j]);
	}
	pinfo->particle->m_ActionList.push_back(action);

	*rval = JSVAL_TRUE;

	return JS_TRUE;
}

static JSBool
clearActions(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	// we can accept one parameter - the index of the action
	if (argc != 0) {
		return JS_FALSE;
	}

	AztecParticleInfo *pinfo = (AztecParticleInfo *)JS_GetInstancePrivate(cx, obj, &aztecParticle_class, NULL);
	if (pinfo == NULL || pinfo->particle == NULL) {
		return JS_FALSE;
	}

	pinfo->particle->m_ActionList.clear();

	*rval = JSVAL_TRUE;	// Failed to delete

	return JS_TRUE;
}

static JSBool
deleteAction(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	// we can accept one parameter - the index of the action
	if (argc != 1) {
		return JS_FALSE;
	}

	AztecParticleInfo *pinfo = (AztecParticleInfo *)JS_GetInstancePrivate(cx, obj, &aztecParticle_class, NULL);
	if (pinfo == NULL || pinfo->particle == NULL) {
		return JS_FALSE;
	}

	int index = JSVAL_TO_INT(argv[0]);
	int num_actions = pinfo->particle->m_ActionList.size();
	if (index < 0 || index >= num_actions) {
		return JS_FALSE;
	}

	pinfo->particle->m_ActionList.erase(pinfo->particle->m_ActionList.begin() + index);

	*rval = JSVAL_TRUE;

	return JS_TRUE;
}

static JSBool
getActionParams(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	// we can accept one parameter - the index of the action
	if (argc != 1) {
		return JS_FALSE;
	}

	AztecParticleInfo *pinfo = (AztecParticleInfo *)JS_GetInstancePrivate(cx, obj, &aztecParticle_class, NULL);
	if (pinfo == NULL || pinfo->particle == NULL) {
		return JS_FALSE;
	}

	int index = JSVAL_TO_INT(argv[0]);
	int num_actions = pinfo->particle->m_ActionList.size();
	if (index < 0 || index >= num_actions) {
		return JS_FALSE;
	}

	// Create an array of floats
	const particleParamType &params = pinfo->particle->m_ActionList[index].m_params;
	unsigned int param_length = params.size();
#if 1
	JSObject *paramArray = JS_NewArrayObject(cx, 0, NULL);
	jsval dval;
	for (unsigned int i=0;i<param_length;i++) {
		JS_NewDoubleValue(cx, params[i], &dval);
		JS_SetElement(cx, paramArray, i, &dval);
	}
#else
	jsval *vector = new jsval[param_length];
	for (unsigned int i=0;i<param_length;i++) {
		double d = (double)params[i];
		vector[i] = DOUBLE_TO_JSVAL(d);
	}
	JSObject *paramArray = JS_NewArrayObject(cx, param_length, vector);
	// delete[] vector;
#endif

	*rval = OBJECT_TO_JSVAL(paramArray);

	return JS_TRUE;
}

static JSBool
getActionType(JSContext *cx, JSObject *obj, uintN argc, jsval *argv, jsval *rval)
{
	// we can accept one parameter - the index of the action
	if (argc != 1) {
		return JS_FALSE;
	}

	AztecParticleInfo *pinfo = (AztecParticleInfo *)JS_GetInstancePrivate(cx, obj, &aztecParticle_class, NULL);
	if (pinfo == NULL || pinfo->particle == NULL) {
		return JS_FALSE;
	}

	int index = JSVAL_TO_INT(argv[0]);
	int num_actions = pinfo->particle->m_ActionList.size();
	if (index < 0 || index >= num_actions) {
		return JS_FALSE;
	}

	*rval = INT_TO_JSVAL(pinfo->particle->m_ActionList[index].action);

	return JS_TRUE;
}

static JSFunctionSpec aztecParticle_methods[] = {
	{"addAction",				addAction,				2},
	{"clearActions",			clearActions,			0},
	{"deleteAction",			addAction,				1},
	{"getActionParams",			getActionParams,		1},
	{"getActionType",			getActionType,			1},
#if 0
	{"addAction",				addAction,				1},
	{"deleteAction",			deleteAction,			1},
	{"calculateNormals",		calculateNormals,		0},
	{"centerMesh",				centerMesh,				0},
	{"collapseVertices",		collapseVertices,		1},
	{"extrudeFaces",			extrudeFaces,			3},
	{"flipNormals",				flipNormals,			1},
	{"getTriangle",				getTriangle,			1},
	{"getVertex",				getVertex,				1},
	{"scaleMesh",				scaleMesh,				1},
	{"weldVertices",			weldVertices,			2},
#endif
	{0}
};

static JSPropertySpec aztecParticle_props[] = {
	{"numActions",		MPrimitiveParticle::actionType::actionEnum::lastActionEnum+1,	JSPROP_ENUMERATE},
	{"actionArray",		MPrimitiveParticle::actionType::actionEnum::lastActionEnum+2,	JSPROP_ENUMERATE},
	{"color",			MPrimitiveParticle::actionType::color,			JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"colorDomain",		MPrimitiveParticle::actionType::colorDomain,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"size",			MPrimitiveParticle::actionType::size		,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"sizeDomain",		MPrimitiveParticle::actionType::sizeDomain,		JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"startingAge",		MPrimitiveParticle::actionType::startingAge,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"timeStep",		MPrimitiveParticle::actionType::timeStep,		JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"velocity",		MPrimitiveParticle::actionType::velocity,		JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"velocityDomain",	MPrimitiveParticle::actionType::velocityDomain,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"vertexB",			MPrimitiveParticle::actionType::vertexB,		JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"vertexBTracks",	MPrimitiveParticle::actionType::vertexBTracks,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"avoid",			MPrimitiveParticle::actionType::avoid,			JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"bounce",			MPrimitiveParticle::actionType::bounce,			JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"copyVertexB",		MPrimitiveParticle::actionType::copyVertexB,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"damping",			MPrimitiveParticle::actionType::damping,		JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"explosion",		MPrimitiveParticle::actionType::explosion,		JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"follow",			MPrimitiveParticle::actionType::follow,			JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"gravitate",		MPrimitiveParticle::actionType::gravitate,		JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"gravity",			MPrimitiveParticle::actionType::gravity,		JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"jet",				MPrimitiveParticle::actionType::jet,			JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"killOld",			MPrimitiveParticle::actionType::killOld,		JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"matchVelocity",	MPrimitiveParticle::actionType::matchVelocity,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"move",			MPrimitiveParticle::actionType::move,			JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"orbit",			MPrimitiveParticle::actionType::orbit,			JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"randomAccel",		MPrimitiveParticle::actionType::randomAccel,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"randomDisplace",	MPrimitiveParticle::actionType::randomDisplace,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"randomVelocity",	MPrimitiveParticle::actionType::randomVelocity,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"restore",			MPrimitiveParticle::actionType::restore,		JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"sink",			MPrimitiveParticle::actionType::sink,			JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"sinkVelocity",	MPrimitiveParticle::actionType::sinkVelocity,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"source",			MPrimitiveParticle::actionType::source,			JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"speedLimit",		MPrimitiveParticle::actionType::speedLimit,		JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"targetColor",		MPrimitiveParticle::actionType::targetColor,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"targetSize",		MPrimitiveParticle::actionType::targetSize,		JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"targetVelocity",	MPrimitiveParticle::actionType::targetVelocity,	JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"vertex",			MPrimitiveParticle::actionType::vertex,			JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"vortex",			MPrimitiveParticle::actionType::vortex,			JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},

	{"PDPoint",			0,												JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"PDLine",			1,												JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"PDTriangle",		2,												JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"PDPlane",			3,												JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"PDBox",			4,												JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"PDSphere",		5,												JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"PDCylinder",		6,												JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"PDCone",			7,												JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"PDBlob",			8,												JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"PDDisc",			9,												JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{"PDRectangle",		10,												JSPROP_ENUMERATE | JSPROP_READONLY | JSPROP_PERMANENT},
	{0}
};

static JSBool
aztecParticle_getProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	AztecParticleInfo *pinfo = (AztecParticleInfo *)JS_GetInstancePrivate(cx, obj, &aztecParticle_class, NULL);
	if (pinfo == NULL || pinfo->particle == NULL) {
		return JS_FALSE;
	}

	if (JSVAL_IS_STRING(id)) {
		char *propStr = JS_GetStringBytes(JS_ValueToString(cx, id));
		if (strcmp(propStr, "numActions") == 0) {
			jsint numA = pinfo->particle->m_ActionList.size();
			*vp = INT_TO_JSVAL(numA);
			return JS_TRUE;
		} else if (strcmp(propStr, "actions") == 0) {
			// jsint numT = minfo->mesh->getNumTris();
			*vp = INT_TO_JSVAL(0);
			return JS_TRUE;
		} else {
			// Look through the list of predefined action enumerated values and
			// see if we can find it.
			for (int index=0;aztecParticle_props[index].name != NULL;index++) {
				if (!strcmp(propStr, aztecParticle_props[index].name)) {
					// Value is in the range of the particle action enumerate type
					*vp = INT_TO_JSVAL(aztecParticle_props[index].tinyid);
					return JS_TRUE;
				}
			}
		}
		
		// May be a built-in method.
		for (int m=0;aztecParticle_methods[m].name != NULL;m++) {
			if (!strcmp(propStr, aztecParticle_methods[m].name)) {
				// Yup - is a built-in function.
				JSFunction *jfunc =
					JS_NewFunction(cx, aztecParticle_methods[m].call,
									aztecParticle_methods[m].nargs,
									0, obj, propStr);
				JSObject *jfobj = JS_GetFunctionObject(jfunc);
				*vp = OBJECT_TO_JSVAL(jfobj);
				return JS_TRUE;
			}
		}

		JS_ReportError(cx, "%s is not defined", propStr);
		return JS_FALSE;
	} else if (JSVAL_IS_INT(id)) {
		int index = JSVAL_TO_INT(id);
		// Look at the non-enumerated values first
		if (index == MPrimitiveParticle::actionType::actionEnum::lastActionEnum+1) {
			*vp = INT_TO_JSVAL(pinfo->particle->m_ActionList.size());
			return JS_TRUE;
		} else if (index == MPrimitiveParticle::actionType::actionEnum::lastActionEnum+2) {
			// TBD: return an object that holds the actual parameters for the action
			*vp = INT_TO_JSVAL(0);
			return JS_TRUE;
		} else if (index >= MPrimitiveParticle::actionType::actionEnum::firstActionEnum &&
			index <= MPrimitiveParticle::actionType::actionEnum::lastActionEnum) {
			*vp = INT_TO_JSVAL(index);
			return JS_TRUE;
		}
#if 0
		// Look through the list of predefined action enumerated values and
		// see if we can find it.
		for (JSPropertySpec *specEntry = aztecParticle_props;
			specEntry != NULL;
			specEntry++) {
			if (index == specEntry->tinyid) {
				// Value is in the range of the particle action enumerate type
				*vp = INT_TO_JSVAL(val);
				return JS_TRUE;
			}
		}
#endif
	}

	return JS_TRUE;
}

static JSBool
aztecParticle_setProperty(JSContext *cx, JSObject *obj, jsval id, jsval *vp)
{
	AztecParticleInfo *pinfo = (AztecParticleInfo *)JS_GetInstancePrivate(cx, obj, &aztecParticle_class, NULL);
	if (pinfo == NULL) {
		return JS_FALSE;
	}

	return JS_FALSE;
}

static JSBool
aztecParticle_newEnumerate(JSContext *cx, JSObject *obj, JSIterateOp enum_op,
	jsval *statep, jsid *idp)
{
	int num_props = (sizeof(aztecParticle_props) / sizeof(aztecParticle_props[0])) - 1;
	switch(enum_op) {
	case JSENUMERATE_INIT:
	{
		AztecEnumState *enumState = new AztecEnumState;
		enumState->numParams = num_props;
		enumState->index = 0;
		*statep = PRIVATE_TO_JSVAL(enumState);
		if (idp != NULL) {
			*idp = INT_TO_JSVAL(num_props);
		}

		return JS_TRUE;
	}

	case JSENUMERATE_NEXT:
	{
		AztecEnumState *enumState = (AztecEnumState *)JSVAL_TO_PRIVATE(*statep);
		if (enumState->index < enumState->numParams) {
			JSString *str = JS_NewStringCopyZ(cx, aztecParticle_props[enumState->index].name);
			JS_ValueToId(cx, STRING_TO_JSVAL(str), idp);
			enumState->index++;
			return JS_TRUE;
		}
		// Drop through to destroy
	}

	case JSENUMERATE_DESTROY:
	{
		AztecEnumState *enumState = (AztecEnumState *)JSVAL_TO_PRIVATE(*statep);
		delete enumState;
        *statep = JSVAL_NULL;
		return JS_TRUE;
	}

	default:
		OutputDebugString("Bad enumeration state\n");
		return JS_FALSE;
	}
}

static void
aztecParticle_finalize(JSContext *cx, JSObject *obj)
{
	AztecParticleInfo *pinfo = (AztecParticleInfo *)JS_GetInstancePrivate(cx, obj, &aztecParticle_class, NULL);
	if (pinfo != NULL) {
		pinfo->particle = NULL;
		delete pinfo;
		JS_SetPrivate(cx, obj, NULL);
	}
}

JSClass aztecParticle_class = { 
	"AztecParticle", JSCLASS_HAS_PRIVATE | JSCLASS_NEW_ENUMERATE, 
		JS_PropertyStub, JS_PropertyStub,				// add/del property
		aztecParticle_getProperty, aztecParticle_setProperty,	// get/set property
		(JSEnumerateOp)aztecParticle_newEnumerate,
		JS_ResolveStub,
		JS_ConvertStub, aztecParticle_finalize
};

static JSBool aztecParticle_constructor(JSContext *cx, JSObject *obj, uintN argc,
                             jsval *argv, jsval *rval)
{
	AztecParticleInfo *pinfo = new AztecParticleInfo();
	if (pinfo == NULL) {
		return NULL;
	}
	pinfo->particle = new MPrimitiveParticle();

	return JS_SetPrivate(cx, obj, pinfo);
}

JSObject *newParticle(JSContext *cx, JSObject *obj, MParticlePtr particle)
{
#if 0
	JSObject *jsparticle = JS_NewObject(cx, &aztecParticle_class, NULL, JS_GetGlobalObject(cx));
#else
	MNamedObjectPtr Obj = AZTEC_CAST(MNamedObject, particle);
	if (Obj == NULL) {
		return NULL;
	}
	//JSObject *jsparticle = JS_NewObject(cx, &aztecParticle_class, NULL, JS_GetGlobalObject(cx));
	JSObject *jsparticle = JS_NewObject(cx, &aztecParticle_class,
		newSceneObject(cx, obj, Obj.m_Ptr), JS_GetGlobalObject(cx));
#endif

	if (jsparticle == NULL) {
		return NULL;
	}

	AztecParticleInfo *pinfo = new AztecParticleInfo();
	if (pinfo == NULL) {
		return NULL;
	}
	pinfo->particle = particle;
	JS_SetPrivate(cx, jsparticle, pinfo);

	return jsparticle;
}

void addParticleClass(JSContext *cx)
{
	// Add a Vertex class (reflection of MVertex3) to the global namespace
	JSObject *iobj =
    JS_InitClass(cx, JS_GetGlobalObject(cx), NULL,
		&aztecParticle_class, aztecParticle_constructor, 0,
		aztecParticle_props, aztecParticle_methods, 0, 0);

}

}
