#if !defined( __RTRANGE_TREE )
#define __RTRANGE_TREE

#include "Range3D.h"

// Data structures and routines for maintaining 3D objects in a 2-3 tree
class RangeKey {
	float bounds[3];
public:
	RangeKey() {
		// bounds[0] = bounds[1] = bounds[2] = 0.0f;
	}
	RangeKey(float a, float b, float c) {
		bounds[0] = a;
		bounds[1] = b;
		bounds[2] = c;
		// Mask off the bottom 3 bits of the floating point
		// number to avoid rounding errors (I'm seeing one
		// bit difference at the bottom leading to errors).
		*((int *)&bounds[0]) &= 0xFFFFFFF8;
		*((int *)&bounds[1]) &= 0xFFFFFFF8;
		*((int *)&bounds[2]) &= 0xFFFFFFF8;
	}
	float operator[](int index) { return bounds[index]; }
	RangeKey &operator=(RangeKey &k) {
		bounds[0] = k.bounds[0];
		bounds[1] = k.bounds[1];
		bounds[2] = k.bounds[2];
		return *this;
	}
	bool operator==(RangeKey &r) {
		return bounds[0] == r.bounds[0] &&
			   bounds[1] == r.bounds[1] &&
			   bounds[2] == r.bounds[2];
	}
};

#ifndef ALLOW_23TREE_PRINTING
#define ALLOW_23TREE_PRINTING 1
#endif

// A symbol table & it's methods.
class Range_Tree {
protected:
	class range_data {
	public:
		range_data(void *data) {
			this->data = data;
			next = NULL;
		}
		void *data;
		range_data *next;
	};

	class node23leaf {
	public:
		node23leaf() { }
		node23leaf(RangeKey &key, void *data);
		RangeKey key;
		range_data *data;
	};

	class node23 {
	public:
		enum leaf_kind { LEAF_NODE, TWO_NODE, THREE_NODE } kind;

		node23(leaf_kind kind) { this->kind = kind; }
		node23(RangeKey &key, Range3D &range, void *data);

		node23leaf *link; // Link to the smallest entry in this branch of the tree
		RangeKey key[2];  // Value(s) used for keeping entries sorted
		Range3D range;    // Range of values stored at this level and below
		// Can have 1 or 2 entries: leaf=1, two=1, three=2
		union {
			node23 *ptrs[3];  // Can have 0, 2, or 3 entries: leaf=0, two=2, three=3
			node23leaf *leaf; // Pointer to actual information (only if this is a leaf)
		};
	};

	typedef class range_data *rangedataptr;
	typedef class Range_Tree::node23 *nodeptr;

public:
	Range_Tree();
	~Range_Tree();

	nodeptr Insert(Range3D &range, void *value); // Insert a value that covers a range
	void Delete(Range3D &range, void *value);    // Remove an element from the tree
	void Delete(nodeptr leaf, void *value);     // Remove an element from the tree
	void Process(Range3D &range, void *data);    // Process everything that overlaps range
	void ProcessAll(void *data);                 // Process everything in the tree

	int InTree(void *value); // Exhaustive search for the number of time value appears

#if ALLOW_23TREE_PRINTING
	void show_two_three(nodeptr tree, int depth);
	void Show();
#endif

protected:
	// Base of the tree used to hold range information
	node23 *Root;
	// How many entries in the tree?
	int entry_count;

	// User supplied tree manipulation
	virtual void process(Range3D &range, void *value, void *data);
	// User supplied cleanup routine
	virtual void cleanup(void *data);

	void shread_two_three(nodeptr intree);

	// Find the smallest element in a 2-3 tree (TBD: this should use the link
	// member to make a single step to the bottom.  The code for managing the
	// links hasn't been tested yet.)
	RangeKey &least_key(node23 *branch);

	// Compare boxes by lower left corners
	int compare(RangeKey &a, RangeKey &b);

private:
	int exhaustive_search(nodeptr tree, void *value);

	void process_two_three(nodeptr tree, Range3D &range, void *data);
	void process_all_nodes(nodeptr tree, void *data);

	// Insertion of entries in the tree
	int split(nodeptr intree, RangeKey &key, Range3D &range, void *data,
			  nodeptr *left_tree, nodeptr *right_tree, nodeptr *leaf);
	int insert3(nodeptr intree, RangeKey &key, Range3D &range, void *data,
				nodeptr *outtree, nodeptr *leaf);

	// Deletion of entries in the tree
	enum delete_result { DeleteFailed, DeleteSucceeded, DeletedBranch };
	delete_result two_three_delete(nodeptr intree, RangeKey &key, Range3D &range,
								   void *value, nodeptr *outtree);

	// Smallest element in a tree (starting from a particular branch)
	rangedataptr two_three_least(nodeptr intree);

};

#endif // __RTRANGE_TREE
