#include "StdAfx.h"

#include "DotGraphExporter.h"

#include "MBaseObject.h"
#include "MSystemManager.h"
#include "MMeshShape.h"

#include <stdio.h>
#include <assert.h>

#include <map>
#include <set>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


DotGraphExporter::DotGraphExporter() {
  m_TabCount = 0;
}

DotGraphExporter::~DotGraphExporter() {
}

// Class related
MTranslatorPtr DotGraphExporter::createNew() const {
  DotGraphExporter   *NewTranslator;
  
  NewTranslator = new DotGraphExporter;
  
  return NewTranslator;
}

typedef std::map<MDAGNode*, std::string> NodeNameMap;
typedef std::set<MDAGNode*> NodeSet;

NodeNameMap nodeNames;
NodeSet nodesDone;
NodeSet nodeNamesPrinted;

const char * getName(MDAGNode *node) {
  NodeNameMap::iterator it = nodeNames.find(node);

  if (it != nodeNames.end()) {
    return it->second.c_str();
  }

  MNamedObject *namedObj = AZTEC_CAST(MNamedObject, node);
  MParameterObject *paramObj = AZTEC_CAST(MParameterObject, node);
  MParameterObjectList *paramList = AZTEC_CAST(MParameterObjectList, node);

  char name[1024];

  if (namedObj != NULL) {
    strcpy(name, (namedObj->getClassName() + "." + namedObj->getName()).c_str());
  } else if (paramObj != NULL) {
    std::string nameStr;
    
    if (paramObj->getOwner() != NULL) {
      namedObj = AZTEC_CAST(MNamedObject, paramObj->getOwner());

      if (namedObj != NULL) {
        nameStr += namedObj->getName();
        nameStr += ".";
      }
    }
    
    nameStr += paramObj->getLongName();

    strcpy(name, nameStr.c_str());
  } else if (paramList != NULL) {
    std::string nameStr;

    namedObj = AZTEC_CAST(MNamedObject, paramList->getOwner());

    if (namedObj != NULL) {
      nameStr += namedObj->getName();
    } else {
      sprintf(name, "%x", node);
      nameStr += name;
    }
    nameStr += ".paramList";

    strcpy(name, nameStr.c_str());
  } else {
    sprintf(name, "%x.%s", node, node->getClassName().c_str());
  }

  return nodeNames.insert(NodeNameMap::value_type(node, name)).first->second.c_str();
}

void writeNodeText(FILE *file, MDAGNode *node) {
  // only bother printing out the node if we hanve't done it yet.
  if (nodeNamesPrinted.find(node) != nodeNamesPrinted.end()) {
    return;
  }

  nodeNamesPrinted.insert(node);

  const char *nodeName = getName(node);

  const char *shape = "box";
  const char *fontsize = "8";
  const char *width = "0.25";
  const char *height = "0.2";
  const char *colour = NULL;
  const char *style = NULL;

  if (AZTEC_CAST(MNamedObject, node) != NULL) {
    style = "filled";
    colour = "lightblue";
  }

  if (AZTEC_CAST(MParameterObject, node) != NULL) {
    style = "filled";
    colour = "pink";
  }

  if (AZTEC_CAST(MSceneObject, node) != NULL) {
    style = "filled";
    colour = "green";
  }

  if (AZTEC_CAST(MParameterObjectList, node) != NULL) {
    style = "filled";
    colour = "lightyellow";
  }


  fprintf(file, "Node%x [label=\"%s\"", node, nodeName);
  
  if (shape != NULL) {
    fprintf(file, ",shape=%s", shape);
  }
  if (fontsize != NULL) {
    fprintf(file, ",fontsize=%s", fontsize);
  }
  if (height != NULL) {
    fprintf(file, ",height=%s", height);
  }
  if (width != NULL) {
    fprintf(file, ",width=%s", width);
  }
  if (style != NULL) {
    fprintf(file, ",style=%s", style);
  }
  if (colour != NULL) {
    fprintf(file, ",color=%s", colour);
  }

  fprintf(file, "]\n");

}

void writeDAGNode(FILE *file, MDAGNode *node) {
  if (nodesDone.find(node) != nodesDone.end()) {
    return;
  }

  MDAGNode **inputs = new MDAGNode*[node->getInputCount()];
  MDAGNode **outputs = new MDAGNode*[node->getOutputCount()];

  node->getInputs(inputs);
  node->getOutputs(outputs);

  // add us to the known set of completed nodes, so we don't go off
  // into an infinite loop writing ourselves out all over again.
  nodesDone.insert(node);

  writeNodeText(file, node);

  // write all the inputs out.
  for (int index = 0; index < node->getInputCount(); ++index) {
    writeDAGNode(file, inputs[index]);
    fprintf(file, "Node%x -> Node%x\n", inputs[index], node);
  }

  // write all the outputs out.
  for (int index = 0; index < node->getOutputCount(); ++index) {
    writeDAGNode(file, outputs[index]);
  }

  delete[] inputs;
  delete[] outputs;
}

void writeOutNamedObject(FILE *file, MNamedObject *obj) {
  if (nodeNamesPrinted.find(obj) != nodeNamesPrinted.end()) {
    return;
  }

  fprintf(file, "subgraph %s {\n", obj->getName().c_str());

  writeNodeText(file, obj);
  writeNodeText(file, &*obj->getParamList());

  for (int i = 0; i < obj->getParamList()->getNumParams(); ++i) {
    writeNodeText(file, &*obj->getParamList()->getParameter(i));
  }

  fprintf(file, "}\n");
}

void printOutImportantNodes(FILE *file, MDAGNode *node) {
  if (nodesDone.find(node) != nodesDone.end()) {
    return;
  }

  MDAGNode **inputs = new MDAGNode*[node->getInputCount()];
  MDAGNode **outputs = new MDAGNode*[node->getOutputCount()];

  node->getInputs(inputs);
  node->getOutputs(outputs);

  // add us to the known set of completed nodes, so we don't go off
  // into an infinite loop writing ourselves out all over again.
  nodesDone.insert(node);

  if (AZTEC_CAST(MNamedObject, node) != NULL) {
    writeOutNamedObject(file, AZTEC_CAST(MNamedObject, node));
  } else {
    writeNodeText(file, node);
  }

  // write all the inputs out.
  for (int index = 0; index < node->getInputCount(); ++index) {
    printOutImportantNodes(file, inputs[index]);
  }

  // write all the outputs out.
  for (int index = 0; index < node->getOutputCount(); ++index) {
    printOutImportantNodes(file, outputs[index]);
  }

  delete[] inputs;
  delete[] outputs;
}

bool DotGraphExporter::exportFile(MStr Filename, MScenePtr Scene) {
  // Open an ascii file, and start writing to it
  FILE      *hFile;

  hFile = fopen(Filename,"wt");

  fprintf(hFile, "digraph G\n");
  fprintf(hFile, "{\n");
  fprintf(hFile, "rankdir=LR\n");

  if (!hFile)
    return false;
  
  // go through every object in the scene, and write out the objects
  // todo: do not just write out meshes, write out actual object data
  MBaseObjectPtr Obj;
  MSceneObjectPtr SceneObj;
  MMeshPtr MeshObj;
  MTransformObjectPtr XFormObj;
  
  Scene->setTime(0);

  nodeNamesPrinted.clear();
  nodesDone.clear();
  nodeNames.clear();

  Scene->getObjectList()->beginIteration();
  // set the time back to the start of the animation
  while ((Obj = Scene->getObjectList()->getNext()) != NULL) {
    MDAGNode *node = AZTEC_CAST(MDAGNode, Obj);
    printOutImportantNodes(hFile, node);    
  }
  Scene->getObjectList()->endIteration();

  nodesDone.clear();

  Scene->getObjectList()->beginIteration();
  // set the time back to the start of the animation
  while ((Obj = Scene->getObjectList()->getNext()) != NULL) {
    MDAGNode *node = AZTEC_CAST(MDAGNode, Obj);
    writeDAGNode(hFile, node);    
  }
  Scene->getObjectList()->endIteration();


  fprintf(hFile, "}\n");
  
  fflush(hFile);
  fclose(hFile);
  
  return false;
}
