#include "StdAfx.h"
#include "MD3Translator.h"
#include "DevilTranslator.h"

#include "MBaseObject.h"
#include "MAnimMesh.h"
#include "MMeshShape.h"

#include <MSystemManager.h>
#include <params/MParameterFactory.h>

#include <stdio.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <assert.h>

#include <set>

#include <math.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


using namespace std;

//---------------------------------------------------------------------------------

MMD3Translator::MMD3Translator()
{
}

MMD3Translator::~MMD3Translator()
{
}

// Class related
MTranslatorPtr MMD3Translator::createNew() const {
  MMD3Translator   *NewTranslator;
  
  NewTranslator = new MMD3Translator;
  
  return NewTranslator;
}

bool MMD3Translator::canImportFile(MStr Filename) {
  MD3Header  Head;
  
  FILE *hFile;
  
  Filename.Replace('/', '\\');
  hFile = fopen(Filename, "rb");
  
  if (!hFile)
    return false;
  
  fread(&Head,sizeof(MD3Header),1,hFile);
  
  if (Head.ID != MD3_FILEID || Head.Version != MD3_VERSION)
  {
    fclose(hFile);
    return false;
  }
  
  fclose(hFile);
  
  return true;
}


bool MMD3Translator::importFile(MStr Filename, MScenePtr Scene) {
  MSceneObjectPtr groupObj;

  // just do the import file without the extra info.
  return importFile(Filename, Scene, groupObj);
}

bool MMD3Translator::importFile(MStr Filename, MScenePtr Scene, MSceneObjectPtr &groupObj) {
  FILE              *hFile;
  
  MSystemManagerPtr SysMan;
  MD3Model md3Model;
  
  SysMan = Aztec::getSystemManager();
  assert(SysMan != NULL);
  
  hFile = fopen(Filename, "rb");
  
  if (!hFile)
    return false;

  md3Model.readFromFile(hFile);
  
  fclose(hFile);

  // Add a single XForm node for all objects to be children to this
  MSceneObjectPtr MasterXForm;
  MTreeObjectNodePtr MasterXFormNode;
  MStr MasterObjName;
  
  {
    MStr     Str;
    char     ObjFilename[256], *ObjName;
    
    strcpy(ObjFilename, (LPCTSTR)Filename);
    ObjName = ObjFilename + strlen(ObjFilename);
    
    while (*ObjName != '.')
      ObjName--;
    
    *ObjName = '\x0';
    
    while (*ObjName != '\\' && ObjName > ObjFilename)
      ObjName--;
    if (*ObjName == '\\')
      ObjName++;
    
    MasterObjName = ObjName;
  }
  
  MasterXForm = new MSceneObject;
  MasterXForm->setName(MasterObjName + MStr("Group"));
  MasterXForm->addParameter(MParameterFactory::createString("q3obj", "q3objname", "Q3 Obj. Name"));
  MasterXForm->setParamByName("q3obj", MasterObjName);
  MasterXFormNode = Scene->addObject(MasterXForm);

  // the MasterXForm object is our main object
  groupObj = MasterXForm;
  
  // Add the actual objects to the scene.
  for (int nMesh=0; nMesh < md3Model.header.numMeshes; nMesh++) {
    MD3Mesh     *FileMesh;
    MSceneObjectPtr animSceneObj;
    MMeshShapePtr meshShape;
    MAnimMeshPtr AnimMesh;
    MEditableMeshPtr SkinMesh;
    MVector3 *Verts;
    MTriangle   *Tris;
    
    DWORD       Frame, nTri;
    
    
    FileMesh = &md3Model.surfaces[nMesh];
    // Convert the vertices and tris for the object to a MVector3 and MTriangle array.
    
    Verts = new MVector3[FileMesh->Header.NumVerts];
    
    AnimMesh = new MAnimMesh;
    meshShape = new MMeshShape(AnimMesh);
    animSceneObj = new MSceneObject(meshShape);

    Scene->addObject(AnimMesh);
    Scene->addObject(meshShape);
    
    for (Frame = 0; Frame<FileMesh->Header.NumFrames;Frame++) {
      DWORD   n;
      for (n=0; n < FileMesh->Header.NumVerts; n++) {
        MVector3    V;
        
        
        V.x = FileMesh->Frames[Frame].Vertices[n].Vec[0]/64.0f;
        V.y = FileMesh->Frames[Frame].Vertices[n].Vec[1]/64.0f;
        V.z = FileMesh->Frames[Frame].Vertices[n].Vec[2]/64.0f;
        
        Verts[n].x = V.x;
        Verts[n].y = V.y;
        Verts[n].z = V.z;
        
      }
      
      // If we are on the first frame, create the animated vertices
      if (Frame == 0) {
        AnimMesh->addVertexArray(Verts, FileMesh->Header.NumVerts);
      } else{
        AnimMesh->setVertexArray(Frame*120, Verts, FileMesh->Header.NumVerts);
      }
    }
    
    
    // Convert the triangles to an MTriangle Array;
    
    Tris = new MTriangle[FileMesh->Header.NumTris];
    
    for (nTri = 0; nTri < FileMesh->Header.NumTris; nTri++) {
      Tris[nTri].setVertex(0, FileMesh->Triangles[nTri][2]);
      Tris[nTri].setVertex(1, FileMesh->Triangles[nTri][1]);
      Tris[nTri].setVertex(2, FileMesh->Triangles[nTri][0]);
    }
    
    
    // do the skin mesh
    {
      MVector3 *SkinVerts;
      DWORD       nVert;
      
      SkinMesh = new MEditableMesh;
      SkinVerts = new MVector3[FileMesh->Header.NumVerts];
      
      for (nVert = 0; nVert < FileMesh->Header.NumVerts; nVert++) {
        SkinVerts[nVert].x = FileMesh->SkinVerts[nVert][0];
        SkinVerts[nVert].y = FileMesh->SkinVerts[nVert][1];
        SkinVerts[nVert].z = 0;
      }
      
      SkinMesh->addVertexArray(SkinVerts, FileMesh->Header.NumVerts);
      delete[] SkinVerts;
      
    }
    
    {
      MMaterialPtr SkinMaterial;
      DevilTranslator TGATrans(DevilTranslator::TGA);
      
      SkinMaterial = new MMaterial;
      
      char              SkinFileName[256];
      
      // Try and locate the skin required.
      {
        char     DirStr[256];
        bool     FoundFile = false;
        int      n;
        
        strcpy(DirStr, (LPCTSTR)Filename);
        
        while (!FoundFile)
        {
          n = strlen(DirStr) - 1;
          while (n--)
          {
            if (DirStr[n] == '\\')
            {
              DirStr[n+1] = '\x0';
              break;
            }
          }
          strcpy(SkinFileName, DirStr);
          
          strcat(SkinFileName, FileMesh->SkinNames[0].name);
          
          if (TGATrans.canImportFile(SkinFileName)) {
            FoundFile = true;
          }
          if (strlen(DirStr)<=3)
            break;
        }
        if (FoundFile) {
          SkinMaterial->setDiffuseMapFilename(SkinFileName);
        }
      }
      
      SkinMaterial->setName(MasterObjName + "_Material");
      Scene->addObject(SkinMaterial);
      AnimMesh->setName(MStr(FileMesh->Header.Name) + "Mesh");
      meshShape->setName(MStr(FileMesh->Header.Name) + "Shape");

      animSceneObj->setName(FileMesh->Header.Name);
      animSceneObj->setTextureMaterial(SkinMaterial);
      animSceneObj->setShapeObject(meshShape);

    }
    
    
    AnimMesh->addTriangleArray(Tris, FileMesh->Header.NumTris);
    SkinMesh->addTriangleArray(Tris, FileMesh->Header.NumTris);
    
    AnimMesh->setTextureMesh(SkinMesh);
    
    MSceneObjectPtr NewTransform;
    MTreeObjectNodePtr NewObjNode;
     
    NewTransform = new MSceneObject;
     
    // Set the names up for the objects
    {
      MStr     Str;
      
      Str = AnimMesh->getName();
      Str += "Group";
      NewTransform->setName((LPCTSTR)Str);
    }
    
    delete[] Tris;
    delete[] Verts;
    
    AnimMesh->weldVertices(0, 0.00);
    
    NewObjNode = Scene->addObject(animSceneObj, MasterXFormNode);
    
    AnimMesh->calculateNormals();
  }
  
  // Add in the Tag transforms
  {
    DWORD   nTag, nFrame;
  
    for (nTag = 0; nTag < md3Model.header.numTags; nTag++) {
      MSceneObjectPtr TagXForm;
      MTreeObjectNodePtr TagXFormNode;
    
      TagXForm = new MSceneObject;
      TagXForm->setName(MasterObjName + MStr(md3Model.tagFrames[0].Tags[nTag].Name) + MStr("_TAG"));
      TagXForm->addParameter(MParameterFactory::createString("q3tag", "Q3TagName", "Q3 Tag Name"));
      TagXForm->setParamByName("q3tag", MStr(md3Model.tagFrames[0].Tags[nTag].Name));
    
      TagXFormNode = Scene->addObject(TagXForm, MasterXFormNode);
    
      for (nFrame = 0; nFrame < md3Model.header.numTagFrames; nFrame++) {
        // Set the position data.
        MVector3    Pos, Rot;
        MMatrix4    Mat;
      
        MD3Tag *curTag = &md3Model.tagFrames[nFrame].Tags[nTag];

        Pos.x = curTag->origin[0];
        Pos.y = curTag->origin[1];
        Pos.z = curTag->origin[2];
      
        Mat.identity();
        Mat.m[0][0] = curTag->Matrix[0][0];
        Mat.m[0][1] = curTag->Matrix[1][0];
        Mat.m[0][2] = curTag->Matrix[2][0];
        Mat.m[1][0] = curTag->Matrix[0][1];
        Mat.m[1][1] = curTag->Matrix[1][1];
        Mat.m[1][2] = curTag->Matrix[2][1];
        Mat.m[2][0] = curTag->Matrix[0][2];
        Mat.m[2][1] = curTag->Matrix[1][2];
        Mat.m[2][2] = curTag->Matrix[2][2];
        Mat.transpose();
      
        Mat.convertToEuler(Rot);
        Rot *= 180 / 3.14159265358979323f;
      
        TagXForm->getTransformObject()->setTranslateVector(Pos);
        TagXForm->getTransformObject()->setRotateVector(Rot);
        TagXForm->getTransformObject()->updateKey(nFrame*120, TRANSFORM_CHAN_TRANSLATE | TRANSFORM_CHAN_ROTATE, true);

        MMatrix4 newMat;
        TagXForm->getTransformObject()->getTransformMatrix(newMat);

      }
    }
  }

  return true;
}

bool canDoObject(MTreeObjectNodePtr objNode, MTreeObjectNodePtr rootNode, const vector<MTreeObjectNodePtr> &exclude) {
  if (objNode == NULL) {
    return false;
  }

  // if we have a root node, and our object isn't a child
  // of that root, we can't export it.
  if (rootNode != NULL && objNode != rootNode && !objNode->isChildOf( rootNode )) {
    return false;
  }

  // now check to see if the object is a child of one of the excluded nodes.
  vector<MTreeObjectNodePtr>::const_iterator it;

  for (it = exclude.begin(); it != exclude.end(); ++it) {
    if (objNode == *it || objNode->isChildOf(*it)) {
      return false;
    }
  }

  // if all of our tests have failed, then we do the object!
  return true;
}

void addSegmentWithNoOverlap(vector<MTimeSegmentPtr> &segments, MTimeSegmentPtr seg) {
  // go through the already stored segments and see if the time 
  // segment is alredy taken care of, or if we only do a fraction
  // of the time segment.
  int segStart, segEnd;

  segStart = seg->getStart();
  segEnd = seg->getEnd();

  for (int i = 0; i < segments.size(); ++i) {
    // see if our segment is totally encompassed
    if (segments[i]->getStart() <= segStart &&
      segments[i]->getEnd() >= segEnd) {
      return;
    }

    // otherwise adjust our segment to fit if we intersect the segments at all.
    if ( (segStart >= segments[i]->getStart() && segStart <= segments[i]->getEnd()) &&
      (segEnd >= segments[i]->getStart() && segEnd <= segments[i]->getEnd()) ) {
         
      if (segments[i]->getEnd() > segStart) {
        segStart = segments[i]->getEnd();
      }
      if (segments[i]->getStart() < segEnd) {
        segEnd = segments[i]->getStart();
      }

      if (segStart == segEnd) {
        return;
      }
    }
  }

  segments.push_back(seg);
}

MStr toString(MNamedObjectPtr obj) {
  MStr result = "'";
  if (obj != NULL) {
    result += obj->getName();
  } else {
    result += "null";
  }
  result += "'";

  return result;
}



/**
** Tkane from q3source/common/mathlib.c
** NormalToLatLong
**
** We use two byte encoded normals in some space critical applications.
** Lat = 0 at (1,0,0) to 360 (-1,0,0), encoded in 8-bit sine table format
** Lng = 0 at (0,0,1) to 180 (0,0,-1), encoded in 8-bit sine table format
**
*/
#define RAD2DEG(r) ((r)/3.14159265358979323*180.0)

void NormalToLatLong( const float normal[3], unsigned char bytes[2] ) {
  // check for singularities
  if ( normal[0] == 0 && normal[1] == 0 ) {
    if ( normal[2] > 0 ) {
      bytes[0] = 0;
      bytes[1] = 0;           // lat = 0, long = 0
    } else {
      bytes[0] = 128;
      bytes[1] = 0;           // lat = 0, long = 128
    }
  } else {
    int     a, b;
    
    a = (int)RAD2DEG( atan2( normal[1], normal[0] ) ) * (255.0f / 360.0f);
    a &= 0xff;
    
    b = (int)RAD2DEG( acos( normal[2] ) ) * ( 255.0f / 360.0f );
    b &= 0xff;
    
    bytes[0] = b;   // longitude
    bytes[1] = a;   // lattitude
  }
}

bool MMD3Translator::exportObject(MStr filename, MScenePtr scene, MSceneObjectPtr rootObj, const vector<MTreeObjectNodePtr> &exclude) {
  if (scene == NULL) {
    return false;
  }

  MBaseObjectPtr obj;  
  MTreeObjectNodePtr rootNode;

  // keep the tags separate.
  vector<MSceneObjectPtr> tags;
  vector<MTreeObjectNodePtr> tagNodes;
  vector<MTreeObjectNodePtr> objects;
  vector<MTimeSegmentPtr> segments;
  vector<MSceneObjectPtr> meshes;

  if (rootObj != NULL) {
    rootNode = scene->getObjectList()->findObject(rootObj);
  }

  // iterate over the scene, and extract all our tags and objects
  scene->getObjectList()->beginIteration();
  while ((obj = scene->getObjectList()->getNext()) != NULL) {
    MTreeObjectNodePtr objNode = scene->getObjectList()->getCurrentNode();
    // we only want to consider scene objects.
    MSceneObjectPtr sceneObj = AZTEC_CAST(MSceneObject, obj);

    if (sceneObj == NULL || !canDoObject(objNode, rootNode, exclude)) {
      continue;
    }

    objects.push_back(objNode);

    // see if we have a mesh
    MShapeObjectPtr shape = sceneObj->getShapeObject();

    if (shape != NULL) {
      MMeshPtr tempMesh = AZTEC_CAST(MMesh, shape->convertToMesh());
      if (tempMesh != NULL) {
        meshes.push_back(sceneObj);
      }
    }

    // see if we have a tag
    MStr tagName;
    if (sceneObj->getParamByName("q3tag", tagName)) {
      tags.push_back(sceneObj);
      tagNodes.push_back(objNode);
    }
  }
  scene->getObjectList()->endIteration();

  // iterate over our segments, and see which are are valid for this model
  for (int i = 0; i < scene->getTimeSegmentCount(); ++i) {
    MTimeSegmentPtr seg = scene->getTimeSegment(i);

    // see if the segment contains any of our objects.
    for (int n = 0; n < seg->getObjectCount(); ++n) {
      bool segmentUsed = false;
      MNamedObjectPtr namedObj = seg->getObject(n);
      MTreeObjectNodePtr namedNode = scene->getObjectList()->findObject(namedObj);

      // iterate over the known objects
      for (int index = 0; index < objects.size(); ++index) {
        MNamedObjectPtr obj = AZTEC_CAST(MNamedObject, objects[index]->getObject());
        if (namedObj == obj) {
          segmentUsed = true;
          break;
        }
      }

      if (segmentUsed) {
        addSegmentWithNoOverlap(segments, seg);
        break;
      }
    }

  }

  if (segments.size() == 0) {
    Aztec::getSystemManager()->logOutput(
      "Warning exporting MD3 file. No Time segments were found "
      "that used the object %s. Exporting a single frame only.", toString(rootObj).c_str());

    MTimeSegmentPtr seg = new MTimeSegment();
    seg->setRange(0,119);
    segments.push_back(seg);
  } else {
    Aztec::getSystemManager()->logOutput("The following segments were found to use the object %s", toString(rootObj).c_str());
    for (int i = 0; i < segments.size(); ++i) {
      Aztec::getSystemManager()->logOutput("    %s", segments[i]->getName().c_str());
    }
  }

  MD3Model  md3Model;
  MD3Header *md3Head = &md3Model.header;
  
  md3Head->ID = MD3_FILEID;
  md3Head->Version = MD3_VERSION;

  md3Head->FileName[0] = '\0';
  md3Head->flags = 0;
  md3Head->numMeshes = meshes.size();
  md3Head->numSkins = 1;
  md3Head->numTags = tags.size();
  md3Head->numTagFrames = 0;
  for (int i = 0; i < segments.size(); ++i) {
    int length = segments[i]->getEnd() - segments[i]->getStart();
    // TODO: When the md3's are imported, the animation keys should be
    // scaled to suit the time segment's fps. 
    // But for now, we use stock standard 30 fps.
    // int fps = segments[i]->getParamInt("fps");
    int fps = 30;
    if (length > 0) {
      md3Head->numTagFrames +=  fps * length / 3600 + 1;
    }
  }

  Aztec::getSystemManager()->logOutput("Outputting %i tag frames.", md3Head->numTagFrames);

  // create our tag frames.
  {
    md3Model.tagFrames = new MD3TagFrame[md3Head->numTagFrames];
  
    for (int frameIndex = 0; frameIndex < md3Head->numTagFrames; frameIndex++) {
      md3Model.tagFrames[frameIndex].Tags = new MD3Tag[md3Head->numTags];
    }
  }

  // create the md3 surfaces.
  md3Model.surfaces = new MD3Mesh[meshes.size()];
  // this is a mapping from the indexth Skin Vertex, to the
  // vertexMapping[index] mesh vertex
  vector< vector<int> > vertexMapping;

  vertexMapping.resize(meshes.size());

  for (int meshIndex = 0; meshIndex < meshes.size(); ++meshIndex) {
    MD3Mesh *surf = &md3Model.surfaces[meshIndex];
    MMeshPtr mesh = AZTEC_CAST(MMesh, meshes[meshIndex]->getShapeObject()->convertToMesh());
    MMeshPtr texMesh = mesh->getTextureMesh();

    // if we don't have a texture mesh, use the mesh itself.
    if (texMesh == NULL) {
      Aztec::getSystemManager()->logOutput("Warning: Object %s doesn't have a texture mesh.", meshes[meshIndex]->getName().c_str());
      texMesh = mesh;
    }

    vertexMapping[meshIndex].resize(texMesh->getNumVerts(), -1);

    if (mesh == NULL) {
      Aztec::getSystemManager()->logOutput("Warning: Object %s didn't produce a mesh properly.", meshes[meshIndex]->getName().c_str());
      continue;
    }

    surf->Header.ID = MD3_FILEID;
    memset(surf->Header.Name, 0, MAX_QPATH);
    strcpy(surf->Header.Name, meshes[meshIndex]->getName().c_str());

    surf->Header.flags = 0;
    surf->Header.NumFrames = md3Head->numTagFrames;
    surf->Header.numShaders = 1; // just use one dummy shader for now.
    surf->Header.NumVerts = texMesh->getNumVerts();
    surf->Header.NumTris = mesh->getNumTris();

    surf->SkinNames = new MD3ShaderHeader[1];
    memset(surf->SkinNames[0].name, 0, MAX_QPATH);
    surf->SkinNames[0].shaderIndex = 0;

    // set up the triangles. Now the triangles we actually use are 
    // generated from the texture mesh
    surf->Triangles = new MD3Triangle[surf->Header.NumTris];

    MD3Triangle *curTri = surf->Triangles;
    for (int triIndex = 0; triIndex < surf->Header.NumTris; ++triIndex) {
      // Quake3's triangle ordering is backwards from us.
      (*curTri)[0] = mesh->getTriangleVertex(triIndex, 2);
      (*curTri)[1] = mesh->getTriangleVertex(triIndex, 1);
      (*curTri)[2] = mesh->getTriangleVertex(triIndex, 0);
      curTri++;
    }

    // set up the mapping from tex vertices to 3d vertices
    // iterate over all the triangles and do them. The mapping
    // should always be n to m where n >= m.
    for (int triIndex = 0; triIndex < surf->Header.NumTris; ++triIndex) {
      vertexMapping[meshIndex][texMesh->getTriangleVertex(triIndex, 0)] = mesh->getTriangleVertex(triIndex, 0);
      vertexMapping[meshIndex][texMesh->getTriangleVertex(triIndex, 1)] = mesh->getTriangleVertex(triIndex, 1);
      vertexMapping[meshIndex][texMesh->getTriangleVertex(triIndex, 2)] = mesh->getTriangleVertex(triIndex, 2);
    }

    // set up the texture coordinates
    surf->SkinVerts = new MD3SkinVertex[surf->Header.NumVerts];

    MD3SkinVertex *curSkinVert = surf->SkinVerts;
    for (int vertIndex = 0; vertIndex < texMesh->getNumVerts(); ++vertIndex) {
      MVector3 vert = texMesh->getVertexPosition(vertIndex);

      (*curSkinVert)[0] = vert.x;
      (*curSkinVert)[1] = vert.y;
      curSkinVert++;
    }

    // create the animation frames, but do not fill them out.
    surf->Frames = new MD3MeshFrame[surf->Header.NumFrames];
    
    for (int frameIndex = 0; frameIndex < surf->Header.NumFrames; ++frameIndex) {
      surf->Frames[frameIndex].Vertices = new MD3Vertex[surf->Header.NumVerts];
      memset(surf->Frames[frameIndex].Vertices, 0, sizeof(MD3Vertex) * surf->Header.NumVerts);
    }
  }


  // now that we have created all our surfaces, go through our animation and 
  // fill out our surface frame animation and tag animation
  int frameIndex = 0;

  for (int i = 0; i < segments.size(); ++i) {
    MTimeSegmentPtr timeSeg = segments[i];
    int length = timeSeg->getEnd() - timeSeg->getStart();
    // TODO: When the md3's are imported, the animation keys should be
    // scaled to suit the time segment's fps. 
    // But for now, we use stock standard 30 fps.
    // int fps = segments[i]->getParamInt("fps");
    int fps = 30;

    float tickStep = 3600.0f / (float)fps;
    int curTime, frameCount = 0;

    // dont do anything if the segment length is zero.
    if (length == 0) {
      continue;
    }

    curTime = timeSeg->getStart();
    scene->setTime(curTime-10);
    while (curTime <= timeSeg->getEnd()) {
      assert(frameIndex < md3Model.header.numTagFrames);
      assert(frameCount < fps * length / 3600 + 1);

      scene->setTime(curTime);

      // fill out our tag information
      {
        MD3TagFrame *tagFrame = &md3Model.tagFrames[frameIndex];
        memset(tagFrame->Creator, 0, 16);
        strcpy(tagFrame->Creator, "(from Aztec)");

        for (int tagIndex = 0; tagIndex < tags.size(); ++tagIndex) {
          MSceneObjectPtr tagObj = tags[tagIndex];
          MD3Tag *curTag = &tagFrame->Tags[tagIndex];
          strcpy(curTag->Name, tagObj->getParamByName("q3tag").c_str());

          // get the origin. The origin will be relative to our root object.
          // if we do not have a root object, it is relative to the world.
          MVector3 pos = tagObj->getTransformObject()->getTranslateVector(curTime);

          curTag->origin[0] = pos.x;
          curTag->origin[1] = pos.y;
          curTag->origin[2] = pos.z;
          
          MMatrix4 matrix;
          scene->getWorldTransformMatrix(tagNodes[tagIndex], matrix);

          // convert the local transformation to one relative to the root object.
          if (rootObj != NULL) {
            MMatrix4 rootMatrix;
            scene->getWorldTransformMatrix(rootNode, rootMatrix);
            rootMatrix.inverse();
            matrix = rootMatrix * matrix;
          }

          for (int i = 0; i < 3; ++i) {
            for (int j = 0; j < 3; ++j) {
              curTag->Matrix[i][j] = matrix.m[i][j];
            }
          }
        } 
      }

      // do our frame's animation for our meshes
      for (int meshIndex = 0; meshIndex < meshes.size(); ++meshIndex) {
        MMeshPtr mesh = AZTEC_CAST(MMesh, meshes[meshIndex]->getShapeObject()->convertToMesh());
        
        if (mesh == NULL) {
          continue;
        }

        MD3Mesh *curMesh = &md3Model.surfaces[meshIndex];
        MD3MeshFrame *curFrame = &curMesh->Frames[frameIndex];
        int numVerts = curMesh->Header.NumVerts;
        MVector3 localVert, worldVert;

        MMatrix4 objectToWorld, worldToObject, transformMat;

        scene->getWorldTransformMatrix(scene->getObjectList()->findObject(meshes[meshIndex]), objectToWorld);
        if (rootNode != NULL) {
          scene->getWorldTransformMatrix(rootNode, worldToObject);
          worldToObject.inverse();
        } else {
          worldToObject.identity();
        }

        transformMat = worldToObject * objectToWorld;

        MVector3 worldZero = transformMat * MVector3(0,0,0);
        for (int vertIndex = 0; vertIndex < numVerts; ++vertIndex) {
          // get our vertex information.
          MVector3 meshVert;
          MVector3 worldNormal;
          int vertexIndex = vertexMapping[meshIndex][vertIndex];
          meshVert = mesh->getVertexPosition(vertexIndex);

          localVert = meshVert;

          worldVert = transformMat * MVector4(localVert);

          curFrame->Vertices[vertIndex].Vec[0] = (short)(worldVert.x * 64.0);
          curFrame->Vertices[vertIndex].Vec[1] = (short)(worldVert.y * 64.0);
          curFrame->Vertices[vertIndex].Vec[2] = (short)(worldVert.z * 64.0);

          // the normal in a quake3 file is a lat/log normal stored in 2 bytes. 
          // The first byte is the latitude, and the second byte is longnitude.
          // If we are decoding the word, we do:
          // lat = (normal >> 8) & 0xff;
          // lng = normal && 0xff;
          // lat *= M_PI/128;
          // lng *= M_PI/128;
          // normal[0] = cos(lat) * sin(lng);
          // normal[1] = sin(lat) * cos(lng);
          // normal[2] = cos(lng);

          worldNormal = transformMat * mesh->getVertexNormal(vertexIndex) - worldZero;

          float norm[3];
          unsigned char byteNorm[2];
          norm[0] = worldNormal.x;
          norm[1] = worldNormal.y;
          norm[2] = worldNormal.z;

          NormalToLatLong(norm, byteNorm);
          curFrame->Vertices[vertIndex].normal = (byteNorm[1] << 8) + byteNorm[0];
        }
      }

      frameIndex++;
      frameCount++;
      curTime = (int)(tickStep * frameCount + timeSeg->getStart());
    }

  }

  // now that we have constructed our model, write it out.
  //int hFile = open((LPCTSTR)filename, O_BINARY | O_WRONLY | O_CREAT | O_TRUNC, S_IREAD | S_IWRITE);
  FILE *hFile = fopen(filename.c_str(), "wb");

  if (!hFile) {
    Aztec::getSystemManager()->logOutput("Could not open the file %s, aborting export.", (LPCTSTR)filename);
  } else {

    md3Model.writeToFile(hFile);
    
    fclose(hFile);
  }

  return true;
}


bool MMD3Translator::exportFile(MStr Filename, MScenePtr Scene) {
  return exportObject(Filename, Scene, NULL, vector<MTreeObjectNodePtr>());
}


MD3Mesh::MD3Mesh()
{
  SkinNames = NULL;
  Triangles = NULL;
  SkinVerts = NULL;
  Frames = NULL;
}

MD3Mesh::~MD3Mesh()
{
  if (SkinNames)
    delete[] SkinNames;
  if (Triangles)
    delete[] Triangles;
  
  if (SkinVerts)
    delete[] SkinVerts;
  
  if (Frames)
  {
    DWORD    n;
    for (n = 0; n < Header.NumFrames; n++)
    {
      if (Frames[n].Vertices)
        delete[] Frames[n].Vertices;
      
    }
    delete[] Frames;
  }
}

MD3Model::MD3Model() {
  tagFrames = NULL;
  surfaces = NULL;
}

MD3Model::~MD3Model() {
  if (tagFrames != NULL) {
    for (int i = 0; i < header.numTagFrames; ++i) {
      if (tagFrames[i].Tags != NULL) {
        delete[] tagFrames[i].Tags;
      }
    }
    delete[] tagFrames;
  }
  if (surfaces != NULL) {
    delete[] surfaces;
  }
}

int MD3Model::readFromFile(FILE *hFile) {
  MD3Header *md3Head;
  MD3TagFrame *TagFrames = NULL;

  md3Head = &header;
  fread(md3Head,sizeof(MD3Header),1,hFile);
  
  if (md3Head->ID != MD3_FILEID || md3Head->Version != MD3_VERSION)
  {
    fclose(hFile);
    return false;
  }
  
  
  {
    DWORD   n, f;
    
    tagFrames = new MD3TagFrame[md3Head->numTagFrames];
    
    // Read in the tag frames
    for (f = 0; f < md3Head->numTagFrames; f++) {
      // read in the tag except for the last 4 bytes which is a 
      // pointer to a bnuch of tags.
      fread(&tagFrames[f],sizeof(MD3TagFrame)-4,1,hFile);
    }
    
    // Read in the tags
    for (f = 0; f < md3Head->numTagFrames; f++) {
      tagFrames[f].Tags = new MD3Tag[md3Head->numTags];
      
      // Read in on the the MD3ObjectHeader part      
      for (n = 0; n < md3Head->numTags; n++) {
        fread(&tagFrames[f].Tags[n], sizeof(MD3Tag),1,hFile);
      }
    }
  }
  
  // Read in all the meshes for the object
  
  DWORD      nMesh;
  
  surfaces = new MD3Mesh[md3Head->numMeshes];

  for (nMesh=0; nMesh<md3Head->numMeshes; nMesh++) {
    MD3Mesh     *Mesh;
    DWORD       n;
    long headPos, curPos;
    
    Mesh = &surfaces[nMesh];
    
    headPos = ftell(hFile);
    fread(&Mesh->Header, sizeof(MD3MeshHeader),1,hFile);
    
    // Read in the skin names
    curPos = ftell(hFile);
   
    // move in the file as much as we need to.
    fseek(hFile, headPos - curPos + (long)Mesh->Header.Shader_Start, SEEK_CUR);

    Mesh->SkinNames = new MD3ShaderHeader[Mesh->Header.numShaders];
    for (n = 0; n< Mesh->Header.numShaders; n++) {
      fread(Mesh->SkinNames[n].name, MAX_QPATH,1,hFile);
      fread(&Mesh->SkinNames[n].shaderIndex, 4,1,hFile);
    }
    
    // Read in the Triangles.
    curPos = ftell(hFile);
    
    // move in the file as much as we need to.
    fseek(hFile, headPos - curPos + (long)Mesh->Header.Triangle_Start, SEEK_CUR);

    Mesh->Triangles = new MD3Triangle[Mesh->Header.NumTris];
    fread(Mesh->Triangles, sizeof(MD3Triangle)*Mesh->Header.NumTris,1,hFile);
    
    curPos = ftell(hFile);
    // move in the file as much as we need to.
    fseek(hFile, headPos - curPos + (long)Mesh->Header.SkinVerts_Start, SEEK_CUR);
    // Read in the Texture Coords.
    Mesh->SkinVerts = new MD3SkinVertex[Mesh->Header.NumVerts];
    fread(Mesh->SkinVerts, sizeof(MD3SkinVertex)*Mesh->Header.NumVerts,1,hFile);
    
    Mesh->Frames = new MD3MeshFrame[Mesh->Header.NumFrames];

    curPos = ftell(hFile);
    fseek(hFile, headPos - curPos + (long)Mesh->Header.Vertex_Start, SEEK_CUR);
    
    for (n = 0; n<Mesh->Header.NumFrames; n++) {
      Mesh->Frames[n].Vertices = new MD3Vertex[Mesh->Header.NumVerts];
      fread(Mesh->Frames[n].Vertices, sizeof(MD3Vertex)*Mesh->Header.NumVerts,1,hFile);
    }

  }

  return 1;
}

int MD3Model::writeToFile(FILE *hFile) {
  // write out the header.
  fwrite(&header, sizeof(MD3Header),1,hFile);

  header.tagStart = ftell(hFile);

  // write out the tags;
  {
    DWORD   n, f;

    for (f = 0; f < header.numTagFrames; f++) {
      fwrite(&tagFrames[f], sizeof(MD3TagFrame)-4,1,hFile);
    }

    for (f = 0; f < header.numTagFrames; f++) {
      for (n = 0; n < header.numTags; n++) {
        fwrite(&tagFrames[f].Tags[n], sizeof(MD3Tag),1,hFile);
      }
    }
  }

  header.meshStart = ftell(hFile);

  // write out the meshes.
  DWORD      nMesh;
  
  for (nMesh=0; nMesh<header.numMeshes; nMesh++) {
    MD3Mesh     *Mesh;
    DWORD       n;
    long headPos, curPos;
    
    Mesh = &surfaces[nMesh];
    
    headPos = ftell(hFile);
    fwrite(&Mesh->Header, sizeof(MD3MeshHeader),1,hFile);
    
    Mesh->Header.Shader_Start = ftell(hFile) - headPos;

    for (n = 0; n< Mesh->Header.numShaders; n++) {
      fwrite(Mesh->SkinNames[n].name, MAX_QPATH,1,hFile);
      fwrite(&Mesh->SkinNames[n].shaderIndex, 4,1,hFile);
    }
    
    Mesh->Header.Triangle_Start = ftell(hFile) - headPos;

    fwrite(Mesh->Triangles, sizeof(MD3Triangle)*Mesh->Header.NumTris,1,hFile);
    
    Mesh->Header.SkinVerts_Start = ftell(hFile) - headPos;

    // Read in the Texture Coords.
    fwrite(Mesh->SkinVerts, sizeof(MD3SkinVertex)*Mesh->Header.NumVerts,1,hFile);
    
    Mesh->Header.Vertex_Start = ftell(hFile) - headPos;

    for (n = 0; n<Mesh->Header.NumFrames; n++) {
      fwrite(Mesh->Frames[n].Vertices, sizeof(MD3Vertex)*Mesh->Header.NumVerts,1,hFile);
    }

    // now that we have updated our mesh header completly. Rewind to that start pos,
    // rewrite the header, and then come back here.
    curPos = ftell(hFile);
    fseek(hFile, headPos, SEEK_SET);
    fwrite(&Mesh->Header, sizeof(MD3MeshHeader),1,hFile);
    fseek(hFile, curPos, SEEK_SET);
  }

  // now that we have finsihed with everything rewind back to the 
  // start of the file, and rewrite the header.
  header.FileSize = ftell(hFile);
  fseek(hFile, 0, SEEK_SET);
  fwrite(&header, sizeof(MD3Header),1,hFile);
  fseek(hFile, header.FileSize, SEEK_SET);

  return 1;
}

//---------
MD3Header::MD3Header() {
  memset(this, 0, sizeof(*this));
}
