
#include "StdAfx.h"
#include "OBJTranslator.h"

#include "MBaseObject.h"
#include "MEditableMesh.h"
#include "MMeshShape.h"
#include "MBone.h"
#include "MLight.h"

#include "MSystemManager.h"
#include "MTextFileStream.h"

#include <stdio.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <assert.h>

#if defined( _DEBUG ) && defined( _MSC_VER )
// Memory leak detection for MS compiler
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


MOBJTranslator::MOBJTranslator()
{
}

MOBJTranslator::~MOBJTranslator()
{
}

// Class related
Aztec::MTranslatorPtr MOBJTranslator::createNew() const {
   MOBJTranslator   *NewTranslator;

   NewTranslator = new MOBJTranslator;

   return NewTranslator;
}

bool MOBJTranslator::canImportFile(MStr Filename) {
   FILE  * hFile;
   bool  Result = false;
   
   Filename.Replace('/', '\\');
   hFile = fopen(Filename, "rb");

   if (!hFile) return false;

   // Split the filename into bits.
   MStr     Drive, Dir, Name, Ext;

   Filename.SplitPath(NULL, NULL, &Name, &Ext);
   if (Ext.compareNoCase(".obj") == 0) Result = true;

   fclose(hFile);

   return Result;
}

bool MOBJTranslator::importFile(MStr Filename, Aztec::MScenePtr scene) {
  Filename.Replace('/', '\\');

  if (scene == NULL) {
    return false;
  }

  // Note from Richard: Currently I can't use MTextFileReader
  //                    because MStr is too limited in functionality..

  Aztec::MBaseObjectPtr    obj;
  Aztec::MEditableMeshPtr  mesh;
  Aztec::MMeshShapePtr     meshShape;
  Aztec::MSceneObjectPtr   sceneObj;


  // Open the input file
  FILE* fp = fopen((LPCTSTR)Filename, "r+t");

  if (fp == NULL) {
    Aztec::getSystemManager()->logOutput("Could not open %s, aborting import.", Filename.c_str());
  }


  std::vector<Aztec::MVector3> verts;
  std::vector<Aztec::MVector3> normals;

  typedef std::vector<int> Polygon;

  std::vector<Polygon> polygons;

  char  line[2048];
  int   linecount = 0;


  while (fgets(line, sizeof(line), fp)) {

    // Increase the line count
    linecount++;

    // TODO: Don't use strtok, use a tokenizer
    char* kw = strtok(line," \t\n");

    if (kw) {

      if (strcmp(kw,"v") == 0) {

        // Vertices [v x y z # no optional vars]
        char* xs = strtok(NULL," \t\n");
        char* ys = strtok(NULL," \t\n");
        char* zs = strtok(NULL," \t\n");

        if (xs && ys && zs) {

          Aztec::MVector3 vert;

          vert.x = (float)atof(xs);
          vert.y = -(float)atof(zs);
          vert.z = (float)atof(ys);

          verts.push_back(vert);

        } else {

          Aztec::getSystemManager()->logOutput("Error at line %d (section: vertices) in %s.", linecount, Filename.c_str());
          fclose(fp);

          return false;
        }

      } else if (strcmp(kw,"vt") == 0) {

        // Texture coords [vt u v w # u + v are optional (1D, 2D, 3D)]
        char* xs = strtok(NULL," \t\n");
        char* ys = strtok(NULL," \t\n");
        char* zs = strtok(NULL," \t\n");

        if (xs || ys || zs) {

          Aztec::MVector3 vert;
          char      num = 0;    // # of dimensions (1, 2 or 3)

          vert.x = (float)atof(xs);            ++num;
          if (ys)  { vert.y = (float)atof(ys); ++num; }
          if (zs)  { vert.z = (float)atof(zs); ++num; }

          // TODO: Add to texcoord array

        } else {

          Aztec::getSystemManager()->logOutput("Error at line %d (section: tex coords) in %s.", linecount, Filename.c_str());
          fclose(fp);

          return false;
        }

      } else if (strcmp(kw, "vn") == 0) {

        // Vertex Normals [vn x y z # no optional vars]
        char* xs = strtok(NULL," \t\n");
        char* ys = strtok(NULL," \t\n");
        char* zs = strtok(NULL," \t\n");

        if (xs && ys && zs) {

          normals.push_back( Aztec::MVector3((float)atof(xs), (float)atof(ys), (float)atof(zs)) );

        } else {

          Aztec::getSystemManager()->logOutput("Error at line %d (section: vertex normals) in %s.", linecount, Filename.c_str());
          fclose(fp);

          return false;
        }

      } else if (strcmp(kw, "g") == 0) {

        // Face Groups [g name01 name02 name03 etc # other vars than name01 are optional]
        char* gs = strtok(NULL, " \t\n");

        //names.push_back(MStr(gs));

        // TODO: Handle other (optional) group names

      } else if (strcmp(kw, "s") == 0) {

        // TODO: Smoothing Groups [s number # no optional vars]

      } else if (strcmp(kw, "f") == 0) {

        // Faces [f index1/tex1/nrml1 index2/tex2/nrml2 index3/tex3/nrml3 # tex + nrml are optional
        //        signed indices are indices into the last specified values (eg. for -1 you should use the last value)]

        char* str;

        // TODO: Polygons can contain more than 3 indices
        //       should be handled correctly and triangulated
        //       (see a few lines further)
        std::vector<int> polyPoints[3];
        int  ii = 0;

        while ((str = strtok(NULL," \t\n"))) {

          char *slash;
          long a = 0;

          do {
            slash = strchr(str,'/');
            if (slash) {
              *slash = 0;
            }

            if (*str) {
              polyPoints[a].push_back(atoi(str));
            }

            ++a;
            str = slash+1;
          } while (slash && (a<3));

          /* i[0][] = Vertices
             i[1][] = Tex Coords
             i[2][] = Vertex Normals */

          // Check for zero based indices
          if (polyPoints[0].back() == 0) {

            Aztec::getSystemManager()->logOutput("Error at line %i, found zero based index.", linecount);
            fclose(fp);

            return false;

          } else if (polyPoints[0].back() < 0) {

            // TODO: Negative index values
          }

          ++ii;
        }

        // TODO: If the polygon contains more than 3 indices
        // we should triangulate it because Aztec doesn't support
        // quads/polygons..
        //
        // NOTE: Quad support should actually being built into
        // Aztec because it lets you create much better looking
        // meshes, especially when using subdivs.

        // for (int j=0; j<a; j++) poly.push_back(..[j]);
        // poly.triangulate(a <-- num of indices)

        // And finally add the indices to our temporary array

        // The indicies from the .obj files start at 1, whereas we always start at zero.

        for (int i = 0; i < polyPoints[0].size(); ++i) {
          --polyPoints[0][i];
        }

        polygons.push_back(polyPoints[0]);
      }

    }
  }

  fclose(fp);



  // TODO: Create multiple objects instead of throwing every
  // mesh group into one mesh
  for (int i=0; i<1/*numberOfMeshes*/; i++)
  {
    mesh      = new Aztec::MEditableMesh;
    meshShape = new Aztec::MMeshShape(mesh);
    sceneObj  = new Aztec::MSceneObject(meshShape);

    scene->addObject(mesh);
    scene->addObject(meshShape);
    scene->addObject(sceneObj);

    mesh->addVertexArray(&verts[0], verts.size());

    for (int j = 0; j < polygons.size(); ++j) {
      mesh->triangulatePolygon(polygons[j]);

    }


    // TODO: Set to group name from file, set the original name
    // for the sceneObject so the user won't be confused with the
    // naming extensions..
    sceneObj->setName("meshObj");
    mesh->setName("meshObjMesh");
    meshShape->setName("meshObjShape");

    // TODO: Compute smoothing groups from the file or assign
    // the normals from the file
    mesh->calculateNormals();
    mesh = NULL;
  }


  return true;
}

bool MOBJTranslator::exportFile(MStr Filename, Aztec::MScenePtr scene) {

  Filename.Replace('/', '\\');
  if (scene == NULL) return false;

  Aztec::MTextFileWriter             file;
  MStr                        name, ext;
  Aztec::MBaseObjectPtr              obj;

  typedef std::pair<Aztec::MSceneObjectPtr, Aztec::MMeshPtr> MeshPair;
  typedef std::vector<MeshPair> MeshList;
  MeshList meshes;
  std::vector<Aztec::MSceneObjectPtr>     sceneObjects;
  std::vector<Aztec::MMaterialPtr>        materials;

  // Standard header comment
  MStr comment = MStr("#\n"\
                      "# Exported by Aztec\n"\
                      "#\n"\
                      "#\n"\
                      "# http://www.diginux.net/aztec\n"\
                      "#\n\n\n");



  // Collect all meshes from the current scene
  scene->getObjectList()->beginIteration();

  while ((obj = scene->getObjectList()->getNext()) != NULL) {

    Aztec::MSceneObjectPtr sceneObj;
    Aztec::MShapeObjectPtr shapeObj;

    sceneObj = AZTEC_CAST(Aztec::MSceneObject, obj);
    if (sceneObj == NULL) {
      Aztec::MMaterialPtr  mat = AZTEC_CAST(Aztec::MMaterial, obj);

      if (mat != NULL) materials.push_back(mat);
      continue;
    }

    shapeObj = sceneObj->getShapeObject();
    if (shapeObj == NULL) continue;

    // Ignore lights and bones (and cameras when they're implemented)
    if (shapeObj->getClassName() == MStr("MLight"))      continue;
    if (shapeObj->getClassName() == MStr("MBoneObject")) continue;

    Aztec::MMeshPtr  tempMesh = AZTEC_CAST(Aztec::MMesh, shapeObj->convertToMesh());

    if (tempMesh != NULL) {
      meshes.push_back(MeshPair(sceneObj, tempMesh));
      sceneObjects.push_back(sceneObj);
    }
  }

  scene->getObjectList()->endIteration();




  // Start writing the materials to a seperate file
  if (!materials.empty())
  {
    std::vector<Aztec::MMaterialPtr>::iterator  it;

    Filename.SplitPath(NULL, NULL, &name, &ext);
    name += MStr(".MTL");

    if (!file.open(name)) {
      Aztec::getSystemManager()->logOutput("Could not open %s, materials will not be exported.", name.c_str());
    } else {

    // --- Write out the material file ---
    file.writeString(comment);

    for (it=materials.begin(); it!=materials.end(); it++)
    {
      Aztec::MVector3  amb, diff, spec;
      float     shin;
      MStr      dtex;
      MStr      str;

      amb  = (*it)->getParamVec("AmbientColor");
      diff = (*it)->getParamVec("DiffuseColor");
      spec = (*it)->getParamVec("SpecularColor");
      shin = (*it)->getParamFloat("Shininess");
      dtex = (*it)->getParamByName("DiffuseMapName");

      // Create a new material
      file.writeString(MStr("newmtl ") + (*it)->getName() + MStr("\n\n"));


      // Compose the material parameters
      str.Format("Ka %f %f %f\n", amb.x,  amb.y,  amb.z);
      file.writeString(str);

      str.Format("Kd %f %f %f\n", diff.x, diff.y, diff.z);
      file.writeString(str);

      str.Format("Ks %f %f %f\n", spec.x, spec.y, spec.z);
      file.writeString(str);

      // Shininess [1.0 - 128.0]
      str.Format("Ns %f\n", shin);
      file.writeString(str);

      // Build a relative path name
      if (dtex.c_str() != NULL) {
        dtex.SplitPath(NULL, NULL, &name, &ext);
        dtex = name + ext;

        str.Format("map_Kd %s\n\n\n", dtex.c_str());
        file.writeString(str);
      }
    }

    }

    file.close();
  }





  if (!file.open(Filename)) {
    Aztec::getSystemManager()->logOutput("Could not open %s, aborting export.", Filename.c_str());
    return false;
  }

  {
    file.writeString(comment);
    file.writeString(MStr("mtllib ") + name + MStr("\n\n"));

    MeshList::iterator  it;
    MStr str;

    // vertices in obj files are considered as one big array, so we will have 
    // to keep track of how many we have done.
    int vertexBase = 1;

    // NOTE: Vertex normals and texcoords are per polygon corner, not per vertex
    for (it=meshes.begin(); it!=meshes.end(); it++)
    {
      // --- Write out the vertices ---
      str.Format("\n# Vertices: %i\n", it->second->getNumVerts());
      file.writeString(str);

      for (int i=0; i<it->second->getNumVerts(); i++)
      {
        Aztec::MVector3 vtx = it->second->getVertexPosition(i);
        Aztec::MVector3 worldPoint;

        worldPoint = scene->objectToWorldSpace(Aztec::MBaseObjectPtr(it->first), vtx);

        str.Format("v %f %f %f\n", worldPoint.x, worldPoint.z, -worldPoint.y);
        file.writeString(str);
      }


      // --- Write out the tex coords ---
      str.Format("\n# Tex Coords: %i\n", it->second->getNumVerts());
      file.writeString(str);

      for (int i=0; i<it->second->getNumVerts(); i++)
      {
        // TODO: Get the tex coords from the texture mesh
        /*MVertex*  vtx = it->second->getVert(i);

        str.Format("vt %f %f %f\n", vtx->x, vtx->y, vtx->z);
        file.writeString(str);*/
      }


      // --- Write out the vertex normals ---
      str.Format("\n# Vertex Normals: %i\n", it->second->getNumVerts());
      file.writeString(str);

      // TODO: See above
      for (int i=0; i<it->second->getNumVerts(); i++)
      {
        Aztec::MVector3 normal = it->second->getVertexNormal(i);

        str.Format("vn %f %f %f\n", normal.x, normal.y, normal.z);
        file.writeString(str);
      }


      // --- Write out the faces ---
      str.Format("\n# Faces: %i\n", it->second->getNumTris());
      file.writeString(str);

      // Mesh group
      file.writeString(MStr("g ") + it->second->getName() + MStr("\n"));

      // Material for this mesh group
      file.writeString(MStr("usemtl ") + sceneObjects[0]->getTextureMaterialName() + MStr("\n"));

      Aztec::MMeshPtr mesh = it->second;

      for (int i=0; i < mesh->getNumTris(); i++)
      {
        file.writeString(MStr("f "));

        // TODO: Could be extended to quads or polygons..
        for (int j=0; j<3; j++)
        {
          // Vertex Index (indices are non-zerobased)
          file.writeInt(mesh->getTriangleVertex(i, j)+vertexBase);
          file.writeString(MStr("//"));

          // TODO: If there are texcoords to be written, the "//" above
          // this comment should be replaced with a single slash
          // file.writeInt(texcoord)
          // file.writeString(MStr("/"));

          // Normal Index
          file.writeInt(mesh->getTriangleVertex(i, j)+vertexBase);
          file.writeString(MStr(" "));
        }

        file.writeString(MStr("\n"));
      }

      // we have done all our vertices, now update our base level.
      vertexBase += it->second->getNumVerts();
    }

  }

  file.close();


  return true;
}
