/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "scaletool.h"

#include "model.h"
#include "log.h"
#include "modelstatus.h"
#include "pixmap/scaletool.xpm"

#include "mq3macro.h"
#include "mq3compat.h"

#include <math.h>

#include <qwidget.h>

ScaleTool::ScaleTool()
{
}

ScaleTool::~ScaleTool()
{
}

const char * ScaleTool::getName( int arg )
{
   return "Scale";
}

bool ScaleTool::getKeyBinding( int arg, int & keyBinding )
{
   return false;
}

void ScaleTool::mouseButtonDown( Parent * parent, int buttonState, int x, int y )
{
   m_vertexCoords.clear();
   m_jointCoords.clear();

   m_allowX = true;
   m_allowY = true;
   m_allowZ = true;

   Model * model = parent->getModel();
   list<int> vertices = model->getSelectedVertices();
   list<int>::iterator it;

   bool firstSet = false;

   for ( it = vertices.begin(); it != vertices.end(); it++ )
   {
      double coords[3];
      model->getVertexCoords( (*it), coords );

      VertexCoords vc;
      vc.index = (*it);
      vc.x = coords[0];
      vc.y = coords[1];
      vc.z = coords[2];

      // update range
      if ( !firstSet )
      {
         m_minX = vc.x;
         m_minY = vc.y;
         m_minZ = vc.z;
         m_maxX = vc.x;
         m_maxY = vc.y;
         m_maxZ = vc.z;
         firstSet = true;
      }
      else
      {
         if ( vc.x < m_minX ) { m_minX = vc.x; }
         if ( vc.x > m_maxX ) { m_maxX = vc.x; }
         if ( vc.y < m_minY ) { m_minY = vc.y; }
         if ( vc.y > m_maxY ) { m_maxY = vc.y; }
         if ( vc.z < m_minZ ) { m_minZ = vc.z; }
         if ( vc.z > m_maxZ ) { m_maxZ = vc.z; }
      }

      m_vertexCoords.push_back( vc );
   }

   list<int> joints = model->getSelectedBoneJoints();

   for ( it = joints.begin(); it != joints.end(); it++ )
   {
      double coords[3];
      model->getBoneJointCoords( (*it), coords );

      VertexCoords vc;
      vc.index = (*it);
      vc.x = coords[0];
      vc.y = coords[1];
      vc.z = coords[2];

      // update range
      if ( !firstSet )
      {
         m_minX = vc.x;
         m_minY = vc.y;
         m_minZ = vc.z;
         m_maxX = vc.x;
         m_maxY = vc.y;
         m_maxZ = vc.z;
         firstSet = true;
      }
      else
      {
         if ( vc.x < m_minX ) { m_minX = vc.x; }
         if ( vc.x > m_maxX ) { m_maxX = vc.x; }
         if ( vc.y < m_minY ) { m_minY = vc.y; }
         if ( vc.y > m_maxY ) { m_maxY = vc.y; }
         if ( vc.z < m_minZ ) { m_minZ = vc.z; }
         if ( vc.z > m_maxZ ) { m_maxZ = vc.z; }
      }

      m_jointCoords.push_back( vc );
   }

   double curX = 0;
   double curY = 0;
   double curZ = 0;
   
   m_startLengthX = 0;
   m_startLengthY = 0;
   m_startLengthZ = 0;

   int index = 0;

   if ( !parent->getXValue( x, y, &curX ) )
   {
      index = 1;
   }
   if ( !parent->getYValue( x, y, &curY ) )
   {
      index = 2;
   }
   if ( !parent->getZValue( x, y, &curZ ) )
   {
      index = 0;
   }

   m_x = curX;
   m_y = curY;
   m_z = curZ;

   if ( m_point == ST_ScalePointFar )
   {
      double minmin = 0;
      double minmax = 0;
      double maxmin = 0;
      double maxmax = 0;

      switch ( index )
      {
         case 0:
            m_farZ = m_minZ;

            minmin = distance( m_minX, m_minY, curX, curY );
            minmax = distance( m_minX, m_maxY, curX, curY );
            maxmin = distance( m_maxX, m_minY, curX, curY );
            maxmax = distance( m_maxX, m_maxY, curX, curY );

            if ( minmin > minmax )
            {
               if ( minmin > maxmin )
               {
                  if ( minmin > maxmax )
                  {
                     m_farX = m_minX;
                     m_farY = m_minY;
                  }
                  else
                  {
                     m_farX = m_maxX;
                     m_farY = m_maxY;
                  }
               }
               else
               {
                  // maxmin > minmin
                  if ( maxmin > maxmax )
                  {
                     m_farX = m_maxX;
                     m_farY = m_minY;
                  }
                  else
                  {
                     m_farX = m_maxX;
                     m_farY = m_maxY;
                  }
               }
            }
            else
            {
               // minmax > minmin
               if ( minmax > maxmin )
               {
                  if ( minmax > maxmax )
                  {
                     m_farX = m_minX;
                     m_farY = m_maxY;
                  }
                  else
                  {
                     m_farX = m_maxX;
                     m_farY = m_maxY;
                  }
               }
               else
               {
                  // maxmin > minmax
                  if ( maxmin > maxmax )
                  {
                     m_farX = m_maxX;
                     m_farY = m_minY;
                  }
                  else
                  {
                     m_farX = m_maxX;
                     m_farY = m_maxY;
                  }
               }
            }
            break;
         case 1:
            m_farX = m_minX;

            minmin = distance( m_minZ, m_minY, curZ, curY );
            minmax = distance( m_minZ, m_maxY, curZ, curY );
            maxmin = distance( m_maxZ, m_minY, curZ, curY );
            maxmax = distance( m_maxZ, m_maxY, curZ, curY );

            if ( minmin > minmax )
            {
               if ( minmin > maxmin )
               {
                  if ( minmin > maxmax )
                  {
                     m_farZ = m_minZ;
                     m_farY = m_minY;
                  }
                  else
                  {
                     m_farZ = m_maxZ;
                     m_farY = m_maxY;
                  }
               }
               else
               {
                  // maxmin > minmin
                  if ( maxmin > maxmax )
                  {
                     m_farZ = m_maxZ;
                     m_farY = m_minY;
                  }
                  else
                  {
                     m_farZ = m_maxZ;
                     m_farY = m_maxY;
                  }
               }
            }
            else
            {
               // minmax > minmin
               if ( minmax > maxmin )
               {
                  if ( minmax > maxmax )
                  {
                     m_farZ = m_minZ;
                     m_farY = m_maxY;
                  }
                  else
                  {
                     m_farZ = m_maxZ;
                     m_farY = m_maxY;
                  }
               }
               else
               {
                  // maxmin > minmax
                  if ( maxmin > maxmax )
                  {
                     m_farZ = m_maxZ;
                     m_farZ = m_minZ;
                  }
                  else
                  {
                     m_farZ = m_maxZ;
                     m_farY = m_maxY;
                  }
               }
            }
            break;
         case 2:
            m_farY = m_minY;

            minmin = distance( m_minX, m_minZ, curX, curZ );
            minmax = distance( m_minX, m_maxZ, curX, curZ );
            maxmin = distance( m_maxX, m_minZ, curX, curZ );
            maxmax = distance( m_maxX, m_maxZ, curX, curZ );

            if ( minmin > minmax )
            {
               if ( minmin > maxmin )
               {
                  if ( minmin > maxmax )
                  {
                     m_farX = m_minX;
                     m_farZ = m_minZ;
                  }
                  else
                  {
                     m_farX = m_maxX;
                     m_farZ = m_maxZ;
                  }
               }
               else
               {
                  // maxmin > minmin
                  if ( maxmin > maxmax )
                  {
                     m_farX = m_maxX;
                     m_farZ = m_minZ;
                  }
                  else
                  {
                     m_farX = m_maxX;
                     m_farZ = m_maxZ;
                  }
               }
            }
            else
            {
               // minmax > minmin
               if ( minmax > maxmin )
               {
                  if ( minmax > maxmax )
                  {
                     m_farX = m_minX;
                     m_farZ = m_maxZ;
                  }
                  else
                  {
                     m_farX = m_maxX;
                     m_farZ = m_maxZ;
                  }
               }
               else
               {
                  // maxmin > minmax
                  if ( maxmin > maxmax )
                  {
                     m_farX = m_maxX;
                     m_farX = m_minX;
                  }
                  else
                  {
                     m_farX = m_maxX;
                     m_farZ = m_maxZ;
                  }
               }
            }
            break;
         default:
            break;
      }

      m_startLengthX = fabs( m_farX - curX );
      m_startLengthY = fabs( m_farY - curY );
      m_startLengthZ = fabs( m_farZ - curZ );
   }
   else
   {
      m_centerX = (m_maxX - m_minX) / 2.0 + m_minX;
      m_centerY = (m_maxY - m_minY) / 2.0 + m_minY;
      m_centerZ = (m_maxZ - m_minZ) / 2.0 + m_minZ;

      m_startLengthX = fabs( m_centerX - curX );
      m_startLengthY = fabs( m_centerY - curY );
      m_startLengthZ = fabs( m_centerZ - curZ );
   }

   model_status( model, StatusNormal, STATUSTIME_SHORT, "Scaling selected primitives" );
}

void ScaleTool::mouseButtonMove( Parent * parent, int buttonState, int x, int y )
{
   LOG_PROFILE();

   Model * model = parent->getModel();

   int index = 0;

   double curX = m_x;
   double curY = m_y;
   double curZ = m_z;

   if ( !parent->getXValue( x, y, &curX ) )
   {
      index = 1;
   }
   if ( !parent->getYValue( x, y, &curY ) )
   {
      index = 2;
   }
   if ( !parent->getZValue( x, y, &curZ ) )
   {
      index = 0;
   }

   if ( buttonState & BS_Shift )
   {
      if ( m_allowX && m_allowY && m_allowZ )
      {
         double ax = fabs( curX - m_x );
         double ay = fabs( curY - m_y );
         double az = fabs( curZ - m_z );

         if ( ax > ay && ax > az )
         {
            m_allowY = false;
            m_allowZ = false;
         }
         if ( ay > ax && ay > az )
         {
            m_allowX = false;
            m_allowZ = false;
         }
         if ( az > ax && az > ay )
         {
            m_allowX = false;
            m_allowY = false;
         }
      }
   }

   if ( !m_allowX )
   {
      curX = m_x;
   }
   if ( !m_allowY )
   {
      curY = m_y;
   }
   if ( !m_allowZ )
   {
      curZ = m_z;
   }


   double spX = ( m_point == ST_ScalePointFar ) ? m_farX : m_centerX;
   double spY = ( m_point == ST_ScalePointFar ) ? m_farY : m_centerY;
   double spZ = ( m_point == ST_ScalePointFar ) ? m_farZ : m_centerZ;

   double lengthX = distance( spX, 0, curX, 0 );
   double lengthY = distance( spY, 0, curY, 0 );
   double lengthZ = distance( spZ, 0, curZ, 0 );

   list<VertexCoords>::iterator it;

   switch ( index )
   {
      case 0:
         for( it = m_vertexCoords.begin(); it != m_vertexCoords.end(); it++ )
         {
            double x = (*it).x;
            double y = (*it).y;
            double z = (*it).z;

            x -= spX;
            y -= spY;
            z -= spZ;

            double xper = (lengthX / m_startLengthX);
            double yper = (lengthY / m_startLengthY);

            if ( m_proportion == ST_ScaleFree )
            {
               x *= xper;
               y *= yper;
            }
            else
            {
               if ( xper > yper )
               {
                  x *= xper;
                  y *= xper;
                  if ( m_proportion == ST_ScaleProportion3D )
                  {
                     z *= xper;
                  }
               }
               else
               {
                  x *= yper;
                  y *= yper;
                  if ( m_proportion == ST_ScaleProportion3D )
                  {
                     z *= yper;
                  }
               }
            }

            x += spX;
            y += spY;
            z += spZ;

            model->moveVertex( (*it).index, x, y, z );
         }
         for( it = m_jointCoords.begin(); it != m_jointCoords.end(); it++ )
         {
            double x = (*it).x;
            double y = (*it).y;
            double z = (*it).z;

            x -= spX;
            y -= spY;
            z -= spZ;

            double xper = (lengthX / m_startLengthX);
            double yper = (lengthY / m_startLengthY);

            if ( m_proportion == ST_ScaleFree )
            {
               x *= xper;
               y *= yper;
            }
            else
            {
               if ( xper > yper )
               {
                  x *= xper;
                  y *= xper;
                  if ( m_proportion == ST_ScaleProportion3D )
                  {
                     z *= xper;
                  }
               }
               else
               {
                  x *= yper;
                  y *= yper;
                  if ( m_proportion == ST_ScaleProportion3D )
                  {
                     z *= yper;
                  }
               }
            }

            x += spX;
            y += spY;
            z += spZ;

            model->moveBoneJoint( (*it).index, x, y, z );
         }
         break;
      case 1:
         for( it = m_vertexCoords.begin(); it != m_vertexCoords.end(); it++ )
         {
            double x = (*it).x;
            double y = (*it).y;
            double z = (*it).z;

            x -= spX;
            y -= spY;
            z -= spZ;

            double zper = (lengthZ / m_startLengthZ);
            double yper = (lengthY / m_startLengthY);

            if ( m_proportion == ST_ScaleFree )
            {
               z *= zper;
               y *= yper;
            }
            else
            {
               if ( zper > yper )
               {
                  z *= zper;
                  y *= zper;
                  if ( m_proportion == ST_ScaleProportion3D )
                  {
                     x *= zper;
                  }
               }
               else
               {
                  z *= yper;
                  y *= yper;
                  if ( m_proportion == ST_ScaleProportion3D )
                  {
                     x *= yper;
                  }
               }
            }

            x += spX;
            y += spY;
            z += spZ;

            model->moveVertex( (*it).index, x, y, z );
         }
         for( it = m_jointCoords.begin(); it != m_jointCoords.end(); it++ )
         {
            double x = (*it).x;
            double y = (*it).y;
            double z = (*it).z;

            x -= spX;
            y -= spY;
            z -= spZ;

            double zper = (lengthZ / m_startLengthZ);
            double yper = (lengthY / m_startLengthY);

            if ( m_proportion == ST_ScaleFree )
            {
               z *= zper;
               y *= yper;
            }
            else
            {
               if ( zper > yper )
               {
                  z *= zper;
                  y *= zper;
                  if ( m_proportion == ST_ScaleProportion3D )
                  {
                     x *= zper;
                  }
               }
               else
               {
                  z *= yper;
                  y *= yper;
                  if ( m_proportion == ST_ScaleProportion3D )
                  {
                     x *= yper;
                  }
               }
            }

            x += spX;
            y += spY;
            z += spZ;

            model->moveBoneJoint( (*it).index, x, y, z );
         }
         break;
      case 2:
         for( it = m_vertexCoords.begin(); it != m_vertexCoords.end(); it++ )
         {
            double x = (*it).x;
            double y = (*it).y;
            double z = (*it).z;

            x -= spX;
            y -= spY;
            z -= spZ;

            double xper = (lengthX / m_startLengthX);
            double zper = (lengthZ / m_startLengthZ);

            if ( m_proportion == ST_ScaleFree )
            {
               z *= zper;
               x *= xper;
            }
            else
            {
               if ( zper > xper )
               {
                  z *= zper;
                  x *= zper;
                  if ( m_proportion == ST_ScaleProportion3D )
                  {
                     y *= zper;
                  }
               }
               else
               {
                  z *= xper;
                  x *= xper;
                  if ( m_proportion == ST_ScaleProportion3D )
                  {
                     y *= xper;
                  }
               }
            }

            x += spX;
            y += spY;
            z += spZ;

            model->moveVertex( (*it).index, x, y, z );
         }
         for( it = m_jointCoords.begin(); it != m_jointCoords.end(); it++ )
         {
            double x = (*it).x;
            double y = (*it).y;
            double z = (*it).z;

            x -= spX;
            y -= spY;
            z -= spZ;

            double xper = (lengthX / m_startLengthX);
            double zper = (lengthZ / m_startLengthZ);

            if ( m_proportion == ST_ScaleFree )
            {
               z *= zper;
               x *= xper;
            }
            else
            {
               if ( zper > xper )
               {
                  z *= zper;
                  x *= zper;
                  if ( m_proportion == ST_ScaleProportion3D )
                  {
                     y *= zper;
                  }
               }
               else
               {
                  z *= xper;
                  x *= xper;
                  if ( m_proportion == ST_ScaleProportion3D )
                  {
                     y *= xper;
                  }
               }
            }

            x += spX;
            y += spY;
            z += spZ;

            model->moveBoneJoint( (*it).index, x, y, z );
         }
         break;
      default:
         break;
   }

   parent->updateAllViews();
}

void ScaleTool::mouseButtonUp( Parent * parent, int buttonState, int x, int y )
{
   model_status( parent->getModel(), StatusNormal, STATUSTIME_SHORT, "Scale complete" );
}

const char ** ScaleTool::getPixmap()
{
   return (const char **) scaletool_xpm;
}

double ScaleTool::distance( double x1, double y1, double x2, double y2 )
{
   double xDiff = x2 - x1;
   double yDiff = y2 - y1;
   return sqrt( xDiff*xDiff + yDiff*yDiff );
}

double ScaleTool::max( double a, double b )
{
   return ( a > b ) ? a : b;
}

void ScaleTool::setProportionValue( int newValue )
{
   m_proportion = newValue;
}

void ScaleTool::setPointValue( int newValue )
{
   m_point = newValue;
}

void ScaleTool::activated( int arg, Model * model, QMainWindow * mainwin )
{
   model_status( model, StatusNormal, STATUSTIME_NONE, "Tip: Hold shift to restrict scaling to one dimension" );
   m_widget = new ScaleToolWidget( this, mainwin );
#ifdef HAVE_QT4
   //mainwin->addDockWindow( m_widget, DockBottom );
#endif
   m_widget->show();
}

void ScaleTool::deactivated()
{
   m_widget->close();
}

