/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */

#include "checkupdate.h"

#include "mm3dport.h"
#include "sysconf.h"
#include "version.h"

#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <string>

#ifndef WIN32
#include <sys/wait.h>
#define CHECK_UPDATE_ENABLED
#endif // WIN32


//-------------------------------------------------------
// Module types
//-------------------------------------------------------

enum CheckUpdateState_e
{
   CUS_None,
   CUS_Checking,
   CUS_Ready,
   CUS_Done,
};

typedef enum CheckUpdateState_e CheckUpdateState;

//-------------------------------------------------------
// Module constants
//-------------------------------------------------------

// This is needed because GCC 4.0.0 doesn't realize that NULL
// is a proper terminator for calls to execlp
#define NULL_SENTINEL ((void *) 0)

//-------------------------------------------------------
// Module statics
//-------------------------------------------------------

int s_major[ CUV_MAX ] = { 0, 0 };
int s_minor[ CUV_MAX ] = { 0, 0 };
int s_patch[ CUV_MAX ] = { 0, 0 };

std::string s_versionString[ CUV_MAX ] = { "", "" };

static CheckUpdateState s_state      = CUS_None;
static int              s_childPid   = 0;
static std::string      s_file       = "";

#ifdef CHECK_UPDATE_ENABLED

//-------------------------------------------------------
// Module private functions
//-------------------------------------------------------

static void _pipe_write( const char * cmdstr, const char * file )
{
   FILE * fpout = fopen( file, "w" );

   if ( fpout )
   {
      FILE * fpin = popen( cmdstr, "r" );
      if ( fpin )
      {
         char input[ PATH_MAX ];
         while ( fgets( input, sizeof(input), fpin ) )
         {
            fprintf( fpout, "%s", input );
         }
         pclose( fpin );
      }
      fclose( fpout );
   }
}

void _read_version_file( const char * filename )
{
   if ( s_state == CUS_Ready )
   {
      s_state = CUS_Done;

      FILE * fp = fopen( filename, "r" );

      if ( fp )
      {
         char input[ 1024 ];
         char verstr[ 1024 ];
         int major = 0;
         int minor = 0;
         int patch = 0;

         while ( fgets( input, sizeof(input), fp ) )
         {
            if ( strncmp( "stable", input, 6 ) == 0 )
            {
               int items = sscanf( &input[6], "%s %d %d %d", verstr, 
                     &major, &minor, &patch );

               if ( items == 4 )
               {
                  s_major[ CUV_Stable ] = major;
                  s_minor[ CUV_Stable ] = minor;
                  s_patch[ CUV_Stable ] = patch;
                  s_versionString[ CUV_Stable ] = verstr;
               }
            }
            if ( strncmp( "devel", input, 5 ) == 0 )
            {
               int items = sscanf( &input[5], "%s %d %d %d", verstr, 
                     &major, &minor, &patch );

               if ( items == 4 )
               {
                  s_major[ CUV_Devel ] = major;
                  s_minor[ CUV_Devel ] = minor;
                  s_patch[ CUV_Devel ] = patch;
                  s_versionString[ CUV_Devel ] = verstr;
               }
            }
         }
         fclose( fp );
      }

      s_state = CUS_Done;
   }
}

//-------------------------------------------------------
// Module private web get functions
//-------------------------------------------------------

static void _start_wget( const char * url, const char * file )
{
   // exec wget -q -O version url
   execlp( "wget", "wget", "-q", "-O", file, url, NULL_SENTINEL );
   exit( -1 ); // should be unecessary, but just in case
}

static void _start_lynx( const char * url, const char * file )
{
   // popen lynx -source url

   char cmdstr[ PATH_MAX ];
   PORT_snprintf( cmdstr, sizeof(cmdstr), "lynx -source %s", url );
   _pipe_write( cmdstr, file );
   exit( 0 ); // this is necessary because of popen
}

static void _start_w3m( const char * url, const char * file )
{
   // popen w3m -dump_source url

   char cmdstr[ PATH_MAX ];
   PORT_snprintf( cmdstr, sizeof(cmdstr), "w3m -dump_source %s", url );
   _pipe_write( cmdstr, file );
   exit( 0 ); // this is necessary because of popen
}

static void _start_curl( const char * url, const char * file )
{
   // exec curl -s -o version url
   execlp( "curl", "curl", "-s", "-o", file, url, NULL_SENTINEL );
   exit( -1 ); // should be unecessary, but just in case
}

#endif // CHECK_UPDATE_ENABLED

//-------------------------------------------------------
// Module public functions
//-------------------------------------------------------

int check_update_start()
{
#ifdef CHECK_UPDATE_ENABLED
   // Must be in Done or None state to start process
   if ( s_state != CUS_Done && s_state != CUS_None )
   {
      return 0;
   }

   std::string url = "http://www.misfitcode.com/misfitmodel3d/version?current=";
   url = url + std::string( VERSION );

   std::string file = getMm3dHomeDirectory();

   file += "/webversion";
   unlink( file.c_str() );

   s_file = file;

   s_childPid  = fork();
   switch ( s_childPid )
   {
      case 0:   // I'm child

         {
            // None of these return if successful
            // try each in order

            _start_wget( url.c_str(), file.c_str() );
            _start_lynx( url.c_str(), file.c_str() );
            _start_w3m(  url.c_str(), file.c_str() );
            _start_curl( url.c_str(), file.c_str() );
         }
         exit( -1 );  // Oops, couldn't launch anything
         break;

      case -1:

         // Fork failed... uh oh.
         s_state = CUS_Done;
         return 0;
         break;

      default:

         // I'm parent
         s_state = CUS_Checking;
         return 1;
         break;
   }

   return 0;
#else
   return 0;
#endif  // CHECK_UPDATE_ENABLED
}

int check_update_is_done()
{
   switch ( s_state )
   {
      case CUS_None:
      case CUS_Done:
      case CUS_Ready:
         return 1;

      case CUS_Checking:
#ifndef WIN32
         {
            int status = 0;
            if ( waitpid( s_childPid, &status, WNOHANG ) )
            {
               s_state = CUS_Ready;
               return 1;
            }
         }
#endif // WIN32
      default:
         return 0;
   }
   return 0;
}

int check_update_should_check()
{
   return ( s_state == CUS_None ) ? 0 : 1;
}

void check_update_clear_check()
{
   s_state = CUS_None;
}

int check_update_get_version( CheckUpdateVersion version, int & major, int & minor, int & patch )
{
#ifndef WIN32
   _read_version_file( s_file.c_str() );

   if ( version < CUV_MAX )
   {
      major = s_major[ version ];
      minor = s_minor[ version ];
      patch = s_patch[ version ];
      return 1;
   }
   else
#endif // WIN32
   {
      return 0;
   }
}

const char * check_update_get_version_string( CheckUpdateVersion version )
{
#ifndef WIN32
   _read_version_file( s_file.c_str() );

   if ( version < CUV_MAX )
   {
      return s_versionString[ version ].c_str();
   }
   else
#endif // WIN32
   {
      return "";
   }
}

//-------------------------------------------------------
// Module testing
//-------------------------------------------------------

#ifdef MODULETEST

int main( int argc, char * argv[] )
{
   check_update_start();

   printf( "waiting...\n" );
   while( ! check_update_is_done() )
   {
      printf( "sleeping...\n" );
      sleep( 1 );
   }

   printf( "done\n" );

   int best = -1;
   int bmajor = VERSION_MAJOR;
   int bminor = VERSION_MINOR;
   int bpatch = VERSION_PATCH;
   const char * bstr = VERSION_STRING;

   int major = 0;
   int minor = 0;
   int patch = 0;
   const char * str = "";

   str = check_update_get_version_string( CUV_Stable );
   check_update_get_version( CUV_Stable, major, minor, patch );
   printf( "stable %d %d %d %s\n", major, minor, patch, str );

   if ( major > bmajor || 
         ((major == bmajor) && 
          ((minor > bminor) || 
           ((minor == bminor) && patch > bpatch ) ) ) )
   {
      printf( "stable %s is more recent than %s\n", str, bstr );

      best = CUV_Stable;
      bmajor = major;
      bminor = minor;
      bpatch = patch;
      bstr = str;
   }

   str = check_update_get_version_string( CUV_Devel );
   check_update_get_version( CUV_Devel, major, minor, patch );
   printf( "devel %d %d %d %s\n", major, minor, patch, str );

   if ( major > bmajor || 
         ((major == bmajor) && 
          ((minor > bminor) || 
           ((minor == bminor) && patch > bpatch ) ) ) )
   {
      printf( "devel %s is more recent than %s\n", str, bstr );

      best = CUV_Devel;
      bmajor = major;
      bminor = minor;
      bpatch = patch;
      bstr = str;
   }

}

#endif // MODULETEST

