/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "cubetool.h"
#include "pixmap/cubetool.xpm"

#include "model.h"
#include "weld.h"
#include "modelstatus.h"
#include "log.h"

#include <math.h>

#include <qwidget.h>

using std::vector;

static void _cubify( bool isCube, double & coord, double & diff_d, double & diff_s1, double & diff_s2 )
{
   if ( isCube )
   {
      if ( fabs(diff_s1) > fabs(diff_s2) )
      {
         if ( (diff_s1 < 0 && diff_s2 > 0) 
               || (diff_s1 > 0 && diff_s2 < 0 ) )
         {
            diff_s2 = -diff_s1;
         }
         else
         {
            diff_s2 =  diff_s1;
         }
      }
      else
      {
         if ( (diff_s1 < 0 && diff_s2 > 0) 
               || (diff_s1 > 0 && diff_s2 < 0 ) )
         {
            diff_s1 = -diff_s2;
         }
         else
         {
            diff_s1 =  diff_s2;
         }
      }
   }

   if ( fabs(diff_s1) < fabs(diff_s2) )
   {
      coord  =  fabs(diff_s1/2.0);
      diff_d = -fabs(diff_s1);
   }
   else
   {
      coord  =  fabs(diff_s1/2.0);
      diff_d = -fabs(diff_s1);
   }
}

CubeTool::CubeTool()
   : m_tracking( false ),
     m_parent( NULL )
{
}

CubeTool::~CubeTool()
{
}

void CubeTool::mouseButtonDown( Parent * parent, int buttonState, int x, int y )
{
   if ( !m_tracking )
   {
      m_parent   = parent; // Keep track of which parent we're serving
      m_tracking = true;
      m_invertedNormals = false;

      m_x1 = 0.0;
      m_y1 = 0.0;
      m_z1 = 0.0;

      parent->getXValue( x, y, &m_x1 );
      parent->getYValue( x, y, &m_y1 );
      parent->getZValue( x, y, &m_z1 );

      Model * model = parent->getModel();

      model->unselectAll();

      CubeVertices cv;

      double x1 = 0.0;
      double y1 = 0.0;
      double x2 = 0.0;
      double y2 = 0.0;
      double z  = 0.0;

      int xindex = 0;
      int yindex = 1;
      int zindex = 2;

      // Front
      for ( unsigned side = 0; side < 6; side++ )
      {
         if ( side & 1 )
         {
            x1 = 1.0; x2 = 0.0;
            y1 = 1.0; y2 = 0.0;
            z  = 1.0;
         }
         else
         {
            x1 = 0.0; x2 = 1.0;
            y1 = 1.0; y2 = 0.0;
            z  = 0.0;
         }

         switch ( side )
         {
            case 0:
            case 1:
               xindex = 0;
               yindex = 1;
               zindex = 2;
               break;
            case 2:
            case 3:
               xindex = 2;
               yindex = 1;
               zindex = 0;
               break;
            case 4:
            case 5:
               xindex = 0;
               yindex = 2;
               zindex = 1;
               break;
            default:
               break;
         }

         for ( unsigned y = 0; y <= m_segments; y++ )
         {
            for ( unsigned x = 0; x <= m_segments; x++ )
            {
               cv.coords[xindex] = x1 + ((x2 - x1) * (double) x / (double) m_segments);
               cv.coords[yindex] = y1 + ((y2 - y1) * (double) y / (double) m_segments);
               cv.coords[zindex] = z;
               cv.v = model->addVertex( m_x1, m_y1, m_z1 );

               log_debug( "adding vertex %d at %f,%f,%f\n", cv.v, cv.coords[0], cv.coords[1], cv.coords[2] );

               m_vertices.push_back( cv );
            }

            if ( y > 0 )
            {
               int row1 = m_vertices.size() - (m_segments + 1) * 2;
               int row2 = m_vertices.size() - (m_segments + 1);
               for ( unsigned x = 0; x < m_segments; x++ )
               {
                  log_debug( "%d,%d,%d,%d\n", row1 + x, row1 + x + 1, row2 + x, row2 + x + 1 );
                  int t1 = model->addTriangle( m_vertices[ row2 + x ].v, m_vertices[ row1 + x + 1].v, m_vertices[ row1 + x ].v );
                  int t2 = model->addTriangle( m_vertices[ row2 + x ].v, m_vertices[ row2 + x + 1].v, m_vertices[ row1 + x + 1].v );
                  m_triangles.push_back( t1 );
                  m_triangles.push_back( t2 );

                  if ( side >= 2 )
                  {
                     model->invertNormals( t1 );
                     model->invertNormals( t2 );
                  }
               }
            }
         }
      }

      unsigned count = m_vertices.size();
      for ( unsigned sv = 0; sv < count; sv++ )
      {
         model->selectVertex( m_vertices[sv].v );
      }

      parent->updateAllViews();

      model_status( model, StatusNormal, STATUSTIME_SHORT, "Cube created" );
   }
}

void CubeTool::mouseButtonUp( Parent * parent, int buttonState, int x, int y )
{
   if ( parent != m_parent )
   {
      log_error( "Can't serve two parents at once\n" );
   }

   if ( m_tracking )
   {
      m_tracking = false;

      bool invert = false;

      double x1 = m_x1;
      double y1 = m_y1;
      double z1 = m_z1;
      double x2 = 0.0;
      double y2 = 0.0;
      double z2 = 0.0;
      double xdiff = 0.0;
      double ydiff = 0.0;
      double zdiff = 0.0;

      bool haveXVal = parent->getXValue( x, y, &x2 );
      bool haveYVal = parent->getYValue( x, y, &y2 );
      parent->getZValue( x, y, &z2 );

      Model * model = parent->getModel();

      if ( !haveXVal )
      {
         ydiff = y2 - m_y1;
         zdiff = z2 - m_z1;

         _cubify( m_isCube, x1, xdiff, ydiff, zdiff );

         if ( m_y1 > y2 )
         {
            invert = !invert;
         }
         if ( z2 < m_z1 )
         {
            invert = !invert;
         }

      }
      else if ( !haveYVal )
      {
         xdiff = x2 - m_x1;
         zdiff = z2 - m_z1;

         _cubify( m_isCube, y1, ydiff, xdiff, zdiff );

         if ( m_x1 > x2 )
         {
            invert = !invert;
         }
         if ( z2 < m_z1 )
         {
            invert = !invert;
         }
      }
      else
      {
         xdiff = x2 - m_x1;
         ydiff = y2 - m_y1;

         _cubify( m_isCube, z1, zdiff, xdiff, ydiff );

         if ( m_y1 < y2 )
         {
            invert = !invert;
         }
         if ( x2 > m_x1 )
         {
            invert = !invert;
         }
      }

      updateVertexCoords( model, x1, y1, z1, xdiff, ydiff, zdiff );

      if ( invert != m_invertedNormals )
      {
         unsigned count = m_triangles.size();
         for ( unsigned t = 0; t < count; t++ )
         {
            model->invertNormals( m_triangles[t] );
         }

         m_invertedNormals = invert;
      }

      weldSelectedVertices( model );
      parent->updateAllViews();

      m_vertices.clear();
      m_triangles.clear();
   }
}

void CubeTool::mouseButtonMove( Parent * parent, int buttonState, int x, int y )
{
   if ( parent != m_parent )
   {
      log_error( "Can't serve two parents at once\n" );
   }

   if ( m_tracking )
   {
      bool invert = false;

      double x1 = m_x1;
      double y1 = m_y1;
      double z1 = m_z1;
      double x2 = 0.0;
      double y2 = 0.0;
      double z2 = 0.0;
      double xdiff = 0.0;
      double ydiff = 0.0;
      double zdiff = 0.0;

      bool haveXVal = parent->getXValue( x, y, &x2 );
      bool haveYVal = parent->getYValue( x, y, &y2 );
      parent->getZValue( x, y, &z2 );

      Model * model = parent->getModel();

      if ( !haveXVal )
      {
         ydiff = y2 - m_y1;
         zdiff = z2 - m_z1;

         _cubify( m_isCube, x1, xdiff, ydiff, zdiff );

         if ( m_y1 > y2 )
         {
            invert = !invert;
         }
         if ( z2 < m_z1 )
         {
            invert = !invert;
         }
      }
      else if ( !haveYVal )
      {
         xdiff = x2 - m_x1;
         zdiff = z2 - m_z1;

         _cubify( m_isCube, y1, ydiff, xdiff, zdiff );

         if ( m_x1 > x2 )
         {
            invert = !invert;
         }
         if ( z2 < m_z1 )
         {
            invert = !invert;
         }
      }
      else
      {
         xdiff = x2 - m_x1;
         ydiff = y2 - m_y1;

         _cubify( m_isCube, z1, zdiff, xdiff, ydiff );

         if ( m_y1 < y2 )
         {
            invert = !invert;
         }
         if ( x2 > m_x1 )
         {
            invert = !invert;
         }
      }

      updateVertexCoords( model, x1, y1, z1, xdiff, ydiff, zdiff );

      if ( invert != m_invertedNormals )
      {
         unsigned count = m_triangles.size();
         for ( unsigned t = 0; t < count; t++ )
         {
            model->invertNormals( m_triangles[t] );
         }

         m_invertedNormals = invert;
      }

      parent->updateAllViews();
   }
}

void CubeTool::updateVertexCoords( Model * model, double x, double y, double z,
      double xlen, double ylen, double zlen )
{
   vector<CubeVertices>::iterator it;

   for ( it = m_vertices.begin(); it != m_vertices.end(); it++ )
   {
      model->moveVertex( (*it).v, 
            (*it).coords[0]*xlen + x, 
            (*it).coords[1]*ylen + y, 
            (*it).coords[2]*zlen + z );
   }
}

void CubeTool::activated( int arg, Model * model, QMainWindow * mainwin )
{
   m_widget = new CubeToolWidget( this, mainwin );
#ifdef HAVE_QT4
   //mainwin->addDockWindow( m_widget, DockBottom );
#endif
   m_widget->show();
}

void CubeTool::deactivated()
{
   m_widget->close();
}

const char ** CubeTool::getPixmap()
{
   return (const char **) cubetool_xpm;
}

void CubeTool::setCubeValue( bool newValue )
{
   m_isCube = newValue;
   log_debug( "isCube = %s\n", newValue ? "true" : "false" );
}

void CubeTool::setSegmentValue( int newValue )
{
   if ( newValue >= 1 )
   {
      m_segments = newValue;
   }
}

