/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "selectfacetool.h"
#include "bounding.h"
#include "decalmgr.h"
#include "log.h"
#include "modelstatus.h"

#include "pixmap/selectfacetool.xpm"

#include <stdio.h>
#include <qnamespace.h>

SelectFaceTool::SelectFaceTool()
   : m_boundingBox( NULL),
     m_tracking( false ),
     m_unselect( false ),
     m_includeBackfacing( true ),
     m_startX( 0 ),
     m_startY( 0 ),
     m_x1( 0.0 ),
     m_y1( 0.0 ),
     m_z1( 0.0 ),
     m_selectionMode( Model::SelectTriangles ),
     m_widget( NULL )
{
}

SelectFaceTool::~SelectFaceTool()
{
}

void SelectFaceTool::activated( int arg, Model * model, QMainWindow * mainwin )
{
   m_widget = new SelectFaceToolWidget( (SelectFaceToolWidget::Observer *) this, (QWidget *) mainwin );
   m_widget->show();
}

void SelectFaceTool::deactivated()
{
   m_widget->close();
}

void SelectFaceTool::mouseButtonDown( Parent * parent, int buttonState, int x, int y )
{
   if ( m_tracking )
   {
      return;
   }

   parent->getModel()->setSelectionMode( Model::SelectTriangles );

   m_boundingBox = new BoundingBox();
   DecalManager::getInstance()->addDecalToParent( m_boundingBox, parent );

   if ( buttonState & BS_Right )
   {
      m_unselect = true;
   }
   else
   {
      m_unselect = false;
   }

   m_tracking = true;
   m_startX = x;
   m_startY = y;

   m_x1 = 0.0;
   m_y1 = 0.0;
   m_z1 = 0.0;

   parent->getXValue( x, y, &m_x1 );
   parent->getYValue( x, y, &m_y1 );
   parent->getZValue( x, y, &m_z1 );

   if ( ! m_unselect && ! (buttonState & BS_Shift) )
   {
      parent->getModel()->unselectAll();
   }

   parent->updateAllViews();
   model_status( parent->getModel(), StatusNormal, STATUSTIME_SHORT, "Starting selection" );
}

class NormalTest : public Model::SelectionTest
{
   public:
      NormalTest( Model * model, double x, double y, double z )
      {
         m_model = model;

         m_vect[0] = x;
         m_vect[1] = y;
         m_vect[2] = z;
      }

      // These should go into one of the Vector classes
      static double dot( const double v1[3], const double v2[3] )
      {
         return (v1[0] * v2[0]) + (v1[1] * v2[1]) + (v1[2] * v2[2]);
      }

      static double * cross( const double v1[3], const double v2[3], double result[3])
      {
         result[0] = v1[1] * v2[2] - v1[2] * v2[1];
         result[1] = v1[0] * v2[2] - v1[2] * v2[0];
         result[2] = v1[0] * v2[1] - v1[1] * v2[0];

         return result;
      }

      bool shouldSelect( void * element )
      {
         Model::Triangle * tri = static_cast<Model::Triangle *>( element );
         if (tri)
         {
            double v0[3];
            m_model->getVertexCoords( tri->m_vertexIndices[0], v0 );
            double v1[3];
            m_model->getVertexCoords( tri->m_vertexIndices[1], v1 );
            double v2[3];
            m_model->getVertexCoords( tri->m_vertexIndices[2], v2 );

            v1[0] -= v0[0]; v1[1] -= v0[1]; v1[2] -= v0[2];
            v2[0] -= v0[0]; v2[1] -= v0[1]; v2[2] -= v0[2];
            double normal[3];
            cross( v1, v2, normal );

            return ( dot( m_vect, normal ) > 0.0 );
         }
         else
         {
            return false;
         }
      }

   private:
      Model * m_model;
      double m_vect[3];
};

void SelectFaceTool::mouseButtonUp( Parent * parent, int buttonState, int x, int y )
{
   if ( m_unselect )
   {
      if ( buttonState & BS_Left )
      {
         // We're waiting for the right button
         return;
      }
   }
   else
   {
      if ( buttonState & BS_Right )
      {
         // We're waiting for the left button
         return;
      }
   }

   if ( m_tracking )
   {
      m_tracking = false;

      double x1 = m_x1;
      double y1 = m_y1;
      double z1 = m_z1;
      double x2 = 0.0;
      double y2 = 0.0;
      double z2 = 0.0;

      Model * model = parent->getModel();
      Model::SelectionTest * test = NULL;
      if ( !m_includeBackfacing )
      {
         Parent::ViewDirection direction = parent->getViewDirection();
         switch ( direction )
         {
            case Parent::ViewFront:
               test = new NormalTest( model, 0.0, 0.0, 1.0 );
               break;
            case Parent::ViewBack:
               test = new NormalTest( model, 0.0, 0.0, -1.0 );
               break;
            case Parent::ViewLeft:
               test = new NormalTest( model, 1.0, 0.0, 0.0 );
               break;
            case Parent::ViewRight:
               test = new NormalTest( model, -1.0, 0.0, 0.0 );
               break;
            case Parent::ViewTop:
               test = new NormalTest( model, 0.0, -1.0, 0.0 );
               break;
            case Parent::ViewBottom:
               test = new NormalTest( model, 0.0, 1.0, 0.0 );
               break;
            case Parent::View3d:
               // No selection in 3d view currently
               break;
         }
      }
      if ( parent->getXValue( x, y, &x2 ) )
      {
         if ( parent->getYValue( x, y, &y2 ) )
         {
            if ( m_unselect )
            {
               model->unselectInVolumeXY( x1, y1, x2, y2, test );
            }
            else
            {
               model->selectInVolumeXY( x1, y1, x2, y2, test );
            }
         }
         else if ( parent->getZValue( x, y, &z2 ) )
         {
            if ( m_unselect )
            {
               model->unselectInVolumeXZ( x1, z1, x2, z2, test );
            }
            else
            {
               model->selectInVolumeXZ( x1, z1, x2, z2, test );
            }
         }
      }
      else
      {
         if ( parent->getYValue( x, y, &y2 ) )
         {
            if ( parent->getZValue( x, y, &z2 ) )
            {
               if ( m_unselect )
               {
                  model->unselectInVolumeYZ( y1, z1, y2, z2, test );
               }
               else
               {
                  model->selectInVolumeYZ( y1, z1, y2, z2, test );
               }
            }
         }
      }

      DecalManager::getInstance()->removeDecal( m_boundingBox );
      m_boundingBox = NULL;

      parent->updateAllViews();
      model_status( parent->getModel(), StatusNormal, STATUSTIME_SHORT, "Selection complete" );
   }
}

void SelectFaceTool::mouseButtonMove( Parent * parent, int buttonState, int x, int y )
{
   if ( m_tracking )
   {
      double x1 = m_x1;
      double y1 = m_y1;
      double z1 = m_z1;
      double x2 = 0.0;
      double y2 = 0.0;
      double z2 = 0.0;

      parent->getXValue( x, y, &x2 );
      parent->getYValue( x, y, &y2 );
      parent->getZValue( x, y, &z2 );
      m_boundingBox->setBounds( x1, y1, z1, x2, y2, z2 );
      parent->updateView();
   }
}

const char ** SelectFaceTool::getPixmap()
{
   return (const char **) selectfacetool_xpm;
}

const char * SelectFaceTool::getName( int arg )
{
   return "Select Faces";
}

bool SelectFaceTool::getKeyBinding( int arg, int & keyBinding )
{
   keyBinding = Qt::Key_F;
   return true;
}

void SelectFaceTool::setBackfacingValue( bool newValue )
{
   m_includeBackfacing = newValue;
   log_debug( "includeBackfacing = %s\n", newValue ? "true" : "false" );
}


