/*  Misfit Model 3D
 * 
 *  Copyright (c) 2004-2005 Kevin Worcester
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
 *  USA.
 *
 *  See the COPYING file for full license text.
 */


#include "statusbar.h"
#include "mm3dport.h"

#include <qlabel.h>
#include <qpixmap.h>
#include <qtooltip.h>
#include <qtimer.h>

#include <stdarg.h>

using std::map;

map<Model *, StatusBar *> StatusBar::s_modelMap;

StatusBar::StatusBar( Model * model, QWidget * parent, const char * name )
   : StatusBarBase( parent, name ),
     m_model( model ),
     m_queueDisplay( false )
{
   s_modelMap[ m_model ] = this;
}

StatusBar::~StatusBar()
{
   s_modelMap.erase( m_model );
}

StatusBar * StatusBar::getStatusBarFromModel( Model * model )
{
   if ( s_modelMap.find( model ) != s_modelMap.end() )
   {
      return s_modelMap[ model ];
   }
   else
   {
      return NULL;
   }
}

void StatusBar::setModel( Model * model )
{
   s_modelMap.erase( m_model );
   m_model = model;
   s_modelMap[ m_model ] = this;
}

void StatusBar::setText( const QString & str )
{
   QToolTip::remove( this );
   if ( str.length() > 72 )
   {
      char * temp = strdup( str.latin1() );
      QToolTip::add( this, temp );
      strcpy( &(temp[70]), "..." );
      m_statusLabel->setText( temp );
      free( temp );
   }
   else
   {
      m_statusLabel->setText( str );
   }
}

void StatusBar::addText( StatusTypeE type, unsigned ms, const QString & str )
{
   if ( m_queueDisplay )
   {
      TextQueueItemT tqi;
      tqi.str  = str;
      tqi.ms   = ms;
      tqi.type = type;
      m_queue.push_back( tqi );
   }
   else
   {
      setText( str );
      QTimer::singleShot( ms, this, SLOT(timerExpired()));
      if ( type == StatusError ) 
      {
         m_statusLabel->setPaletteForegroundColor( QColor( 255, 255, 255 ) );
         m_statusLabel->setPaletteBackgroundColor( QColor( 255, 0, 0 ) );
      }
      m_queueDisplay = true;
   }
}

void StatusBar::timerExpired()
{
   m_statusLabel->unsetPalette();
   if ( !m_queue.empty() )
   {
      TextQueueItemT tqi = m_queue.front();
      m_queue.pop_front();

      setText( tqi.str );

      m_queueDisplay = true;
      if ( tqi.type == StatusError ) 
      {
         m_statusLabel->setPaletteForegroundColor( QColor( 255, 255, 255 ) );
         m_statusLabel->setPaletteBackgroundColor( QColor( 255, 0, 0 ) );
      }

      if ( tqi.ms > 0 )
      {
         QTimer::singleShot( tqi.ms, this, SLOT(timerExpired()));
      }
      else
      {
         timerExpired();
      }
   }
   else
   {
      m_queueDisplay = false;
   }
}

void StatusBar::setVertices( unsigned v, unsigned sv )
{
   QString str;
   if ( sv )
   {
      str.sprintf( "V:%d/%d", sv, v );
   }
   else
   {
      str.sprintf( "V:%d", v );
   }

   m_vertexLabel->setText( str );
}

void StatusBar::setFaces( unsigned f, unsigned sf )
{
   QString str;
   if ( sf )
   {
      str.sprintf( "F:%d/%d", sf, f );
   }
   else
   {
      str.sprintf( "F:%d", f );
   }

   m_faceLabel->setText( str );
}

void StatusBar::setGroups( unsigned g, unsigned sg )
{
   QString str;
   if ( sg )
   {
      str.sprintf( "G:%d/%d", sg, g );
   }
   else
   {
      str.sprintf( "G:%d", g );
   }

   m_groupLabel->setText( str );
}

void StatusBar::setBoneJoints( unsigned b, unsigned sb )
{
   QString str;
   if ( sb )
   {
      str.sprintf( "B:%d/%d", sb, b );
   }
   else
   {
      str.sprintf( "B:%d", b );
   }

   m_boneLabel->setText( str );
}

void StatusBar::setTextures( unsigned t, unsigned st )
{
   QString str;
   if ( st )
   {
      str.sprintf( "M:%d/%d", st, t );
   }
   else
   {
      str.sprintf( "M:%d", t );
   }

   m_textureLabel->setText( str );
}

extern "C" void model_status( Model * model, StatusTypeE type, unsigned ms, const char * fmt, ... )
{
   static char temp[1024];
   va_list ap;
   va_start( ap, fmt );
   PORT_vsnprintf( temp, sizeof(temp), fmt, ap );
   StatusBar * bar = StatusBar::getStatusBarFromModel( model );
   if ( bar )
   {
      bar->addText( type, ms, temp );
   }
}

