unit AsciiExportPlugin;

interface

uses windows,
    sysutils,
    mslib,
    msPlugin,
    dialogs,
    classes;

type
    TmsAsciiExportPlugin = class(TmsPlugin)
    public
        function Execute(pModel: PmsModel): integer; override;
    end;

var
    Plug: TmsAsciiExportPlugin;

implementation

function TmsAsciiExportPlugin.Execute(pModel: PmsModel): integer;
var
    str: TStringlist;
    i, j: integer;
    pmesh: PmsMesh;
    pVertex: PmsVertex;
    Vertex, normal, Position, Rotation: TmsVec3;
    uv: TmsVec2;
    pTriangle: PmsTriangle;
    nIndices: TWordArray;
    fname: string;
    name: array[0..MS_MAX_NAME] of char;
    pMaterial: PmsMaterial;
    Vec4: TmsVec4;
    Texture: array[0..MS_MAX_PATH] of char;
    pBone: PmsBone;
    pPosKey: PmsPositionKey;
    pRotKey: PmsRotationKey;
    nNumComments, nCommentLength: integer;
    Comment: array of char;
begin
    result := 0;
    if (msModel_GetMeshCount(pModel) = 0) then
    begin
        MessageBox(0, 'The model is empty!  Nothing exported!', '(Delphi SDK) MilkShape 3D ASCII Export', MB_OK or MB_ICONWARNING);
        exit;
    end;

    // choose filename
    with TsaveDialog.create(nil) do
    begin
        filter := 'MilkShape 3D ASCII Files (*.txt)|*.txt|All Files|*.*';
        Options := [ofHideReadOnly, ofPathMustExist, ofFileMustExist, ofEnableSizing];
        defaultext := 'txt';
        title := 'Export MilkShape 3D ASCII';
        try
            if execute then
            begin
                fname := filename;
            end
            else
                exit;
        finally
            free;
        end;
    end;

    str := TStringlist.create;

    with str do
    begin
        add('// MilkShape 3D ASCII');
        add('');
        add(format('Frames: %d', [msModel_GetTotalFrames(pModel)]));
        add(format('Frame: %d', [msModel_GetFrame(pModel)]));
        add('');
        add(format('Meshes: %d', [msModel_GetMeshCount(pModel)]));
        for i := 0 to msModel_GetMeshCount(pModel) - 1 do
        begin
            pMesh := msModel_GetMeshAt(pModel, i);
            msMesh_GetName(pMesh, @Name, MS_MAX_NAME);
            if strlen(Name) = 0 then
                strcopy(Name, ' ');
            add(format('"%s" %d %d', [Name, msMesh_GetFlags(pMesh), msMesh_GetMaterialIndex(pMesh)]));

            // vertices
            add(format('%d', [msMesh_GetVertexCount(pMesh)]));

            for j := 0 to msMesh_GetVertexCount(pMesh) - 1 do
            begin
                pVertex := msMesh_GetVertexAt(pMesh, j);

                msVertex_GetVertex(pVertex, Vertex);
                msVertex_GetTexCoords(pVertex, uv);

                add(format('%d %f %f %f %f %f %d',
                    [msVertex_GetFlags(pVertex),
                    Vertex[0], Vertex[1], Vertex[2],
                        uv[0], uv[1],
                        msVertex_GetBoneIndex(pVertex)]));
            end;

            // vertex normals
            add(format('%d', [msMesh_GetVertexNormalCount(pMesh)]));
            for j := 0 to msMesh_GetVertexNormalCount(pMesh) - 1 do
            begin
                msMesh_GetVertexNormalAt(pMesh, j, Normal);
                add(format('%f %f %f', [Normal[0], Normal[1], Normal[2]]));
            end;

            // triangles
            setlength(nIndices, 3);
            add(format('%d', [msMesh_GetTriangleCount(pMesh)]));
            for j := 0 to msMesh_GetTriangleCount(pMesh) - 1 do
            begin
                pTriangle := msMesh_GetTriangleAt(pMesh, j);

                msTriangle_GetVertexIndices(pTriangle, nIndices);

                add(format('%d %d %d %d %d %d %d %d',
                    [msTriangle_GetFlags(pTriangle),
                    nIndices[0], nIndices[1], nIndices[2],
                        pTriangle^.nNormalIndices[0], pTriangle^.nNormalIndices[1], pTriangle^.nNormalIndices[2],
                        msTriangle_GetSmoothingGroup(pTriangle)])
                        );
            end;
            setlength(nIndices, 0);
        end;

        // materials
        add('');
        add(format('Materials: %d', [msModel_GetMaterialCount(pModel)]));
        for i := 0 to msModel_GetMaterialCount(pModel) - 1 do
        begin
            pMaterial := msModel_GetMaterialAt(pModel, i);
            msMaterial_GetName(pMaterial, Name, MS_MAX_NAME);
            if strlen(Name) = 0 then
                strcopy(Name, ' ');
            add(format('"%s"', [Name]));

            msMaterial_GetAmbient(pMaterial, vec4);
            add(format('%f %f %f %f', [vec4[0], vec4[1], vec4[2], vec4[3]]));
            msMaterial_GetDiffuse(pMaterial, vec4);
            add(format('%f %f %f %f', [vec4[0], vec4[1], vec4[2], vec4[3]]));
            msMaterial_GetSpecular(pMaterial, vec4);
            add(format('%f %f %f %f', [vec4[0], vec4[1], vec4[2], vec4[3]]));
            msMaterial_GetEmissive(pMaterial, vec4);
            add(format('%f %f %f %f', [vec4[0], vec4[1], vec4[2], vec4[3]]));
            add(format('%f', [msMaterial_GetShininess(pMaterial)]));
            add(format('%f', [msMaterial_GetTransparency(pMaterial)]));

            msMaterial_GetDiffuseTexture(pMaterial, Texture, MS_MAX_PATH);
            add(format('"%s"', [Texture]));
            msMaterial_GetAlphaTexture(pMaterial, Texture, MS_MAX_PATH);
            add(format('"%s"', [Texture]));
        end;

        // bones
        add('');
        add(format('Bones: %d', [msModel_GetBoneCount(pModel)]));
        for i := 0 to msModel_GetBoneCount(pModel) - 1 do
        begin
            pBone := msModel_GetBoneAt(pModel, i);
            msBone_GetName(pBone, Name, MS_MAX_NAME);
            if (strlen(Name) = 0) then
                strcopy(Name, ' ');
            add(format('"%s"', [Name]));
            msBone_GetParentName(pBone, Name, MS_MAX_NAME);
            add(format('"%s"', [Name]));
            msBone_GetPosition(pBone, Position);
            msBone_GetRotation(pBone, Rotation);
            add(format('%d %f %f %f %f %f %f',
                [msBone_GetFlags(pBone),
                Position[0], Position[1], Position[2],
                    Rotation[0], Rotation[1], Rotation[2]]));

            add(format('%d', [msBone_GetPositionKeyCount(pBone)]));
            for j := 0 to msBone_GetPositionKeyCount(pBone) - 1 do
            begin
                pPosKey := msBone_GetPositionKeyAt(pBone, j);

                add(format('%f %f %f %f',
                    [pPosKey^.fTime, pPosKey^.Position[0], pPosKey^.Position[1], pPosKey^.Position[2]]));
            end;

            add(format('%d', [msBone_GetRotationKeyCount(pBone)]));
            for j := 0 to msBone_GetRotationKeyCount(pBone) - 1 do
            begin
                pRotKey := msBone_GetRotationKeyAt(pBone, j);
                add(format('%f %f %f %f',
                    [pRotKey^.fTime, pRotKey^.Rotation[0], pRotKey^.Rotation[1], pRotKey^.Rotation[2]]));
            end;
        end;

        // group comments
        nNumComments := 0;
        for i := 0 to msModel_GetMeshCount(pModel) - 1 do
        begin
            pMesh := msModel_GetMeshAt(pModel, i);
            nCommentLength := msMesh_GetComment(pMesh, nil, 0);
            if (nCommentLength > 0) then
                inc(nNumComments);
        end;

        add(format('GroupComments: %d', [nNumComments]));

        for i := 0 to msModel_GetMeshCount(pModel) - 1 do
        begin
            pMesh := msModel_GetMeshAt(pModel, i);
            nCommentLength := msMesh_GetComment(pMesh, nil, 0);
            if (nCommentLength > 0) then
            begin
                add(format('%d', [i]));
                setlength(comment, nCommentLength + 1);
                msMesh_GetComment(pMesh, @Comment, nCommentLength);
                Comment[nCommentLength] := #0;
                // while (true)
                // {
                //    char * p = strstr(pszComment, "\r\n");
                //    if (p)
                //    {
                //    p[0] = '\\';
                //    p[1] = 'n';
                // }
                // else
                //    break;
                add(format('%s', [Comment]));
                setlength(Comment, 0);
            end;
        end;

        // material comments
        nNumComments := 0;
        for i := 0 to msModel_GetMaterialCount(pModel) - 1 do
        begin
            pMaterial := msModel_GetMaterialAt(pModel, i);
            nCommentLength := msMaterial_GetComment(pMaterial, nil, 0);
            if (nCommentLength > 0) then
                inc(nNumComments);
        end;

        add(format('MaterialComments: %d', [nNumComments]));

        for i := 0 to msModel_GetMaterialCount(pModel) - 1 do
        begin
            pMaterial := msModel_GetMaterialAt(pModel, i);
            nCommentLength := msMaterial_GetComment(pMaterial, nil, 0);
            if (nCommentLength > 0) then
            begin
                add(format('%d', [i]));
                setlength(comment, nCommentLength + 1);
                msMaterial_GetComment(pMaterial, @Comment, nCommentLength);
                Comment[nCommentLength] := #0;
                // while (true)
                // {
                //    char * p = strstr(pszComment, "\r\n");
                //    if (p)
                //    {
                //    p[0] = '\\';
                //    p[1] = 'n';
                // }
                // else
                //    break;
                add(format('%s', [Comment]));
                setlength(comment, 0);
            end;
        end;

        // joint comments
        nNumComments := 0;
        for i := 0 to msModel_GetBoneCount(pModel) - 1 do
        begin
            pBone := msModel_GetBoneAt(pModel, i);
            nCommentLength := msBone_GetComment(pBone, nil, 0);
            if (nCommentLength > 0) then
                inc(nNumComments);
        end;

        add(format('BoneComments: %d', [nNumComments]));

        for i := 0 to msModel_GetBoneCount(pModel) - 1 do
        begin
            pBone := msModel_GetBoneAt(pModel, i);
            nCommentLength := msBone_GetComment(pBone, nil, 0);
            if (nCommentLength > 0) then
            begin
                add(format('%d', [i]));
                setlength(comment, nCommentLength + 1);
                msBone_GetComment(pBone, @Comment, nCommentLength);
                Comment[nCommentLength] := #0;
                // while (true)
                // {
                //    char * p = strstr(pszComment, "\r\n");
                //    if (p)
                //    {
                //    p[0] = '\\';
                //    p[1] = 'n';
                // }
                // else
                //    break;
                add(format('%s', [Comment]));
                setlength(comment, 0);
            end;
        end;

        // model comments
        nNumComments := 0;
        nCommentLength := msModel_GetComment(pModel, nil, 0);
        if (nCommentLength > 0) then
            nNumComments := 1;
        add(format('ModelComment: %d', [nNumComments]));

        if (nCommentLength > 0) then
        begin
            setlength(comment, nCommentLength + 1);
            msModel_GetComment(pModel, @Comment, nCommentLength);
            Comment[nCommentLength] := #0;
            // while (true)
            // {
            //    char * p = strstr(pszComment, "\r\n");
            //    if (p)
            //    {
            //    p[0] = '\\';
            //    p[1] = 'n';
            // }
            // else
            //    break;
            add(format('%s', [Comment]));
            setlength(comment, 0);
        end;

        // dont' forget to destroy the model
        msModel_Destroy(pModel);
    end;
    showmessage('finished');
    str.savetofile(fname);
    str.free;
end;

end.

