#include <stdbool.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <dirent.h>
#include <sys/stat.h>
#include <time.h>
#include <math.h>
#include <ctype.h>//isdigit

#define MAX_NAME 32

// Quake II valid flag and content masks
#define Q2_FLAG_MASK     0x000003FF   // Light, Sky, Trans33, Trans66, Warp, Flowing, NoDraw, Hint, Skip
#define Q2_CONTENT_MASK  0x0000003F   // Solid, Window, Aux, Lava, Slime, Water, Mist, AreaPortal

// Quake II palette
static const uint8_t quake2_palette[256][3] = {
    {  0,   0,   0},
    { 15,  15,  15},
    { 31,  31,  31},
    { 47,  47,  47},
    { 63,  63,  63},
    { 75,  75,  75},
    { 91,  91,  91},
    {107, 107, 107},
    {123, 123, 123},
    {139, 139, 139},
    {155, 155, 155},
    {171, 171, 171},
    {187, 187, 187},
    {203, 203, 203},
    {219, 219, 219},
    {235, 235, 235},
    { 99,  75,  35},
    { 91,  67,  31},
    { 83,  63,  31},
    { 79,  59,  27},
    { 71,  55,  27},
    { 63,  47,  23},
    { 59,  43,  23},
    { 51,  39,  19},
    { 47,  35,  19},
    { 43,  31,  19},
    { 39,  27,  15},
    { 35,  23,  15},
    { 27,  19,  11},
    { 23,  15,  11},
    { 19,  15,   7},
    { 15,  11,   7},
    { 95,  95, 111},
    { 91,  91, 103},
    { 91,  83,  95},
    { 87,  79,  91},
    { 83,  75,  83},
    { 79,  71,  75},
    { 71,  63,  67},
    { 63,  59,  59},
    { 59,  55,  55},
    { 51,  47,  47},
    { 47,  43,  43},
    { 39,  39,  39},
    { 35,  35,  35},
    { 27,  27,  27},
    { 23,  23,  23},
    { 19,  19,  19},
    {143, 119,  83},
    {123,  99,  67},
    {115,  91,  59},
    {103,  79,  47},
    {207, 151,  75},
    {167, 123,  59},
    {139, 103,  47},
    {111,  83,  39},
    {235, 159,  39},
    {203, 139,  35},
    {175, 119,  31},
    {147,  99,  27},
    {119,  79,  23},
    { 91,  59,  15},
    { 63,  39,  11},
    { 35,  23,   7},
    {167,  59,  43},
    {159,  47,  35},
    {151,  43,  27},
    {139,  39,  19},
    {127,  31,  15},
    {115,  23,  11},
    {103,  23,   7},
    { 87,  19,   0},
    { 75,  15,   0},
    { 67,  15,   0},
    { 59,  15,   0},
    { 51,  11,   0},
    { 43,  11,   0},
    { 35,  11,   0},
    { 27,   7,   0},
    { 19,   7,   0},
    {123,  95,  75},
    {115,  87,  67},
    {107,  83,  63},
    {103,  79,  59},
    { 95,  71,  55},
    { 87,  67,  51},
    { 83,  63,  47},
    { 75,  55,  43},
    { 67,  51,  39},
    { 63,  47,  35},
    { 55,  39,  27},
    { 47,  35,  23},
    { 39,  27,  19},
    { 31,  23,  15},
    { 23,  15,  11},
    { 15,  11,   7},
    {111,  59,  23},
    { 95,  55,  23},
    { 83,  47,  23},
    { 67,  43,  23},
    { 55,  35,  19},
    { 39,  27,  15},
    { 27,  19,  11},
    { 15,  11,   7},
    {179,  91,  79},
    {191, 123, 111},
    {203, 155, 147},
    {215, 187, 183},
    {203, 215, 223},
    {179, 199, 211},
    {159, 183, 195},
    {135, 167, 183},
    {115, 151, 167},
    { 91, 135, 155},
    { 71, 119, 139},
    { 47, 103, 127},
    { 23,  83, 111},
    { 19,  75, 103},
    { 15,  67,  91},
    { 11,  63,  83},
    {  7,  55,  75},
    {  7,  47,  63},
    {  7,  39,  51},
    {  0,  31,  43},
    {  0,  23,  31},
    {  0,  15,  19},
    {  0,   7,  11},
    {  0,   0,   0},
    {139,  87,  87},
    {131,  79,  79},
    {123,  71,  71},
    {115,  67,  67},
    {107,  59,  59},
    { 99,  51,  51},
    { 91,  47,  47},
    { 87,  43,  43},
    { 75,  35,  35},
    { 63,  31,  31},
    { 51,  27,  27},
    { 43,  19,  19},
    { 31,  15,  15},
    { 19,  11,  11},
    { 11,   7,   7},
    {  0,   0,   0},
    {151, 159, 123},
    {143, 151, 115},
    {135, 139, 107},
    {127, 131,  99},
    {119, 123,  95},
    {115, 115,  87},
    {107, 107,  79},
    { 99,  99,  71},
    { 91,  91,  67},
    { 79,  79,  59},
    { 67,  67,  51},
    { 55,  55,  43},
    { 47,  47,  35},
    { 35,  35,  27},
    { 23,  23,  19},
    { 15,  15,  11},
    {159,  75,  63},
    {147,  67,  55},
    {139,  59,  47},
    {127,  55,  39},
    {119,  47,  35},
    {107,  43,  27},
    { 99,  35,  23},
    { 87,  31,  19},
    { 79,  27,  15},
    { 67,  23,  11},
    { 55,  19,  11},
    { 43,  15,   7},
    { 31,  11,   7},
    { 23,   7,   0},
    { 11,   0,   0},
    {  0,   0,   0},
    {119, 123, 207},
    {111, 115, 195},
    {103, 107, 183},
    { 99,  99, 167},
    { 91,  91, 155},
    { 83,  87, 143},
    { 75,  79, 127},
    { 71,  71, 115},
    { 63,  63, 103},
    { 55,  55,  87},
    { 47,  47,  75},
    { 39,  39,  63},
    { 35,  31,  47},
    { 27,  23,  35},
    { 19,  15,  23},
    { 11,   7,   7},
    {155, 171, 123},
    {143, 159, 111},
    {135, 151,  99},
    {123, 139,  87},
    {115, 131,  75},
    {103, 119,  67},
    { 95, 111,  59},
    { 87, 103,  51},
    { 75,  91,  39},
    { 63,  79,  27},
    { 55,  67,  19},
    { 47,  59,  11},
    { 35,  47,   7},
    { 27,  35,   0},
    { 19,  23,   0},
    { 11,  15,   0},
    {  0, 255,   0},
    { 35, 231,  15},
    { 63, 211,  27},
    { 83, 187,  39},
    { 95, 167,  47},
    { 95, 143,  51},
    { 95, 123,  51},
    {255, 255, 255},
    {255, 255, 211},
    {255, 255, 167},
    {255, 255, 127},
    {255, 255,  83},
    {255, 255,  39},
    {255, 235,  31},
    {255, 215,  23},
    {255, 191,  15},
    {255, 171,   7},
    {255, 147,   0},
    {239, 127,   0},
    {227, 107,   0},
    {211,  87,   0},
    {199,  71,   0},
    {183,  59,   0},
    {171,  43,   0},
    {155,  31,   0},
    {143,  23,   0},
    {127,  15,   0},
    {115,   7,   0},
    { 95,   0,   0},
    { 71,   0,   0},
    { 47,   0,   0},
    { 27,   0,   0},
    {239,   0,   0},
    { 55,  55, 255},
    {255,   0,   0},
    {  0,   0, 255},
    { 43,  43,  35},
    { 27,  27,  23},
    { 19,  19,  15},
    {235, 151, 127},
    {195, 115,  83},
    {159,  87,  51},
    {123,  63,  27},
    {235, 211, 199},
    {199, 171, 155},
    {167, 139, 119},
    {135, 107,  87},
    {159,  91,  83}
};
// Kingpin palette
static const uint8_t kingpin_palette[256][3] = {
    {  0,   0,   0},
    { 15,  15,  15},
    { 31,  31,  31},
    { 47,  47,  47},
    { 63,  63,  63},
    { 75,  75,  75},
    { 91,  91,  91},
    {107, 107, 107},
    {123, 123, 123},
    {139, 139, 139},
    {155, 155, 155},
    {171, 171, 171},
    {187, 187, 187},
    {203, 203, 203},
    {219, 219, 219},
    {235, 235, 235},
    {219, 222, 255},
    {207, 210, 241},
    {195, 197, 227},
    {182, 185, 212},
    {170, 172, 198},
    {157, 159, 184},
    {145, 147, 169},
    {133, 134, 155},
    {120, 122, 141},
    {108, 109, 126},
    { 95,  96, 112},
    { 83,  84,  98},
    { 71,  71,  83},
    { 58,  59,  69},
    { 46,  46,  55},
    { 33,  33,  40},
    {206, 173, 156},
    {200, 167, 149},
    {193, 160, 142},
    {186, 153, 134},
    {179, 147, 127},
    {172, 140, 120},
    {166, 133, 112},
    {159, 126, 105},
    {152, 120,  98},
    {145, 113,  90},
    {138, 106,  83},
    {131,  99,  75},
    {127,  91,  71},
    {121,  87,  68},
    {114,  82,  64},
    {107,  77,  60},
    {101,  72,  57},
    { 94,  68,  53},
    { 87,  63,  49},
    { 81,  58,  45},
    { 74,  53,  42},
    { 67,  48,  38},
    { 61,  44,  34},
    { 54,  39,  30},
    { 47,  34,  27},
    { 41,  29,  23},
    { 34,  24,  19},
    { 27,  20,  15},
    { 21,  15,  12},
    { 14,  10,   8},
    {  7,   5,   4},
    {  0,   0,   0},
    {100, 130, 226},
    { 99, 127, 219},
    { 95, 127, 211},
    { 91, 123, 203},
    { 87, 119, 195},
    { 83, 115, 187},
    { 83, 115, 179},
    { 79, 111, 171},
    { 75, 107, 163},
    { 71, 103, 155},
    { 67,  99, 147},
    { 63,  95, 139},
    { 59,  91, 131},
    { 59,  87, 123},
    { 55,  83, 115},
    { 51,  79, 107},
    { 48,  73,  99},
    { 47,  67,  91},
    { 47,  63,  87},
    { 43,  59,  79},
    { 39,  55,  75},
    { 39,  47,  67},
    { 35,  43,  63},
    { 31,  39,  55},
    { 31,  35,  51},
    { 27,  31,  43},
    { 23,  27,  39},
    { 19,  23,  31},
    { 15,  19,  27},
    { 15,  15,  23},
    { 11,  11,  15},
    {  7,   7,  11},
    {165, 196, 125},
    {159, 191, 123},
    {151, 187, 115},
    {147, 179, 111},
    {139, 171, 107},
    {131, 167, 103},
    {127, 159,  99},
    {119, 155,  95},
    {115, 147,  91},
    {107, 143,  87},
    {103, 135,  83},
    { 95, 131,  79},
    { 91, 123,  75},
    { 83, 119,  71},
    { 79, 111,  67},
    { 75, 107,  63},
    { 71,  99,  59},
    { 67,  91,  55},
    { 63,  87,  51},
    { 59,  79,  51},
    { 55,  75,  47},
    { 51,  71,  43},
    { 47,  63,  43},
    { 43,  59,  39},
    { 39,  51,  35},
    { 35,  47,  31},
    { 31,  43,  27},
    { 27,  35,  23},
    { 23,  31,  23},
    { 19,  23,  19},
    { 15,  19,  15},
    {  8,  12,   8},
    {175,  31,   7},
    {163,  27,   7},
    {155,  23,   7},
    {143,  19,   7},
    {131,  15,   0},
    {123,  15,   0},
    {111,  11,   0},
    {103,   7,   0},
    { 91,   7,   0},
    { 83,   7,   0},
    { 71,   0,   0},
    { 63,   0,   0},
    { 51,   0,   0},
    { 43,   0,   0},
    { 31,   0,   0},
    { 22,   0,   0},
    {171,  91,  35},
    {159,  87,  35},
    {147,  83,  35},
    {139,  79,  35},
    {127,  75,  35},
    {119,  71,  35},
    {107,  67,  35},
    { 95,  63,  31},
    { 87,  55,  31},
    { 75,  51,  27},
    { 67,  43,  23},
    { 55,  39,  23},
    { 43,  31,  19},
    { 35,  23,  15},
    { 23,  15,  11},
    { 15,  11,   7},
    {170, 147, 135},
    {165, 143, 131},
    {160, 138, 127},
    {154, 133, 122},
    {149, 129, 118},
    {143, 124, 114},
    {138, 119, 109},
    {132, 114, 105},
    {127, 110, 101},
    {121, 105,  96},
    {116, 100,  92},
    {110,  95,  88},
    {105,  91,  83},
    { 99,  86,  79},
    { 94,  81,  75},
    { 88,  76,  70},
    { 83,  72,  66},
    { 77,  67,  61},
    { 72,  62,  57},
    { 66,  57,  53},
    { 61,  53,  48},
    { 55,  48,  44},
    { 50,  43,  40},
    { 44,  38,  35},
    { 39,  34,  31},
    { 33,  29,  27},
    { 28,  24,  22},
    { 22,  19,  18},
    { 17,  15,  14},
    { 11,  10,   9},
    {  6,   5,   5},
    {  0,   0,   0},
    {255, 255, 255},
    {255, 244, 213},
    {255, 231, 173},
    {255, 221, 137},
    {255, 211, 102},
    {255, 204,  67},
    {252, 190,  48},
    {249, 185,  36},
    {247, 176,  23},
    {244, 174,  13},
    {234, 155,  10},
    {229, 148,  10},
    {224, 138,  10},
    {219, 131,  10},
    {216, 125,   8},
    {206, 113,   8},
    {201, 107,   8},
    {198,  98,   6},
    {193,  92,   5},
    {192,  90,   3},
    {173,  75,   5},
    {165,  65,   3},
    {150,  54,   3},
    {140,  48,   2},
    {124,  39,   3},
    {114,  32,   3},
    { 99,  25,   3},
    { 89,  20,   2},
    { 73,  15,   3},
    { 66,   9,   0},
    { 51,   5,   0},
    { 45,   5,   0},
    {147, 179, 255},
    { 91, 135, 239},
    { 39,  95, 223},
    {  0,  56, 209},
    {255, 188, 190},
    {247, 123, 123},
    {243,  59,  63},
    {239,   0,   7},
    {143, 255, 139},
    { 95, 247,  87},
    { 55, 239,  43},
    { 12, 229,   0},
    {255, 246, 168},
    {255, 235, 147},
    {255, 223, 123},
    {255, 211, 103},
    { 75, 235, 203},
    { 79, 211, 171},
    { 83, 187, 143},
    { 83, 163, 123},
    { 79, 139, 103},
    { 71, 115,  83},
    { 63,  91,  67},
    { 51,  67,  51},
    { 32,  42,  35},
    {235, 183, 255},
    {198, 150, 220},
    {161, 116, 184},
    {123,  83, 148},
    { 86,  49, 112},
    { 48,  15,  76},
    {252,  74, 193}
};


// --- WAL header ---
typedef struct {
    char name[MAX_NAME];
    uint32_t width;
    uint32_t height;
    uint32_t offsets[4];
    char nextName[MAX_NAME];
    uint32_t flags;
    uint32_t contents;
    uint32_t value;
} miptex_t;

// --- TGA header (24-bit uncompressed only) ---
#pragma pack(push,1)
typedef struct {
    uint8_t idLength;
    uint8_t colorMapType;
    uint8_t imageType;
    uint16_t cmFirstEntry;
    uint16_t cmLength;
    uint8_t cmEntrySize;
    uint16_t xOrigin;
    uint16_t yOrigin;
    uint16_t width;
    uint16_t height;
    uint8_t pixelDepth;
    uint8_t imageDesc;
} TGAHeader;
#pragma pack(pop)

// --- PAK header ---
typedef struct {
    char name[56];
    int32_t filepos;
    int32_t filelen;
} pakentry_t;

TGAHeader th;

// Global flags
char rootpath[1024] = ".";
int debug = 0;                 // FREDZ reduce log spam

int iniFound = 0;              // set when any textureinfo.ini is successfully opened
int skipIni = 0;               // default: read INI
int onlyIni = 0;               // default: not restricted to INI entries

int skipAlpha = 0;             // default: use alpha channel when present
int onlyAlpha = 0;             // default: not alpha-only

int q2FlagsOnly = 0;           // default: allow all flags
int requirePowerOf2 = 0;       // default: no PoT requirement

int alphaThreshold = 255;      // Quake II: alpha < 255 -> transparent
int useAlphaThreshold = 0;     // set only if -alphathresh is passed

int useSemiAlphaDetect = 0;
int alphaSemiMax = 127;        // default for -alphathreshsemi

int deletedMode = 0;           // 0 = normal convert, 1 = delete .wal, 2 = delete .wal_json, 3 = delete both
int scanMode = 0;              // 0 = no scan, 1 = -scantga, 2 = -scanini, 3 = -scanalpha
int fixMode = 0;

int skipoverwrite = 0;

int walJsonPerTexture = 0;     // 1 = write per-texture .wal_json sidecars
int use_q2_palette = 0;
int use_kp_palette = 0;

void load_quake2_palette(uint8_t palette[256][3]) {
    memcpy(palette, quake2_palette, sizeof(quake2_palette));
}

void load_kingpin_palette(uint8_t palette[256][3]) {
    memcpy(palette, kingpin_palette, sizeof(kingpin_palette));
}


int extract_colormap_from_pak(const char *pakfile, uint8_t palette[256][3]) {
    FILE *f = fopen(pakfile, "rb");
    if (!f) return -1;

    // Read header
    char magic[4];
    int32_t diroffset, dirlen;
    fread(magic, 1, 4, f);
    fread(&diroffset, 4, 1, f);
    fread(&dirlen, 4, 1, f);

    if (strncmp(magic, "PACK", 4) != 0) {
        fclose(f);
        return -1;
    }

    int numfiles = dirlen / sizeof(pakentry_t);
    pakentry_t *entries = malloc(dirlen);
    fseek(f, diroffset, SEEK_SET);
    fread(entries, sizeof(pakentry_t), numfiles, f);

    // Look for pics/colormap.pcx
    for (int i = 0; i < numfiles; i++) {
        if (strcmp(entries[i].name, "pics/colormap.pcx") == 0) {
            // Read file into buffer
            uint8_t *buf = malloc(entries[i].filelen);
            fseek(f, entries[i].filepos, SEEK_SET);
            fread(buf, 1, entries[i].filelen, f);

            // Parse PCX palette (last 769 bytes: 0x0C + 256*3)
            if (entries[i].filelen >= 769 && buf[entries[i].filelen - 769] == 0x0C) {
                memcpy(palette, buf + entries[i].filelen - 768, 256 * 3);
                free(buf);
                free(entries);
                fclose(f);
                return 0; // success
            }

            free(buf);
            break;
        }
    }

    free(entries);
    fclose(f);
    return -1; // not found
}

// --- Load palette from colormap.pcx ---
int load_or_extract_colormap(uint8_t palette[256][3]) {
	static const char *paths[] = {
		"colormap.pcx",                // current directory
		"main/pics/colormap.pcx",      // Kingpin
		"baseq2/pics/colormap.pcx"     // Quake II
	};

	// --- Try disk paths first ---
	for (int i = 0; i < 3; i++) {
		FILE *f = fopen(paths[i], "rb");
		if (!f)
			continue;
		fseek(f, -769, SEEK_END);
		uint8_t marker = fgetc(f);
		if (marker == 0x0C && fread(palette, 3, 256, f) == 256) {
			fclose(f);
			printf("Loaded palette from %s\n", paths[i]);
			return 0;
		}
		fclose(f);
	}

	// --- Try pak0.pak extraction ---
	if (extract_colormap_from_pak("main/pak0.pak", palette) == 0) {
		printf("Extracted colormap.pcx from main/pak0.pak\n");
		return 0;
	}
	if (extract_colormap_from_pak("baseq2/pak0.pak", palette) == 0) {
		printf("Extracted colormap.pcx from baseq2/pak0.pak\n");
		return 0;
	}

	// --- If all fail ---
	fprintf(stderr,
		"Error: Could not locate a valid 'colormap.pcx'.\n"
		"Searched in:\n"
		"  - Current directory\n"
		"  - main/pics/\n"
		"  - baseq2/pics/\n"
		"  - main/pak0.pak\n"
		"  - baseq2/pak0.pak\n"
		);
	return -1;
}

// --- RGB to palette index ---
int rgb_to_index(uint8_t r, uint8_t g, uint8_t b, uint8_t palette[256][3]) {
	int best = 0, bestDist = 999999;
	for (int i = 0; i < 256; i++) {
		int dr = r - palette[i][0];
		int dg = g - palette[i][1];
		int db = b - palette[i][2];
		int dist = dr*dr + dg*dg + db*db;
		if (dist < bestDist) { bestDist = dist; best = i; }
	}
	return best;
}

// --- Apply textureinfo.ini values ---
int apply_textureinfo(const char *dir, const char *texname, miptex_t *wal) {
	char inipath[512];
	snprintf(inipath, sizeof(inipath), "%s/textureinfo.ini", dir);

	FILE *f = fopen(inipath, "r");
	if (!f)
        return 0;

    iniFound = 1;  // INI exists in this directory

	if (debug)
		printf("Applying textureinfo.ini from %s\n", dir);

	// Normalize slashes and extract basename
	char cleanname[MAX_NAME];
	strncpy(cleanname, texname, MAX_NAME);
	cleanname[MAX_NAME - 1] = '\0';
	for (int i = 0; cleanname[i]; i++) {
		if (cleanname[i] == '\\') cleanname[i] = '/';
	}
	const char *basename = strrchr(cleanname, '/');
	if (!basename) basename = cleanname;
	else basename++; // skip the slash

	char line[256];
	int match = 0;
	int found = 0;
	while (fgets(line, sizeof(line), f)) {
		line[strcspn(line, "\r\n")] = 0; // trim newline

		if (line[0] == '[') {
			char section[64];
			sscanf(line, "[%63[^]]]", section);
			match = (strcmp(section, basename) == 0);
			if (match) {
				found = 1;
				printf("Matched [%s] in %s/textureinfo.ini\n", section, dir);
			}
		}
		else if (match) {
			unsigned int val;
			if (sscanf(line, "Flags=%u", &val) == 1) {
				wal->flags = val;
				printf("  -> Flags = %u\n", val);
				continue;
			}
			if (sscanf(line, "Value=%u", &val) == 1) {
				wal->value = val;
				printf("  -> Value = %u\n", val);
				continue;
			}
			if (sscanf(line, "Contents=%u", &val) == 1) {
				wal->contents = val;
				printf("  -> Contents = %u\n", val);
				continue;
			}
		}
	}
	fclose(f);
	return found;
}

void generate_mipmap(uint8_t *src, int w, int h, uint8_t *dst, uint8_t palette[256][3])
{
	int nw = w / 2, nh = h / 2;
	for (int y = 0; y < nh; y++) {
		for (int x = 0; x < nw; x++) {
			double r = 0, g = 0, b = 0;
			for (int dy = 0; dy < 2; dy++) {
				for (int dx = 0; dx < 2; dx++) {
					int idx = src[(y * 2 + dy)*w + (x * 2 + dx)];
					double lr = pow(palette[idx][0] / 255.0, 2.2);
					double lg = pow(palette[idx][1] / 255.0, 2.2);
					double lb = pow(palette[idx][2] / 255.0, 2.2);
					r += lr; g += lg; b += lb;
				}
			}
			r = pow(r / 4.0, 1.0 / 2.2);
			g = pow(g / 4.0, 1.0 / 2.2);
			b = pow(b / 4.0, 1.0 / 2.2);
			dst[y*nw + x] = rgb_to_index((int)(r * 255), (int)(g * 255), (int)(b * 255), palette);
		}
	}
}

void flip_vertical(uint8_t *data, int width, int height) {
	for (int y = 0; y < height / 2; y++) {
		uint8_t *row_top = data + y * width;
		uint8_t *row_bottom = data + (height - 1 - y) * width;
		for (int x = 0; x < width; x++) {
			uint8_t temp = row_top[x];
			row_top[x] = row_bottom[x];
			row_bottom[x] = temp;
		}
	}
}

int textureinfo_has_entry(const char *dir, const char *texname)
{
    char inipath[512];
    snprintf(inipath, sizeof(inipath), "%s/textureinfo.ini", dir);
    FILE *f = fopen(inipath, "r");
    if (!f)
        return 0; // no INI in this directory

    iniFound = 1;  // INI exists in this directory

    // Normalize slashes and extract basename
    char cleanname[MAX_NAME];
    strncpy(cleanname, texname, MAX_NAME);
    cleanname[MAX_NAME - 1] = 0;
    for (int i = 0; cleanname[i]; i++) {
        if (cleanname[i] == '\\')
            cleanname[i] = '/';
    }

    const char *basename = strrchr(cleanname, '/');
    if (!basename)
        basename = cleanname;
    else
        basename++; // skip the slash

    // Strip extension so INI sections use names without .tga/.wal
    char baseNoExt[MAX_NAME];
    strncpy(baseNoExt, basename, MAX_NAME - 1);
    baseNoExt[MAX_NAME - 1] = 0;
    char *dot = strrchr(baseNoExt, '.');
    if (dot)
        *dot = '\0';

    char line[256];
    while (fgets(line, sizeof(line), f)) {
        line[strcspn(line, "\r\n")] = 0;
        if (line[0] == '[') {
            char section[64];
            if (sscanf(line, "[%63[^]]", section) == 1) {
                if (strcmp(section, baseNoExt) == 0) {
                    fclose(f);
                    return 1; // found
                }
            }
        }
    }

    fclose(f);
    return 0; // not found
}


int scan_tga(const char *filename)
{
	FILE *f = fopen(filename, "rb");
	if (!f) {
		fprintf(stderr, "Error: cannot open %s\n", filename);
		return 0;
	}

	if (fread(&th, sizeof(TGAHeader), 1, f) != 1) {
		fprintf(stderr, "Error: %s truncated (cannot read header)\n", filename);
		fclose(f);
		return 0;
	}

    // Check supported type:
    //  2 = uncompressed RGB
    //  3 = uncompressed grayscale (optional)
    // 10 = RLE-compressed RGB
    if (th.imageType != 2 && th.imageType != 3 && th.imageType != 10) {
        fprintf(stderr, "Error: %s has unsupported TGA type (%d)\n", filename, th.imageType);
        fclose(f);
        return 0;
    }

    if (th.imageType == 10) {
        // Shallow RLE validation: make sure we can walk exactly width*height pixels.
        int w = th.width;
        int h = th.height;
        int size = w * h;
        int bytesPerPixel = th.pixelDepth / 8;

        // Skip ID field if present
//        if (th.idLength > 0)//FREDZ create a bug disable with corrupt tga.
//            fseek(f, th.idLength, SEEK_CUR);

        int pixelsDecoded = 0;
        while (pixelsDecoded < size) {
            uint8_t packetHeader;
            if (fread(&packetHeader, 1, 1, f) != 1) {
                fprintf(stderr, "Error %s: truncated RLE packet header\n", filename);
                fclose(f);
                return 0;
            }

            int count = (packetHeader & 0x7F) + 1;
            if (count <= 0 || pixelsDecoded + count > size) {
                fprintf(stderr, "Error %s: invalid RLE packet run (count=%d)\n", filename, count);
                fclose(f);
                return 0;
            }

            if (packetHeader & 0x80) {
                // RLE packet: one pixel repeated 'count' times
                uint8_t pixel[4];
                if (fread(pixel, bytesPerPixel, 1, f) != 1) {
                    fprintf(stderr, "Error %s: truncated RLE pixel data\n", filename);
                    fclose(f);
                    return 0;
                }
            } else {
                // Raw packet: count pixels follow
                if (fseek(f, (long)count * bytesPerPixel, SEEK_CUR) != 0) {
                    fprintf(stderr, "Error: %s truncated: cannot skip raw RLE data\n", filename);
                    fclose(f);
                    return 0;
                }
            }

            pixelsDecoded += count;
        }

        // If we reach here, RLE stream at least structurally covers all pixels
        fclose(f);
        return 1;
    }


	// Check pixel depth
	if (th.pixelDepth != 24 && th.pixelDepth != 32) {
		fprintf(stderr, "Error: %s has unsupported pixel depth (%d)\n", filename, th.pixelDepth);
		fclose(f);
		return 0;
	}

	// Check origin flag (must be bottom-left)
	if (th.imageDesc & 0x20) {
		fprintf(stderr, "Error: %s has top-left origin. Only bottom-left origin supported.\n", filename);
		fclose(f);
		return 0;
	}

    // Check file size vs expected (uncompressed only)
    struct stat st;
    if (stat(filename, &st) == 0) {
        if (th.imageType == 2 || th.imageType == 3) {
            size_t expected = sizeof(TGAHeader) + th.idLength
                            + (size_t)th.width * th.height * th.pixelDepth / 8;
            if (st.st_size < expected) {
                fprintf(stderr, "Error: %s truncated: expected %zu bytes, got %lld\n", filename, expected, (long long)st.st_size);
                fclose(f);
                return 0;
            }
        } else {
            // For RLE (type 10) just ensure it's not smaller than header + ID
            size_t minSize = sizeof(TGAHeader) + th.idLength;
            if (st.st_size < minSize) {
                fprintf(stderr, "Error: %s truncated RLE TGA (too small)\n", filename);
                fclose(f);
                return 0;
            }
        }
    }

	fclose(f);
	return 1; // valid
}

int scan_tga_alpha(const char *filename, int *hasAlpha, int *hasSemi)
{
    *hasAlpha = 0;
    *hasSemi  = 0;

    FILE *f = fopen(filename, "rb");
    if (!f) {
        fprintf(stderr, "Error: cannot open %s\n", filename);
        return 0;
    }

    if (fread(&th, sizeof(TGAHeader), 1, f) != 1) {
        fprintf(stderr, "Error: %s truncated (cannot read header)\n", filename);
        fclose(f);
        return 0;
    }

    // Only care about RGB (2) and RLE RGB (10), and 32-bit (with alpha)
    if ((th.imageType != 2 && th.imageType != 10) || th.pixelDepth != 32) {
        fclose(f);
        return 1; // valid but no alpha channel we care about
    }

    int w = th.width;
    int h = th.height;
    int size = w * h;
    int bytesPerPixel = th.pixelDepth / 8;

    // Skip ID field if present
//    if (th.idLength > 0)//FREDZ create a bug disable with corrupt tga.
//        fseek(f, th.idLength, SEEK_CUR);

    uint8_t *pixels = (uint8_t *)malloc(size * bytesPerPixel);
    if (!pixels) {
        fclose(f);
        return 0;
    }

    if (th.imageType == 2) {
        // Uncompressed
        if (fread(pixels, bytesPerPixel, size, f) != (size_t)size) {
            fprintf(stderr, "Error %s truncated: cannot read pixel data\n", filename);
            free(pixels);
            fclose(f);
            return 0;
        }
    } else if (th.imageType == 10) {
        // RLE compressed, same logic as in converttgatowal
        int pixelsDecoded = 0;
        while (pixelsDecoded < size) {
            uint8_t packetHeader;
            if (fread(&packetHeader, 1, 1, f) != 1) {
                fprintf(stderr, "Skipping %s: truncated RLE packet header\n", filename);
                free(pixels);
                fclose(f);
                return 0;
            }
            int count = (packetHeader & 0x7F) + 1;
            if (count <= 0 || pixelsDecoded + count > size) {
                fprintf(stderr, "Skipping %s: invalid RLE packet run count %d\n", filename, count);
                free(pixels);
                fclose(f);
                return 0;
            }

            if (packetHeader & 0x80) {
                // RLE packet: one pixel repeated 'count' times
                uint8_t pixel[4];
                if (fread(pixel, bytesPerPixel, 1, f) != 1) {
                    fprintf(stderr, "Skipping %s: truncated RLE pixel data\n", filename);
                    free(pixels);
                    fclose(f);
                    return 0;
                }
                for (int j = 0; j < count; j++)
                    memcpy(pixels + (pixelsDecoded + j) * bytesPerPixel, pixel, bytesPerPixel);
            } else {
                // Raw packet: count pixels follow
                if (fread(pixels + pixelsDecoded * bytesPerPixel, bytesPerPixel, count, f) != (size_t)count) {
                    fprintf(stderr, "Skipping %s: truncated raw RLE data\n", filename);
                    free(pixels);
                    fclose(f);
                    return 0;
                }
            }
            pixelsDecoded += count;
        }
    }

    fclose(f);

    // Scan alpha channel (percentage-based semi detection)
    // ignore tiny edge-noise near 0/255 (tune eps = 1,2,4,8)
    const int eps = 1;
    int semiCount = 0;
    int alphaCount = 0;

    for (int i = 0; i < size; i++) {
        uint8_t a = pixels[i * bytesPerPixel + 3];

        if (a < 255) alphaCount++;

        if (a > eps && a < 255 - eps)
            semiCount++;
    }

    *hasAlpha = (alphaCount > 0);

    // require at least X% semi pixels
    const float semiPct = (size > 0) ? (100.0f * (float)semiCount / (float)size) : 0.0f;
    *hasSemi = (semiPct >= 10.0f);   // 1% is a decent start (tune) //FREDZ put on 10% maybe need more tweaking?

    // FREDZ see actualy %
    if (debug)
         printf("%s semiPct=%.3f%% semiCount=%d/%d\n", filename, semiPct, semiCount, size);

    free(pixels);
    return 1;
}


int fix_tga(const char *infile, const char *outfile)
{
    FILE *fin = fopen(infile, "rb");
    if (!fin) { perror("open TGA"); return 0; }

    if (debug)
        printf("FIX_TGA CALLED: %s\n", infile);

    TGAHeader th;
    if (fread(&th, sizeof(TGAHeader), 1, fin) != 1) {
        fprintf(stderr, "Error: %s truncated (cannot read header)\n", infile);
        fclose(fin);
        return 0;
    }

    int w = th.width;
    int h = th.height;
    int size = w * h;
    int bytesPerPixel = th.pixelDepth / 8;

    if (debug)
        printf("DEBUG: %s type=%d depth=%d\n", infile, th.imageType, th.pixelDepth);

    // Only support types 1,2,3,10 and depths 8,24,32
    if ((th.imageType != 1 && th.imageType != 2 && th.imageType != 3 && th.imageType != 10) ||
        (th.pixelDepth != 8 && th.pixelDepth != 24 && th.pixelDepth != 32)) {
        fprintf(stderr, "Skipping %s: unsupported format (type %d, depth %d)\n", infile, th.imageType, th.pixelDepth);
        fclose(fin);
        return 0;
    }

    uint8_t *pixels = NULL;
    uint8_t *colormap = NULL;
    int cmap_bytes = 0;

    // --- Indexed TGA: read palette and pixel indices ---
    if (th.imageType == 1 && th.pixelDepth == 8) {
        // Read palette
        cmap_bytes = th.cmLength * (th.cmEntrySize / 8);
        colormap = malloc(cmap_bytes);
        if (!colormap) { fclose(fin); return 0; }
        fseek(fin, sizeof(TGAHeader) + th.idLength, SEEK_SET);
        if (fread(colormap, 1, cmap_bytes, fin) != (size_t)cmap_bytes) {
            fprintf(stderr, "Error: %s truncated (cannot read palette)\n", infile);
            free(colormap); fclose(fin); return 0;
        }
        // Read pixel indices
        pixels = malloc(size);
        if (!pixels) { free(colormap); fclose(fin); return 0; }
        if (fread(pixels, 1, size, fin) != (size_t)size) {
            fprintf(stderr, "Error: %s truncated (cannot read pixel indices)\n", infile);
            free(pixels); free(colormap); fclose(fin); return 0;
        }
        bytesPerPixel = 1;
    }
    // --- Uncompressed grayscale ---
    else if (th.imageType == 3 && th.pixelDepth == 8) {
        fseek(fin, sizeof(TGAHeader) + th.idLength, SEEK_SET);
        pixels = malloc(size);
        if (!pixels) { fclose(fin); return 0; }
        if (fread(pixels, 1, size, fin) != (size_t)size) {
            fprintf(stderr, "Error: %s truncated (cannot read grayscale)\n", infile);
            free(pixels); fclose(fin); return 0;
        }
        bytesPerPixel = 1;
    }
    // --- Uncompressed RGB(A) ---
    else if (th.imageType == 2 && (th.pixelDepth == 24 || th.pixelDepth == 32)) {
        fseek(fin, sizeof(TGAHeader) + th.idLength, SEEK_SET);
        pixels = malloc(size * bytesPerPixel);
        if (!pixels) { fclose(fin); return 0; }
        if (fread(pixels, bytesPerPixel, size, fin) != (size_t)size) {
            fprintf(stderr, "Error: %s truncated (cannot read pixels)\n", infile);
            free(pixels); fclose(fin); return 0;
        }
    }
    // --- RLE-compressed RGB(A) ---
    else if (th.imageType == 10 && (th.pixelDepth == 24 || th.pixelDepth == 32)) {
        fseek(fin, sizeof(TGAHeader) + th.idLength, SEEK_SET);
        pixels = malloc(size * bytesPerPixel);
        if (!pixels) { fclose(fin); return 0; }
        int pixelsDecoded = 0;
        while (pixelsDecoded < size) {
            uint8_t packetHeader;
            if (fread(&packetHeader, 1, 1, fin) != 1) {
                fprintf(stderr, "Error: %s truncated (RLE header)\n", infile);
                free(pixels); fclose(fin); return 0;
            }
            int count = (packetHeader & 0x7F) + 1;
            if (count <= 0 || pixelsDecoded + count > size) {
                fprintf(stderr, "Error: %s invalid RLE packet run (count=%d)\n", infile, count);
                free(pixels); fclose(fin); return 0;
            }
            if (packetHeader & 0x80) {
                // RLE packet: one pixel repeated
                uint8_t pixel[4];
                if (fread(pixel, bytesPerPixel, 1, fin) != 1) {
                    fprintf(stderr, "Error: %s truncated (RLE pixel)\n", infile);
                    free(pixels); fclose(fin); return 0;
                }
                for (int j = 0; j < count; j++)
                    memcpy(pixels + (pixelsDecoded + j) * bytesPerPixel, pixel, bytesPerPixel);
            } else {
                // Raw packet: count pixels follow
                if (fread(pixels + pixelsDecoded * bytesPerPixel, bytesPerPixel, count, fin) != (size_t)count) {
                    fprintf(stderr, "Error: %s truncated (RLE raw data)\n", infile);
                    free(pixels); fclose(fin); return 0;
                }
            }
            pixelsDecoded += count;
        }
    } else {
        fprintf(stderr, "Skipping %s: unsupported or corrupt TGA\n", infile);
        fclose(fin);
        return 0;
    }

    fclose(fin);

    // --- Convert to 24-bit RGB ---
    uint8_t *rgb_pixels = malloc(size * 3);
    if (!rgb_pixels) { free(pixels); if (colormap) free(colormap); return 0; }

    if (th.imageType == 1 && th.pixelDepth == 8) {
        // Indexed: use palette
        for (int i = 0; i < size; i++) {
            int idx = pixels[i];
            if (idx >= th.cmLength) idx = 0;
            rgb_pixels[i * 3 + 0] = colormap[idx * (th.cmEntrySize / 8) + 0]; // B
            rgb_pixels[i * 3 + 1] = colormap[idx * (th.cmEntrySize / 8) + 1]; // G
            rgb_pixels[i * 3 + 2] = colormap[idx * (th.cmEntrySize / 8) + 2]; // R
        }
        printf("Fixed %s: converted 8-bit indexed -> 24-bit RGB\n", infile);
    } else if (th.imageType == 3 && th.pixelDepth == 8) {
        // Grayscale: replicate value
        for (int i = 0; i < size; i++) {
            uint8_t gray = pixels[i];
            rgb_pixels[i * 3 + 0] = gray;
            rgb_pixels[i * 3 + 1] = gray;
            rgb_pixels[i * 3 + 2] = gray;
        }
        printf("Fixed %s: converted 8-bit grayscale -> 24-bit RGB\n", infile);
    } else if (th.pixelDepth == 24) {
        memcpy(rgb_pixels, pixels, size * 3);
    } else if (th.pixelDepth == 32) {
        for (int i = 0; i < size; i++) {
            rgb_pixels[i * 3 + 0] = pixels[i * 4 + 0]; // B
            rgb_pixels[i * 3 + 1] = pixels[i * 4 + 1]; // G
            rgb_pixels[i * 3 + 2] = pixels[i * 4 + 2]; // R
        }
        printf("Fixed %s: converted 32-bit -> 24-bit\n", infile);
    }

    if (colormap) free(colormap);
    free(pixels);

    // --- Flip vertically if needed ---
    if (th.imageDesc & 0x20) {
        for (int y = 0; y < h / 2; y++) {
            uint8_t *row_top = rgb_pixels + y * w * 3;
            uint8_t *row_bottom = rgb_pixels + (h - 1 - y) * w * 3;
            for (int x = 0; x < w * 3; x++) {
                uint8_t temp = row_top[x];
                row_top[x] = row_bottom[x];
                row_bottom[x] = temp;
            }
        }
        printf("Fixed %s: flipped top-left -> bottom-left\n", infile);
    }

    // --- Write fixed TGA (always uncompressed, bottom-left, 24-bit) ---
    FILE *fout = fopen(outfile, "wb");
    if (!fout) {
        perror("open output");
        free(rgb_pixels);
        return 0;
    }

    // --- Prepare header for standard 24-bit TGA ---
    th.imageType = 2;      // Uncompressed RGB
    th.pixelDepth = 24;
    th.imageDesc &= ~0x20; // Bottom-left origin
    th.idLength = 0;       // No ID field
    th.colorMapType = 0;   // No palette
    th.cmLength = 0;
    th.cmEntrySize = 0;
    th.cmFirstEntry = 0;

    fwrite(&th, sizeof(TGAHeader), 1, fout);
    fwrite(rgb_pixels, 3, size, fout);
    fclose(fout);

    free(rgb_pixels);
    return 1;
}

// --- Check if TGA needs fixing ---
int tga_needs_fix(const char *filename)
{
    int needsfix = 0;

    FILE *f = fopen(filename, "rb");
    if (!f) return 0; // can't open, skip silently

    if (fread(&th, sizeof(TGAHeader), 1, f) != 1) {
        fclose(f);
        fprintf(stderr, "Error: %s truncated (cannot read header)\n", filename);
        return -1; // ERROR - count as failed
    }

    // Accept types 1,2,3,10 and depths 8,24,32
    if ((th.imageType != 1 && th.imageType != 2 && th.imageType != 10 && th.imageType != 3) ||
        (th.pixelDepth != 8 && th.pixelDepth != 24 && th.pixelDepth != 32)) {
        fprintf(stderr, "Skipping %s: unsupported format (type %d, depth %d)\n",
                filename, th.imageType, th.pixelDepth);
        fclose(f);
        return -1;
    }

    // Always mark 8-bit indexed and grayscale TGAs as needing fix
    if ((th.imageType == 1 && th.pixelDepth == 8) ||
        (th.imageType == 3 && th.pixelDepth == 8)) {
        needsfix = 1;
    }

    // Check: top-left origin (needs flip)
    if (th.imageDesc & 0x20) {
        needsfix = 1;
    }

    // ... RLE validation ...
    if (th.imageType == 10) {
        int w = th.width;
        int h = th.height;
        int size = w * h;
        int bytesPerPixel = th.pixelDepth / 8;
        int pixelsDecoded = 0;

        while (pixelsDecoded < size) {
            uint8_t packetHeader;
            if (fread(&packetHeader, 1, 1, f) != 1) {
                fprintf(stderr, "Skipping %s: truncated RLE stream\n", filename);
                fclose(f);
                return -1;  // ← ERROR: truncated
            }

            int count = (packetHeader & 0x7F) + 1;
            if (count <= 0 || pixelsDecoded + count > size) {
                fprintf(stderr, "Skipping %s: corrupt RLE packet (invalid run count %d)\nPlease open and re-save this file in an image editor to repair it.\n", filename, count);
                fclose(f);
                return -1;  // ← ERROR: corrupt packet
            }

            if (packetHeader & 0x80) {
                if (fseek(f, bytesPerPixel, SEEK_CUR) != 0) break;
            } else {
                if (fseek(f, (long)count * bytesPerPixel, SEEK_CUR) != 0) break;
            }
            pixelsDecoded += count;
        }
    }

    fclose(f);
    return needsfix;  // 0 = good, 1 = needs fix
}

// --- Convert one TGA to WAL ---
int convert_tga_to_wal(const char *infile, const char *outfile, uint8_t palette[256][3])
{
    FILE *f = fopen(infile, "rb");
    if (!f) {
        perror("open TGA");
        return 0;
    }

    fread(&th, sizeof(TGAHeader), 1, f);

    int w = th.width, h = th.height;
    int size = w * h;
    int bytesPerPixel = th.pixelDepth / 8;

    uint8_t *pixels = malloc(size * bytesPerPixel);
    uint8_t *base   = malloc(size);
    if (!pixels || !base)
    {
        fclose(f);
        free(pixels);
        free(base);
        return 0;
    }

	// --- Prepare WAL name and dirpath ---
    char walname[256];
    strncpy(walname, infile, sizeof(walname)-1);
    walname[sizeof(walname)-1] = '\0';

    // Normalize slashes (convert '\' to '/')
    for (char *p = walname; *p; p++) {
        if (*p == '\\') *p = '/';
    }

    // Strip extension
    char *dot = strrchr(walname, '.');
    if (dot) *dot = '\0';

    // Find "textures/" in the path
    const char *relpath = walname;
    char *texpos = strstr(walname, "textures/");
    if (texpos)
        relpath = texpos + 9; // skip "textures/"


    // Extract directory path from infile
    char dirpath[512];
    strncpy(dirpath, infile, sizeof(dirpath)-1);
    dirpath[sizeof(dirpath)-1] = '\0';
    char *slash = strrchr(dirpath, '/');
    if (!slash)
        slash = strrchr(dirpath, '\\');
    if (slash)
        *slash = '\0'; // terminate at last slash

    // --- Read pixel data ---
    if (th.imageType == 2) {
        fread(pixels, bytesPerPixel, size, f);
    }
    else if (th.imageType == 10) {
        int pixelsDecoded = 0;
        while (pixelsDecoded < size) {
            uint8_t packetHeader;
            if (fread(&packetHeader, 1, 1, f) != 1) {
                fprintf(stderr, "Skipping %s: truncated RLE packet header\n", infile);
                free(pixels);
                free(base);
                return 0;
            }

            int count = (packetHeader & 0x7F) + 1;
            if (count <= 0 || pixelsDecoded + count > size) {
                fprintf(stderr, "Skipping %s: invalid RLE packet run (count=%d)\n", infile, count);
                free(pixels);
                free(base);
                return 0;
            }

            if (packetHeader & 0x80) {
                // RLE packet: one pixel repeated 'count' times
                uint8_t pixel[4];
                if (fread(pixel, bytesPerPixel, 1, f) != 1) {
                    fprintf(stderr, "Skipping %s: truncated RLE pixel data\n", infile);
                    free(pixels);
                    free(base);
                    return 0;
                }
                for (int j = 0; j < count; j++) {
                    memcpy(pixels + (pixelsDecoded + j) * bytesPerPixel, pixel, bytesPerPixel);
                }
            } else {
                // Raw packet: 'count' pixels follow
                if (fread(pixels + pixelsDecoded * bytesPerPixel, bytesPerPixel, count, f) != (size_t)count)
                {
                    fprintf(stderr, "Skipping %s: truncated raw RLE data\n", infile);
                    free(pixels);
                    free(base);
                    return 0;
                }
            }

            pixelsDecoded += count;
        }
    }

    fclose(f);

    // --- Detect alpha usage ---
    int hasAlpha = 0;
    int hasSemiAlpha  = 0;
    int ignoreAlphaThisTexture = skipAlpha;  // start from global -skipalpha

    if (th.pixelDepth == 24) {
        if (onlyAlpha)
        {
            if (debug)
                fprintf(stderr, "Skipping %s (24-bit RGB has no alpha channel)\n", infile);
            free(pixels);
            free(base);
            return 0;
        }
    }
    else if (th.pixelDepth == 32) {
        for (int i = 0; i < size; i++) {
            uint8_t a = pixels[i*4 + 3];
            if (a < 255)
                hasAlpha = 1;
            if (useSemiAlphaDetect && a > 0 && a < alphaSemiMax)
                hasSemiAlpha = 1;
        }
        if (skipAlpha) {
            fprintf(stderr, "Skipping alpha on %s: -skipalpha set (RGB only)\n", infile);
            ignoreAlphaThisTexture = 1;
        }
        else if (useSemiAlphaDetect && hasSemiAlpha) {
            fprintf(stderr,
                    "Note %s has semi-transparent alpha (0..%d); ignoring alpha channel (no pink cutout)\n", infile, alphaSemiMax);
            ignoreAlphaThisTexture = 1;
        }
        else if (onlyAlpha) {
            if (!hasAlpha) {
                fprintf(stderr, "Skipping %s (no alpha channel data)\n", infile);
                free(pixels); free(base);
                return 0;
            }
            if (debug)
                fprintf(stderr, "Note: %s is 32-bit RGBA, using alpha channel only (ignoring RGB)\n", infile);
        }
    }
    else
    {
        fprintf(stderr, "Skipping %s (TGA must be 24-bit RGB or 32-bit RGBA)\n", infile);
        free(pixels);
        free(base);
        return 0;
    }

    // --- Dimension check ---
    if (requirePowerOf2 && ((w & (w - 1)) != 0 || (h & (h - 1)) != 0)) {
        fprintf(stderr, "Skipping %s (dimensions %dx%d not power-of-two)\n", infile, w, h);
        free(pixels);
        free(base);
        return 0;
    }

    // --- Conversion loop ---
    for (int i = 0; i < size; i++) {
        uint8_t r = pixels[i*bytesPerPixel+2];
        uint8_t g = pixels[i*bytesPerPixel+1];
        uint8_t b = pixels[i*bytesPerPixel+0];
        int idx = rgb_to_index(r, g, b, palette);

        if (bytesPerPixel == 4) {
            uint8_t a = pixels[i*4+3];

            if (onlyAlpha) {
                base[i] = (a < alphaThreshold) ? 255 : 0;
            }
            else if (ignoreAlphaThisTexture) {
                // Treat as opaque RGB only (for -skipalpha or -alphathreshsemi case)
                base[i] = idx;
            }
            else
            {
                base[i] = (a < alphaThreshold ? 255 : idx);
            }
        }
        else {
            base[i] = idx;
        }
    }

	// Prepare WAL header
	miptex_t wal;
	memset(&wal, 0, sizeof(wal));

    // Copy into WAL header (truncate if needed)
    int len = strlen(relpath);
    if (len >= MAX_NAME) len = MAX_NAME - 1;
    memcpy(wal.name, relpath, len);
    wal.name[len] = '\0';

    wal.width = w;
    wal.height = h;
    wal.offsets[0] = sizeof(miptex_t);
    wal.offsets[1] = wal.offsets[0] + size;
    wal.offsets[2] = wal.offsets[1] + size/4;
    wal.offsets[3] = wal.offsets[2] + size/16;

    // --- Apply textureinfo.ini (merged logic) ---
    int matched = 0;

    if (!skipIni || onlyIni) {
        matched = apply_textureinfo(dirpath, wal.name, &wal);
    }

    if (onlyIni && !matched)
    {
        if (debug)
            fprintf(stderr, "Skipping %s (not listed in %s/textureinfo.ini)\n",
                    infile, dirpath);
        free(pixels); free(base);
        return 0;
    }

    // If requested, write per-texture .wal_json file when INI matched
    if (walJsonPerTexture && matched) {
        char jsonpath[1024];

        // Start from the source image path (infile), so .wal_json sits next to the TGA/PNG
        strncpy(jsonpath, infile, sizeof(jsonpath) - 1);
        jsonpath[sizeof(jsonpath) - 1] = '\0';

        char *dot = strrchr(jsonpath, '.');
        if (dot)
            strcpy(dot, ".wal_json");
        else
            strncat(jsonpath, ".wal_json",
                    sizeof(jsonpath) - strlen(jsonpath) - 1);

        FILE *jf = fopen(jsonpath, "w");
        if (!jf) {
            fprintf(stderr, "Warning: could not write wal_json %s\n", jsonpath);
        } else {
            // Minimal wal_json content: metadata only
            fprintf(jf,
                    "{\n"
                    "  \"flags\": %u,\n"
                    "  \"contents\": %u,\n"
                    "  \"value\": %u\n"
                    "}\n",
                    wal.flags, wal.contents, wal.value);
            fclose(jf);
        }
    }

	// Strip down to Quake II only if requested
	if (q2FlagsOnly) {
		wal.flags &= Q2_FLAG_MASK;
		wal.contents &= Q2_CONTENT_MASK;
	}

    // Generate mipmaps
    uint8_t *mip1 = malloc(size / 4);
    uint8_t *mip2 = malloc(size / 16);
    uint8_t *mip3 = malloc(size / 64);
    generate_mipmap(base, w, h, mip1, palette);
    generate_mipmap(mip1, w/2, h/2, mip2, palette);
    generate_mipmap(mip2, w/4, h/4, mip3, palette);

    flip_vertical(base, w, h);
    flip_vertical(mip1, w/2, h/2);
    flip_vertical(mip2, w/4, h/4);
    flip_vertical(mip3, w/8, h/8);

    // Write WAL file
    FILE *fout = fopen(outfile, "wb");
    if (!fout) {
        perror("open WAL");
        free(pixels); free(base); free(mip1); free(mip2); free(mip3);
        return 0;
    }

    fwrite(&wal, sizeof(miptex_t), 1, fout);
    fwrite(base, 1, size, fout);
    fwrite(mip1, 1, size/4, fout);
    fwrite(mip2, 1, size/16, fout);
    fwrite(mip3, 1, size/64, fout);

    fclose(fout);
    free(pixels); free(base); free(mip1); free(mip2); free(mip3);

    printf("Converted %s -> %s\n", infile, outfile);

    return 1;
}

// Delete the file corresponding to a TGA
static void delete_for_tga_ext(const char *infile, const char *ext, const char *label)
{
    char out[512];
    strncpy(out, infile, sizeof(out) - 1);
    out[sizeof(out) - 1] = 0;

    for (char *p = out; *p; ++p) if (*p == '\\') *p = '/';

    char *dot = strrchr(out, '.');
    if (dot) *dot = 0;

    strncat(out, ext, sizeof(out) - strlen(out) - 1);

    if (remove(out) == 0)
        printf("Deleted %s %s\n", label, out);
    else if (errno != ENOENT)
        fprintf(stderr, "Failed to delete %s: %s (%s)\n", label, out, strerror(errno));
}

// --- Recursively process directories ---
void process_directory(const char *path, uint8_t palette[256][3], int *converted, int *skippedExists, int *skippedInvalid)
{
    DIR *dir = opendir(path);
    if (!dir) return;

    struct dirent *entry;
    while ((entry = readdir(dir)) != NULL) {
        if (strcmp(entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0)
            continue;

        char fullpath[1024];
        snprintf(fullpath, sizeof(fullpath), "%s/%s", path, entry->d_name);

        struct stat st;
        if (stat(fullpath, &st) == 0) {
            if (S_ISDIR(st.st_mode))
            {
                process_directory(fullpath, palette, converted, skippedExists, skippedInvalid);
            }
            else {
                const char *ext = strrchr(entry->d_name, '.');
                if (ext && strcasecmp(ext, ".tga") == 0)
                {
                    char outpath[1024];
                    strcpy(outpath, fullpath);
                    char *dot = strrchr(outpath, '.');
                    if (dot)
                        strcpy(dot, ".wal");

                    // Skip if WAL exists and -overwrite IS set  (inverted behavior)
                    if (skipoverwrite) {
                        struct stat stwal;
                        if (stat(outpath, &stwal) == 0) { // file exists
                            fprintf(stderr, "Skipping %s (WAL exists: %s)\n", fullpath, outpath);
                            (*skippedExists)++;
                            continue;
                        }
                    }

                    if (deletedMode)
                    {
                        if (deletedMode == 1)
                            delete_for_tga_ext(fullpath, ".wal", "WAL");
                        else if (deletedMode == 2)
                            delete_for_tga_ext(fullpath, ".waljson", "waljson");
                        else if (deletedMode == 3)
                        {
                            delete_for_tga_ext(fullpath, ".wal", "WAL");
                            delete_for_tga_ext(fullpath, ".waljson", "waljson");
                        }
                        continue;              // no scan/convert in deleted modes
                    }
                    else {
                        if (!scan_tga(fullpath))
                        {
                            if (debug)
                                fprintf(stderr, "Skipped %s: invalid or unsupported TGA\n", fullpath);
                            (*skippedInvalid)++;
                            continue;
                        }

                        if (!convert_tga_to_wal(fullpath, outpath, palette)) {
                            // optional: treat failed conversion as skipped
                            (*skippedInvalid)++;
                        } else {
                            (*converted)++;
                        }
                    }

                }
            }
        }
    }
    closedir(dir);
}

void scan_directory(const char *path, int *scanned, int *invalid)
{
    DIR *dir = opendir(path);
    if (!dir)
        return;

    struct dirent *entry;
    while ((entry = readdir(dir)) != NULL) {
        if (strcmp(entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0)
            continue;

        char fullpath[1024];
        snprintf(fullpath, sizeof(fullpath), "%s/%s", path, entry->d_name);

        struct stat st;
        if (stat(fullpath, &st) == 0) {
            if (S_ISDIR(st.st_mode)) {
                scan_directory(fullpath, scanned, invalid);
            } else {
                const char *ext = strrchr(entry->d_name, '.');
                if (ext && strcasecmp(ext, ".tga") == 0) {
                    (*scanned)++;
                    if (!scan_tga(fullpath))
                    {
                        if (debug)
                            fprintf(stderr, "Corrupt or unsupported TGA: %s\n", fullpath);
                        (*invalid)++;
                    }
                    // valid TGAs are silent
                }
            }
        }
    }
    closedir(dir);
}

void scan_ini_directory(const char *path, int *totalIniEntries, int *missingTga)
{
    DIR *dir = opendir(path);
    if (!dir)
        return;

    struct dirent *entry;
    while ((entry = readdir(dir)) != NULL) {
        if (strcmp(entry->d_name, ".") == 0 ||
            strcmp(entry->d_name, "..") == 0)
            continue;

        char fullpath[1024];
        snprintf(fullpath, sizeof(fullpath), "%s/%s", path, entry->d_name);

        struct stat st;
        if (stat(fullpath, &st) == 0 && S_ISDIR(st.st_mode)) {
            // Recurse into subdirectory
            scan_ini_directory(fullpath, totalIniEntries, missingTga);
        }
    }

    // After scanning children, check for textureinfo.ini in this directory
    char inipath[1024];
    snprintf(inipath, sizeof(inipath), "%s/textureinfo.ini", path);

    FILE *f = fopen(inipath, "r");
    if (!f) {
        closedir(dir);
        return;
    }

    printf("Scanning %s\n", inipath);

    char line[256];
    char section[64];

    while (fgets(line, sizeof(line), f)) {
        // trim newline
        line[strcspn(line, "\r\n")] = '\0';
        if (!line[0])
            continue;

        if (line[0] == '[') {
            // Parse [section]
            if (sscanf(line, "[%63[^]]", section) == 1) {
                (*totalIniEntries)++;

                // section may contain subdirs like "sub/brick01"
                char texpath[1024];
                snprintf(texpath, sizeof(texpath), "%s/%s.tga", path, section);

                struct stat tst;
                if (stat(texpath, &tst) != 0) {
                    (*missingTga)++;
                    fprintf(stderr,
                        "Missing TGA for INI entry: [%s] in %s (expected %s)\n",
                        section, inipath, texpath);
                }
            }
        }
    }

    fclose(f);
    closedir(dir);
}

void scan_alpha_directory(const char *path, int *alphaCount, int *semiCount)
{
    DIR *dir = opendir(path);
    if (!dir)
        return;

    struct dirent *entry;
    while ((entry = readdir(dir)) != NULL) {
        if (strcmp(entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0)
            continue;

        char fullpath[1024];
        snprintf(fullpath, sizeof(fullpath), "%s/%s", path, entry->d_name);

        struct stat st;
        if (stat(fullpath, &st) == 0 && S_ISDIR(st.st_mode)) {
            scan_alpha_directory(fullpath, alphaCount, semiCount);
        } else {
            const char *ext = strrchr(entry->d_name, '.');
            if (ext && strcasecmp(ext, ".tga") == 0) {
                int hasAlpha = 0, hasSemi = 0;
                if (!scan_tga_alpha(fullpath, &hasAlpha, &hasSemi))
                    continue; // error already reported

                if (hasAlpha)
                {
                    (*alphaCount)++;
                    if (hasSemi)
                        (*semiCount)++;

                    printf("%s : alpha=YES, semi=%s\n", fullpath, hasSemi ? "YES" : "NO");
                }
            }
        }
    }

    closedir(dir);
}


// --- Recursively fix TGAs ---
void fix_directory(const char *path, int *fixed, int *failed)
{
    DIR *dir = opendir(path);
    if (!dir) return;

    struct dirent *entry;
    while ((entry = readdir(dir)) != NULL) {
        if (strcmp(entry->d_name, ".") == 0 || strcmp(entry->d_name, "..") == 0)
            continue;

        char fullpath[1024];
        snprintf(fullpath, sizeof(fullpath), "%s/%s", path, entry->d_name);

        struct stat st;
        if (stat(fullpath, &st) == 0) {
            if (S_ISDIR(st.st_mode)) {
                fix_directory(fullpath, fixed, failed);
            } else
            {
                const char *ext = strrchr(entry->d_name, '.');
                if (ext && strcasecmp(ext, ".tga") == 0) {
                    int needs = tga_needs_fix(fullpath);
                    if (needs > 0) {  // 1 = needs fixing
                        if (fix_tga(fullpath, fullpath)) {
                            (*fixed)++;
                        } else {
                            (*failed)++;
                        }
                    } else if (needs < 0) {  // -1 = error/unsupported
                        (*failed)++;
                    }
                    // needs == 0: good file, skip silently
                }
            }
        }
    }
    closedir(dir);
}

// --- Main ---
int main(int argc, char **argv)
{
    iniFound = 0;

    time_t now = time(NULL);
    char date[64];
    strftime(date, sizeof(date), "%Y-%m-%d", localtime(&now));

    // Default root depending on game folder
    int pathArgIndex = -1;
    bool pathProvided = false;

    // Parse arguments
    for (int j = 1; j < argc; j++)
    {
        // Skip the directory argument itself
        if (pathProvided && j == pathArgIndex)
            continue;

        if (strcmp(argv[j], "-alphathresh") == 0) {
            if (j + 1 >= argc) {
                fprintf(stderr, "Error: -alphathresh requires a numeric value 0-255.\n");
                return 1;
            }

            // Require the value to start with a digit (or +/-digit if you want)
            const char *val = argv[j + 1];
            if (!isdigit((unsigned char)val[0])) {
                fprintf(stderr, "Error: -alphathresh requires a numeric value 0-255, got '%s'.\n", val);
                return 1;
            }

            alphaThreshold = atoi(argv[++j]);
            useAlphaThreshold = 1;
            printf("Alpha threshold set to %d\n", alphaThreshold);

            if (alphaThreshold < 0 || alphaThreshold > 255) {
                fprintf(stderr,
                    "Warning: alpha threshold %d out of range 0-255. Clamping.\n",
                    alphaThreshold);
                alphaThreshold = (alphaThreshold < 0) ? 0 : 255;
            }
        }
        else if (strcmp(argv[j], "-alphathreshsemi") == 0)
        {
            if (j + 1 >= argc) {
                fprintf(stderr, "Error: -alphathreshsemi requires a numeric value 1-255.\n");
                return 1;
            }
            const char *val = argv[++j];
            if (!isdigit((unsigned char)val[0])) {
                fprintf(stderr, "Error: -alphathreshsemi requires a numeric value 1-255, got '%s'.\n", val);
                return 1;
            }
            alphaSemiMax = atoi(val);
            if (alphaSemiMax <= 0 || alphaSemiMax > 255)
                alphaSemiMax = 127;  // clamp to something sane

            useSemiAlphaDetect = 1;
            printf("Semi-alpha detection enabled: textures with alpha in (0..%d) will ignore -alphathresh\n", alphaSemiMax);
        }
        else if (strcmp(argv[j], "-skipalpha") == 0) {
            skipAlpha = 1; // ignore alpha channel completely
            printf("Alpha channel will be ignored (RGB only)\n");
        }
        else if (strcmp(argv[j], "-onlyalpha") == 0) {
            onlyAlpha = 1; // ignore RGB, use alpha only
            printf("Using alpha channel only (ignoring RGB)\n");
        }
        else if (strcmp(argv[j], "-skipini") == 0)
        {
            skipIni = 1;
            printf("Skipping textureinfo.ini\n");
        }
        else if (strcmp(argv[j], "-onlyini") == 0) {
            onlyIni = 1;
            printf("INI-only mode: textures will be generated from textureinfo.ini without pixel conversion\n");
        }
        else if (strcmp(argv[j], "-powerof2") == 0) {
            requirePowerOf2 = 1;
        }
        else if (strcmp(argv[j], "-q2flags") == 0) {
            q2FlagsOnly = 1;
        }
        else if (strcmp(argv[j], "-q2palette") == 0) {
            use_q2_palette = 1;
        }
        else if (strcmp(argv[j], "-kppalette") == 0) {
            use_kp_palette = 1;
        }
        else if (strcmp(argv[j], "-deletedwal") == 0) {
            deletedMode |= 1;  // bit 0 -> delete .wal
            printf("Deleted mode: WALs corresponding to TGAs will be removed\n");
        }
        else if (strcmp(argv[j], "-deletedwal_json") == 0) {
            deletedMode |= 2;  // bit 1 -> delete .wal_json
            printf("Deleted mode: .wal_json sidecars corresponding to TGAs will be removed\n");
        }
        else if (strcmp(argv[j], "-scantga") == 0) {
            if (scanMode != 0) {
                fprintf(stderr, "Error -scantga cannot be combined with -scanini or -scanalpha.\n");
                return 1;
            }
            scanMode = 1;
        }
        else if (strcmp(argv[j], "-scanini") == 0) {
            if (scanMode != 0) {
                fprintf(stderr, "Error -scanini cannot be combined with -scantga or -scanalpha.\n");
                return 1;
            }
            scanMode = 2;
        }
        else if (strcmp(argv[j], "-scanalpha") == 0) {
            if (scanMode != 0) {
                fprintf(stderr, "Error -scanalpha cannot be combined with -scantga or -scanini.\n");
                return 1;
            }
            scanMode = 3;
        }
        else if (strcmp(argv[j], "-fixtga") == 0) {
            fixMode = 1;
        }
        else if (strcmp(argv[j], "-wal_json") == 0) {
            walJsonPerTexture = 1;
            printf("wal_json mode: will write per-texture .wal_json files "
                   "for textures with textureinfo.ini metadata\n");
        }
        else if (strcmp(argv[j], "-skipoverwrite") == 0)
        {
            skipoverwrite = 1;
        }
        else if (
            strcmp(argv[j], "-h") == 0 ||
            strcmp(argv[j], "--help") == 0 ||
            strcmp(argv[j], "?") == 0 ||
            strcmp(argv[j], "/?") == 0 ||
            strcmp(argv[j], "help") == 0
        ) {
            printf("Usage: %s [options] [directory]\n", argv[0]);
            printf("If no directory is given, current directory is used.\n\n");
            printf("Options:\n");
            printf("  -alphathresh <N>      : set alpha threshold (default 255).\n");
            printf("                          Pixels with alpha < N map to palette index 255 (transparent).\n");
            printf("  -alphathreshsemi <N>  : detect semi-transparent 32-bit TGAs.\n");
            printf("                          If any alpha in (0..N), ignore alpha for that texture (no pink cutout).\n");
            printf("  -onlyalpha            : use alpha channel only (ignore RGB images)\n");
            printf("  -skipalpha            : ignore alpha channel (RGB images only)\n");
            printf("  -skipini              : ignore textureinfo.ini completely\n");
            printf("  -onlyini              : convert only TGAs listed in textureinfo.ini (apply INI flags/contents)\n");
            printf("  -powerof2             : skip textures that are not power-of-two (e.g. 64x64, 128x256)\n");
            printf("  -q2flags              : only apply Quake II-compatible flags\n");
            printf("  -q2palette            : Use hardcoded Quake II palette\n");
            printf("  -kppalette            : Use hardcoded Kingpin palette\n");
            printf("  -deletedwal           : delete WALs corresponding to TGAs\n");
            printf("  -deletedwal_json      : delete .wal_json files corresponding to TGAs\n");
            printf("  -wal_json             : write per-texture .wal_json files next to the TGA\n");
            printf("  -scantga              : scan TGA header and report issues (no conversion)\n");
            printf("  -scanini              : scan textureinfo.ini for entries with no matching .tga (no conversion)\n");
            printf("  -scanalpha            : scan TGAs that have alpha and report whether alpha is semi-transparent\n");
            printf("  -fixtga               : attempt to repair TGA and overwrite file.\n");
            printf("  -skipoverwrite        : skip if .wal exists (default: overwrite existing .wal files)\n");
            printf("  -h, --help            : show this help message\n");

            printf("\nBy default, the tool auto-detects the textures folder:\n");
            printf("      Kingpin   main/textures\n");
            printf("      Quake II  baseq2/textures\n");
            printf("If run inside kingpin/main or quake2/baseq2, it will look for the local 'textures' subfolder.\n");
            printf("If no textures folder is found, the tool exits with an error.\n");

            printf("\nCreated by FREDZ on %s\n", date);
            return 0;
        }
        else if (argv[j][0] == '-')
        {
            fprintf(stderr, "Unknown option: %s\n", argv[j]);
            fprintf(stderr, "See %s -h or --help for usage information.\n", argv[0]);
            return 1;
        }
    }

    // Fixed root handling - path first, then options
    for (int i = 1; i < argc; i++) {
        // Skip the value that belongs to -alphathresh
        if (strcmp(argv[i], "-alphathresh") == 0) {
            if (i + 1 < argc) {
                i++;    // skip its numeric parameter
            }
            continue;
        }

        if (argv[i][0] != '-') {
            strncpy(rootpath, argv[i], sizeof(rootpath) - 1);
            rootpath[sizeof(rootpath) - 1] = 0;

            char testpath[2048];
            snprintf(testpath, sizeof(testpath), "%s/textures", rootpath);
            if (access(testpath, 0) != 0) {
                fprintf(stderr, "Error: could not find textures folder at %s\n", testpath);
                return 1;
            }

            pathArgIndex = i;
            pathProvided = true;
            break;
        }
    }


    // Auto-detect if no path provided
    if (!pathProvided) {
        if (access("main/textures", 0) == 0) strcpy(rootpath, "main");
        else if (access("baseq2/textures", 0) == 0) strcpy(rootpath, "baseq2");
        else if (access("textures", 0) == 0) strcpy(rootpath, ".");
        else {
            fprintf(stderr,
                "Error: could not find textures folder (main/textures or baseq2/textures).\n"
                "Run this tool from your Kingpin/Quake2 folder,\n"
                "or pass a path like \"kingpin\\main\" or \"quake2\\baseq2\".\n");
            return 1;
        }
    }

    int conversionOption = useAlphaThreshold || useSemiAlphaDetect || onlyAlpha || skipAlpha || skipIni || onlyIni ||
            requirePowerOf2 || q2FlagsOnly ||
            walJsonPerTexture || skipoverwrite;
    int paletteOptions = use_q2_palette || use_kp_palette;

	//conflict cmd error check.

    if (use_kp_palette && use_q2_palette)
    {
		fprintf(stderr,
          "Error: -kppalette and -q2palette cannot be used together.\n"
            "Please specify only one palette option.\n");
		return 1;
	}

	if (skipIni && walJsonPerTexture) {
        fprintf(stderr,
            "Error: -wal_json and -skipini cannot be used together. "
            "wal_json needs textureinfo.ini.\n");
        return 1;
    }

    if (skipIni && onlyIni)
    {
		fprintf(stderr,
			"Error: -skipini and -onlyini cannot be used together.\n"
			"Choose either to skip ini or to use ini-only mode.\n");
		return 1;
	}

	if (skipAlpha && onlyAlpha)
    {
		fprintf(stderr,
			"Error: -skipalpha and -onlyalpha cannot be used together.\n"
			"Choose either to ignore alpha (RGB only) or to use alpha only.\n");
		return 1;
	}

	if (skipAlpha && (useAlphaThreshold || useSemiAlphaDetect))
    {
		fprintf(stderr,
			"Error: -skipalpha and -alphathresh / -alphathreshsemi cannot be used together.\n");
		return 1;
	}

	// --- Standalone: Deletion ---
    if (deletedMode)
    {
        if (conversionOption || scanMode || fixMode || paletteOptions)
        {
            fprintf(stderr,
                "Error: -deletedwal / -deletedwal_json are standalone modes. "
                "All conversion options are disabled in deleted modes.\n");
            return 1;
        }
    }

    // --- Standalone: Scan ---
    if (scanMode)
    {
        if (conversionOption || deletedMode || fixMode || paletteOptions)
        {
            fprintf(stderr,
            "Error: scan modes (-scantga / -scanini /-scanalpha) are standalone. "
            "All conversion options are disabled in scan modes.\n");
            return 1;
        }
        else
        {
            if (scanMode == 1) {
                int scanned = 0, invalid = 0;
                scan_directory(rootpath, &scanned, &invalid);
                printf("%d TGAs, %d corrupt/unsupported\n", scanned, invalid);
                return 0;
            }

            if (scanMode == 2) {
                int totalIniEntries = 0;
                int missingTga = 0;
                scan_ini_directory(rootpath, &totalIniEntries, &missingTga);
                printf("%d INI entries, %d missing TGAs\n", totalIniEntries, missingTga);
                return (missingTga > 0) ? 1 : 0;
            }
            if (scanMode == 3) {
                int alphaCount = 0;
                int semiCount  = 0;
                scan_alpha_directory(rootpath, &alphaCount, &semiCount);
                printf("%d TGAs with alpha, %d with semi-transparent alpha\n",
                       alphaCount, semiCount);
                return 0;
            }
            return 0;
        }
    }

    // --- Standalone: Fix ---
	if (fixMode)
    {
		if (conversionOption || scanMode || deletedMode || fixMode || paletteOptions)
		{
			fprintf(stderr, "Error: -fixtga is a standalone mode. All conversion options are disabled in fix mode.\n");
			return 1;
		}
		else
		{
			int fixed = 0, failed = 0;
			fix_directory(rootpath, &fixed, &failed);
			printf("\nFixed %d TGAs, %d failed\n", fixed, failed);
			return 0;
		}
	}

    // --- Only load palette if needed ---
    uint8_t palette[256][3];

    if (use_q2_palette) {
        load_quake2_palette(palette);
        printf("Loaded Quake II hardcoded palette\n");
    } else if (use_kp_palette) {
        load_kingpin_palette(palette);
        printf("Loaded Kingpin hardcoded palette\n");
    }
    else if (!scanMode && !fixMode && deletedMode == 0)
    {
        // First try pak0.pak under the selected rootpath (e.g. "kingpin\\main")
        char pakPath[1024];
        snprintf(pakPath, sizeof(pakPath), "%s/pak0.pak", rootpath);
        pakPath[sizeof(pakPath) - 1] = '\0';

        // If that fails, fall back to the original logic (current dir, main/pics, baseq2/pics, main/baseq2 pak0.pak)
        if (extract_colormap_from_pak(pakPath, palette) == 0) {
            printf("Extracted colormap.pcx from %s\n", pakPath);
        } else {
            if (load_or_extract_colormap(palette) != 0) {
                return 1;
            }
        }
    }

	// Run conversion
	int converted = 0;
	int skippedExists = 0;
	int skippedInvalid = 0;

    //FREDZ make sure it's textures.
    char textures_path[1024];
    snprintf(textures_path, sizeof(textures_path), "%s/textures", rootpath);

    struct stat st;
    if (stat(textures_path, &st) == 0 && S_ISDIR(st.st_mode)) {
        // Found textures folder, process all .tga files in it (and subfolders)
        process_directory(textures_path, palette, &converted, &skippedExists, &skippedInvalid);
    } else {
        fprintf(stderr, "No 'textures' folder found in %s\n", rootpath);
    }


    printf("\nConverted %d textures, skipped %d existing WAL, skipped %d invalid/failed\n", converted, skippedExists, skippedInvalid);

    if (onlyIni && !iniFound) {
        fprintf(stderr,
            "Warning: -onlyini requested, but no textureinfo.ini was found.\n");
    }
    if (walJsonPerTexture && !iniFound) {
        fprintf(stderr,
            "Warning: -wal_json requested, but no textureinfo.ini was found; "
            "no .wal_json files were written.\n");
    }

    printf("rootpath = '%s'\n", rootpath);
    printf("Created by FREDZ on %s\n", date);
    return 0;
}
