// prefs.c

#include "syshdrs.h"

#include "TraySpy.h"
#include "qserver.h"
#include "prefs.h"
#include "resource.h"
#include "Strn.h"
#include "wsock.h"

extern HINSTANCE ghInstance;
extern HWND gMainWnd;
extern TCHAR gWndClass[32];
extern int gRefreshOnNextIncrement;

// Giant structure of preferences and state information.
//
Preferences gPrefs;

// This is the "root" key we use.
TCHAR gRegistryKey[256];

// Try to save the current window position.
RECT gSavedWRect;



// Get the Preferences (configuration) dialog ready.
//
static void InitPrefsDlg(HWND hDlg)
{
	HWND ctrl;
	char buf[260];
	int i;

	ctrl = GetDlgItem(hDlg, IDC_SERVER);
	if (ctrl != NULL) {
		SetWindowText(ctrl, gPrefs.serverAddrStr);
	}

	ctrl = GetDlgItem(hDlg, IDC_RCON_PASSWORD);
	if (ctrl != NULL) {
		SetWindowText(ctrl, gPrefs.rconPassword);
	}

	ctrl = GetDlgItem(hDlg, IDC_REFRESH);
	if (ctrl != NULL) {
		sprintf(buf, "%d", gPrefs.refreshIntervalSecs);
		SetWindowText(ctrl, buf);
	}

	for (i=0; i<MAX_VISIBLE_BUDDIES; i++) {
		ctrl = GetDlgItem(hDlg, IDC_BUDDY1 + i);
		if (ctrl != NULL) {
			SetWindowText(ctrl, gPrefs.buddies[i].name);
		}
		ctrl = GetDlgItem(hDlg, IDC_SOUND1 + i);
		if (ctrl != NULL) {
			SetWindowText(ctrl, gPrefs.buddies[i].sound);
		}
		ctrl = GetDlgItem(hDlg, IDC_PROGRAM1 + i);
		if (ctrl != NULL) {
			SetWindowText(ctrl, gPrefs.buddies[i].program);
		}
	}

}	// InitPrefsDlg




// The user clicked "OK", so we should save the changes
// the user made in the dialog.
//
static void PrefsDlgOK(HWND hDlg)
{
	HWND ctrl;
	char buf[260];
	char oldServerAddrStr[64];
	int b, i;

	// We make a copy of this so we can tell if the value
	// actually changed.
	//
	STRNCPY(oldServerAddrStr, gPrefs.serverAddrStr);
	ctrl = GetDlgItem(hDlg, IDC_SERVER);
	if (ctrl != NULL) {
		GetWindowText(ctrl, gPrefs.serverAddrStr, sizeof(gPrefs.serverAddrStr) - 1);
	}

	ctrl = GetDlgItem(hDlg, IDC_RCON_PASSWORD);
	if (ctrl != NULL) {
		ZeroMemory(gPrefs.rconPassword, sizeof(gPrefs.rconPassword));
		GetWindowText(ctrl, gPrefs.rconPassword, sizeof(gPrefs.rconPassword) - 1);
	}

	ctrl = GetDlgItem(hDlg, IDC_REFRESH);
	if (ctrl != NULL) {
		GetWindowText(ctrl, buf, sizeof(buf) - 1);
		buf[sizeof(buf) - 1] = '\0';
		sscanf(buf, "%d", &gPrefs.refreshIntervalSecs);
		if (gPrefs.refreshIntervalSecs < MINIMUM_REFRESH)
			gPrefs.refreshIntervalSecs = MINIMUM_REFRESH;
	}

	for (i=0; i<MAX_VISIBLE_BUDDIES; i++) {
		ctrl = GetDlgItem(hDlg, IDC_BUDDY1 + i);
		if (ctrl != NULL) {
			GetWindowText(ctrl, gPrefs.buddies[i].name, sizeof(gPrefs.buddies[i].name) - 1);
			if (gPrefs.buddies[i].name[0] == '\0')
				sprintf(gPrefs.buddies[i].name, "Buddy%d", i + 1);
		}
		ctrl = GetDlgItem(hDlg, IDC_SOUND1 + i);
		if (ctrl != NULL) {
			GetWindowText(ctrl, gPrefs.buddies[i].sound, sizeof(gPrefs.buddies[i].sound) - 1);
		}
		ctrl = GetDlgItem(hDlg, IDC_PROGRAM1 + i);
		if (ctrl != NULL) {
			GetWindowText(ctrl, gPrefs.buddies[i].program, sizeof(gPrefs.buddies[i].program) - 1);
		}
	}

	// Enable/Disable items
	SetMenuStates();

	if (strcmp(oldServerAddrStr, gPrefs.serverAddrStr) != 0) {
		// New server was specified.
		CloseQuakeSocket();
		for (b=0; b<MAX_BUDDIES; b++) {
			gPrefs.buddies[b].playing = 0;
			gPrefs.buddies[b].quitPlayingAt = 0;
		}

		if (SetQuakeServer(gPrefs.serverAddrStr) < 0) {
			ErrBox("Could not get resolve the IP address and port number for \"%s\".", gPrefs.serverAddrStr);
		}
		gRefreshOnNextIncrement = 1;
	} else {
		// Using the same server.
		//
		// But the buddy list, etc., may have changed, so do a refresh
		// anyway to reflect those changes in the status window.
		//
		gRefreshOnNextIncrement = 1;
	}
}	// PrefsDlgOK




static void BrowseSound(HWND hDlg, int i)
{
	OPENFILENAME o;
	HWND ctrl;
	char fn[260];
	char basedir[260], *basedircp, *cp;

	ZeroMemory(fn, sizeof(fn));
	basedircp = NULL;
	ctrl = GetDlgItem(hDlg, IDC_SOUND1 + i);
	if (ctrl != NULL) {
		GetWindowText(ctrl, basedir, sizeof(basedir) - 1);
		basedir[sizeof(basedir) - 1] = '\0';
		cp = strrchr(basedir, '\\');
		if (cp != NULL) {
			*cp++ = '\0';
			basedircp = basedir;
			STRNCPY(fn, cp);
		}
	}

	ZeroMemory(&o, sizeof(o));
	o.lStructSize = sizeof(o);
	o.Flags				= OFN_FILEMUSTEXIST  | OFN_PATHMUSTEXIST;
	o.hwndOwner         = hDlg;
	o.lpstrFile         = fn;
	o.lpstrFilter       = _T("Sound Files (*.wav)\0*.wav\0All Files (*.*)\0*.*\0\0";);
	o.lpstrDefExt		= _T("wav");
	o.lpstrInitialDir	= basedircp;
	o.nMaxFile          = sizeof(fn);
	if (GetOpenFileName(&o)) {
		ctrl = GetDlgItem(hDlg, IDC_SOUND1 + i);
		if (ctrl != NULL) {
			SetWindowText(ctrl, fn);
			PlaySoundFile(fn);
		}
	}
}	// BrowseSound




static void BrowseProgram(HWND hDlg, int i)
{
	OPENFILENAME o;
	HWND ctrl;
	char fn[260];

	ZeroMemory(fn, sizeof(fn));
	ZeroMemory(&o, sizeof(o));
	o.lStructSize = sizeof(o);
	o.Flags				= OFN_FILEMUSTEXIST  | OFN_PATHMUSTEXIST;
	o.hwndOwner         = hDlg;
	o.lpstrFile         = fn;
	o.lpstrFilter       = _T("Programs (*.exe)\0*.exe\0All Files (*.*)\0*.*\0\0";);
	o.lpstrDefExt		= _T("exe");
	o.nMaxFile          = sizeof(fn);
	if (GetOpenFileName(&o)) {
		ctrl = GetDlgItem(hDlg, IDC_PROGRAM1 + i);
		if (ctrl != NULL) {
			SetWindowText(ctrl, fn);
		}
	}
}	// BrowseProgram




#pragma warning(disable : 4100)		// warning C4100: unreferenced formal parameter
static BOOL CALLBACK PrefsDlgProc(HWND hDlg, UINT iMsg, WPARAM w, LPARAM l_unused)
{
	WORD id;

	switch (iMsg) {
	case WM_INITDIALOG:
		InitPrefsDlg(hDlg);
		return TRUE;

	case WM_COMMAND:
		id = LOWORD(w);
		switch (id) {
			case IDOK:
				PrefsDlgOK(hDlg);
				EndDialog(hDlg, 0);
				return TRUE;

			case IDCANCEL:
				EndDialog(hDlg, 0);
				return TRUE;

			case IDC_BROWSE_SOUND1:
				BrowseSound(hDlg, 0);
				return TRUE;

			case IDC_BROWSE_PROGRAM1:
				BrowseProgram(hDlg, 0);
				return TRUE;

			case IDC_BROWSE_SOUND2:
				BrowseSound(hDlg, 1);
				return TRUE;

			case IDC_BROWSE_PROGRAM2:
				BrowseProgram(hDlg, 1);
				return TRUE;

			case IDC_BROWSE_SOUND3:
				BrowseSound(hDlg, 2);
				return TRUE;

			case IDC_BROWSE_PROGRAM3:
				BrowseProgram(hDlg, 2);
				return TRUE;

			case IDC_BROWSE_SOUND4:
				BrowseSound(hDlg, 3);
				return TRUE;

			case IDC_BROWSE_PROGRAM4:
				BrowseProgram(hDlg, 3);
				return TRUE;
		}
		break;
	}
	return (FALSE);
}	// PrefsDlgProc
#pragma warning(default : 4100)		// warning C4100: unreferenced formal parameter




// Pop-up the configuration dialog for the user.
//
void ShowPreferencesDlg(void)
{
	DialogBox(ghInstance, MAKEINTRESOURCE(IDD_PREFERENCES), gMainWnd, PrefsDlgProc);
}	// ShowPreferencesDlg



 

void FindQuakePath(void)
{
	char d;
	int i, rc;
	char path[260];
	FILE *fp;

	if (gPrefs.gamePath[0] != '\0')
		return;

	for (d='C'; d<='Z'; d++) {
		path[0] = d;
		path[1] = ':';
		path[2] = '\0';

		rc = GetDriveType(path);
		if (rc != DRIVE_FIXED) {
			continue;
		}

		for (i=0; i<3; i++) {
			switch (i) {
			case 0:
				STRNCAT(path, "\\quake2\\quake2.exe");
				break;
			case 1:
				STRNCAT(path, "\\games\\quake2\\quake2.exe");
				break;
			case 2:
				STRNCAT(path, "\\Program Files\\quake2\\quake2.exe");
				break;
			}

			fp = fopen(path, "r");
			if (fp == NULL) {
				path[2] = '\0';
				continue;
			}
			fclose(fp);
			STRNCPY(gPrefs.gamePath, path);
			return;
		}
	}

	STRNCPY(gPrefs.gamePath, "unknown");
}	// FindQuakePath




// Set sane defaults if the user has not run this before,
// or we can't load their preferences.
//
void InitPreferences(void)
{
	int i;

	_tcscpy(gRegistryKey, _T("Software\\TraySpy\\"));
	if (gWndClass[0] != '\0')
		_tcscat(gRegistryKey, gWndClass);
	else
		_tcscat(gRegistryKey, _T("default"));

	ZeroMemory(&gPrefs, sizeof(gPrefs));
	ZeroMemory(&gSavedWRect, sizeof(gSavedWRect));

#ifdef DEFAULT_SERVER
	// For testing only
	strcpy(gPrefs.serverAddrStr, DEFAULT_SERVER);
#endif
	gPrefs.refreshIntervalSecs = DEFAULT_REFRESH;

	for (i=0; i<MAX_BUDDIES; i++) {
		sprintf(gPrefs.buddies[i].name, "Buddy%d", i + 1);
//		sprintf(gPrefs.buddies[i].sound, "snd%d.wav", i + 1);
//		sprintf(gPrefs.buddies[i].program, "prg%d.exe", i + 1);
	}
}	// InitPreferences




// Overwrite the defaults with saved information from
// the registry.
//
void LoadPreferences(void)
{
	HKEY hkey = 0;
	TCHAR str[512], var[64], q;
	DWORD len, type, dw, j;
	int i;

	if (RegOpenKey(HKEY_CURRENT_USER, (LPCTSTR) gRegistryKey, &hkey) == ERROR_SUCCESS) {
		ZeroMemory(str, sizeof(str));

		len = (sizeof(str) / sizeof(TCHAR)) - 1;
		if (RegQueryValueEx(hkey, "ServerName", (DWORD *) 0, &type, (BYTE *) str, &len) == ERROR_SUCCESS) {
			str[len] = '\0';
			STRNCPY(gPrefs.serverAddrStr, str);
		}

		len = (sizeof(str) / sizeof(TCHAR)) - 1;
		if (RegQueryValueEx(hkey, "GamePath", (DWORD *) 0, &type, (BYTE *) str, &len) == ERROR_SUCCESS) {
			str[len] = '\0';
			STRNCPY(gPrefs.gamePath, str);
		}

		ZeroMemory(gPrefs.rconPassword, sizeof(gPrefs.rconPassword));
		len = (sizeof(str) / sizeof(TCHAR)) - 1;
		if (RegQueryValueEx(hkey, "RconP", (DWORD *) 0, &type, (BYTE *) str, &len) == ERROR_SUCCESS) {
			if (len > 1) {
				len--;		// Do not use nul-terminator
				q = (TCHAR) 'q';
				for (j=0; j<len; j++)
					str[j] = (TCHAR) (str[j] ^ q);						// Poor Man's decryption
				str[len] = '\0';
				STRNCPY(gPrefs.rconPassword, str);
			}
		}

		len = (sizeof(str) / sizeof(TCHAR)) - 1;
		if (RegQueryValueEx(hkey, "Refresh", (DWORD *) 0, &type, (BYTE *) &dw, &len) == ERROR_SUCCESS) {
			gPrefs.refreshIntervalSecs = (int) dw;
			if (gPrefs.refreshIntervalSecs < MINIMUM_REFRESH)
				gPrefs.refreshIntervalSecs = MINIMUM_REFRESH;
		}
		
		// Note:  Although the configuration screen only
		// displays MAX_VISIBLE_BUDDIES buddies, a poweruser
		// could actually manually enter up to MAX_BUDDIES.
		//
		for (i=0; i<MAX_BUDDIES; i++) {
			_stprintf(var, _T("Buddy%dName"), i + 1);
			len = (sizeof(str) / sizeof(TCHAR)) - 1;
			if (RegQueryValueEx(hkey, var, (DWORD *) 0, &type, (BYTE *) str, &len) == ERROR_SUCCESS) {
				str[len] = '\0';
				STRNCPY(gPrefs.buddies[i].name, str);
			}

			_stprintf(var, _T("Buddy%dSound"), i + 1);
			len = (sizeof(str) / sizeof(TCHAR)) - 1;
			if (RegQueryValueEx(hkey, var, (DWORD *) 0, &type, (BYTE *) str, &len) == ERROR_SUCCESS) {
				str[len] = '\0';
				STRNCPY(gPrefs.buddies[i].sound, str);
			}

			_stprintf(var, _T("Buddy%dProgram"), i + 1);
			len = (sizeof(str) / sizeof(TCHAR)) - 1;
			if (RegQueryValueEx(hkey, var, (DWORD *) 0, &type, (BYTE *) str, &len) == ERROR_SUCCESS) {
				str[len] = '\0';
				STRNCPY(gPrefs.buddies[i].program, str);
			}
		}
		
		RegCloseKey(hkey);
	}


	if (gPrefs.serverAddrStr[0] != '\0') {
		if (SetQuakeServer(gPrefs.serverAddrStr) < 0) {
			// SetQuakeServer zeroed it out for us.
			ErrBox("Invalid server address from saved settings.  Please configure a valid address.");
		}
	}

	FindQuakePath();
}	// LoadPreferences




// Try to save the changes the user may have made to the
// configuration into the registry, so we can reload them
// again next time.
//
void SavePreferences(void)
{
	HKEY hkey = 0;
	TCHAR var[128], q;
	DWORD dw, j;
	int i;

	if (
		(RegOpenKey(HKEY_CURRENT_USER, (LPCTSTR) gRegistryKey, &hkey) == ERROR_SUCCESS) ||
		(RegCreateKey(HKEY_CURRENT_USER, (LPCTSTR) gRegistryKey, &hkey) == ERROR_SUCCESS)
		)
	{
		(void) RegSetValueEx(
			hkey,
			"ServerName",
			(DWORD) 0,
			(DWORD) REG_SZ,
			(const BYTE *) gPrefs.serverAddrStr,
			(DWORD) strlen(gPrefs.serverAddrStr) + 1
			);
		
		(void) RegSetValueEx(
			hkey,
			"GamePath",
			(DWORD) 0,
			(DWORD) REG_SZ,
			(const BYTE *) gPrefs.gamePath,
			(DWORD) strlen(gPrefs.gamePath) + 1
			);
		
		STRNCPY(var, gPrefs.rconPassword);
		dw = (DWORD) strlen(var);
		if (dw > 0) {
			q = (TCHAR) 'q';
			for (j=0; j<dw; j++)
				var[j] = (TCHAR) (var[j] ^ q);						// Poor Man's encryption
		}
		
		(void) RegSetValueEx(
			hkey,
			"RconP",
			(DWORD) 0,
			(DWORD) REG_BINARY,
			(const BYTE *) var,
			dw + 1
			);

		dw = (DWORD) gPrefs.refreshIntervalSecs;
		(void) RegSetValueEx(
			hkey,
			"Refresh",
			(DWORD) 0,
			(DWORD) REG_DWORD,
			(const BYTE *) &dw,
			(DWORD) sizeof(dw)
			);

		
		for (i=0; i<MAX_BUDDIES; i++) {
			_stprintf(var, _T("Buddy%dName"), i + 1);			
			(void) RegSetValueEx(
				hkey,
				var,
				(DWORD) 0,
				(DWORD) REG_SZ,
				(const BYTE *) gPrefs.buddies[i].name,
				(DWORD) strlen(gPrefs.buddies[i].name) + 1
				);
			
			_stprintf(var, _T("Buddy%dSound"), i + 1);			
			(void) RegSetValueEx(
				hkey,
				var,
				(DWORD) 0,
				(DWORD) REG_SZ,
				(const BYTE *) gPrefs.buddies[i].sound,
				(DWORD) strlen(gPrefs.buddies[i].sound) + 1
				);
			
			_stprintf(var, _T("Buddy%dProgram"), i + 1);			
			(void) RegSetValueEx(
				hkey,
				var,
				(DWORD) 0,
				(DWORD) REG_SZ,
				(const BYTE *) gPrefs.buddies[i].program,
				(DWORD) strlen(gPrefs.buddies[i].program) + 1
				);
		}

		RegCloseKey(hkey);
	}
}	// SavePreferences




void RestoreWindowPosition(int *x, int *y, int *dx, int *dy)
{
	HKEY hkey = 0;
	char str[64];
	DWORD len, type;
	int x2, y2;
	HWND desktopWnd;
	HDC desktopDC;
	int hRes, vRes;

	len = sizeof(str) - 1;
	type = (DWORD) REG_SZ;
	if (RegOpenKey(HKEY_CURRENT_USER, (LPCTSTR) gRegistryKey, &hkey) == ERROR_SUCCESS) {
		if (RegQueryValueEx(hkey, "WindowPosition", (DWORD) 0, &type, (BYTE *) str, &len) == ERROR_SUCCESS) {
			str[len] = '\0';
			sscanf(str, "%d %d %d %d", &x2, &y2, dx, dy);
			
			// Read the window position, but don't use it
			// if it wouldn't be visible on screen.
			// This could happen if the user changed
			// to a smaller resolution.
			//
			desktopWnd = GetDesktopWindow();
			if (desktopWnd != (HWND) 0) {
				desktopDC = GetDC(desktopWnd);
				if (desktopDC != (HDC) 0) {
					hRes = GetDeviceCaps(desktopDC, HORZRES);
					vRes = GetDeviceCaps(desktopDC, VERTRES);
					
					if ((x2 <= hRes) && (y2 <= vRes) && (x2 >= 0) && (y2 >= 0)) {
						*x = x2;
						*y = y2;
					}
				}
			}
		}
		RegCloseKey(hkey);
	}
}	// RestoreWindowPosition




void SaveWindowRect(HWND hwnd)
{
	RECT rect;

	if ((GetWindowRect(hwnd, &rect)) && (rect.left > 0) && (rect.top > 0)) {
		gSavedWRect = rect;
	}
}	// SaveWindowRect




void SaveWindowPosition(HWND hwnd)
{
	HKEY hkey = 0;
	char str[64];

	SaveWindowRect(hwnd);
	if ((gSavedWRect.left > 0) && (gSavedWRect.top > 0)) {
		if (
			(RegOpenKey(HKEY_CURRENT_USER, (LPCTSTR) gRegistryKey, &hkey) == ERROR_SUCCESS) ||
			(RegCreateKey(HKEY_CURRENT_USER, (LPCTSTR) gRegistryKey, &hkey) == ERROR_SUCCESS)
			)
		{
			sprintf(str, "%d %d %d %d",
				gSavedWRect.left,
				gSavedWRect.top,
				(gSavedWRect.right - gSavedWRect.left),
				(gSavedWRect.bottom - gSavedWRect.top)
				);
			(void) RegSetValueEx(
				hkey,
				"WindowPosition",
				(DWORD) 0,
				(DWORD) REG_SZ,
				(BYTE *) str,
				(DWORD) strlen(str)
				);
			
			RegCloseKey(hkey);
		}		
	}
}	// SaveWindowPosition




// This should probably be in a separate file...
//
static const char *gPrompt;
static char *gAnswer;
static int gAnswerSize;

static void InitSimpleInputDlg(HWND hDlg)
{
	HWND ctrl;

	ctrl = GetDlgItem(hDlg, IDC_PROMPT);
	if (ctrl != NULL) {
		SetWindowText(ctrl, gPrompt);
	}

	ctrl = GetDlgItem(hDlg, IDC_SIMPLE_EDIT);
	if (ctrl != NULL) {
		SetWindowText(ctrl, gAnswer);
	}
}	// InitSimpleInputDlg




static void SimpleInputDlgOK(HWND hDlg)
{
	HWND ctrl;

	ctrl = GetDlgItem(hDlg, IDC_SIMPLE_EDIT);
	if (ctrl != NULL) {
		ZeroMemory(gAnswer, gAnswerSize);
		GetWindowText(ctrl, gAnswer, gAnswerSize - 1);
	}
}	// SimpleInputDlgOK




#pragma warning(disable : 4100)		// warning C4100: unreferenced formal parameter
static BOOL CALLBACK SimpleInputDlgProc(HWND hDlg, UINT iMsg, WPARAM w, LPARAM l_unused)
{
	WORD id;

	switch (iMsg) {
	case WM_INITDIALOG:
		InitSimpleInputDlg(hDlg);
		return TRUE;

	case WM_COMMAND:
		id = LOWORD(w);
		switch (id) {
			case IDOK:
				SimpleInputDlgOK(hDlg);
				EndDialog(hDlg, IDOK);
				return TRUE;
			case IDCANCEL:
				EndDialog(hDlg, IDCANCEL);
				return TRUE;
		}
		break;
	}
	return (FALSE);
}	// SimpleInputDlgProc
#pragma warning(default : 4100)		// warning C4100: unreferenced formal parameter




BOOL InputDlg(const char *const prompt, char *answer, int asize)
{
	UINT rc;

	gPrompt = prompt;
	gAnswer = answer;
	gAnswerSize = asize;
	rc = DialogBox(ghInstance, MAKEINTRESOURCE(IDD_SIMPLE_INPUT), gMainWnd, SimpleInputDlgProc);
	return (rc == IDOK);
}	// InputDlg
